/*
 * File Name: filemodel.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

#include <ctype.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>

#include "ctb_log.h"
#include "filemodel.h"
#include "filetypes.h"
#include "i18n.h"
#include "ipc.h"
#include "menu.h"
#include "shortcut.h"
#include "db.h"
#include "ctb_actions.h"
#include "views.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

static const char* order_names[N_CTB_SORT_ORDER] =
        {
                // TRANSLATORS: this is a document sorting option
                [CTB_SORT_BY_NAME]       = N_("- by title"),
                // TRANSLATORS: this is a document sorting option
                [CTB_SORT_BY_TYPE]       = N_("- by type"),
                // TRANSLATORS: this is a document sorting option
                [CTB_SORT_BY_SIZE]       = N_("- by size"),
                // TRANSLATORS: this is a document sorting option
                [CTB_SORT_BY_DATE_ADDED] = N_("- by date added"),
                // TRANSLATORS: this is a document sorting option
                [CTB_SORT_BY_DATE_READ]  = N_("- recently opened"),
                // TRANSLATORS: this is a document sorting option
                [CTB_SORT_BY_AUTHOR]     = N_("- by author"),
        };

typedef struct
        {
            gchar   *parent_dir;
            gchar   *last_item;
        } stack_entry_t;

//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

static const char *THUMBNAIL_IMAGE_TYPE = "png";
static int NUM_RECENT_ITEMS = (19 + 19); // two pages (TODO: make configurable)


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkListStore          *g_filestore    = NULL;            // details of files
static gint                  g_items_per_page     = 1;               // max. number of items to be displayed
static gint                  g_item_offset   = 0;               // offset of first item to be displayed
static gint                  g_total_items   = 0;               // total number of items
static gboolean              g_has_next_page = FALSE;           // do we have a next page to be displayed?
static int                   g_curpage       = 0;               // shown in pagecounter
static int                   g_numpages      = 0;               // shown in pagecounter
static gboolean              g_boundary_check = TRUE;           // on TRUE greys out nav. arrows, FALSE for reflowable docs
static ctb_viewmodes_t       g_viewmode      = BROWSE_MODE;
static int                   g_desktop_offset = 0;
static int                   g_desktop_items  = 0;

static ctb_sort_order_t      g_sort_order    = -1;
static gboolean              g_is_sort_asc   = TRUE;            // item sorting order ascending or descending
static filemodel_thumbsize_t g_thumbnail_size= MODTHUMB_MEDIUM; // thumbail size to retrieve from database

static gchar                 *g_current_dir  = NULL;            // current directory
static metadata_table        *g_values       = NULL;            // results from metadb
static GSList                *toggled_list   = NULL;            // list of toggled items
static gchar                 *g_delete_text  = NULL;            // text shown on first item in delete mode
static time_t                g_last_modified = 0;               // when global.db was last modified

static GList                 *g_dir_stack    = NULL;            // stack of directories
// value 0 = not active, 1 has dir with initial, 2 has dir with initial, 3 has both
static char g_alpha_list[FILEMODEL_NUM_ALPHA]; // [0='#', 1='a', 26='z']
static GSList                *desktop_names  = NULL;            // names of visible desktop items

static gchar                 *g_search_filter = NULL;
static const gchar           *g_subtitle     = NULL;

//MH: bug in original code, (sorting on last read date is not working)
static filelist_entry_t      *g_last_read      = NULL;
#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
static metadata_table        *g_last_read_results = NULL;
#endif

// init: when ctb is started its window is on top
static gboolean g_window_is_on_top = TRUE;                      // needed for page bar updates
static ViewMode              g_viewmode2 = -1;

//============================================================================
// Local Function Definitions
//============================================================================

static gchar*         format_size                       ( const gint64 bytes );
static void           load_items_in_model               ( void );
static void           clear_toggled_list                ( void );
static gboolean       is_delete_allowed(const gchar* filename, const gchar* dirpath);
static void           scroll_to_filename(const gchar *filename);
static void           update_alpha_list();
static void           clear_dir_stack();
static GdkPixbuf*     get_thumbnail(const metadata_cell *cell);
static GdkPixbuf*     create_delete_overlay(const GdkPixbuf* source, gboolean toggled);
static int            num_specials_per_page();
static const char*    viewmode_get_title(int viewmode);
static ctb_sort_order_t viewmode_get_order(int viewmode);
static void           viewmode_set_order(int viewmode, ctb_sort_order_t sort_order);
static void           viewmode_set_mode(int viewmode, int mode);
static gboolean       viewmode_get_fixed_order(int viewmode);


//============================================================================
// Functions Implementation
//============================================================================

void filemodel_init()
{
    filetypes_init(TRUE);
    views_init();
    get_filemodel();
    filemodel_set_viewmode2(DESKTOP_VIEW);
}


erMetadb filemodel_get_database()
{
    return get_database();
}


static void clear_last_read()
{
#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
    filelist_entry_free(g_last_read);
    g_last_read = NULL;
    metadata_table_free(g_last_read_results);
    g_last_read_results = NULL;
#endif
}


void filemodel_quit()
{
    close_database();
    if (g_values) metadata_table_free(g_values);
    g_values = NULL;
    clear_last_read();
    clear_toggled_list();
}


GtkTreeModel* get_filemodel()
{
    if (g_filestore == NULL)
    {
        g_filestore = gtk_list_store_new(N_FILEMODEL_COLUMNS, FILEMODEL_COLUMN_TYPES);
    }
    return GTK_TREE_MODEL(g_filestore);
}


static const char* get_row_filename(const metadata_cell *row)
{
    const metadata_cell *cell = row; // first item already
    if (cell->type != METADATA_TEXT) {
        ERRORPRINTF("illegal cell type [%d] for filename", cell->type);
        return NULL;
    }
    return cell->value.v_text->str;
}


static const char* get_row_dirpath(const metadata_cell *row)
{
    const metadata_cell *cell = row + 1;        // see create_query()
    if (cell->type != METADATA_TEXT) {
        ERRORPRINTF("illegal cell type [%d] for directory_path", cell->type);
        return NULL;
    }
    return cell->value.v_text->str;
}


static gboolean get_row_is_directory(const metadata_cell *row)
{
    const metadata_cell *cell = row + 2;        // see create_query()
    if (cell->type != METADATA_INT64) {
        ERRORPRINTF("illegal cell type [%d] for is_directory", cell->type);
        return FALSE;
    }
    gboolean is_directory = FALSE;
    gint64 i64 = cell->value.v_int64;
    if (i64 == 1) {
        is_directory = TRUE;
    } else if (i64 != 0) {
        ERRORPRINTF("illegal value [%lld] for is_directory", i64);
    }
    return is_directory;
}


static const char* get_row_filetype(const metadata_cell *row)
{
    const metadata_cell *cell = row + 3;        // see create_query()
    if (cell->type != METADATA_TEXT) {
        ERRORPRINTF("illegal cell type [%d] for file_type", cell->type);
        return NULL;
    }
    return cell->value.v_text->str;
}


static gint64 get_row_filesize(const metadata_cell *row)
{
    const metadata_cell *cell = row + 4;        // see create_query()
    if (cell->type != METADATA_INT64)
    {
        if (cell->type != METADATA_NULL) {
            ERRORPRINTF("illegal cell type [%d] for filesize", cell->type);
        }
        return 0;
    }
    gint64 i64 = cell->value.v_int64;
    g_assert(i64 >= 0);
    return i64;
}


static gint64 get_row_filedate(const metadata_cell *row)
{
    const metadata_cell *cell = row + 5;        // see create_query()
    if (cell->type != METADATA_INT64)
    {
        if (cell->type != METADATA_NULL) {
            ERRORPRINTF("illegal cell type [%d] for filedate", cell->type);
        }
        return 0;
    }
    gint64 i64 = cell->value.v_int64;
    g_assert(i64 >= 0);
    return i64;
}


static const char* get_row_title(const metadata_cell *row)
{
    const metadata_cell *cell = row + 6;        // see create_query()
    if (cell->type != METADATA_TEXT) {
        if (cell->type != METADATA_NULL) {
            ERRORPRINTF("illegal cell type [%d] for title", cell->type);
        }
        return NULL;
    }
    return cell->value.v_text->str;
}


static const char* get_row_author(const metadata_cell *row)
{
    const metadata_cell *cell = row + 7;        // see create_query()
    if (cell->type != METADATA_TEXT) {
        if (cell->type != METADATA_NULL) {
            ERRORPRINTF("illegal cell type [%d] for author", cell->type);
        }
        return NULL;
    }
    return cell->value.v_text->str;
}


static const guchar* get_row_thumbnail(const metadata_cell *row, int* size)
{
    const metadata_cell *cell = row + 8;        // see create_query()
    if (cell->type != METADATA_BLOB) {
        if (cell->type != METADATA_NULL) {
            ERRORPRINTF("illegal cell type [%d] for thumbnail", cell->type);
        }
        *size = 0;
        return NULL;
    }
    *size = cell->value.v_blob.len;
    return (guchar*)cell->value.v_blob.data;
}


static int get_thumbnail_column()
{
    switch (g_thumbnail_size) {
        case MODTHUMB_MINI:
            return COL_THUMB_MINI;
        case MODTHUMB_SMALL:
            return COL_THUMB_SMALL;
        case MODTHUMB_MEDIUM:
            return COL_THUMB_MEDIUM;
        case MODTHUMB_LARGE:
            return COL_THUMB_LARGE;
        default:
            ERRORPRINTF("unknown g_thumbnail_size [%d]", g_thumbnail_size);
            return -1;
    }
}


static int create_query()
{
    int thumbcolumn = get_thumbnail_column();
    if (thumbcolumn == -1) return ER_FAIL;

    // NOTE if order is changed, change get_row_xxx() methods too!
    int rc= db_query_create(COL_FILENAME,
                        COL_DIRECTORY_PATH,
                        COL_IS_DIRECTORY,
                        COL_FILETYPE,
                        COL_FILESIZE,
                        COL_FILETIME_MODIFIED,
                        COL_TITLE,
                        COL_AUTHOR,
                        thumbcolumn,
                        -1);
    if (rc != ER_OK)  ERRORPRINTF("error creating query");
    return rc;
}


static time_t get_last_modified()
{
    const gchar* mountpoint = ipc_get_media();
    if (mountpoint == NULL) return 0;

    char database_file[256];
    sprintf(database_file, "%s/%s", mountpoint, ERMETADB_GLOBAL_DATABASE_FILE);

    struct stat statbuf;
    stat(database_file, &statbuf);
    time_t date_modified = statbuf.st_mtime;
    return date_modified;
}


static gboolean is_database_changed()
{
    return g_last_modified != get_last_modified();
}


// NOTE g_values can be zero afterwards if no items present
static void load_dir_from_metadb()
{
    LOGPRINTF("%s() thumbsize=%d", __func__, g_thumbnail_size);

#if (TIMING_ON)
    struct timeval t1, t2;
    gettimeofday(&t1, NULL);
#endif

    if (g_current_dir == NULL) goto out;

    // cleanup old results
    if (g_values) metadata_table_free(g_values);
    g_values = NULL;
    g_total_items = 0;
    g_item_offset = 0;
    clear_toggled_list();
    bzero(g_alpha_list, sizeof(g_alpha_list));

    if (filemodel_current_is_desktop()) goto out;

    g_last_modified = get_last_modified();

    int rc = create_query();
    if (rc != ER_OK) goto out;

    switch (viewmode_get_style(g_viewmode2)) {
        case DESKTOP_VIEW:
            // already handled
            break;
        case SETTINGS_VIEW:
            // get normal view (settings)
            rc = db_query_execute(g_sort_order,
                                  g_is_sort_asc,
                                  &g_values,
                                  NULL);
            break;
        case BOOKS_VIEW:
        case NEWS_VIEW:
        case IMAGES_VIEW:
        case PERSONAL_VIEW:
        case HELP_VIEW:
        case NOTES_VIEW:
            // get tag-filtered view (Books, Images, News, Notes)
            g_assert(viewmode_get_tag_filter(g_viewmode2));
            rc = db_query_execute(g_sort_order,
                                  g_is_sort_asc,
                                  &g_values,
                                  viewmode_get_tag_filter(g_viewmode2));
            break;
        case DIR_VIEW:
            // get directory-filtered view (SD Card)
            rc = db_query_execute_path_filter(g_sort_order,
                                              g_is_sort_asc,
                                              &g_values,
                                              g_current_dir,
                                              TRUE);
            break;
        case SHORTCUT_VIEW:
            // get directory-filtered view (SD Card)
            rc = db_query_execute_path_filter(g_sort_order,
                                              g_is_sort_asc,
                                              &g_values,
                                              g_current_dir,
                                              FALSE);
            break;
        case RECENT_VIEW:
            rc = db_query_execute_recent(g_sort_order, &g_values, NUM_RECENT_ITEMS);
            break;
        case SEARCH_VIEW:
            // search results (when user searches)
            rc = db_query_execute_search_filter(g_sort_order,
                                                g_is_sort_asc,
                                                &g_values,
                                                g_search_filter);
            break;
        default:
            ERRORPRINTF("Unkown style for viewmode: %d", g_viewmode2);
            break;
        
    }

    if (rc != ER_OK)
    {
        ERRORPRINTF("cannot select metadata");
        goto out;
    }
    // note: values is NULL when no files/folders selected
    if (g_values)
    {
        g_total_items = metadata_table_n_rows(g_values);
        update_alpha_list();
    }
out:
#if (TIMING_ON)
    gettimeofday(&t2, NULL);
    float cost = (float) (((t2.tv_sec - t1.tv_sec) * 1000 * 1000) + (t2.tv_usec - t1.tv_usec));
    printf("%s() items=%d  duration=%4.1lf ms\n", __func__, g_total_items, cost/1000);
#endif
    return;
}


static void set_sortorder(ctb_sort_order_t sort_order)
{
    if (g_sort_order == sort_order) return;

    static const gboolean is_sort_ascending[N_CTB_SORT_ORDER] =
                {
                    [CTB_SORT_BY_NAME]       = TRUE,
                    [CTB_SORT_BY_TYPE]       = TRUE,
                    [CTB_SORT_BY_SIZE]       = FALSE,
                    [CTB_SORT_BY_DATE_ADDED] = FALSE,
                    [CTB_SORT_BY_DATE_READ]  = FALSE,
                    [CTB_SORT_BY_AUTHOR]     = TRUE,
                };

    g_sort_order = sort_order;
    g_is_sort_asc = is_sort_ascending[sort_order];
    menu_select_sort_order(sort_order);
    g_subtitle = gettext (order_names[sort_order]);
}


gboolean filemodel_set_sortorder (ctb_sort_order_t sort_order,
                                  const gchar* cursor_item,
                                  gboolean updateScreen)
{
    if (g_sort_order == sort_order) return FALSE;

    g_assert(!viewmode_get_fixed_order(g_viewmode2));
    viewmode_set_order(g_viewmode2, sort_order);
    set_sortorder(sort_order);

    if (updateScreen) {
        load_dir_from_metadb();
        if (g_values) scroll_to_filename(cursor_item);
        load_items_in_model();
    }
    return TRUE;
}


gboolean filemodel_show_alphabar() {
    // depends on viewmode
    if (viewmode_get_style(g_viewmode2) == DESKTOP_VIEW || viewmode_get_style(g_viewmode2) == SETTINGS_VIEW) return FALSE;

    // otherwise depend on sortorder
    if (g_sort_order == CTB_SORT_BY_NAME || g_sort_order == CTB_SORT_BY_AUTHOR) {
        return TRUE;
    }
    return FALSE;
}


gboolean filemodel_has_searchbutton()
{
    if (viewmode_get_style(g_viewmode2) == DESKTOP_VIEW || viewmode_get_style(g_viewmode2) == SETTINGS_VIEW) return FALSE;
    return TRUE;
}


void filemodel_set_viewmode2(ViewMode newmode)
{
    LOGPRINTF("viewmode %d -> %d", g_viewmode2, newmode);
    if (g_viewmode2 != newmode) {
        g_viewmode2 = newmode;
        set_sortorder(viewmode_get_order(newmode));
        const char* category = NULL;
        if (newmode != DESKTOP_VIEW) {
            category = viewmode_get_title(newmode);
        }
        menu_set_category(category);
    }
}

ViewMode filemodel_get_viewmode2(void)
{
    return g_viewmode2;
}

void filemodel_set_search_filter(const gchar* search_string)
{
    g_free(g_search_filter);
    g_search_filter = NULL;
    if (search_string == NULL || strcmp(search_string, "") == 0) return;
    g_search_filter = g_strdup(search_string);
}


const gchar* filemodel_get_title()
{
    return _(viewmode_get_title(g_viewmode2));
}


const gchar* filemodel_get_subtitle()
{
    if (viewmode_get_fixed_order(g_viewmode2)) return "";
    else return g_subtitle;
}


static gboolean is_dir_on_media(const char* dir)
{
    const char* media = ipc_get_media();
    if (!media) return FALSE;

    int len = strlen(media);
    if (strncmp(dir, media, len) == 0 &&
       (dir[len] == '/' || dir[len] == '\0')) return TRUE;
    return FALSE;
}


static gboolean filemodel_current_is_media()
{
    return is_dir_on_media(g_current_dir);
}


const gchar* filemodel_get_menu_content()
{
    if (g_viewmode == DELETE_MODE) return MENU_CONTENT_DELETE_MODE;

    if (viewmode_get_style(g_viewmode2) == RECENT_VIEW) return MENU_CONTENT_RECENT_MODE;

    if (filemodel_current_is_media()) return MENU_CONTENT_MEDIA;

    return MENU_CONTENT;
}


gboolean filemodel_set_viewsize (const int n_items, gboolean updateScreen)
{
    LOGPRINTF("items=%d->%d  updateScreen=%d", g_items_per_page, n_items, updateScreen);
    g_return_val_if_fail(n_items > 0  && n_items < 100, FALSE);
    if (n_items == g_items_per_page) return FALSE;

    g_items_per_page = n_items;

    // round g_desktop_offset down to new pagination
    int desktop_page = g_desktop_offset / g_items_per_page;
    g_desktop_offset = desktop_page * g_items_per_page;

    // round g_item_offset down to new pagination
    int page = g_item_offset / g_items_per_page;
    g_item_offset = page * g_items_per_page;

    if (updateScreen) {
        load_items_in_model();
    }
    return TRUE;
}


void filemodel_set_viewmode(ctb_viewmodes_t newmode, gboolean updateScreen)
{
    LOGPRINTF("mode: %d->%d  update=%d", g_viewmode, newmode, updateScreen);
    g_viewmode = newmode;
    clear_toggled_list();
    if (updateScreen) {
        load_items_in_model();
    }
    menu_update_view_mode(g_viewmode);
    ipc_menu_set_pagecounter(g_curpage, g_numpages, g_boundary_check);
}


void filemodel_set_thumbsize(const filemodel_thumbsize_t thumb_size, gboolean reload)
{
    LOGPRINTF("thumb_size [%d]", thumb_size);

    if (thumb_size == g_thumbnail_size) return;

    g_thumbnail_size = thumb_size;
    if (reload) load_dir_from_metadb();
    // NOTE: 'fixes' wrong height in contentview
    gtk_list_store_clear(g_filestore);
}


// returns number of shown items (NOT special items like Up, etc) per page
// NOTE: the minimum is 1
static int num_items_per_page()
{
    int num_per_page = g_items_per_page;
    num_per_page -= num_specials_per_page();
    if (num_per_page <= 0) num_per_page = 1;
    return num_per_page;
}


gboolean filemodel_has_prev_page()
{
    if (filemodel_current_is_desktop()) return (g_desktop_offset > 0);
    else return (g_item_offset > 0);
}


gboolean filemodel_has_next_page()
{
    return g_has_next_page;
}


void filemodel_page_previous()
{
    if (filemodel_current_is_desktop()) {
        if (g_desktop_offset > 0) {
            g_desktop_offset -= g_items_per_page;
            if (g_desktop_offset < 0) g_desktop_offset = 0;
            load_items_in_model();
        }
    } else {
        if (g_item_offset > 0) {
            g_item_offset -= num_items_per_page();
            if (g_item_offset < 0) g_item_offset = 0;
            load_items_in_model();
        }
    }
}


void filemodel_page_next()
{
    if (g_has_next_page)
    {
        LOGPRINTF("entry: old offset [%d] page size [%d]", g_item_offset, g_items_per_page);
        if (filemodel_current_is_desktop()) {
            g_desktop_offset += g_items_per_page;
            g_assert(g_desktop_offset < g_desktop_items);
        } else {
            g_item_offset += num_items_per_page();
            g_assert(g_item_offset < g_total_items);
        }
        load_items_in_model();
    }
}


void filemodel_update_pagecounter()
{
    ipc_menu_set_pagecounter(g_curpage, g_numpages, g_boundary_check);
}


static void scroll_to_filename(const gchar *filename)
{
    LOGPRINTF("filename=%s", filename);
    gint item_idx  = 0;

    if (g_values && filename && *filename)
    {
        int row;
        int columns = g_values->n_columns;
        const metadata_cell *cell = (const metadata_cell*) (g_values->cell_data->data);
        for (row = 0; row < g_total_items; row++) {
            const gchar* name = get_row_filename(cell);
            if (strcmp( filename, name) == 0) {
                item_idx = row;
                break;
            }
            cell += columns;
        }
    }

    int num_per_page = num_items_per_page();
    int page = item_idx / num_per_page;
    g_item_offset = page * num_per_page;
}


static const gchar* get_sorted_name(const metadata_cell* row)
{
    const gchar* name = NULL;
    if (g_sort_order == CTB_SORT_BY_NAME) {
        if (viewmode_get_style(g_viewmode2) == DIR_VIEW) {
            name = get_row_filename(row);
        } else {
            name = get_row_title(row);
            if (name == NULL) name = get_row_filename(row);
        }
    } else if (g_sort_order == CTB_SORT_BY_AUTHOR) {
        name = get_row_author(row);
    }
    return name;
}


static unsigned int alpha2index(char alpha)
{
    if (isalpha(alpha)) {
        int letter = toupper(alpha);
        return (1 + letter - 'A');  // normal letter
    } else {
        return 0;   // index for '#'
    }
}


int filemodel_scroll_to_letter(gchar letter, gboolean jump_to_dir, gboolean* same_page)
{
    g_assert(g_values);
    gint item_idx  = 0;
    gboolean found = FALSE;

    gboolean first_ok = TRUE;
    int index = alpha2index(letter);
    if (g_alpha_list[index] == (ALPHA_HAS_DIR|ALPHA_HAS_FILE)) first_ok = FALSE;

    int row;
    const metadata_cell *cell = (const metadata_cell*) (g_values->cell_data->data);
    for (row = 0; row < g_total_items; row++) {
        const gchar* name = get_sorted_name(cell);
        if (name) {
            char current = name[0];
            if (current == letter || current == tolower(letter) ||  // normal char (a-z)
                (letter == '#' && !isalpha(current))) {             // special char (0-9,!,?, etc)
                // check whether it's the preferred type
                if (first_ok || jump_to_dir == get_row_is_directory(cell)) {
                    item_idx = row;
                    found = TRUE;
                    break;
                }
            }
        }
        cell += g_values->n_columns;
    }

    int num_per_page = num_items_per_page();
    if (item_idx >= g_item_offset && item_idx < (g_item_offset + num_per_page)) {
        *same_page = TRUE;
    } else {
        *same_page = FALSE;
    }

    int page = item_idx / num_per_page;
    g_item_offset = page * num_per_page;
    
    if (found) {
        item_idx = item_idx % num_per_page;
        item_idx += num_specials_per_page();
    } else {
        // shouldn't really happen!
        item_idx = -1;
    }
    if (!*same_page) load_items_in_model();
    return item_idx;
}


int filemodel_get_first_alpha_index()
{
    if (g_total_items == 0) return -1;
    g_assert(g_values);

    const metadata_cell *cell = (const metadata_cell*) (g_values->cell_data->data);
    cell += (g_item_offset * g_values->n_columns);
    const gchar* name = get_sorted_name(cell);
    if (name == NULL) return -1;
    return alpha2index(name[0]);
}


void filemodel_scroll_to_filename (const gchar *filename)
{
    scroll_to_filename(filename);
    load_items_in_model();
}


int filemodel_get_display_index(const gchar *filename)
{
    if (!filename || !*filename) return 0;

    // special for desktop
    if (filemodel_current_is_desktop()) {
        int index = 0;
        GSList *iter = desktop_names;
        while (iter) {
            if (strcmp(filename, (const char*)iter->data) == 0) return index;
            index++;
            iter = g_slist_next(iter);
        }
    }
    else if (g_values)
    {
        int num_per_page = num_items_per_page();
        int columns = g_values->n_columns;
        const metadata_cell *cell = (const metadata_cell*) (g_values->cell_data->data);
        // skip g_item_offset rows
        cell += (g_item_offset * g_values->n_columns);
        // search within currently visible window
        int row;
        for (row=g_item_offset; row < (g_item_offset + num_per_page) && row < g_total_items ; row++)
        {
            g_assert(cell->type == METADATA_TEXT);
            if (strcmp( filename, cell->value.v_text->str) == 0) {
                int item_idx = row - g_item_offset;
                item_idx += num_specials_per_page();
                return item_idx;
                break;
            }
            cell += columns;
        }
    }

    return 0;
}


// loads items from metadata.db and load them into model for GUI
// the dir-stack has already been modified for new dir
static int filemodel_chdir(const gchar *dir, const gchar *cursor_item)
{
    LOGPRINTF("dir=%s  cursor=%s", dir, cursor_item ? cursor_item : "NULL");
    g_assert(dir && *dir);

    int ret = ER_OK;
    if (filemodel_current_is_desktop()) {
        if (ipc_get_media() == NULL) close_database();
        // do nothing
    } else {
        if (viewmode_get_style(g_viewmode2) == SETTINGS_VIEW) {
            ret = open_global_database(dir);    // Settings
        } else {
            const char* card = ipc_get_media();
            ret = open_global_database(card);
        }
    }
    if (ret != ER_OK)
    {
        ERRORPRINTF("cannot synchronise database, keeping model unchanged");
        return ER_FAIL;
    }

    g_free(g_current_dir);
    g_current_dir = g_strdup(dir);
    menu_set_current_is_media(filemodel_current_is_media());
    load_dir_from_metadb();
    if (g_values) scroll_to_filename(cursor_item);
    load_items_in_model();

    return ER_OK;
}


int filemodel_chdir_desktop()
{
    clear_dir_stack();
    g_desktop_offset = 0;
    filemodel_set_viewmode2(DESKTOP_VIEW);
    return filemodel_chdir(DIR_DESKTOP_INTERNAL, NULL);
}


static void free_stack_entry(stack_entry_t *entry)
{
    g_assert(entry);
    g_free(entry->parent_dir);
    g_free(entry->last_item);
    g_free(entry);
}


static stack_entry_t* create_stack_entry(const gchar* parent, const gchar* item)
{
    stack_entry_t *entry = g_new0(stack_entry_t, 1);
    entry->parent_dir = g_strdup(parent);
    entry->last_item = g_strdup(item);
    return entry;
}


static void clear_dir_stack()
{
    GList *iter = g_dir_stack;
    while (iter) {
        free_stack_entry(iter->data);
        iter = g_list_next(iter);
    }
    g_list_free(g_dir_stack);
    g_dir_stack = NULL;
}


int filemodel_chdir_down(const gchar *dir, const gchar *last_item)
{
    LOGPRINTF("dir=%s  last_item=%s", dir, last_item);
    g_dir_stack = g_list_append(g_dir_stack, create_stack_entry(g_current_dir, last_item));

    filemodel_chdir(dir, NULL);
    return 0;
}


gchar *filemodel_chdir_up()
{
    LOGPRINTF("");
    g_assert(g_dir_stack);
    GList* last = g_list_last(g_dir_stack);
    stack_entry_t *entry = (stack_entry_t*)last->data;
    g_dir_stack = g_list_delete_link(g_dir_stack, last);

    if (g_dir_stack)
    {
        // still can go up more
        filemodel_chdir(entry->parent_dir, entry->last_item);
    }
    else
    {
        // on desktop
        filemodel_set_viewmode2(DESKTOP_VIEW);
        filemodel_chdir(DIR_DESKTOP_INTERNAL, NULL);
    }
    gchar* last_item = g_strdup(entry->last_item);
    free_stack_entry(entry);

    return last_item;
}


gboolean filemodel_resync(gboolean force_reload)
{
    if (!force_reload && !is_database_changed()) return FALSE;

    if (filemodel_current_is_desktop()) {
        if (g_desktop_items <= g_items_per_page) g_desktop_offset = 0;
    }
    int old_offset = g_item_offset;
    load_dir_from_metadb();

    // set new offset as close to old one as possible
    g_item_offset = old_offset;
    int num_per_page = num_items_per_page();
    while (g_item_offset >= g_total_items) {
        g_item_offset -= num_per_page;
    }
    if (g_item_offset < 0) g_item_offset = 0;
    return TRUE;
}


static GdkPixbuf* get_thumbnail_from_data(const guchar* data, int size, const gchar* filename)
{
    // load pixbuf with data from blob
    GdkPixbufLoader *thumb_ldr = gdk_pixbuf_loader_new_with_type(THUMBNAIL_IMAGE_TYPE, NULL);
    g_assert(thumb_ldr);
    GdkPixbuf* thumbnail = NULL;

    GError *err = NULL;
    gboolean ok = gdk_pixbuf_loader_write(thumb_ldr, data, size, &err);
    if (!ok)
    {
        ERRORPRINTF("cannot load thumbnail for file [%s] - [%s]", filename, err->message);
        goto unref;
    }

    ok = gdk_pixbuf_loader_close(thumb_ldr, &err);
    if (!ok)
    {
        ERRORPRINTF("cannot close thumbnail for file [%s] - [%s]", filename, err->message);
        goto unref;
    }

    // get a pointer to the resulting pixbuf, if any
    thumbnail = gdk_pixbuf_loader_get_pixbuf(thumb_ldr);
    if (thumbnail) {
        // take a reference on the object
        g_object_ref(thumbnail);
    } else {
        ERRORPRINTF("cannot get thumbnail pixbuf for file [%s]", filename);
    }
unref:
    g_clear_error(&err);
    g_object_unref(thumb_ldr);
    return thumbnail;
}


gboolean filemodel_current_is_desktop()
{
    return (g_dir_stack == NULL);
}


gboolean filemodel_window_is_on_top()
{
    return g_window_is_on_top;
}


void filemodel_set_window_is_on_top(gboolean is_on_top )
{
    g_window_is_on_top = is_on_top;
}


const gchar* filemodel_get_current_dir()
{
    return g_current_dir;
}


int filemodel_get_dir_depth()
{
    int depth = 0;
    if (g_dir_stack != NULL)
    {
        depth = g_list_length(g_dir_stack);
    }
    return depth;
}


static void update_alpha_list()
{
    bzero(g_alpha_list, sizeof(g_alpha_list));

    g_assert(g_values);

    if (g_sort_order != CTB_SORT_BY_NAME && g_sort_order != CTB_SORT_BY_AUTHOR) return;

    const metadata_cell *cell = (const metadata_cell*) (g_values->cell_data->data);

    int row;
    for (row=0; row<g_total_items; row++) {
        const gchar* name = get_sorted_name(cell);
        if (name != NULL) {
            int index = alpha2index(name[0]);
            gboolean is_dir = get_row_is_directory(cell);
            g_alpha_list[index] |= (is_dir ? ALPHA_HAS_DIR : ALPHA_HAS_FILE);
        }

        cell += g_values->n_columns;
    }
}


const gchar* filemodel_get_alpha_list()
{
    return g_alpha_list;
}


static int is_row_toggled(int row, const gchar* filename, const gchar* dirpath)
{
    if (g_viewmode == DELETE_MODE) {
        // check if allowed
        if (!is_delete_allowed(filename, dirpath)) return -1;

        // check if we have previous state
        GSList *iter = toggled_list;
        while (iter) {
            if (row == (int)iter->data) return 1;
            iter = g_slist_next(iter);
        }
    }
    return 0;
}


static void clear_toggled_list()
{
    g_slist_free(toggled_list);
    toggled_list = NULL;
}


static gint compare_toggled_entries(gconstpointer left, gconstpointer right)
{
    return ((int)left > (int)right);
}


void filemodel_toggle_entry(int index, int state, GtkTreeIter *iter)
{
    index -= num_specials_per_page();
    int item_index = g_item_offset + index;
    
    if (state == 0) {
        toggled_list = g_slist_remove(toggled_list, (int*)item_index);
    } else {
        toggled_list = g_slist_prepend(toggled_list, (int*)item_index);
    }

    GtkTreeModel *model = GTK_TREE_MODEL(g_filestore);

    // add overlay for delete mode in iconView
    if (g_viewmode == DELETE_MODE && g_thumbnail_size == MODTHUMB_MEDIUM)
    {
        const metadata_cell *cell = (const metadata_cell*) (g_values->cell_data->data);
        cell += (item_index * g_values->n_columns);
        GdkPixbuf *thumbnail = get_thumbnail(cell);
        GdkPixbuf *thumbnail_delete = create_delete_overlay(thumbnail, state);

        gtk_list_store_set (GTK_LIST_STORE(model), iter, MODCOL_THUMBNAIL, thumbnail_delete, -1);
        g_object_unref(thumbnail);
        g_object_unref(thumbnail_delete);
    }
    gtk_list_store_set (GTK_LIST_STORE(model), iter, MODCOL_TOGGLED, state, -1);
}


static void delete_entry_at_offset(int offset)
{
    g_assert(offset < g_total_items);
    const metadata_cell *cell = (const metadata_cell*) (g_values->cell_data->data);
    // skip g_item_offset rows
    cell += (offset * g_values->n_columns);

    const char* filename = get_row_filename(cell);
    const char *dirpath = get_row_dirpath(cell);
    gboolean is_directory = get_row_is_directory(cell);
    const char *filetype = get_row_filetype(cell);

    // check last_read item
#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
    if (g_last_read &&
        strcmp(filename, g_last_read->filename->str) == 0 &&
        strcmp(dirpath, g_last_read->directory_path->str) == 0)
    {
        clear_last_read();
    }
#endif

    filelist_entry_t *entry = filelist_entry_new();
    entry->filename         = g_string_new(filename);
    entry->filename_display = NULL;
    entry->filetype         = g_string_new(filetype);
    entry->directory_path   = g_string_new(dirpath);
    entry->is_directory     = is_directory;
    delete_item(entry);
    filelist_entry_free(entry);
}


void filemodel_delete_toggled()
{
    if (toggled_list) { // if none toggled do nothing
        toggled_list = g_slist_sort(toggled_list, compare_toggled_entries);
        GSList *iter = toggled_list;
        while (iter) {
            int item_offset = (int)iter->data;
            delete_entry_at_offset(item_offset);
            iter = g_slist_next(iter);
        }
        clear_toggled_list();
        filemodel_resync(FALSE);
    }
}


int filemodel_num_toggled()
{
    int count = 0;
    GSList *iter = toggled_list;
    while (iter) {
        count++;
        iter = g_slist_next(iter);
    }
    return count;
}


void filemodel_set_delete_text(const gchar* text)
{
    if (g_delete_text != NULL && strcmp(g_delete_text, text) == 0) return;
    g_free(g_delete_text);
    g_delete_text = g_strdup(text);
    if (g_viewmode == DELETE_MODE) {
        GtkTreeIter first;
        GtkTreeModel *model = GTK_TREE_MODEL(g_filestore);
        gboolean valid = gtk_tree_model_get_iter_first(model, &first);
        if (valid) {
            gtk_list_store_set (GTK_LIST_STORE(model), &first,
                                MODCOL_TITLE, g_delete_text, -1);
        }
    }
}


static gboolean is_delete_allowed(const gchar* filename, const gchar* dirpath) {
    // dont allow System dir to be deleted
    if (strcmp(dirpath, DIR_LIBRARY) == 0 && strcmp(filename, "System") == 0) return FALSE;

    const gchar *dir = dirpath;
    if (strcmp(dirpath, ".") == 0) dir = g_current_dir;
    char real_dir[PATH_MAX];
    char *cp = realpath(dir, real_dir);
    if (cp == real_dir) {
        // not allowed to delete items from internal desktop
        if (strcmp(real_dir, DIR_DESKTOP_INTERNAL) == 0) return FALSE;

        // not allow to delete items outside mountpoint (Settings, etc)
        if (!is_dir_on_media(real_dir)) return FALSE;
    }

    return TRUE;
}


static void add_previous_dir(const gchar* title, const gchar* subtitle, const gchar* iconname)
{
    gtk_list_store_insert_with_values(
                    g_filestore,
                    NULL,
                    g_items_per_page,
                    MODCOL_FILENAME,         "..",
                    MODCOL_TITLE,            title,
                    MODCOL_FILETYPE,         SPECIAL_ITEM_NAME,
                    MODCOL_FILETYPE_DISPLAY, "",
                    MODCOL_DIRECTORY_PATH,   ".",
                    MODCOL_IS_DIRECTORY,     TRUE,
                    MODCOL_FILESIZE,         "",
                    MODCOL_FILEDATE,         "",
                    MODCOL_SUBTITLE,         subtitle,
                    MODCOL_THUMBNAIL,        get_icon_from_file(iconname, g_thumbnail_size),
                    MODCOL_TOGGLED,          -1,
                    -1 );
}


static int num_browse_mode_specials()
{
    switch (viewmode_get_style(g_viewmode2)) {
        case DESKTOP_VIEW:
            return 0;
        case SETTINGS_VIEW:
        case BOOKS_VIEW:
        case NEWS_VIEW:
        case IMAGES_VIEW:
        case PERSONAL_VIEW:
        case HELP_VIEW:
        case SEARCH_VIEW:
        case RECENT_VIEW:
            return 1;       // Previous dir
        case NOTES_VIEW:
            return 2;       // Previous dir + New Note
        case DIR_VIEW:
        case SHORTCUT_VIEW:
            return 1;       // Up a level
        default:
            ERRORPRINTF("Unkown style for viewmode: %d", g_viewmode2);
            return 0;
    }
    return 0;       // for compiler warning
}


static void add_browse_mode_specials()
{
    switch (viewmode_get_style(g_viewmode2)) {
        case DESKTOP_VIEW:
            // dont insert anything
            break;
        case SETTINGS_VIEW:
        case BOOKS_VIEW:
        case NEWS_VIEW:
        case IMAGES_VIEW:
        case PERSONAL_VIEW:
        case HELP_VIEW:
        case SEARCH_VIEW:
        case RECENT_VIEW:
            add_previous_dir(_("to Home"), _("Return to the Home screen"), "back");
            break;
        case NOTES_VIEW:
            add_previous_dir(_("to Home"), _("Return to the Home screen"), "back");
            gtk_list_store_insert_with_values(
                            g_filestore,
                            NULL,
                            g_items_per_page,
                            // NOTE should match filename in check for shortcut_allowed()
                            MODCOL_FILENAME,         "new",
                            MODCOL_TITLE,            _("New Note"),
                            MODCOL_FILETYPE,         "note",
                            MODCOL_FILETYPE_DISPLAY, "",
                            MODCOL_DIRECTORY_PATH,   "",
                            MODCOL_IS_DIRECTORY,     FALSE,
                            MODCOL_FILESIZE,         "",
                            MODCOL_FILEDATE,         "",
                            MODCOL_SUBTITLE,         _("Open a new notepad to write on"),
                            MODCOL_THUMBNAIL,        get_icon_from_file("new-note", g_thumbnail_size),
                            MODCOL_TOGGLED,          -1,
                            -1 );
            break;
        case DIR_VIEW:
        case SHORTCUT_VIEW:
            add_previous_dir(_("Up a Level"), "", "parent-dir");
            break;
        default:
            ERRORPRINTF("Unkown style for viewmode: %d", g_viewmode2);
            break;
    }
}


static void add_delete_mode_specials()
{
    gtk_list_store_insert_with_values(
                    g_filestore,
                    NULL,
                    g_items_per_page,
                    MODCOL_FILENAME,         "DELETE-ACTION",
                    MODCOL_TITLE,            g_delete_text,
                    MODCOL_FILETYPE,         "",
                    MODCOL_FILETYPE_DISPLAY, "",
                    MODCOL_DIRECTORY_PATH,   "",
                    MODCOL_IS_DIRECTORY,     FALSE,
                    MODCOL_FILESIZE,         "",
                    MODCOL_FILEDATE,         "",
                    MODCOL_SUBTITLE,           "",
                    MODCOL_THUMBNAIL,        get_icon_from_file("delete-action", g_thumbnail_size),
                    MODCOL_TOGGLED,          -1,
                    -1 );
}


// depends on viewmode (browse/delete) and viewmode2
static int num_specials_per_page()
{
    switch (g_viewmode) {
    case BROWSE_MODE:
        return num_browse_mode_specials();
    case DELETE_MODE:
        return 1;
    }
    return 0;
}


static void add_special_items()
{
    switch (g_viewmode) {
    case BROWSE_MODE:
        add_browse_mode_specials();
        break;
    case DELETE_MODE:
        add_delete_mode_specials();
        break;
    }
}


// sets g_curpage + g_numpages and reports to ipc
// NOTE items_per_page does NOT incude special items (Up a dir, Delete, etc)
static void update_numpages(int items_per_page)
{
    LOGPRINTF("items_per_page=%d", items_per_page);
    int first = 0;
    int total = 0;
    if (filemodel_current_is_desktop()) {
        first = MIN(g_desktop_offset + 1, g_desktop_items);
        total = g_desktop_items;
        g_has_next_page = (g_desktop_offset + items_per_page < total);
    } else {
        // NOTE: g_item offset must be multiple of items_per_page
        //       so fix if needed. Always round down
        if (g_item_offset % items_per_page != 0) {
            int offset = g_item_offset % items_per_page;
            g_item_offset -= offset;
            if (g_item_offset < 0) g_item_offset = 0;
        }

        first = MIN(g_item_offset + 1, g_total_items);
        total = g_total_items;
        g_has_next_page = (g_item_offset + items_per_page < total);
    }

    g_curpage = (first / items_per_page) + 1;   // start at 1
    g_numpages = total / items_per_page;
    if (total % items_per_page != 0) g_numpages++;
    if (g_numpages == 0) g_numpages = 1;

    if (filemodel_current_is_desktop() && g_numpages == 1) {
        g_curpage = 0;
        g_numpages = 0;
    }

    if (filemodel_window_is_on_top())
    {
        ipc_menu_set_pagecounter(g_curpage, g_numpages, g_boundary_check);
    }
}


static GdkPixbuf* get_thumbnail_for_shortcut(shortcut_t *shortcut,
                                      gboolean is_directory,
                                      const gchar* filetype,
                                      const gchar* filename,
                                      const gchar* dir)
{
    GdkPixbuf* thumbnail = NULL;
    // get default icon
    switch (shortcut->type)
    {
        case SHORTCUT_TO_FILE:
        {
            // get icon for extension of target file
            const gchar *ext = g_extension_pointer(shortcut->details.file.filename);
            thumbnail = get_icon_from_file_extension( ext,
                                                      is_directory,
                                                      g_thumbnail_size );
            break;
        }
        case SHORTCUT_TO_FOLDER:
        {
            // get icon for library
            char *cp = shortcut->details.folder.directory;
            if (   cp
                && shortcut->details.folder.filename == NULL )
            {
                if ( strcmp(cp, DIR_LIBRARY) == 0 )
                {
                    // internal library root
                    thumbnail = get_icon_library( g_thumbnail_size );
                }
                else
                {
                    int n = strlen(DIR_STORAGE_MNT);
                    if (   strncmp(cp, DIR_STORAGE_MNT, n) == 0
                        && cp[n] == '/' )
                    {
                        // path: /media/
                        cp = strchr( cp + n + 1, '/' );
                        if ( cp == NULL )
                        {
                            // external library root
                            thumbnail = get_icon_library( g_thumbnail_size );
                        }
                    }
                }
            }

            // get icon for folder
            if (thumbnail == NULL)
            {
                thumbnail = get_icon_from_file_extension( "",
                                                          TRUE,        // is_directory
                                                          g_thumbnail_size );
            }
            break;
        }
        case SHORTCUT_TO_APPLICATION:
            // get icon for application
            thumbnail = get_icon_application( g_thumbnail_size );
            break;
        
        case SHORTCUT_TO_WEB_LOCATION:
            thumbnail = get_icon_from_file_extension( "html",
                                                      FALSE,
                                                      g_thumbnail_size );
        
            break;

        default:
            WARNPRINTF("unknown shortcut type [%d] file [%s]", shortcut->type, filename);

            // get icon from file extension
            thumbnail = get_icon_from_file_extension( filetype,
                                                      is_directory,
                                                      g_thumbnail_size );
    }

    // add shortcut overlay
    if (thumbnail)
    {
        if ( strcmp(dir, DIR_DESKTOP_INTERNAL) != 0 )
        {
            thumbnail = gdk_pixbuf_copy(thumbnail);
            apply_icon_overlay_shortcut(g_thumbnail_size, thumbnail);
        }
        else
        {
            g_object_ref(thumbnail);
        }
    }
    return thumbnail;
}


static gboolean desktop_item_visible(int index)
{
    if (index >= g_desktop_offset &&
        index < g_desktop_offset + g_items_per_page) return TRUE;
    return FALSE;
}


static void add_desktop_item(int index,
                             const gchar* filename,
                             const gchar* title,
                             const gchar* subtitle,
                             const gchar* path,
                             const gchar* iconname)
{
    if (!desktop_item_visible(index)) return;
    gtk_list_store_insert_with_values(
                    g_filestore,
                    NULL,
                    g_items_per_page,
                    MODCOL_FILENAME,         filename,
                    MODCOL_TITLE,            title,
                    MODCOL_FILETYPE,         SPECIAL_ITEM_NAME,
                    MODCOL_FILETYPE_DISPLAY, "",
                    MODCOL_DIRECTORY_PATH,   path,
                    MODCOL_IS_DIRECTORY,     TRUE,
                    MODCOL_FILESIZE,         "",
                    MODCOL_FILEDATE,         "",
                    MODCOL_SUBTITLE,           subtitle,
                    MODCOL_THUMBNAIL,        get_icon_from_file(iconname, g_thumbnail_size),
                    MODCOL_TOGGLED,          -1,
                    -1 );
    desktop_names = g_slist_append(desktop_names, g_strdup(filename));
}


#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
static gboolean add_desktop_last_read_item(int index)
{
    if (!g_last_read) return FALSE;

    g_assert(g_last_read->filename);
    g_assert(g_last_read->filetype);
    g_assert(g_last_read->directory_path);
    g_assert(g_last_read->filename_display);

    if (open_global_database(DIR_LIBRARY) != ER_OK)
    {
        ERRORPRINTF("cannot open database on card");
        return FALSE;
    }

    int rc = create_query();
    if (rc != ER_OK) return FALSE;

    if (g_last_read_results) {
        metadata_table_free(g_last_read_results);
        g_last_read_results = NULL;
    }
    rc = db_query_get_metadata(g_last_read->filename->str, g_last_read->directory_path->str, &g_last_read_results);
    if (rc != ER_OK) {
        ERRORPRINTF("cannot get metadata for database");
        return FALSE;
    }

    // NOTE: values is NULL when no results
    if (g_last_read_results)
    {
        if (!desktop_item_visible(index)) return TRUE;

        const metadata_cell *cell = (const metadata_cell*) (g_last_read_results->cell_data->data);
        const char *title = get_row_title(cell);

        // get thumbnail
        int thumbsize = 0;
        const guchar* thumbdata = get_row_thumbnail(cell, &thumbsize);
        GdkPixbuf *thumbnail = NULL;
        if (thumbdata != NULL) {
            thumbnail = get_thumbnail_from_data(thumbdata, thumbsize, g_last_read->filename->str);
        } else {
            thumbnail = get_icon_from_file_extension(g_last_read->filetype->str, FALSE, g_thumbnail_size);
            if (thumbnail) g_object_ref(thumbnail);
        }

        gtk_list_store_insert_with_values(
                        g_filestore,
                        NULL,
                        g_items_per_page,
                        MODCOL_FILENAME,         g_last_read->filename->str,
                        MODCOL_TITLE,            _("Continue Reading"),
                        MODCOL_FILETYPE,         g_last_read->filetype->str,
                        MODCOL_FILETYPE_DISPLAY, "",
                        MODCOL_DIRECTORY_PATH,   g_last_read->directory_path->str,
                        MODCOL_IS_DIRECTORY,     FALSE,
                        MODCOL_FILESIZE,         "",
                        MODCOL_FILEDATE,         "",
                        MODCOL_SUBTITLE,           title,
                        MODCOL_THUMBNAIL,        thumbnail,
                        MODCOL_TOGGLED,          -1,
                        -1 );
        if (thumbnail) g_object_unref(thumbnail);
        desktop_names = g_slist_append(desktop_names, g_strdup(g_last_read->filename->str));
        return TRUE;
    } else {
        // entry doesn't exist anymore, remove last read entry
        clear_last_read();
        return FALSE;
    }
}
#endif


#if MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
static void add_desktop_store_item(int index)
{
    if (!desktop_item_visible(index)) return;
    // add browser ('Shop') link
    const char* filename = "ebookmall.desktop";
    gtk_list_store_insert_with_values(
                    g_filestore,
                    NULL,
                    g_items_per_page,
                    MODCOL_FILENAME,         filename,
                    MODCOL_TITLE,            _("eBook Mall"),
                    MODCOL_FILETYPE,         "desktop",
                    MODCOL_FILETYPE_DISPLAY, "",
                    MODCOL_DIRECTORY_PATH,   DIR_DESKTOP_TEMPLATE,
                    MODCOL_IS_DIRECTORY,     FALSE,
                    MODCOL_FILESIZE,         "",
                    MODCOL_FILEDATE,         "",
                    MODCOL_SUBTITLE,           _("Browse the eBook Mall for new books and much more"),
                    MODCOL_THUMBNAIL,        get_icon_from_file("shop", g_thumbnail_size),
                    MODCOL_TOGGLED,          -1,
                    -1 );
    desktop_names = g_slist_append(desktop_names, g_strdup(filename));
}
#endif


static void clear_desktop_names()
{
    GSList *iter = desktop_names;
    while (iter) {
        g_free((char*)iter->data);
        iter = g_slist_next(iter);
    }
    g_slist_free(desktop_names);
    desktop_names = NULL;
}


static int add_desktop_items()
{
    clear_desktop_names();

    const gchar* mountpoint = ipc_get_media();
    
    // Determine which items should NOT be shown for a particular device
    viewmode_set_mode(DESKTOP_VIEW, 0);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    viewmode_set_mode(LASTREAD_VIEW, 0);
    viewmode_set_mode(STORE_VIEW, 0);
#endif
#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
    viewmode_set_mode(SHORTCUT_VIEW, 0);
#endif
    // Now handle all Desktop items in sort order
    // when data is refreshed from gconf
    views_prepare_views();
    
    unsigned int i = 0;
    ViewModeInfoExt* view = view_find_info_on_index(i);
    int idx = 0;
    while(view)
    {
        //printf("[%d:%d] %s ", i, view->mode, view->filename);
        if (view->mode & VIEW_ENABLE) // TODO: this allows settings to be disabled!!
        {
            // Handle special cases
            if (view->style == LASTREAD_VIEW)
            {
#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
                if (add_desktop_last_read_item(idx)) idx++;
#endif
            }
            else if (view->style == STORE_VIEW)
            {
#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
                add_desktop_store_item(idx++);
#endif
            }
            else
            {
                char buffer[512];
                // Allow special mount points e.g. "Shortcuts"
                if (view->style == SETTINGS_VIEW)
                {
                    sprintf(buffer, "%s", "/usr/share/ctb/settings");
                }
                else if (view->path != NULL)
                {
                    sprintf(buffer, "%s/%s", mountpoint, view->path);
                }
                else
                {
                    sprintf(buffer, "%s", mountpoint);
                }
                
                //printf("add_desktop_item: %s %s %s %s %s\n",view_info->filename, 
                //                 view_info->title, 
                //                 view_info->subtitle, 
                //                 buffer, 
                //                 view_info->filename_icon);
                //printf("add_desktop_item: [%s];",view->filename);
                add_desktop_item(idx++, 
                                 view->filename, 
                                 view->title, 
                                 view->subtitle, 
                                 buffer, 
                                 view->filename_icon);
            }
        }
        i++;
        view = view_find_info_on_index(i);
    }

    return idx;
}


static GdkPixbuf *get_thumbnail(const metadata_cell *cell)
{
    const char *filename = get_row_filename(cell);
    gboolean is_directory = get_row_is_directory(cell);
    const char *filetype = get_row_filetype(cell);

    // try to read from metadata results
    int thumbsize = 0;
    const guchar* thumbdata = get_row_thumbnail(cell, &thumbsize);
    GdkPixbuf *thumbnail = NULL;
    if (thumbdata != NULL) {
        thumbnail = get_thumbnail_from_data(thumbdata, thumbsize, filename);
    }

    // special for shortcuts
    if (thumbnail == NULL && is_shortcut_file_extension(filetype)) {
        shortcut_t *shortcut = NULL;
        const char *dirpath = get_row_dirpath(cell);
        const gchar *dir = dirpath;
        if (strcmp(dirpath, ".") == 0) dir = g_current_dir;
        parse_shortcut_file(dir, filename, &shortcut);
        if (shortcut) {
            thumbnail = get_thumbnail_for_shortcut(shortcut, is_directory,
                    filetype, filename, dir);
            shortcut_free(shortcut);
        }
    }

    // get default icon when no thumbnail available
    if (thumbnail == NULL) {
        thumbnail = get_icon_from_file_extension(filetype, is_directory,
                    g_thumbnail_size);
        if (thumbnail) g_object_ref(thumbnail);
    }

    return thumbnail;
}


static GdkPixbuf *create_delete_overlay(const GdkPixbuf* source, gboolean toggled)
{
    GdkPixbuf *result = NULL;
    if (gdk_pixbuf_get_width(source) != 120 || gdk_pixbuf_get_height(source) != 120) {
        // overlay on good image
        GdkPixbuf * empty = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, 120, 120);
        gdk_pixbuf_fill(empty, 0x00000000);
        int x_offset = (120 - gdk_pixbuf_get_width(source)) / 2;
        int y_offset = (120 - gdk_pixbuf_get_height(source)) / 2;
        gdk_pixbuf_copy_area(source,
                             0, 0,      // src x,y
                             gdk_pixbuf_get_width(source),   // dest width
                             gdk_pixbuf_get_height(source),  // dest height
                             empty,
                             x_offset, y_offset);    // dest x,y
        apply_icon_overlay_delete(MODTHUMB_MEDIUM, empty, toggled);
        result = empty;
    } else {
        GdkPixbuf *source_copy = gdk_pixbuf_copy(source);
        apply_icon_overlay_delete(MODTHUMB_MEDIUM, source_copy, toggled);
        return source_copy;
    }
    return result;
}


static char* get_title(const shortcut_t *shortcut, const char* title, const char* filename)
{
    // in DIR_VIEW we show filenames
    // MH: for shortcuts always show the name, not the filename
    if (!shortcut && (viewmode_get_style(g_viewmode2) == DIR_VIEW)) return g_strdup(filename);

    if (title && title[0]) return g_strdup(title);

    if (shortcut) return g_strdup(shortcut->name);

    return g_strdup(filename);
}


static char* get_subtitle(const shortcut_t *shortcut, const char* author)
{
    if (!shortcut) {
        if (author && author[0]) {
            /* TRANSLATORS: %s in this string will be replaced by the author of the book.
               For example: 'by Charles Dickens' */
            return g_strdup_printf(_("by %s"), author);
        } else {
            return NULL;
        }
    }

    // MH: for shortcuts always show the name, not the filename
    if (!shortcut && (viewmode_get_style(g_viewmode2) == DIR_VIEW)) return g_strdup(shortcut->name);

    if (shortcut->comment) return g_strdup(shortcut->comment);

    if (shortcut->type == SHORTCUT_TO_APPLICATION) return NULL;

    if (author) {
        /* TRANSLATORS: %s in this string will be replaced by the filename.
           For Example: 'shortcut to files/foo/bar.pdf' */
        return g_strdup_printf(_("shortcut to %s"), author);
    }

    return NULL;
}


// retrieve items from g_values and load these in file model for GUI
// it will start at g_item_offset
static void load_items_in_model()
{
    LOGPRINTF( "total=%d  offset=%d  items_per_page=%d",
               g_total_items, g_item_offset, g_items_per_page);

    if (g_values == NULL && g_total_items != 0) return;
    if (g_current_dir == NULL) return;

#if (TIMING_ON)
    struct timeval t1, t2;
    gettimeofday(&t1, NULL);
#endif

    // remove old items from filestore
    gtk_list_store_clear(g_filestore);

    if (filemodel_current_is_desktop()) {
        const gchar* mountpoint = ipc_get_media();
        if (mountpoint == NULL) goto out;
        g_desktop_items = add_desktop_items();
        if (g_desktop_offset >= g_desktop_items) {
            // items were removed while not on desktop, show page 0
            gtk_list_store_clear(g_filestore);
            g_desktop_offset = 0;
            g_desktop_items = add_desktop_items();
        }
        update_numpages(g_items_per_page);
        goto out;
    }

    g_assert(g_item_offset <= g_total_items);

    int items_per_page = num_items_per_page();
    update_numpages(items_per_page);
    add_special_items();
    if (g_total_items == 0) goto out;

    // add new items to filestore
    const metadata_cell *cell = (const metadata_cell*) (g_values->cell_data->data);

    // skip g_item_offset rows
    cell += (g_item_offset * g_values->n_columns);

    int row;
    for (row=g_item_offset; row < (g_item_offset + items_per_page) && row < g_total_items ; row++)
    {
        const char *filename = get_row_filename(cell);
        const char *dirpath = get_row_dirpath(cell);
        gboolean is_directory = get_row_is_directory(cell);
        const char *filetype = get_row_filetype(cell);
        const char *title = get_row_title(cell);
        const char *author = get_row_author(cell);

        const char *filetype_display = "";
        if (filetype) {
            filetype_display = get_type_descr_from_file_extension(filetype, is_directory);
        }

        gint64 size = get_row_filesize(cell);
        gchar *filesize = NULL;
        if (!is_directory) filesize = format_size(size);

        time_t date = get_row_filedate(cell);
        char filedate[100];
        filedate[0] = '\0';
        /* TRANSLATORS: This is a date and time format specifier, according to the strftime(3) man page.
           The result will be used to identify date and time of a file, e.g. '02/03/2010 11:13:01'.
           If in doubt, please keep unchanged. '%x %X' already takes the locale into account. */
        // xgettext:no-c-format
        strftime(filedate, sizeof(filedate), _("%x %X"), localtime(&date));  // xgettext:no-c-format

        gboolean is_shortcut = is_shortcut_file_extension(filetype);

        shortcut_t *shortcut = NULL;
        if (is_shortcut) {
            const gchar *dir = dirpath;
            if (strcmp(dirpath, ".") == 0) dir = g_current_dir;
            parse_shortcut_file(dir, filename, &shortcut);
        }

        // title
        char *title_display = get_title(shortcut, title, filename);

        // subtitle
        gchar* subtitle_display = NULL;
        if (is_shortcut && shortcut == NULL) {
            subtitle_display = g_strdup(_("broken shortcut"));
        } else {
#if MACHINE_HAS_ACSM
            if (strcmp("acsm", filetype) == 0) {
                subtitle_display = g_strdup(_("This file will be downloaded to your device."));
            } else
#endif
            {
                subtitle_display = get_subtitle(shortcut, author);
            }
        }

        shortcut_free(shortcut);

        gint toggled = is_row_toggled(row, filename, dirpath);

        GdkPixbuf *thumbnail = get_thumbnail(cell);

        // add overlay for delete mode in iconView
        if (g_viewmode == DELETE_MODE && g_thumbnail_size == MODTHUMB_MEDIUM && toggled != -1)
        {
            GdkPixbuf *thumbnail_delete = create_delete_overlay(thumbnail, toggled);
            g_object_unref(thumbnail);
            thumbnail = thumbnail_delete;
        }

        // add to filestore
        gtk_list_store_insert_with_values( g_filestore,
                                           NULL,
                                           g_items_per_page,
                                           MODCOL_FILENAME,         filename,
                                           MODCOL_TITLE,            title_display,
                                           MODCOL_FILETYPE,         filetype,
                                           MODCOL_FILETYPE_DISPLAY, filetype_display,
                                           MODCOL_DIRECTORY_PATH,   dirpath,
                                           MODCOL_IS_DIRECTORY,     is_directory,
                                           MODCOL_FILESIZE,         filesize,
                                           MODCOL_FILEDATE,         filedate,
                                           MODCOL_SUBTITLE,         subtitle_display,
                                           MODCOL_THUMBNAIL,        thumbnail,
                                           MODCOL_TOGGLED,          toggled,
                                           -1 );

        if (thumbnail) g_object_unref(thumbnail);
        g_free(filesize);
        g_free(title_display);
        g_free(subtitle_display);
        cell += g_values->n_columns;
    }
out:
#if (TIMING_ON)
    gettimeofday(&t2, NULL);
    float cost = (float) (((t2.tv_sec - t1.tv_sec) * 1000 * 1000) + (t2.tv_usec - t1.tv_usec));
    printf("%s() duration=%4.1lf ms\n", __func__, cost/1000);
#endif
    return; // needed to avoid empty label
}


// format filesize into a string
static gchar* format_size(const gint64 bytes)
{
    gchar       *size = NULL;   // return value
    long        n_int;
    long        n_frac;

    const int   MB = 1024 * 1024;
    const int   KB = 1024;

    if (bytes >= MB)
    {
        if (bytes >= 10 * MB)
        {
            n_int = (bytes + (MB / 2)) / MB;                    // MB round upward
            size = g_strdup_printf( "%ld %s",
                                     n_int,
                                          _("MB") );
        }
        else
        {
            n_int  = bytes / MB;                                //     MB round downward
            n_frac = ((bytes % MB) + (MB / 20)) / (MB / 10);    // 0.1 MB round upward
            n_frac = MIN(n_frac, 9);
            size = g_strdup_printf( "%ld.%ld %s",
                                     n_int,
                                         n_frac,
                                             _("MB") );
        }
    }
    else if (bytes >= KB)
    {
        if (bytes >= 10 * KB)
        {
            n_int = (bytes + (KB / 2)) / KB;                    // KB round upward
            size = g_strdup_printf( "%ld %s",
                                     n_int,
                                          _("KB") );
        }
        else
        {
            n_int  = bytes / KB;                                //     KB round downward
            n_frac = ((bytes % KB) + (KB / 20)) / (KB / 10);    // 0.1 KB round upward
            n_frac = MIN(n_frac, 9);
            size = g_strdup_printf( "%ld.%ld %s",
                                     n_int,
                                         n_frac,
                                             _("KB") );
        }
    }
    else
    {
        size = g_strdup_printf( "%lld %s",
                                 bytes,
                                    _("B") );
    }

    return size;
}


//------------------------------------------------------------------------------
// Functions to manipulate data in filestore
//------------------------------------------------------------------------------

filelist_entry_t* iter_to_entry(GtkTreeModel *model, GtkTreeIter *iter)
{
    gchar    *filename         = NULL;
    gchar    *filename_display = NULL;
    gchar    *filetype         = NULL;
    gchar    *directory_path   = NULL;
    gboolean is_directory      = FALSE;

    gtk_tree_model_get( model, iter, MODCOL_FILENAME,         &filename,
                                     MODCOL_TITLE, &filename_display,
                                     MODCOL_FILETYPE,         &filetype,
                                     MODCOL_DIRECTORY_PATH,   &directory_path,
                                     MODCOL_IS_DIRECTORY,     &is_directory,
                                     -1 );

    filelist_entry_t *entry = filelist_entry_new();
    entry->filename         = g_string_new(filename);
    entry->filename_display = g_string_new(filename_display);
    entry->filetype         = g_string_new(filetype);
    entry->directory_path   = g_string_new(directory_path);
    entry->is_directory     = is_directory;

    g_free(filename);
    g_free(filename_display);
    g_free(filetype);
    g_free(directory_path);

    return entry;
}


filelist_entry_t* filelist_entry_new()
{
    filelist_entry_t *thiz = g_new0(filelist_entry_t, 1);
    g_assert(thiz);
    thiz->is_directory = FALSE;

    return thiz;
}


void filelist_entry_free(filelist_entry_t *thiz)
{
    if (thiz)
    {
        if (thiz->filename) g_string_free(thiz->filename, TRUE);
        if (thiz->filename_display) g_string_free(thiz->filename_display, TRUE);
        if (thiz->filetype) g_string_free(thiz->filetype, TRUE);
        if (thiz->directory_path) g_string_free(thiz->directory_path, TRUE);
        g_free(thiz);
    }
}


filelist_entry_t* filelist_entry_copy ( const filelist_entry_t *src )
{

    g_assert(src);
    g_assert(src->filename         && src->filename->str);
    g_assert(src->filename_display && src->filename_display->str);
    g_assert(src->filetype         && src->filetype->str);
    g_assert(src->directory_path   && src->directory_path->str);

    filelist_entry_t *thiz = filelist_entry_new();
    thiz->filename = g_string_new(src->filename->str);
    thiz->directory_path = g_string_new(src->directory_path->str);
    thiz->filename_display = g_string_new(src->filename_display->str);
    thiz->filetype = g_string_new(src->filetype->str);
    thiz->is_directory = src->is_directory;

    return thiz;
}


//MH: bug in original code, (sorting on last read date is not working)
//#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
// NOTE only compare dir + filename, since that's unique
static gboolean filelist_entry_equal(const filelist_entry_t *left,
                                     const filelist_entry_t *right)
{
    if (left == NULL || right == NULL) return FALSE;
    if (strcmp(left->filename->str, right->filename->str) != 0) return FALSE;
    if (strcmp(left->directory_path->str, right->directory_path->str) != 0) return FALSE;
    return TRUE;
}


void filemodel_update_last_read(const filelist_entry_t *fileinfo)
{
    // skip entries without directory (like new note)
    if (fileinfo->directory_path->str[0] == 0) return;

    if (!filelist_entry_equal(g_last_read, fileinfo)) {
        clear_last_read();
        g_last_read = filelist_entry_copy(fileinfo);

        time_t now = time(NULL);
        int rc = db_query_update_lastread(fileinfo->filename, fileinfo->directory_path, now);
        if (rc != ER_OK) ERRORPRINTF("cannot update database");
    }
}
//#endif


#if (TIMING_ON)
static u_int64_t get_time_now()
{
    struct timespec now;
    clock_gettime(CLOCK_MONOTONIC, &now);
    u_int64_t now64 = now.tv_sec;
    now64 *= 1000000;
    now64 += (now.tv_nsec/1000);
    return now64;
}

static u_int64_t t1 = 0;
static u_int64_t t2 = 0;

void start_duration_timer()
{
    LOGPRINTF("");
    if (t1 == 0) t1 = get_time_now();
}


void stop_duration_timer()
{
    if (t1 != 0) {
        t2 = get_time_now();
        float duration = t2 - t1;
        printf("ACTION DURATION = %4.1lf ms\n\n", duration / 1000);
        t1 = 0;
    }
}
#endif

// return the number of the user view
// TODO: give this function a proper name!!
int filemodel_is_user_view(const char* filename)
{
    ViewModeInfoExt* view_info = view_find_info_with_name(filename);
    if (view_info) return view_info->number;
    else return -1;
}

static const char* viewmode_get_title(int viewmode)
{
    ViewModeInfoExt* view_info = view_find_info_with_number(viewmode);
    g_assert(view_info);
    return view_info->title;
}

const char* viewmode_get_tag_filter(int viewmode)
{
    ViewModeInfoExt* view_info = view_find_info_with_number(viewmode);
    g_assert(view_info);
    return view_info->tag_filter;
}

static ctb_sort_order_t viewmode_get_order(int viewmode)
{
    ViewModeInfoExt* view_info = view_find_info_with_number(viewmode);
    g_assert(view_info);
    return view_info->order;
}

static void viewmode_set_order(int viewmode, ctb_sort_order_t sort_order)
{
    ViewModeInfoExt* view_info = view_find_info_with_number(viewmode);
    g_assert(view_info);
    view_info->order = sort_order;
}

int viewmode_get_mode(int viewmode)
{
    ViewModeInfoExt* view_info = view_find_info_with_number(viewmode);
    g_assert(view_info);
    return view_info->mode;
}

static void viewmode_set_mode(int viewmode, int mode)
{
    ViewModeInfoExt* view_info = view_find_info_with_number(viewmode);
    g_assert(view_info);
    view_info->mode = mode;
}

static gboolean viewmode_get_fixed_order(int viewmode)
{
    ViewModeInfoExt* view_info = view_find_info_with_number(viewmode);
    g_assert(view_info);
    return view_info->fixed_order;
}

ViewMode viewmode_get_style(int viewmode)
{
    ViewModeInfoExt* view_info = view_find_info_with_number(viewmode);
    g_assert(view_info);
    return view_info->style;
}
