# CalibreKindleCollections.py
#
# Script for managing Kindle collections and creating/updating them based on metadata from Calibre

import sys,os.path,os,json,time
from hashlib import sha1
from optparse import OptionParser
from xml.sax.saxutils import escape

##################################
# Metadata rules for collections
#
# This metadata augments the metadata stored in Calibre.
# For strings which contain lists, separate list items with a "|" character.
##################################

# Minimum number of books which an author/tag/series must have in order to have a collection created
minBooksForAuthor = 4
minBooksForTag = 1
minBooksForSeries = 2

# Any authors for whom you ALWAYS wish to create collections, even if they have fewer than minBooksForAuthor books
includedAuthorsCSV =    "A E van Vogt|Bret Easton Ellis|H P Lovecraft|Kenneth E Baker|Martin H Greenberg|Raymond Chandler|"     \
                        "Roger MacBride Allen|Steven Barnes|William F Wu|Martin Gardner|Lewis Carroll"

# Any authors for whom you do not wish to create collections even if they have at least minBooksForAuthor books
excludedAuthorsCSV =    "Bruce Bethke|D H Lawrence|Fyodor Dostoyevsky|Geoffrey Chaucer|George Eliot|Henry Fielding|Jane Austen|Joseph Conrad|"  \
                        "Marcel Proust|Nathaniel Hawthorne|Plato|Robert Louis Stevenson|William Shakespeare|Emile Zola|Helena Blavatsky|"       \
                        "Christopher Fowler|Edward Bryant|Sheherezad"

# Any series for which you ALWAYS wish to create collections, even if they have fewer than minBooksForSeries books
includedSeriesCSV = ""

# Any series for which you do not wish to create collections even if they have at least minBooksForSeries books
excludedSeriesCSV = ""


# When counting the number of books by an author to decide whether to create a collection, exclude any books in these series
excludedSeriesAuthorsCSV =  "Sector General|Dexter|Donald Strachey|Resident Evil|The Dark is Rising Sequence"

# When counting the number of books by an author to decide whether to create a collection, exclude any books with these tags
excludedTagsAuthorsCSV = "- Isaac Asimov (Robots-Empire-Foundation)|Doctor Who"

excludeAnthologiesWithMultipleAuthors = True        # If True, exclude from count any anthologies which contain stories by more than one author

excludeSeriesWithMultipleAuthors = True             # If True, exclude from count any books from series written by multiple authors UNLESS
                                                    # series name partially or wholly matches an entry in the list in multiauthorSeriesExceptionsCSV
multiauthorSeriesExceptionsCSV = "Nebula Awards|Booker Winner"

##################################################################
# Define and create Miscellany-by-Genre collections
##################################################################

def createMiscellanyCollections():

    if options.verboseMessages:
        print "[[PHASE: CREATING MISCELLANY COLLECTIONS]]\n"

    # Miscellany-by-genre collections
    #
    # These define collections which mop-up specific genres of books which were written, for example,
    # by authors who have not been allocated their own collection.
    #
    # Order of the three params is:
    #
    #   1) The name of the collection to create
    #   2) The tags which identify books to put in this collection (books may match any of the tags given, don't have to match all)
    #   3) Exclude books if they're already in any of the collections listed OR they have any of the tags listed
    #   4) If True, exclude books if they're already in a by-author collection
    #   5) If True, exclude books if they're already in a by-tags collection
    #   6) If True, exclude books if they're already in a by-series collection
    #
    # First three params are strings and params 2 and 3 can be lists, with each item in the list separated with a "|" character.
    # Note that it is perfectly allowable for a given book to be mopped up by multiple genre-specific miscellany collections
    # unless explicitly specified otherwise (i.e. by putting a previously-created collection's name in the list in the third param).
    #
    #                          collection name                  included tags        exclude collections or tags
    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Short Stories",               "Short Story",      "- Arthur Conan Doyle|"                                         \
                                                                                    "- Clive Barker|"                                               \
                                                                                    "- Isaac Asimov (Robots-Empire-Foundation)|"                    \
                                                                                    "- Larry Niven's Known Space|"                                  \
                                                                                    "- Thieves' World|"                                             \
                                                                                    "- Anthologies",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Anthologies",                 "Anthology",        "- Arthur Conan Doyle|"                                         \
                                                                                    "- Clive Barker|"                                               \
                                                                                    "- Isaac Asimov (Robots-Empire-Foundation)|"                    \
                                                                                    "- Larry Niven's Known Space|"                                  \
                                                                                    "- Short Stories|"                                              \
                                                                                    "- Thieves' World|"                                             \
                                                                                    "- Anthologies",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Doctor Who Miscellany",       "Doctor Who",       "- Doctor Who: Eighth Doctor Adventures|"                       \
                                                                                    "- Doctor Who: New Series Adventures|"                          \
                                                                                    "- Doctor Who: Past Doctor Adventures|"                         \
                                                                                    "- Doctor Who: Target series|"                                  \
                                                                                    "- Doctor Who: Virgin Missing Adventures|"                      \
                                                                                    "- Doctor Who: Virgin New Adventures|"                          \
                                                                                    "- Torchwood",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Space Opera",                 "Space Opera",      "- Perry Rhodan",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Shared Worlds",               "Shared World",     "- Perry Rhodan|"                                               \
                                                                                    "eZine",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Science Fiction Miscellany",  "Science Fiction",  "- Anne McCaffrey|"                                             \
                                                                                    "- Anthologies|"                                                \
                                                                                    "- Dark is Rising (Susan Cooper)|"                              \
                                                                                    "- Frank Herbert|"                                              \
                                                                                    "- Isaac Asimov|"                                               \
                                                                                    "- Isaac Asimov (Robots-Empire-Foundation)|"                    \
                                                                                    "- Larry Niven|"                                                \
                                                                                    "- Larry Niven's Known Space|"                                  \
                                                                                    "- Lensman series|"                                             \
                                                                                    "- Nebula Awards stories|"                                      \
                                                                                    "- Perry Rhodan|"                                               \
                                                                                    "- Resident Evil (S D Perry)|"                                  \
                                                                                    "- Robert Asprin|"                                              \
                                                                                    "- Romance|"                                                    \
                                                                                    "- Sector General series (James White)|"                        \
                                                                                    "- Shared Worlds|"                                              \
                                                                                    "- Short Stories|"                                              \
                                                                                    "- Space Opera|"                                                \
                                                                                    "Doctor Who|"                                                   \
                                                                                    "eZine",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Horror Miscellany",           "Horror Stories",   "- Resident Evil (S D Perry)",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Fantasy Miscellany",          "Fantasy Stories",  "- Anne McCaffrey|"                                             \
                                                                                    "- Oz|"                                                         \
                                                                                    "- Shared Worlds|"                                              \
                                                                                    "- Terry Pratchett|"                                            \
                                                                                    "eZine",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Humour Miscellany",           "Humour",           "- Terry Pratchett",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Children's Books Miscellany", "Juvenalia",        "- Isaac Asimov (Robots-Empire-Foundation)|"                    \
                                                                                    "- Oz|"                                                         \
                                                                                    "- The Three Investigators|"                                    \
                                                                                    "Doctor Who",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Classics Miscellany",         "Classics",        "",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Adventure Stories Miscellany","Adventure Stories","",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Thrillers Miscellany",        "Thrillers",        "- Dexter (Jeff Lindsay)|"                                      \
                                                                                    "- Robert Ludlum",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Mysteries Miscellany",        "Mystery Stories",  "- Agatha Christie|"                                            \
                                                                                    "- Dexter (Jeff Lindsay)|"                                      \
                                                                                    "- Donald Strachey Mysteries|"                                  \
                                                                                    "- The Three Investigators",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Biographies",                 "Biography|Autobiography",      "",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Plays",                       "Play",             "- William Shakespeare",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Poetry",                      "Poetry",           "",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Spiritualism and the Occult", "Spiritualism|Occult","",
                                                                                    False,False,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Religion",                    "Religion",         "- Spiritualism",
                                                                                    True,True,False)

    #                          collection name                  included tags        exclude collections or tags
    createMiscellanyCollection("- Non-Fiction Miscellany",      "Non-Fiction",      "- Biographies|"                                                \
                                                                                    "- Non-Fiction about Writing|"                                  \
                                                                                    "- Philosophy|"                                                 \
                                                                                    "- Philosophy of Mind|"                                         \
                                                                                    "- Politics|"                                                   \
                                                                                    "- Religion|"                                                   \
                                                                                    "- Science|"                                                    \
                                                                                    "- Spiritualism",
                                                                                    True,True,False)

    # Generic Miscellany, to mop up anything not assigned to a collection into a "Miscellany" collection
    # First, get a list of all books which are in a Kindle collection
    booksInAnyCollection = {}
    for coll in kindleC:
        for asin in kindleC[coll]["items"]:
            if asin not in booksInAnyCollection:
                booksInAnyCollection[asin] = { 'asin': asin }

    ## Don't put the kindle user guide into the Miscellany collection
    booksInAnyCollection[UserGuideAsin] = { 'asin' : UserGuideAsin }

    # Next, add any books in the Calibre database but not in a kindle collection to the miscellany collection
    startAddBooks(miscellanyCollectionName)

    for book in calibreB:

        asin = getAsin(book['lpath'])
        if asin not in booksInAnyCollection:
            AddBookToCollection(book)

    endAddBooks()


#######################
# Configurable values
#######################

# These values may be changed to suit your own preferences

# Name to use for "Miscellaneous" collection (dumping ground for everything not assigned to a collection)
miscellanyCollectionName = "- Miscellany"

# Prefix to use for author/tag/series collection names
prefixAuthor = "- "
prefixTag = "- "
prefixSeries = "- "

# Tag prefix indicating that a collection is to be created of all tagged items.
# Note that this prefix is stripped off the front of the tag name and replaced with
# the value of prefixTag to form the name of the collection
collectionTagPrefix = "-"

#########################
# Initialisation code
#########################

# Relative path to collection json file
COLLECTIONS = 'system/collections.json'
# Relative path to documents folder
DOCUMENTS = 'documents'
# Relative path to Calibre's on-kindle metadata database
CALIBRE = 'metadata.calibre'

# for the computation of the sha1 identifier
KINDEL_ABS = '/mnt/us'

# Replace these values with the actual filename and asin of the Amazon Kindle user guide on your Kindle if it's different
amazonKindleUserGuideFilename = KINDEL_ABS + "/" + DOCUMENTS + "/Kindle User s Guide-asin_B003O86FMM-type_EBOK-v_0.azw"
amazonKindleUserGuideAsin = "#B003O86FMM^EBOK"

# We don't want to put the kindle user guide into a collection as the kindle will
# just move it out again and we may well end up with an empty collection as a result.
#
# So this is set to amazonKindleUserGuideAsin if user guide found on Kindle
UserGuideAsin = ""

# Dictionaries for storing collections
kindleC = {}
calibreB = {}
calibreC = {}
allBooks = {}
booksInAuthorCollection= {}
booksInSeriesCollection= {}
booksInTagCollection= {}

# Flag indicating whether the Kindle collections have changed and need to be saved back to the device
changeMade = False;


# Dictionary for command-line options
options = {}

def setup():
    '''
    In case the script is not run in the root folder of the Kindle device, the Kindle Mount Point has to be updated because relative paths won`t work anymore
    '''
    global COLLECTIONS,CALIBRE,DOCUMENTS,options,excludedAuthors,excludedSeriesAuthors,excludedTagsAuthors,excludedSeries
    global includedAuthors,includedSeries,multiauthorSeriesExceptions

    # parse commmand line for options
    parser = OptionParser()
    parser.add_option("--na","--noauthors", action="store_false", dest="useAuthors", default=True, help="Do not use authors to create collections")
    parser.add_option("--c","--cleanupdeadfiles", action="store_true", dest="cleanupDeadFiles", default=False, help="Clean collections to remove references to files which are no longer on the kindle")
    parser.add_option("--e","--cleanupemptycollections", action="store_true", dest="cleanupEmptyCollections", default=False, help="Delete empty collections from the kindle")
    parser.add_option("--nm","--nomiscellanycollection", action="store_false", dest="useMiscellanyCollection", default=True, help="Do not put all your uncollected books into the '%s' collection"%miscellanyCollectionName )
    parser.add_option("--nr","--noreport", "--noreportcollections", action="store_false", dest="reportCollections", default=True, help="Do NOT generate a report of the collections on this Kindle")
    parser.add_option("--ns","--noseries", action="store_false", dest="useSeries", default=True, help="Do not use series to create collections")
    parser.add_option("--nt","--notags", action="store_false", dest="useTags", default=True, help="Do not use tags to create collections. By default, all tags which start with a dash (-) character are used to create collections")
    parser.add_option("--nu","--noupdate", "--noupdatecollections", action="store_false", dest="updateCollections", default=True, help="Do NOT update Kindle collections from Calibre metadata")

    parser.add_option("--rb","--rebuild", "--rebuildcollections", action="store_true", dest="rebuildCollections", default=False, help="Rebuild Kindle collections completely from scratch")
    parser.add_option("--s","--sort", "--sortcollections", action="store_true", dest="sortCollections", default=False, help="Sort ALL Kindle collections into alphabetical order by adjusting their lastAccess times")

    parser.add_option("--v","--verbose", action="store_true", dest="verboseMessages", default=False, help="Produces more messages for logging purposes")
    parser.add_option("--q","--quiet", action="store_true", dest="quietMessages", default=False, help="Suppresses output except for the reports on the collections on the Kindle and errors encountered")

    # change default for your use. use '' if u want to set it completely from command line
    parser.add_option("-m","--mnt", dest="mntPoint", default=".", help="Required if script is not run from the root folder of Kindle.")

    (options,args) = parser.parse_args()

    # Use the mount point
    COLLECTIONS = os.path.join(options.mntPoint,COLLECTIONS)
    DOCUMENTS = os.path.join(options.mntPoint,DOCUMENTS)
    CALIBRE = os.path.join(options.mntPoint,CALIBRE)

    # Check if mount path is correct
    if not (os.path.exists(CALIBRE)):
        try:
            print 'ERROR: unknown path to Kindle mounting point. Please set `%s` to correct path (e.g. /Volumes/Kindle)'%(options.mntPoint)
        except Exception, err:
            sys.stderr.write('ERROR setup: %s\n'%str(err))
        sys.exit()

    # Parse inclusions and exclusions
    excludedAuthors = excludedAuthorsCSV.split("|")
    excludedSeriesAuthors = excludedSeriesAuthorsCSV.split("|")
    excludedTagsAuthors = excludedTagsAuthorsCSV.split("|")
    excludedSeries = excludedSeriesCSV.split("|")
    includedAuthors = includedAuthorsCSV.split("|")
    includedSeries = includedSeriesCSV.split("|")
    multiauthorSeriesExceptions = multiauthorSeriesExceptionsCSV.split("|")


    # If both update and rebuild specified, default to rebuild
    if options.updateCollections and options.rebuildCollections:
        options.updateCollections = False

    # If both quiet and verbose specified, default to verbose
    if options.quietMessages and options.verboseMessages:
        options.quietMessages = False

    # When rebuilding the entire collections structure from scratch, automatically sort them
    # so they show in alphabetical order in "collections" view
    if options.rebuildCollections:
        options.sortCollections = True


###################
# Create Kindle collections from Calibre metadata
###################

def createCollectionsFromCalibre():

    global addbooksTitlesAddedList,booksInAuthorCollection,booksInSeriesCollection,booksInTagCollection

    if options.verboseMessages:
        print "[[PHASE: PLACING BOOKS IN COLLECTIONS BASED ON CALIBRE METADATA]]\n"

    for coll in calibreC:

        collType = coll['collType']

        if options.useSeries or collType != "series":

            asins = coll['asins']
            authors = coll['authors']
            collName = coll['collName']
            rawName = coll['rawName']
            titles = coll['titles']

            # Calculate collection names and minimum number of books requirement
            if collType == 'series':

                 if rawName in includedSeries:
                    minBooks = 1
                 else:
                    minBooks = minBooksForSeries

                 if len(authors) == 1:
                    cName = fixupCollname(collName + " - " + authors[0])
                 else:
                    for author in authors:
                        cName = fixupCollname(collName + " - " + author)
                        if cName in kindleC:
                            break;

                 numBooks = len(asins)
            else:
                cName = fixupCollname(collName)
                if collType == 'tag':

                    minBooks = minBooksForTag
                    numBooks = len(asins)

                elif collType == 'author':
                    if rawName in includedAuthors:
                        minBooks = 1
                        numBooks = len(asins)
                    else:
                        minBooks = minBooksForAuthor

                        numBooks = 0
                        for asin in asins:

                            countBook = True
                            book = getBookByAsin(asin)
                            if book:
                                series = book["series"]

                                if series != None:
                                    if book["series"] in excludedSeriesAuthors:
                                        countBook = False

                                    if countBook and excludeSeriesWithMultipleAuthors and series.find(" Awards") == -1:
                                        for scoll in calibreC:
                                            if scoll["collType"] == "series" and scoll["rawName"] == series and len(scoll["authors"]) > 1:
                                                countBook = False
                                                for sname in multiauthorSeriesExceptions:
                                                    if series.find(sname) != -1:
                                                        countBook = True

                                if countBook:
                                    tags = book["tags"]
                                    if tags != None:
                                        for tag in tags:
                                            if tag in excludedTagsAuthors:
                                                countBook = False

                                            if countBook and excludeAnthologiesWithMultipleAuthors and tag == "Anthology" and len(book["authors"]) > 1:
                                                countBook = False

                            if countBook:
                                numBooks = numBooks + 1

            # Add books if there are sufficient books in the collection
            if numBooks >= minBooks:

                startAddBooks(cName)

                for asin in asins:
                    AddBookToCollectionByAsin(asin)

                endAddBooks()

                # Record data about the kind of collection these books were added to
                for asin in kindleC[addbooksCollName]['items']:
                    if collType == "author":
                        booksInAuthorCollection[asin] = { 'asin': asin }
                    elif collType == "series":
                        booksInSeriesCollection[asin] = { 'asin': asin }
                    elif collType == "tag":
                        booksInTagCollection[asin] = { 'asin': asin }

            elif options.verboseMessages and numBooks != len(asins) and len(asins) > minBooks:
                try:
                    print ('COLLECTION NOT CREATED "%s" for author (has %u books, %u of which count):'%(collName,len(asins),numBooks)).encode("utf-8")
                except Exception, err:
                    sys.stderr.write('ERROR createCollectionsFromCalibre A: %s\n'%str(err))
                for asin in asins:
                    book = getBookByAsin(asin)
                    try:
                        print ('\t%s'%(getBookDesc(book))).encode("utf-8")
                    except Exception, err:
                        sys.stderr.write('ERROR createCollectionsFromCalibre B: %s\n'%str(err))

# Create a collection of all books which have the tags in tagsIncludedInCollectionCSV AND is not in a collsExcludedFromCollectionCSV collection
# AND are not already in a by-author, by-series and/or by-tags collection (as determined by the checkFoo flags: True means exclude if in set)
def createMiscellanyCollection(collName, tagsIncludedInCollectionCSV, collsExcludedFromCollectionCSV, checkAuthor, checkSeries, checkTags ):

    tagsIncludedInCollection = tagsIncludedInCollectionCSV.split("|")

    if len(tagsIncludedInCollection) > 0:

        collsExcludedFromCollection = collsExcludedFromCollectionCSV.split("|")

        startAddBooks(collName)

        for book in calibreB:

            ## Don't put the kindle user guide into the Miscellany collection
            asin = getAsin(book['lpath'])

            addMe = True;

            # Check if book is in a forbidden type of collection
            # Must satisfy all three conditions
            if checkAuthor and asin in booksInAuthorCollection:
                addMe = False;

            if addMe and checkSeries and asin in booksInSeriesCollection:
                addMe = False;

            if addMe and checkTags and asin in booksInTagCollection:
                addMe = False;

            if addMe:

                addMe = False;

                # Check tags to see if book has one of the desired tags
                tags = book['tags']
                for tag in tagsIncludedInCollection:
                    if tag in tags:
                        addMe = True

                # Check to see if book in in one of the specifically excluded collections
                # (also treat these as tags to mark items which are to be excluded)
                if addMe and len(collsExcludedFromCollection) > 0:
                    for coll in collsExcludedFromCollection:
                        if addMe and coll in tags:
                            addMe = False
                        if addMe:
                            nName = fixupCollname(coll)
                            if nName in kindleC:
                                if asin in kindleC[nName]["items"]:
                                    addMe = False

                # If book passes all criteria, add it to the collection
                if addMe:
                    AddBookToCollection(book)

        endAddBooks()


##################
# Add books to collection suite:
#
# 1) Call startAddBooks(collName), where collName is the name of the collection to which books are to be added
# 2) Call AddBookToCollection(book) or AddBookToCollectionByAsin(asin) for each book to add, where book is a calibreB strucure
# 3) When done, call endAddBooks()
##################

addbooksNumTitles = 0
addbooksTitlesAddedList = ""
addbooksCollName = ""
addbooksCollRawName = ""

# Call this before an AddBookToCollection/AddBookToCollectionByAsin sequence is used to add book(s) to a given collection
def startAddBooks(collName):

    global addbooksNumTitles,addbooksTitlesAddedList,addbooksCollName,addbooksCollRawName

    addbooksNumTitles = 0
    addbooksTitlesAddedList = "\n"
    addbooksCollName = fixupCollname(collName)
    addbooksCollRawName = addbooksCollName[0:addbooksCollName.find('@en')].encode('utf-8')

# Call this after an AddBookToCollection/AddBookToCollectionByAsin sequence to display log of changes made to the collection
def endAddBooks():

    global kindleC

    # print a description of the collection if it's been changed
    if addbooksNumTitles > 0 and addbooksCollName in kindleC:
        kindleC[addbooksCollName]['lastAccess'] = lastAccess()
        if not options.quietMessages:
            if addbooksNumTitles == 1:
                thissuffix = ""
            else:
                thissuffix = "s"
            collDesc = 'ADDED %u item%s to collection "%s"'%(addbooksNumTitles,thissuffix,addbooksCollRawName)
            if addbooksTitlesAddedList and options.verboseMessages:
                collDesc = collDesc + ':\n%s'%(addbooksTitlesAddedList)
            try:
                print collDesc.encode('utf-8')
            except Exception, err:
                sys.stderr.write('ERROR endAddBooks: %s\n'%str(err))

# Adds the specified Calibre book object to the Kindle collection using AddBookToCollectionByAsin()
def AddBookToCollection(book):
    AddBookToCollectionByAsin(getAsin(book['lpath']))

# Adds the specified book (given by asin) to the Kindle collection collName, creating that collection if it does not already exist
def AddBookToCollectionByAsin(asin):

    global KindleC,changeMade,addbooksNumTitles,addbooksTitlesAddedList

    addedMsg = ""
    book = getBookByAsin(asin)

    if not book:
        print '[[UNIDENTIFIED]] Unknown asin provided to AddBookToCollectionByAsin()'%(asin)
    else:

        # create the kindle collection if it does not exist and update access time
        if addbooksCollName not in kindleC:
            kindleC[addbooksCollName] = {'items':[], 'lastAccess':lastAccess()}
            changeMade = True
            if options.verboseMessages:
                try:
                    print 'CREATED COLLECTION "%s"'%(addbooksCollRawName.encode('utf-8'))
                except Exception, err:
                    sys.stderr.write('ERROR AddBookToCollectionByAsin: %s\n'%str(err))

        # if the document is not already in the collection we add it
        if asin not in kindleC[addbooksCollName]['items']:
            kindleC[addbooksCollName]['items'].append(asin)

            changeMade = True
            addbooksNumTitles = addbooksNumTitles + 1
            if options.verboseMessages:
                addbooksTitlesAddedList = addbooksTitlesAddedList + '\t[%s] %s\n'%(asin,getBookDesc(book))


###################
# Functions to view and manipulate (clean, sort and report on) Kindle collections
###################

# Cleanup dead files from collections and detect and remove empty collections
def cleanupCollections():

    global kindleC,changeMade

    # Now check for items in collections but no longer on the kindle
    collsToRemove = {}
    if options.cleanupDeadFiles:

        if options.verboseMessages:
            print "[[PHASE: ANALYSING DATA FROM KINDLE FILESYSTEM]]\n"

        if not options.quietMessages:
            print 'Retrieving list of all books on kindle. This may take a little while.\n'

        booksOnKindle = get_kindle_books_asins(DOCUMENTS)

        if not options.quietMessages:
            print 'Checking for any books which are in collections but are no longer on the kindle.\n'

        for coll in kindleC:
            numBooks = 0
            for asin in kindleC[coll]['items']:
                # Count books in collection and remove any books in the collection but no longer on the kindle
                if asin in booksOnKindle:
                    numBooks = numBooks + 1
                else:
                    # Never delete the Kindle user guide
                    if asin != UserGuideAsin:
                        kindleC[coll]['items'].remove(asin)
                        book = getBookByAsin(asin)
                        changeMade = True
                        try:
                            print 'Removed book asin [%s] from collection `%s`'%(asin,coll.encode('utf-8'))
                        except Exception, err:
                            sys.stderr.write('ERROR cleanupCollections A: %s\n'%str(err))
                        if book:
                            try:
                                print ("\t[%s] is %s"%(asin,getBookDesc(book))).encode('utf-8')
                            except Exception, err:
                                sys.stderr.write('ERROR cleanupCollections B: %s\n'%str(err))

            # Build list of any empty collections
            if numBooks == 0:
                collsToRemove[coll] = { 'colls': coll }

    elif options.cleanupEmptyCollections:
        # Check for empty collections
        for coll in kindleC:
            if len(kindleC[coll]['items']) == 0:
                collsToRemove[coll] = { 'colls': coll }

    # Delete any empty collections
    if not options.cleanupEmptyCollections:

        if options.verboseMessages:
            print "[[PHASE: DETECTING AND REMOVING EMPTY COLLECTIONS]]\n"

        for coll in collsToRemove:
            kindleC.pop(coll)
            try:
                print 'Deleted empty collection `%s`'%(coll[0:coll.find('@en')].encode('utf-8'))
            except Exception, err:
                sys.stderr.write('ERROR cleanupCollections C: %s\n'%str(err))
            changeMade = True


# Sort the Kindle collections alphabetically (case-insensitive)
def sortCollections():

    global kindleC,changeMade

    if options.verboseMessages:
        print "[[PHASE: SORTING KINDLE COLLECTIONS]]\n"

    # Fake collection lastAccess times so that they initially show up in alphabetical order in the kindle's "collections" view
    fakeAccess = lastAccess()
    for coll in sorted(kindleC.iterkeys(), key=lambda x: x.lower()):
        kindleC[coll]['lastAccess'] = fakeAccess
        if options.verboseMessages:
            try:
                print 'SET lastAccess to %u for collection "%s"'%(fakeAccess,coll[0:coll.find('@en')].encode('utf-8'))
            except Exception, err:
                sys.stderr.write('ERROR sortCollections: %s\n'%str(err))
        fakeAccess = fakeAccess - 1000
        changeMade = True

def getLastAccess(collName):
    return kindleC[collName]["lastAccess"]


# Output descriptions of the Kindle collections
def showCollectionsReport():

    # Produce summary report of collections on kindle
    if options.verboseMessages:
        print "[[PHASE: SUMMARY REPORT]]\n"

    collDesc = ""
    for coll in sorted(kindleC.iterkeys(), key=getLastAccess, reverse=True):
        numTitles = len(kindleC[coll]["items"])
        if numTitles == 0:
            collDesc = collDesc + "\tEMPTY COLLECTION\n"
        else:
            if numTitles == 1:
                thissuffix = ""
            else:
                thissuffix = "s"
            collDesc = collDesc + '\t%s (%u item%s)\n'%(coll[0:coll.find('@en')],numTitles,thissuffix)

    if collDesc:
        numColls = len(kindleC)
        if numColls == 1:
            thisverb = "is"
            thissuffix = ""
        else:
            thisverb = "are"
            thissuffix = "s"
        collDesc = '\n%s collection%s %s now defined on this Kindle, as visible in "By Collections" view on the Kindle home page:\n%s'%(numColls,thissuffix,thisverb,collDesc)
    else:
        collDesc = "\nThere are now no collections defined on this Kindle."

    try:
        print collDesc.encode('utf-8')
    except Exception, err:
        sys.stderr.write('ERROR showCollectionsReport A: %s\n'%str(err))

    if calibreB:

        # Produce detailed report of collections on kindle
        if options.verboseMessages:
            print "[[PHASE: DETAILED REPORT]]\n"

        collDesc = ""
        for coll in sorted(kindleC.iterkeys(), key=getLastAccess, reverse=True):
            numTitles = len(kindleC[coll]["items"])
            if numTitles == 1:
                thissuffix = ""
            else:
                thissuffix = "s"
            collDesc = collDesc + '\t%s (%u item%s)\n'%(coll[0:coll.find('@en')],numTitles,thissuffix)
            for asin in kindleC[coll]["items"]:
                book = getBookByAsin(asin)
                if book:
                    collDesc = collDesc + '\t\t%s\n'%(getBookDesc(book))
                else:
                    try:
                        print ('[[UNIDENTIFIED]] Unable to locate Calibre metadata for [%s] in collection "%s"'%(asin,coll[0:coll.find('@en')])).encode("utf-8")
                    except Exception, err:
                        sys.stderr.write('ERROR showCollectionsReport B: %s\n'%str(err))
        if collDesc:
            numColls = len(kindleC)
            if numColls == 1:
                thissuffix = ""
            else:
                thissuffix = "s"
            collDesc = '\nDetailed report for the %s collection%s now defined on this Kindle:\n%s'%(numColls,thissuffix,collDesc)
        try:
            print collDesc.encode('utf-8')
        except Exception, err:
            sys.stderr.write('ERROR showCollectionsReport C: %s\n'%str(err))


#########################
# Load and save the Kindle collections data
#########################

# Load Kindle collections structure from json file
def loadCollections():

    global kindleC

    if options.rebuildCollections:
        if options.rebuildCollections:
            print '\nNot loading existing Kindle collections: Starting from a blank slate'
    elif options.sortCollections or options.updateCollections or options.useMiscellanyCollection or options.reportCollections:
        if options.verboseMessages:
            print 'Loading existing Kindle collections'

        # Loads Kindle collections to a dictionary
        try:
            cf = open(COLLECTIONS,'r')
            kindleC = json.load(cf)
            cf.close()
        except:
            print 'WARNING loadCollections: %s could not be loaded. Creating a new version.'%(COLLECTIONS.encode('utf-8'))

# Saves kindle collections back into json file
def saveCollections():
    cf = open(COLLECTIONS,'wb')
    json.dump(kindleC,cf)
    cf.close()

    print   "\nREMINDER:\nCOLLECTIONS WILL NOT BE UPDATED UNLESS YOU RESTART YOUR KINDLE NOW!\n"    \
            "Hold the power switch for about 20 seconds and then release. Then wait 20 seconds.\n"  \
            "The screen will flash and then the device will restart.\n"                             \
            "\n"                                                                                    \
            "Alternatively, go to Home->Menu->Settings->Menu->Restart on the Kindle."


#########################
# Get data from the Kindle file system
#########################

# Gets a list of asins of all documents physically present on the kindle
def get_kindle_books_asins(root,
                    ignored_names=("metadata.db",),
                    ignore_extensions=(".jpg", ".gif", ".mbp")):
    asins = []

    def grab_file(arg, dirname, fnames):
        for short_name in fnames:
            lpath = os.path.join(dirname, short_name)
            if os.path.isfile(lpath):
                if any(short_name.endswith(ext) for ext in ignore_extensions):
                    continue
                lpath = os.path.relpath(lpath,options.mntPoint)
                asin = getAsin(lpath)
                asins.append(asin)
                book = getBookByAsin(asin)
                if options.verboseMessages:
                    try:
                        print ('%s\n\t[%s]'%(('%s/%s'%(KINDEL_ABS,lpath.replace('\\','/'))),asin)).encode('utf-8')
                    except Exception, err:
                        sys.stderr.write('ERROR get_kindle_books_asins A: %s\n'%str(err))
                    if book:
                        try:
                            print ("\tMETADATA: %s"%(getBookDesc(book))).encode('utf-8')
                        except Exception, err:
                            sys.stderr.write('ERROR get_kindle_books_asins B: %s\n'%str(err))
                if calibreB and not book:
                    try:
                        print ('[[UNIDENTIFIED]] Unable to locate Calibre metadata for file [%s] "%s"'%(asin,lpath)).encode("utf-8")
                    except Exception, err:
                        sys.stderr.write('ERROR get_kindle_books_asins C: %s\n'%str(err))

    os.path.walk(DOCUMENTS, grab_file, None)
    return asins


#########################
# Get data from the Calibre database on the Kindle
#########################

# Given an asin, identifies and returns the corresponding book metadata
def getBookByAsin(asin):

    if asin in allBooks:
        return allBooks[asin]["book"]
    return False

# Loads Calibre metadata into calibreB and creates the allBooks asin-to-book lookup table
def loadCalibre():

    global calibreB,allBooks

    try:
        cf = open(CALIBRE,'r')
        calibreB = json.load(cf)
        cf.close()

        for book in calibreB:
            asin = getAsin(book['lpath'])
            allBooks[asin] = { 'asin': asin, "book": book }

    except Exception, err:
        sys.stderr.write('ERROR loadCalibre: %s\n'%str(err))
        exit()

# Parse Calibre metadata into calibreC, a description of possible collections which could be derived directly from the raw Calibre metadata
def parseCalibreMetadata():

    global calibreC,UserGuideAsin

    UserGuideAsin = ""

    if options.verboseMessages:
        print "[[PHASE: PARSING METADATA FROM CALIBRE DATABASE]]\n"

    colls = {}

    for book in calibreB:

        title = book['title']
        authors = book['authors']
        lpath = book['lpath']
        asin = getAsin(lpath);

        # If we detect the kindle user guide, make a note of its asin so we can ignore it later on
        if asin == amazonKindleUserGuideAsin: 
            UserGuideAsin = asin
            if options.verboseMessages:
                print "Detected Kindle User Guide with asin [%s]\n"%(UserGuideAsin)
        else:

            tags = book['tags']
            series = book['series']

            # Cope with slightly corrupt metadata
            if len(authors) == 1 and authors[0].find(";") > -1:
                authors = authors[0].split(";")

            if options.verboseMessages:
                bookDesc = '\n\t%s\n\tLPATH %s\n\tASIN [%s]\n\tSERIES: "%s"\n\tTAGS: %s'%(getBookDesc(book),lpath,asin,series,",".join(tags))

            # Process authors first in case there are series or tags which have the same name as authors (such as DargonZine)
            if options.useAuthors:

                for author in authors:

                    if author not in excludedAuthors:

                        if author not in colls:
                            colls[author] = { 'rawName': author, 'titles': [], 'collName': prefixAuthor + author,'collType': 'author', 'authors': [author], 'asins': [] }

                        colls[author]['asins'].append(asin)
                        colls[author]['titles'].append(title)

                        if options.verboseMessages:
                            try:
                                print ('AUTHOR [%s]%s'%(author,bookDesc)).encode('utf-8')
                            except Exception, err:
                                sys.stderr.write('ERROR parseCalibreMetadata AUTHOR: %s\n'%str(err))

            if options.useTags:
                 for tag in tags:

                     tag = fixupNameString(tag);
                     if tag.startswith(collectionTagPrefix):

                         if tag not in colls:
                             colls[tag] = { 'rawName': tag, 'collName': prefixTag + tag[len(collectionTagPrefix):], 'collType': 'tag', 'authors': [], 'asins': [], 'titles': [] }

                         for author in authors:
                             if author not in colls[tag]['authors']:
                                 colls[tag]['authors'].append(author)

                         colls[tag]['asins'].append(asin)
                         colls[tag]['titles'].append(title)

                         if options.verboseMessages:
                            try:
                                print ('TAG [%s]%s'%(tag,bookDesc)).encode('utf-8')
                            except Exception, err:
                                sys.stderr.write('ERROR parseCalibreMetadata TAG: %s\n'%str(err))

            if options.useSeries or excludeSeriesWithMultipleAuthors:
                 if series != None:

                     series = fixupNameString(series);

                     if series not in excludedSeries:

                         if series not in colls:
                             colls[series] = { 'rawName': series, 'collName': prefixSeries + series, 'collType': 'series', 'authors': [], 'asins': [], 'titles': [] }

                         for author in authors:
                             if author not in colls[series]['authors']:
                                 colls[series]['authors'].append(author)

                         colls[series]['asins'].append(asin)
                         colls[series]['titles'].append(title)

                         if options.verboseMessages:
                            try:
                                print ('SERIES [%s]%s'%(series,bookDesc)).encode('utf-8')
                            except Exception, err:
                                sys.stderr.write('ERROR parseCalibreMetadata SERIES: %s\n'%str(err))

    calibreC = colls.values()

###################
# Low-level utility functions
###################

def getAsin(lpath):

    # Standardise lpath to kindle absolute path
    lpath = '%s/%s'%(KINDEL_ABS,lpath.replace('\\','/'));

    # Check for "rogue" asin values
    # These are cases where the getAsin() formula does not work but instead the kindle apparently extracts the asin id from the book's metadata
    if lpath == amazonKindleUserGuideFilename:
        return amazonKindleUserGuideAsin
    else:
        return '*%s'%sha1(lpath).hexdigest()        # No rogue value found, so use the generic formula to calculate the asin

# Returns a standardised display description for a book's metadata
def getBookDesc(book):
    if book:
        if options.verboseMessages:
            return '[%s] "%s" by %s'%(getAsin(book["lpath"]),book["title"]," & ".join(book["authors"]))
        else:
            return '"%s" by %s'%(book["title"]," & ".join(book["authors"]))
    return ""

# Generates and returns a lastaccess value in milliseconds
def lastAccess():
    return int(time.time()*1000)

def fixupCollname(s):
    if s.find('@en') == -1:
        return '%s@en-US'%(fixupNameString(s))
    else:
        return s

def fixupNameString(s):
    return escape(s.strip().replace('  ',' '))


#########################
# Entry point
#########################

if __name__ == '__main__':

    # Set up
    setup()

    if options.verboseMessages:
        print   "CalibreKindleCollections.py log\n"     \
                "===============================\n"     \
                "options.updateCollections %s\n"        \
                "options.rebuildCollections %s\n"       \
                "options.useMiscellanyCollection %s\n"  \
                "options.cleanupDeadFiles %s\n"         \
                "options.cleanupEmptyCollections %s\n"  \
                "options.reportCollections %s\n"  \
                "options.sortCollections %s\n"%(options.updateCollections,options.rebuildCollections,
                    options.useMiscellanyCollection,options.cleanupDeadFiles,options.cleanupEmptyCollections,
                    options.reportCollections,options.sortCollections)

    # Load existing Kindle collections
    loadCollections()

    # Load Calibre metadata
    if options.updateCollections or options.rebuildCollections or options.useMiscellanyCollection or options.reportCollections:
        loadCalibre()

    # Analyse Calibre metadata
    if options.updateCollections or options.rebuildCollections or options.useMiscellanyCollection:
        parseCalibreMetadata()

    # Add new files to the collections based on Calibre metadta
    if options.updateCollections or options.rebuildCollections:
        createCollectionsFromCalibre()

    # Create miscellany collections based on Kindle collections state and calibre metadata
    if options.useMiscellanyCollection:
        createMiscellanyCollections()

    # Cleanup Kindle collections
    if options.cleanupDeadFiles or options.cleanupEmptyCollections:
        cleanupCollections()

    # Sort Kindle collections
    if options.sortCollections:
        sortCollections()

    # Output description of Kindle collections
    if options.reportCollections:
        showCollectionsReport()

    # Save updated collections back to Kindle
    if changeMade:
        saveCollections()
    else:
        print "\nNo changes made to Kindle collections.\n"  \
                "No need to restart your kindle (assuming you've restarted it since the last time you ran this script)."

# EOF CalibreKindleCollections.py
