--[[
	This file is part of luaplot.

	luaplot is free software: you can redistribute it and/or modify
	it under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	luaplot is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public License
	along with luaplot.  If not, see <http://www.gnu.org/licenses/>.
    
    Copyright (C) 2009-2010 Lucas Hermann Negri
--]]

require('lgob.cairo')
require('lgob.pango')
require('lgob.pangocairo')

require('luaplot.utils')

---
-- Base class, that handles some basic rendering features.
luaplot.plot = {}
local plotmt = {__index = plot}

-- Blue rocks
local always_blue = luaplot.always{0, 0, 1}

---
-- Constructor.
--
-- @param args Table with the configuration
function luaplot.plot.new(args)
	local self = {}
	setmetatable(self, plotmt)
	
	self.width 		= args.width  or 400
	self.height 	= args.height or 400
	self.handler	= args.handler
	
	self.label_color= args.label_color or {0, 0, 0}
	self.label_font = args.label_font or 'Sans'
	self.label_size = args.label_size or 8
	self.label_scale= args.label_scale
	self.h_labels 	= args.h_labels
	self.v_labels 	= args.v_labels
	
	self.line_color = args.line_color or {0.5, 0.5, 0.5}
	self.line_width = args.line_width or 1
	self.line_scale = args.line_scale
	
	self.bg_color   = args.bg_color or {1, 1, 1, 1}
	self.bg_pattern = args.bg_pattern
	
	self.data 		= args.data or {}
	self.series_colors = args.series_colors or always_blue
	self.gradient	= args.gradient ~= false and true
	
	-- text extents from the labels
	self.h_labels_width		= {}
	self.h_labels_height	= {}
	self.v_labels_width		= {}
	self.v_labels_height	= {}
	
	-- starting point of the drawing, ie, point to be assumed as (0,0)
	self.h_offset 	= args.h_offset or 0
	self.v_offset 	= args.v_offset or 0
	
	-- margins, used to aux the positioning of legends
	self.right_margin = args.right_margin or 0
	self.bottom_margin = args.bottom_margin or 0
	
	-- bounds
	self.top_bound      = args.top_bound or 0
	self.bottom_bound   = args.bottom_bound or 0
	
	-- sizes of the labels
	self.h_max_width 	= 0
	self.h_max_height	= 0
	self.v_max_width	= 0
	self.v_max_height	= 0
	
	-- callbacks
	self.pre_callback = args.pre_callback
	self.pos_callback = args.pos_callback
	
	-- childs	
	self.child = args.child or {}
	
	return self	
end 

---
-- plots the bounding box (border around the plotting).
function luaplot.plot:render_bounding_box()
	local cr = self.cr
	cr:set_source_rgb(unpack(self.line_color))
	cr:set_line_width(self.line_width)
	local v2 = 2 * self.line_width
	cr:rectangle(self.line_width + self.h_offset, self.line_width + self.v_offset, self.width - v2, self.height - v2)
	cr:stroke()
end

---
-- Renders the background.
function luaplot.plot:render_background()
	local cr = self.cr
	
	if self.bg_pattern then
		cr:set_source(self.bg_pattern)
	else
		cr:set_source_rgba(unpack(self.bg_color))
	end
	
	cr:rectangle(self.h_offset, self.v_offset, self.width, self.height)
	cr:fill()
end

---
-- Draws a rounded rectangle path.
--
-- @param T Roundness of the rectangle (between 1 and 20)
function luaplot.plot:draw_rectangle(x0, y0, x1, y1, T)
	local cr = self.cr
    
    -- to handle negative values
    if(y0 > y1) then y0, y1 = y1, y0 end
	
	cr:arc(x0 + T, y0 + T, T, -math.pi, -math.pi / 2)
	cr:line_to(x1 - T, y0)
	cr:arc(x1 - T, y0 + T, T, -math.pi/ 2, 0)
	cr:line_to(x1, y1 - T)
	cr:arc(x1 - T, y1 - T, T, 0, math.pi / 2)
	cr:line_to(x0 + T, y1)
	cr:arc(x0 + T, y1 - T, T, math.pi / 2, math.pi)
	cr:line_to(x0, y0 + T)
	cr:close_path()
end

---
-- Calculates the extents of the labels.
function luaplot.plot:calc_extents()

	-- horizontal
	if self.h_labels then
		local max_width, max_height = 0, 0
		  
		for i, label in pairs(self.h_labels) do
			self.text_layout:set_text(label, -1)
			
			local width, height = self.text_layout:get_size()
			width, height = width / pango.SCALE, height / pango.SCALE
			
			self.h_labels_width[i] = width
			self.h_labels_height[i] = height
			
			if width > max_width then max_width = width end
			if height > max_height then max_height = height end
		end

		self.h_max_width = max_width
		self.h_max_height = max_height
	end
	
	-- vertical
	if self.v_labels then
		local max_width, max_height = 0, 0
		  
		for i, label in pairs(self.v_labels) do
			self.text_layout:set_text(label, -1)
			
			local width, height = self.text_layout:get_size()
			width, height = width / pango.SCALE, height / pango.SCALE
			
			self.v_labels_width[i] =  width
			self.v_labels_height[i] =  height
			
			if width > max_width then max_width = width end
			if height > max_height then max_height = height end
		end
		
		self.v_max_width = max_width
		self.v_max_height = max_height
	end
	
end

---
-- Calculates the horizontal and vertical data bounds.
function luaplot.plot:calc_bounds()
	-- pass
end

---
-- Calculates the border.
function luaplot.plot:calc_borders()
	self.h_border = self.h_offset
	self.v_border = self.v_offset
end

---
-- Clips the input.
local function clip(size)
	if size < 20 then return math.ceil(size) end
	return size - size % 5
end

-- Calculates the line and font sizes to match the plot size
function luaplot.plot:calc_sizes()
	local smallest = math.min(self.width, self.height)
	
	if self.label_scale then
		self.r_label_size = clip(self.label_size * smallest *  0.003)
	else
		self.r_label_size = self.label_size
	end
	
	if self.line_scale then
		self.r_line_width = self.line_width * smallest * 0.0025
	else
		self.r_line_width = self.line_width
	end
end

---
-- Set the sizes of the line, labels and etc.
function luaplot.plot:set_sizes()
	self.cr:set_line_width(self.r_line_width)
	self.font_desc:set_size(self.r_label_size * pango.SCALE)
	self.text_layout:set_font_description(self.font_desc)
end

---
-- Renders the axis.
function luaplot.plot:render_axis()
	local cr = self.cr
	local y = self.height - self.v_border - self.bottom_margin
	
	cr:set_source_rgb(unpack(self.line_color))
	
	cr:move_to(self.h_border, y)
	cr:line_to(self.h_border, 0)
	cr:stroke()

	cr:move_to(self.h_border, y)
	cr:line_to(self.width - self.right_margin, y)
	cr:stroke()
end

---
-- Render the plotting.
function luaplot.plot:render(x, y, w, h)
	-- pre-callback
	if self.pre_callback then self:pre_callback() end
	
	-- Prepare the cairo stuff
	self.handler:prepare(self)
	
	-- Clipping?
	if x then
		self.cr:rectangle(x, y, w, h)
		self.cr:clip()
	end
	
	-- Calculates and sets the sizes
	self:calc_sizes()
	self:set_sizes()
	
	-- Calculate the bounds
	self:calc_extents()
	self:calc_bounds()
	self:calc_borders()
	
	-- Render stuff
	self:render_background()
	self:render_bounding_box()
	self:render_labels()
	self:render_plot()
	
	-- Render the child
	for n, child in ipairs(self.child) do
		child:render() -- shared context (already clipped)
	end
	
	-- pos-callback
	if self.pos_callback then self:pos_callback() end
	
	-- Commit the changes
	self.handler:commit(self)
end
