--[[
	This file is part of luaplot.

	luaplot is free software: you can redistribute it and/or modify
	it under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	luaplot is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public License
	along with luaplot.  If not, see <http://www.gnu.org/licenses/>.
    
    Copyright (C) 2009-2010 Lucas Hermann Negri
--]]

require('luaplot.plot')

---
-- Dot line plotting class.
luaplot.dotlineplot = {}
setmetatable(luaplot.dotlineplot, {__index = luaplot.plot})
local dotlineplotmt = {__index = luaplot.dotlineplot}
local super = luaplot.plot

local always_two = luaplot.always(2)

---
-- Constructor.
function luaplot.dotlineplot.new(args)
	local self = luaplot.plot.new(args)
	setmetatable(self, dotlineplotmt)
	
	self.series_widths  = args.series_widths or always_two
	self.series_scale   = args.series_scale
	
	self.grid           = args.grid ~= false and true
	self.grid_color     = self.line_color
	
	self.axis           = args.axis ~= false and true
	self.dots           = args.dots ~= false and true
	self.h_label_angle  = args.h_label_angle or 1.2566370614359 -- math.pi / 2.5
	
	return self
end

---
-- Calculates the bounds of the plotting.
function luaplot.dotlineplot:calc_bounds()
	-- horizontal bounds
	local max = 1
		
	for i, series in pairs(self.data) do
		if #series > max then max = #series end
	end
		
	self.h_bounds = {0, max}
	
	-- vertical bounds
	local max = self.top_bound
    local min = self.bottom_bound
		
	for i, series in pairs(self.data) do
		for j, item in pairs(series) do
			if item > max then max = item end
			if item < min then min = item end
		end
	end
		
	self.v_bounds = {min, max}
end

---
-- Calculates the font extents.
function luaplot.dotlineplot:calc_extents()
	super.calc_extents(self)
	if self.v_max_height < 10 then self.v_max_height = 10 end 
end

---
-- Calculates the sizes of plotting elements.
function luaplot.dotlineplot:calc_sizes(use_margin)
	super.calc_sizes(self)
	
	if self.series_scale then
		local t = {}
		local factor = math.min(self.height, self.width) / 400
		
		for i = 1, #self.data do
			t[i] = self.series_widths[i] * factor
		end
		
		self.r_series_widths =  t
	else
		self.r_series_widths = self.series_widths
	end
end

---
-- Calculates the border.
function luaplot.dotlineplot:calc_borders()
	self.h_border = self.h_offset + self.v_max_width + 20
	self.v_border = self.v_offset + self.h_max_width + 20
end

---
-- Draws the labels of the pieces.
function luaplot.dotlineplot:render_labels()
	local cr, grid  = self.cr, self.grid 
	local width     = self.width - self.right_margin
	local height    = self.height - self.bottom_margin
	
	-- horizontal
	if self.h_labels then
        local step = (width - self.h_border) / #self.h_labels
        local x, angle = self.h_border, self.h_label_angle 
        
		for i, item in ipairs(self.h_labels) do
            cr:move_to(x + self.h_labels_height[i] / 2, height - self.v_border + 10)
			
			cr:rotate(angle)
			cr:set_source_rgb( unpack(self.label_color) )
			self.text_layout:set_text(item, -1)
			pangocairo.show_layout(cr, self.text_layout)
			cr:rotate(-angle)
			
			if grid and x ~= self.h_border then
                cr:set_source_rgb( unpack(self.grid_color) )
                cr:move_to(x, height - self.v_border)
                cr:line_to(x, 0)
                cr:stroke()
			end
			
			x = x + step
		end
	end
	
	-- vertical
	if self.v_labels then
		local height    = height - self.v_max_height
		local step      = (height - self.v_border) / (#self.v_labels - 1)
		local y         = self.v_max_height
		local count     = #self.v_labels
		
		for i = count, 1, -1 do
			local item = self.v_labels[i]
			cr:set_source_rgb(r, g, b)
			cr:move_to(self.h_border - self.v_labels_width[i] - 5, y - self.v_labels_height[i] / 2)
			
			cr:set_source_rgb( unpack(self.label_color) )
			self.text_layout:set_text(item, -1)
			pangocairo.show_layout(cr, self.text_layout)
			
			if grid then
				cr:set_source_rgb( unpack(self.grid_color) )
				cr:move_to(self.h_border, y)
				cr:line_to(width, y)
				cr:stroke()
			end
			
			y = y + step
		end
	end
end

---
-- Renders the plot.
function luaplot.dotlineplot:render_plot()
	if self.axis then self:render_axis() end
	
	local width     = self.width	- self.h_border - self.right_margin
	local height    = self.height   - self.v_border - self.v_max_height - self.bottom_margin
	local top       = self.height	- self.v_border - self.bottom_margin
    
	local amplitude = self.v_bounds[2] - self.v_bounds[1]
	local h_step    = width / self.h_bounds[2]
	local v_step    = height / amplitude
	local cr, dots  = self.cr, self.dots
	
	local angle = 2.1 * math.pi
	
	for i, series in ipairs(self.data) do
		cr:set_source_rgb( unpack(self.series_colors[i]) )
		local x, last = self.h_border
		local w = self.r_series_widths[i]
		
		for j, value in pairs(series) do
			if last then
				cr:move_to(x - h_step, top - (last - self.v_bounds[1]) * v_step)
				cr:line_to(x, top - (value - self.v_bounds[1]) * v_step)
				cr:set_line_width(w)
				cr:stroke()	
			end
			
			if dots then
				cr:new_path()
				cr:arc(x, top - (value - self.v_bounds[1]) * v_step, w * 2, 0, angle)
				cr:close_path()
				cr:fill()
			end
			
			x = x + h_step
			last = value
		end
	end
end
