#!/media/mmcblk0p1/Programs/_lua/bin/lua

----------------------------------------------------------------------
-- run_debug.lua: run & debug programs in iRex DR series eReaders
--
-- Copyright (C) 2010  Iñigo Serna <inigoserna@gmail.com>
-- Released under GPL v3+
--
-- Time-stamp: <2010-05-19 02:22:18 inigo>
----------------------------------------------------------------------


require "lgob.gtk"
require "lgob.gdk"


----------------------------------------------------------------------
-- constants
local INITIAL_PATH = "/media/mmcblk0p1/Programs"
local LUA_PATH = "/media/mmcblk0p1/Programs/_lua/bin/lua"
local PYTHON_PATH = "/media/mmcblk0p1/Programs/_python26/bin/python"
local SHELL_PATH = "/bin/bash"
local LUADIR = "/media/mmcblk0p1/Programs/_lua"
local OUTPUT_FILE_PATH = "/media/mmcblk0p1/"
local FILETYPES = { lua='Lua script', py='Python script', sh='Shell script',
                    desktop='Desktop file', other='Generic application' }
local DEFAULT_STYLE = [[
      style "edit-style" { font_name = "mono 7" }
      class "GtkTextView" style "edit-style"
]]
local DOC = [[
<i>This application lets run some other programs and get the output. It is specially useful to execute command line based scripts or to check why some applications don't start properly.

It understands some special type of files like lua, python or shell scripts. Even it can run the proper command if you select a .desktop file!

To run:
- select the program or script to run
- select the path where you want to start (optional)
- specify some arguments to be passed (optional)
- click on "execute"

... and that's it!</i>


(C) 2010, Iñigo Serna  &lt;inigoserna@gmail.com&gt;
Released under GPL v3 or later.

Programmed in lua language with marvelous lgob gtk+ bindings.
]]


----------------------------------------------------------------------
-- global vars
local fbtnProgram, fbtnPath, eArgs, lFileType, btnRun, btnSave, btnHelp, lCmd, tvOutputBuffer
local program = "(no selected)"
local path = INITIAL_PATH
local args = ""
local cmd = nil
local filetype = nil


----------------------------------------------------------------------
-- gtk+ helper functions
local mygtk = {
    -- create a label with markup
    Markup_new = function(txt)
                     local label = gtk.Label.new()
                     label:set_markup(txt)
                     return label
                 end,
    -- add alignment and padding to a widget
    align = function(widget, xalign, pad)
                xalign = xalign or 0
                pad = pad or {0, 0, 0, 0}
                local alignment = gtk.Alignment.new()
                alignment:set("xalign", xalign)
                alignment:set("yalign", 0.5)
                alignment:set_padding(pad[1], pad[2], pad[3], pad[4])
                alignment:add(widget)
                return alignment
            end
}

-- utils
function extract_exec_from_desktop(desktopfile)
    local f, err, errno = io.open(desktopfile, "r")
    local exec
    if f then
        while true do
            local line = f:read("*l")
            if line == nil then break end
            _, _, exec = line:find("Exec=(.+)$")
            if exec ~= nil then break end
        end
        f:close()
    end
    return exec
end

function ui_update_output_data()
    local prog, prog_str = "", ""
    if filetype ~= nil then
        if filetype == "lua" then
            prog = LUA_PATH .. " " .. program
            prog_str = "[lua] " .. program
        elseif filetype == "py" then
            prog = PYTHON_PATH .. " " .. program
            prog_str = "[python] " .. program
        elseif filetype == "sh" then
            prog = SHELL_PATH .. " " .. program
            prog_str = "[shell] " .. program
        elseif filetype == "desktop" then
            prog = extract_exec_from_desktop(program)
            prog_str = prog
        else
            prog = program
            prog_str = prog
        end
        cmd = string.format('cd "%s"; %s %s 2>&1', path, prog, args)
        local cmd_str = string.format('# cd %s\n# %s %s', path, prog_str, args)
        lCmd:set_markup("<span size='small' style='italic'>" .. cmd_str .. "</span>")
        tvOutputBuffer:set_text("-- No output --", -1)
        btnSave:set("sensitive", false)
    end
end


----------------------------------------------------------------------
-- callbacks
function cb_quit()
    fbtnProgram:grab_focus()
    gtk.main_quit()
end

function cb_program_changed()
    local oldprogram = program
    program = fbtnProgram:get_filename()
    if program ~= nil and program ~= oldprogram then
        for ext in program:gmatch("%.(%w+)") do filetype = ext end
        ftype = FILETYPES[filetype] or FILETYPES['other']
        lFileType:set_markup("<span style='italic'>" .. ftype .. "</span>")
        btnRun:set("sensitive", true)
        ui_update_output_data()
    end
end

function cb_path_changed()
    path = fbtnPath:get_filename() or INITIAL_PATH
    ui_update_output_data()
end

function cb_args_changed()
    args = eArgs:get_text() or ""
    ui_update_output_data()
end

function cb_run(win)
    btnRun:set("sensitive", false)
    local prog = ""
    if filetype == "Lua script" then
        prog = LUA_PATH .. " " .. program
    elseif filetype == "Desktop file" then
        prog = extract_exec_from_desktop(program)
    else
        prog = program
    end
    tvOutputBuffer:set_text("Please wait while executing", -1)
    -- win:get("window"):set_cursor(gdk.Cursor.new(gdk.WATCH))
    while gtk.events_pending() do
        gtk.main_iteration()
    end
    local f = io.popen(cmd)
    local buf = f:read("*a"):gsub("^%s*(.-)%s*$", "%1")
    f:close()
    if buf ~= "" then
        tvOutputBuffer:set_text(buf, -1)
        btnSave:set("sensitive", true)
    else
        tvOutputBuffer:set_text("-- No output --", -1)
    end
    btnRun:set("sensitive", true)
    -- win:get("window"):set_cursor(gdk.Cursor.new(gdk.LEFT_PTR))
end

function cb_save(win)
    local istart, iend = gtk.TextIter.new(), gtk.TextIter.new()
    tvOutputBuffer:get_bounds(istart, iend)
    local buf = tvOutputBuffer:get_text(istart, iend)
    local fname = ""
    for txt in program:gmatch("%/([^%/]+)") do fname = txt end
    local fname = OUTPUT_FILE_PATH .. fname .. '-' .. os.date('%Y%m%d%H%M%S') .. '.log'
    local f = io.open(fname, 'w')
    f:write(buf)
    f:close()
    local dlg = gtk.MessageDialog.new(win, gtk.DIALOG_MODAL, gtk.MESSAGE_INFO,
                                      gtk.BUTTONS_OK, "Output saved")
    dlg:set_markup('<span size="large" weight="bold">Output saved!</span>\n\n' ..
                   'in file <i>"' .. fname .. '"</i>')
    dlg:run()
    dlg:destroy()
    btnSave:set("sensitive", false)
end

function cb_help(win)
    local dlg = gtk.MessageDialog.new(win, gtk.DIALOG_MODAL, gtk.MESSAGE_INFO,
                                      gtk.BUTTONS_OK, "Run &amp; Debug help")
    dlg:set_markup('<span size="large" weight="bold">Run &amp; Debug help</span>\n\n' .. DOC)
    dlg:run()
    dlg:destroy()
end


----------------------------------------------------------------------
-- UI
function ui_init()
    local win = gtk.Window.new()
    win:set("title", "Run & Debug", "border-width", 20)
    win:connect("delete-event", cb_quit)
    local vbox = gtk.VBox.new(false, 0)

    -- header: title and quit button
    local hbox = gtk.HBox.new(false, 10)
    local evbox = gtk.EventBox.new()
    evbox:modify_bg(gtk.STATE_NORMAL, gdk.color_parse("black"))
    local title = mygtk.Markup_new("<span size='xx-large' color='white' weight='bold'>Run &amp; Debug</span>")
    evbox:add(title)
    hbox:pack_start(evbox, true, true, 0)
    local evbox_quit = gtk.EventBox.new()
    evbox_quit:set_events(gdk.BUTTON_PRESS_MASK)
    evbox_quit:connect("button-press-event", cb_quit)
    local img = gtk.Image.new_from_file(LUADIR .. "/share/pixmaps/quit.png")
    evbox_quit:add(img)
    hbox:pack_end(evbox_quit, false, false, 0)
    vbox:pack_start(hbox, false, false, 10)

    -- table: program, path and arguments
    local table = gtk.Table.new(3, 2, false)
    table:set_row_spacings(10)
    table:set_col_spacings(10)
    local label = mygtk.Markup_new("<b>Program:</b>")
    table:attach(mygtk.align(label), 0, 1, 0, 1, gtk.FILL, gtk.FILL, 0, 0)
    fbtnProgram = gtk.FileChooserButton.new('Select a program to run',
                                            gtk.FILE_CHOOSER_ACTION_OPEN)
    fbtnProgram:set_current_folder(INITIAL_PATH)
    fbtnProgram:connect("selection-changed", cb_program_changed)
    table:attach(fbtnProgram, 1, 2, 0, 1, gtk.EXPAND+gtk.FILL, gtk.EXPAND+gtk.FILL, 0, 0)
    local label = mygtk.Markup_new("<b>Path:</b>")
    table:attach(mygtk.align(label), 0, 1, 1, 2, gtk.FILL, gtk.FILL, 0, 0)
    fbtnPath = gtk.FileChooserButton.new('Select the path',
                                         gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER)
    fbtnPath:connect("selection-changed", cb_path_changed)
    fbtnPath:set_current_folder(INITIAL_PATH)
    table:attach(fbtnPath, 1, 2, 1, 2, gtk.EXPAND+gtk.FILL, gtk.EXPAND+gtk.FILL, 0, 0)
    local label = mygtk.Markup_new("<b>Arguments:</b>")
    table:attach(mygtk.align(label), 0, 1, 2, 3, gtk.FILL, gtk.FILL, 0, 0)
    eArgs = gtk.Entry.new()
    eArgs:connect("notify::text", cb_args_changed)
    table:attach(eArgs, 1, 2, 2, 3, gtk.EXPAND+gtk.FILL, gtk.EXPAND+gtk.FILL, 0, 0)
    vbox:pack_start(table, false, false, 10)

    -- hbox: filetype, execute button
    local hbox = gtk.HBox.new(false, 35)
    local label = mygtk.Markup_new("<b>File type:</b>")
    hbox:pack_start(label, false, false, 0)
    lFileType = mygtk.Markup_new("<span style='italic'>(no file selected)</span>")
    hbox:pack_start(lFileType, false, false, 0)
    btnHelp = gtk.Button.new_from_stock("gtk-help")
    btnHelp:connect("clicked", cb_help, win)
    hbox:pack_end(btnHelp, false, false, 0)
    btnSave = gtk.Button.new_from_stock("gtk-save")
    btnSave:set("sensitive", false)
    btnSave:connect("clicked", cb_save, win)
    hbox:pack_end(btnSave, false, false, 0)
    btnRun = gtk.Button.new_from_stock("gtk-execute")
    btnRun:set("sensitive", false)
    btnRun:connect("clicked", cb_run, win)
    hbox:pack_end(btnRun, false, false, 0)
    vbox:pack_start(hbox, false, false, 5)

    -- output header
    local hbox = gtk.HBox.new(false, 10)
    local label = mygtk.Markup_new("<span size='large' weight='bold'>Output of:</span>")
    hbox:pack_start(label, false, false, 0)
    lCmd = gtk.Label.new()
    hbox:pack_start(lCmd, false, false, 0)
    vbox:pack_start(hbox, false, false, 5)

    -- output
    local scrollwin = gtk.ScrolledWindow.new()
    scrollwin:set("hscrollbar-policy", gtk.POLICY_AUTOMATIC,
                  "vscrollbar-policy", gtk.POLICY_AUTOMATIC,
                  "shadow-type", gtk.SHADOW_IN)
    local tvOutput = gtk.TextView.new()
    tvOutput:set("editable", false, "cursor-visible", false,
                 "left-margin", 10, "right-margin", 10)
    tvOutputBuffer = tvOutput:get("buffer")
    tvOutputBuffer:set_text("-- No output --", -1)
    scrollwin:add(tvOutput)
    vbox:pack_start(scrollwin, true, true, 10)

    -- show window
    win:add(vbox)
    win:fullscreen()
    win:show_all()
end


----------------------------------------------------------------------
-- main
gtk.rc_parse_string(DEFAULT_STYLE)
ui_init()
gtk.main()


----------------------------------------------------------------------
-- TODO:
--   . BUG: clicking on arguments and quitting let the keyboard visible
