#!/media/mmcblk0p1/Programs/_lua/bin/lua

----------------------------------------------------------------------
-- icons.lua:  in iRex DR series eReaders
--
-- Copyright (C) 2010  Iñigo Serna <inigoserna@gmail.com>
-- Released under GPL v3+
--
-- Time-stamp: <2010-12-14 00:07:18 inigo>
----------------------------------------------------------------------


require("ldbi.sqlite")
require("lgob.gdk")

require "lxp"     -- luaexpat
require "pl"      -- penlight
stringx.import()  -- pl.stringx
require "base64"  -- base64

require("print_r")


----------------------------------------------------------------------
----- Constants
local DBFILE = "/media/mmcblk0p1/global.db"
-- local DBFILE = "/media/ARCHIVE/BackUps/dr800_sd/global.db"
local SMALL, MEDIUM = 60, 120
local ARGS = {programs=false, fb2_files=false}


----------------------------------------------------------------------
----- Global Variables
local conn
local files


----------------------------------------------------------------------
----- Database

-- open database
function db_open(db_file)
    local err, msg
    conn, err, msg = sqlite.connect({ ["dbname"] = db_file })
    if not conn then
        print(msg)
        os.exit(1)
    end
end


-- close database
function db_close()
    conn:close()
end


-- base get items from database
function db_get_items(sql_stmt, prog)
    local cur, err, msg, row
    local items, num = {}, 0
    cur, err, msg = conn:execute(sql_stmt)
    row = cur:fetch({})
    while row do
        local fullpath = row[3] .. '/' .. row[2]
        local img_file = prog(fullpath)
        if img_file then
            items[row[1]] = {id=row[1], filename=row[2], path=row[3], img_file=img_file}
            num = num + 1
        else
            print(fullpath, ': skipped, image not found')
        end
        row = cur:fetch(row)
    end
    cur:close()
    return items, num
end


-- get applications in Programs folder, only those with icon defined
function db_get_programs()
    sql_stmt = [[SELECT * FROM file_metadata WHERE file_type == "desktop" AND directory_path LIKE "/media/mmcblk0p1/Programs%"]]
    return db_get_items(sql_stmt, program_get_icon)
end


-- get fb2 files, only those with cover image
function db_get_fb2_files()
    sql_stmt = [[SELECT * FROM file_metadata WHERE file_type == "fb2"]]
    return db_get_items(sql_stmt, fb2_file_get_image)
end


-- update icons
function db_insert_or_update_icons(id, pix_small, pix_medium)
    local cur, err, msg, row
    cur, err, msg = conn:execute([[SELECT * FROM thumbnails WHERE file_id == $1 ]], 1, {id})
    row = cur:fetch({})
    cur:close()
    if row then -- icon exits => update
        conn:execute([[UPDATE thumbnails SET thumb_data_small = $1, thumb_data_medium = $2 WHERE file_id = $3]],
                     3, {sqlite.binary(pix_small), sqlite.binary(pix_medium), id})
    else        -- icon doesn't exist => insert
        conn:execute([[INSERT INTO thumbnails("file_id", "thumb_data_mini", "thumb_data_small", "thumb_data_medium", "thumb_data_large") VALUES ($1, $2, $3, $4, $5)]], 5,
                     {id, nil, sqlite.binary(pix_small), sqlite.binary(pix_medium), nil})
    end
end


----------------------------------------------------------------------
----- Utils

-- save data into file
function file_save(filename, data)
    local f = io.open(filename, "wb")
    f:write(data)
    f:close()
end


-- get icon path from program.desktop file
function program_get_icon(fullpath)
    local f, err, errno = io.open(fullpath, "r")
    if f then
        local buf = f:read("*a")
        f:close()
        local _, _, icon = buf:find("Icon=(.+\.png)")
        return icon
    else
        return nil
    end
end


-- get cover image from a fb2 file
function fb2_file_get_image(fullpath)
    local cover_id, first_img_id = nil, nil
    local images = {}

    function fb2_images(filename)
        local VALID_TYPES = List({"image/jpg", "image/jpeg", "image/png", "image/bmp"})
        local in_coverpage = false
        local binary = nil
        local callbacks = {
            StartElement = function (parser, elm, attrs)
                if in_coverpage and elm == "image" then
                    cover_id = attrs["l:href"]
                    if cover_id:startswith('#') then
                        cover_id = cover_id:sub(2, #cover_id)
                    end
                end
                if elm == "coverpage" then
                    in_coverpage = true
                elseif elm == "binary" then
                    binary = {id=attrs["id"], type=attrs["content-type"]}
                end
            end,
            EndElement = function (parser, elm)
                if elm == "coverpage" then
                    in_coverpage = false
                elseif elm == "binary" then
                    binary = nil
                end
            end,
            CharacterData = function (parser, buf)
                if binary and VALID_TYPES:contains(binary["type"]) then
                    if not images[binary["id"]] then
                        if not first_img_id then
                            first_img_id = binary["id"]
                        end
                        images[binary["id"]] = base64.decode(buf)
                    end
                end
            end
        }

        local f = io.open(filename, "r")
        local buf = f:read("*a")
        f:close()
        local p = lxp.new(callbacks)
        ok, errmsg, line, col, pos = p:parse(buf)
        if not ok then
            utils.printf("Error parsing %s:%d:%d, %s\n", filename, line, col, errmsg)
        else
            p:parse()
            p:close()
        end
    end

    fb2_images(fullpath)
    coverfile = fullpath.."-cover.img"
    if cover_id then
        img = images[cover_id]
    elseif first_img_id then
        img = images[first_img_id]
    else
        return nil
    end
    if img then
        file_save(coverfile, img)
    	return coverfile
    else
	return nil
    end
end


-- for each app, get icon, resize and insert into database
function create_and_add_icons(files, mode)
    for id, data in pairs(files) do
        local err = true
        local pixbuf_small, pixbuf_medium, res1, pix_small, res2, pix_medium
        utils.printf('Adding icon for %s [%d]\n', data.filename, id)
        -- get images and resize
        pixbuf_small = gdk.Pixbuf.new_from_file_at_size(data.img_file, SMALL, SMALL)
        pixbuf_medium = gdk.Pixbuf.new_from_file_at_size(data.img_file, MEDIUM, MEDIUM)
        if mode == "fb2_files" then
            os.remove(data.img_file)
        end
        if pixbuf_small and pixbuf_medium then
            -- print(id, 'x', pixbuf_small, pixbuf_medium)
            res1, pix_small, _ = gdk.Pixbuf.save_to_buffer(pixbuf_small, 'png')
            res2, pix_medium, _ = gdk.Pixbuf.save_to_buffer(pixbuf_medium, 'png')
            -- print (id, 'y', res1, pix_small, res2, pix_medium)
            if res1 and res2 then
                db_insert_or_update_icons(id, pix_small, pix_medium)
                err = false
            end
        end
        if err then
            utils.printf('Error adding icon for %s [%d]\n', data.filename, id)
        end
    end
end



----------------------------------------------------------------------
----- Main

print("\nGenerate icons & covers")
print("=======================")
-- parse command line arguments
for _, elm in ipairs(arg) do
    if elm == "--all" or elm == "-a" then
        ARGS = {programs=true, fb2_files=true}
        break
    elseif elm == "--programs" then
        ARGS["programs"] = true
    elseif elm == "--fb2" then
        ARGS["fb2_files"] = true
    else
        utils.printf("Error: invalid argument <%s>\n", elm)
    end
end

-- open database and analyze files
db_open(DBFILE)
if ARGS["programs"] then
    files, num_files = db_get_programs()
    utils.printf("=> Analyzing Programs: %d found\n", num_files)
    create_and_add_icons(files, 'programs')
end
if ARGS["fb2_files"] then
    files, num_files = db_get_fb2_files()
    utils.printf("=> Analyzing FB2 files: %d found\n", num_files)
    create_and_add_icons(files, 'fb2_files')
end
db_close()
