--[[
	This file is part of luaplot.

	luaplot is free software: you can redistribute it and/or modify
	it under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	luaplot is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public License
	along with luaplot.  If not, see <http://www.gnu.org/licenses/>.
    
    Copyright (C) 2009-2010 Lucas Hermann Negri
--]]

require('luaplot.plot')

---
-- Pie plotting class.
luaplot.pieplot = {}
setmetatable(luaplot.pieplot, {__index = luaplot.plot})
local pieplotmt = {__index = luaplot.pieplot}
local super = luaplot.plot

---
-- Constructor.
function luaplot.pieplot.new(args)
	local self = luaplot.plot.new(args)
	setmetatable(self, pieplotmt)
	
	self.shadow = args.shadow ~= false and true
	
	return self
end

---
-- Calculates the sizes of plotting elements.
function luaplot.pieplot:calc_sizes(use_margin)
	super.calc_sizes(self)
	
	local width, height = self.width, self.height
	
	if use_margin then
		width = width - self.right_margin
		height = height - self.bottom_margin
	end
	
	self.center_x   = width / 2
	self.center_y   = height / 2
	self.radius     = math.min(width / 3, height / 3)
	self.shadow_size= self.shadow and 5 * self.radius * 0.005 or 0
end

---
-- Calculates the bounds of the plotting.
function luaplot.pieplot:calc_bounds()
	local total = 0
	
	for i, v in pairs(self.data) do
		total = total + v
    end
	
	self.total = total
end

---
-- Draws a pie piece path.
function luaplot.pieplot:draw_piece(angle, next_angle)
	local cr = self.cr
	
	cr:move_to(self.center_x, self.center_y)
	cr:line_to(self.center_x + self.radius * math.cos(angle), self.center_y + self.radius * math.sin(angle))
	cr:arc(self.center_x, self.center_y, self.radius, angle, next_angle)
	cr:line_to(self.center_x, self.center_y)
	cr:close_path()
end

---
-- Draws the labels of the pieces.
function luaplot.pieplot:render_labels()
	-- OK background rendered, now recalc the sizes using the margins
	self:calc_sizes(true)

	if not self.v_labels then return end
	
	local angle, next_angle = 0, 0
	local x0, y0 = self.center_x, self.center_y
	local cr = self.cr
	local r = self.radius * 1.1
	
	for n, key in ipairs(self.v_labels) do
		next_angle = angle + 2 * math.pi * self.data[n] / self.total
		cr:set_source_rgb(unpack(self.series_colors[n]))
		local tmp = (angle + next_angle) / 2
		
		if tmp < math.pi / 2 or tmp > 3 * math.pi / 2 then
			cr:move_to(x0 + r * math.cos(tmp) + self.shadow_size, y0 + r * math.sin(tmp) - self.v_labels_height[n] / 2)
		else
			cr:move_to(x0 + r * math.cos(tmp) - self.v_labels_width[n] - self.shadow_size, y0 + r * math.sin(tmp) - self.v_labels_height[n] / 2)
		end
		
		self.text_layout:set_text(key, -1)
		pangocairo.show_layout(cr, self.text_layout)
		
		angle = next_angle 
	end
end

---
-- Renders a shadow in the pie border.
function luaplot.pieplot:render_shadow()
	self.cr:set_source_rgba(0, 0, 0, 0.5)
	self.cr:arc(self.center_x + self.shadow_size, self.center_y + self.shadow_size, self.radius, 0, 2 * math.pi)
	self.cr:fill()
end

---
-- Renders the plot.
function luaplot.pieplot:render_plot()
	if self.shadow then self:render_shadow() end
	
	local angle, next_angle = 0, 0
	local x0, y0 = self.center_x, self.center_y
	
	local cr, gr, g_color = self.cr, self.gradient
	
	for n, series in ipairs(self.data) do
		next_angle = angle + 2 * math.pi * series / self.total
		
		if gr then
			g_color = cairo.Pattern.create_radial(self.center_x, self.center_y, 0, self.center_x, self.center_y, self.radius)
			local r, g, b = unpack(self.series_colors[n])
			g_color:add_color_stop_rgb(0.3, r, g, b, a)
			g_color:add_color_stop_rgb(1, r * 0.7, g * 0.7, b * 0.7)
			cr:set_source(g_color)
		else		
			cr:set_source_rgb(unpack(self.series_colors[n]))
		end
		
		self:draw_piece(angle, next_angle)
		cr:fill()
		
		cr:set_source_rgb(1, 1, 1)
		self:draw_piece(angle, next_angle)
		cr:stroke()
		
		if g_color then
			g_color:destroy()
			g_color = nil
		end
		
		angle = next_angle
	end
end
