--[[
	This file is part of luaplot.

	luaplot is free software: you can redistribute it and/or modify
	it under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	luaplot is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public License
	along with luaplot.  If not, see <http://www.gnu.org/licenses/>.
    
    Copyright (C) 2009-2010 Lucas Hermann Negri
--]]

require('luaplot.plot')

---
-- Bar plotting class.
luaplot.barplot = {}
setmetatable(luaplot.barplot, {__index = luaplot.plot})
local barplotmt = {__index = luaplot.barplot}

---
-- Constructor.
function luaplot.barplot.new(args)
	local self = luaplot.plot.new(args)
	setmetatable(self, barplotmt)
	
	self.grid       = args.grid ~= false and true
	self.round_rect = args.round_rect ~= false and true
	self.axis       = args.axis ~= false and true
	self.grid_color = args.grid_color or {0.7, 0.7, 0.7, 1}
	self.spacing    = args.spacing or 0.4
	
	return self
end

---
-- Draws a grid.
function luaplot.barplot:render_grid()
	local cr = self.cr
	local height = self.height - self.bottom_margin
	cr:set_source_rgb(unpack(self.grid_color))
	
	local lines     = self.v_labels and #self.v_labels or 10
	local v_step    = (height - 2 * self.v_border ) / (lines -1)
	local y         = self.v_border
	
	for x = 1, lines do
		cr:move_to(self.h_border, y)
		cr:line_to(self.width, y)
		cr:stroke()
		y = y + v_step
	end
end

---
-- Calculates the horizontal and vertical data bounds.
function luaplot.barplot:calc_bounds()
	-- horizontal bounds
	self.h_bounds = {0, #self.data}
	
	-- vertical bounds
	local max = self.top_bound
	local min = self.bottom_bound
		
	for i, series in pairs(self.data) do
		for j, item in pairs(series) do
			if item > max then max = item end
			if item < min then min = item end
		end
	end
		
	self.v_bounds = {min, max}
end

---
-- Calculates the border.
function luaplot.barplot:calc_borders()
	self.h_border = self.h_offset + self.v_max_width + 10
	self.v_border = self.v_offset + 2 * self.h_max_height
end

---
-- Renders both horizontal and vertical labels, if they exists.
function luaplot.barplot:render_labels()
	local cr        = self.cr
	cr:set_source_rgb(unpack(self.label_color))
	local width     = self.width - self.right_margin
	local height    = self.height - self.bottom_margin
	
	-- horizontal
	if self.h_labels then
		local step  = (width - self.h_border) / #self.h_labels
		local x     = self.h_border + step / 2
		
		for i, item in ipairs(self.h_labels) do
			cr:move_to(x - self.h_labels_width[i] / 2, height - self.v_border + 3)
			
			self.text_layout:set_text(item, -1)
			pangocairo.show_layout(cr, self.text_layout)
			x = x + step
		end
	end
	
	-- vertical
	if self.v_labels then
		local step  = (height - 2 * self.v_border) / (#self.v_labels - 1)
		local y     = self.v_border
		local count = #self.v_labels
		
		for i = count, 1, -1 do
			local item = self.v_labels[i]
			cr:move_to(self.h_border - self.v_labels_width[i] - 5, y - self.v_labels_height[i] / 2)
			
			self.text_layout:set_text(item, -1)
			pangocairo.show_layout(cr, self.text_layout)
			y = y + step
		end
	end
end

---
-- Renders the bars.
function luaplot.barplot:render_plot()
	if self.grid then self:render_grid() end
	if self.axis then self:render_axis() end

	local cr        = self.cr
	local plot_w    = self.width - self.h_border - self.right_margin
	local plot_h    = self.height - self.bottom_margin - (2 * self.v_border) 
	
	local amplitude = self.v_bounds[2] - self.v_bounds[1]	
	local h_step    = plot_w / #self.data
	local v_step    = plot_h / amplitude	
	local rc        = self.round_rect
	local gr        = self.gradient
    local y0        = self.v_border + self.v_bounds[1] * v_step
	
	-- for each series (each series have the same space allocation)
	for i, series in ipairs(self.data) do
		i = i - 1
		local i_step = h_step / (#series + self.spacing)
		local x0 = self.h_border + i * h_step + 0.2 * i_step
		local linear, r, g, b, rn
	
		-- for each series item (items from the same series have the same space
		-- allocation, but itens from different series can have different space
		-- allocations
		for n, key in ipairs(series) do
			local tmp = key * v_step / 2
			
			if gr then
				linear = cairo.Pattern.create_linear(x0, tmp, x0 + i_step, tmp)
				r, g, b = unpack(self.series_colors[n])
				linear:add_color_stop_rgb(0, 6 * r / 10, 6 * g / 10, 6 * b / 10)
				linear:add_color_stop_rgb(1, r, g, b)
				cr:set_source(linear)
			else
				cr:set_source_rgb(unpack(self.series_colors[n]))
			end
			
			if rc and key ~= 0 then
				rn = i_step / 10
				self:draw_rectangle(x0, y0 + (amplitude - key) * v_step, x0 + i_step, y0 + amplitude * v_step, rn)
			else
				cr:rectangle(x0, y0 + (amplitude - key) * v_step, i_step, key * v_step)
			end
			
			cr:fill()
			
			if linear then
				linear:destroy()
				linear = nil
			end
			
			x0 = x0 + i_step
		end
	end
end
