#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2010, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

from calibre.customize import InterfaceActionBase

#History:
# Version 1.0 Initial release
# Version 1.1 Added support for multiple row selections to launching multiple searches
 
class ActionSearchTheInternet(InterfaceActionBase):
    name                = 'Search The Internet'
    description         = 'Search various websites for this book'
    supported_platforms = ['windows', 'osx', 'linux'] # Platforms this plugin will run on
    author              = 'Grant Drake'
    version             = (1, 2, 0)

    def load_actual_plugin(self, gui):
        import webbrowser
        from functools import partial
        from PyQt4.Qt import QMenu, QToolButton
        from calibre.utils.config import tweaks
        from calibre.gui2.actions import InterfaceAction
        from calibre.constants import preferred_encoding
        from urllib import quote_plus

        class SearchTheInternetAction(InterfaceAction):


            name = 'Search The Internet'
            action_spec = (_('Search the internet'), None, None, None)
            popup_type = QToolButton.InstantPopup
            action_type = 'current'

            DEFAULT_MENUS = [('FantasticFiction for Author', 'stip_fantastic_fiction.png', 'http://www.fantasticfiction.co.uk/search/?searchfor=author&keywords={author}', 'utf-8'),
                             ('FantasticFiction for Title',  'stip_fantastic_fiction.png', 'http://www.fantasticfiction.co.uk/search/?searchfor=book&keywords={title}', 'utf-8'),
                             (None, None, None, None),
                             ('Amazon.com for Book',    'stip_amazon.png', 'http://www.amazon.com/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1'),
                             (None, None, None, None),
                             ('Google.com for Book',    'stip_google.png', 'http://www.google.com/#sclient=psy&q={author}+"{title}"', 'utf-8'),
                             ('Google images for Book', 'stip_google.png', 'http://www.google.com/images?q={author}+"{title}"', 'utf-8'),
                             (None, None, None, None),
                             ('Wikipedia for Author',   'stip_wikipedia.png', 'http://en.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}', 'utf-8')]
                             
            TOKEN_AUTHOR = '{author}'
            TOKEN_TITLE  = '{title}'
            TOKEN_ISBN   = '{isbn}'
                               
            def genesis(self):
                m = QMenu(self.gui)
                search_internet_menus = None
                if tweaks.has_key('stip_search_internet_menus'):
                    search_internet_menus = tweaks['stip_search_internet_menus']
                if search_internet_menus is None:
                    search_internet_menus = self.DEFAULT_MENUS

                try:
                    for menu_text, icon, tokenised_url, encoding in search_internet_menus:
                        self.create_menu_item(m, menu_text, icon, tokenised_url, encoding)
                except ValueError:
                    # For users upgrading to v1.2 they may have insufficient columns in their tweaks
                    # We will "play nice" and just assume a default encoding
                    print "WARNING: Please update your 'stip_search_internet_menus' to have a correct number of columns"
                    for menu_text, icon, tokenised_url in search_internet_menus:
                        self.create_menu_item(m, menu_text, icon, tokenised_url)
                
                self.qaction.setMenu(m)
                self.search_internet_menu = m
                
            def create_menu_item(self, m, menu_text, icon, tokenised_url, encoding='utf-8'):
                if menu_text is None:
                    m.addSeparator()
                else:
                    ac = self.create_action(spec=(menu_text, icon, None, None),
                        attr=tokenised_url)
                    m.addAction(ac)
                    ac.triggered.connect(partial(self.search_web_link, tokenised_url, encoding))
                     
            def search_web_link(self, tokenised_url, encoding):
                rows = self.gui.library_view.selectionModel().selectedRows()
                if not rows or len(rows) == 0:
                    return
                self.db = self.gui.library_view.model().db
                for row in rows:
                    self.search_web_for_book(row.row(), tokenised_url, encoding)
                    
            def search_web_for_book(self, row, tokenised_url, encoding):
                if encoding is None:
                    encoding = 'utf-8'
                # Will only use the first author for the lookup if there are multiple
                if tokenised_url.find(self.TOKEN_AUTHOR) > -1:
                    author = self.db.authors(row).split(',')[0]
                    safe_author = self.convert_to_search_text(author, encoding)
                    tokenised_url = tokenised_url.replace(self.TOKEN_AUTHOR, safe_author)
                
                if tokenised_url.find(self.TOKEN_TITLE) > -1:
                    title = self.db.title(row)
                    safe_title = self.convert_to_search_text(title, encoding)
                    tokenised_url = tokenised_url.replace(self.TOKEN_TITLE, safe_title)
                    
                if tokenised_url.find(self.TOKEN_ISBN) > -1:
                    isbn = self.db.isbn(row)
                    tokenised_url = tokenised_url.replace(self.TOKEN_ISBN, isbn)

                webbrowser.open(tokenised_url)

            def convert_to_search_text(self, text, encoding):
                # First we strip characters we will definitely not want to pass through.
                # Comma separated author will be pipe delimited in Calibre database
                text = text.replace('|', ' ')
                # Periods from author initials etc do not need to be supplied
                text = text.replace('.', '')
                # Now encode the text using Python function with chosen encoding
                text = quote_plus(text.encode(encoding))
                # If we ended up with double spaces as plus signs (++) replace them
                text = text.replace('++','+')
                return text
                
        return SearchTheInternetAction(gui, None)