'''
A simple script for handling collections on the Amazon Kindle.

The Kindle is a great reading device, unfortunately so far it is rather painful to use it for a large collection of documents -
for example when dealing with many research papers. 

Uses the metadata.calibre file generated by Calibre.
 
As Amazon doesn't seem to support this functionality yet, this is a hack which also requires that the Kindle be restarted to update the system.
This is quite annoying, but there doesn't seem to be another way until Amazon releases its `Kindle Development Kit`.
At which point this script will hopefully become obsolete.

Usage

    $python2.6 CalibreKindleCollections.py [options]

Use the line below to find the available options

    $python2.6 CalibreKindleCollections.py -h

Note that if the script is saved in the root folder of the Kindle (e.g. /Volumes/Kindle), the mount point is not necessary. 

Please feel free to use and extend the script in any way you want. If you already have a valued collection of collections,
it might also be a good idea to backup this file before you give this a try.


What this script does
=====================
Authors: A collection will be created for each author of the name "<prefixAuthor><authorname>", provided that:
    a) "--na" is NOT specified on the command line
    b) The author's collection would contain at least minBooksForAuthor books

Tags: A collection will be created for each tag of the name "<prefixTag><tagname>", provided that:
    a) "--nt" is NOT specified on the command line
    b) Only tags which start with the string given in collectionTagPrefix will produce collections
    c) The tag's collection would contain at least minBooksForTag books

Series: A collection will be created for each series of the name "<prefixSeries><seriesname> - <authorname>", provided that:
    a) "--ns" is NOT specified on the command line
    b) The series's collection would contain at least minBooksForSeries books
    c) Note that the <authorname> is the name of one random author of book(s) in the series (and once chosen will persist across updates to collections)

Miscellany: Any files on the Kindle which are not in a collection when this script has run are automatically placed in the collection
miscRawName, which will be created if necessary. This behaviour may be switched off by specifying the command-line option "--nm"

By default, Kindle collections are updated: Additions may be made to existing collections and new collections created but pre-existing collections
will be left alone (unless they're empty, in which case see below). However, specifying the "--rb" command-line option will cause existing
collections to be wiped and completely replaced with new collections built based on the data from Calibre.

By default, this script will attempt to remove from collections any references to books which no longer exist in the Kindle's file system. This
behaviour can be disabled by specifying the "--nc" option on the command-line.

By default, any collections which are empty after the above operations have been carried out will be deleted. To diable this behaviour, specify
the "--ne" option on the command-line.

If you are rebuilding all of your collections then this script automatically sets their lastAccess time to ensure that they will (initially at least)
show up sorted into alphabetical order on the kindle's home page in "collections" view. If you're updating collections then the lastAccess time
for your collections will simply be set to the current date/time on those collections which have been changed; or you can use the command-line
option "--sort" to resort ALL of the collections on your kindle into alphabetical order. Note, though, that this WILL overwrite any existing
lastAccess data for all of your collections.


Running this Script
===================
The simplest way to run the script is to copy it to your Kindle's root folder (the one above the "documents" folder) and run it from there.
If you're using Windows, just download and install Python 2.6 or 2.7 and you can then run the script simply by double-clicking on it. Or,
if you wish to use the command-line options, you might want to create a batch file with a line like this in it:

C:\Python27\python.exe "K:\CalibreKindleCollections.py" --rebuildcollections --noseries > CalibreKindleCollections_log.txt

That line will run the script with the options to recreate all collections from scratch, not to create collections based on series (so just
authors and tags) and will send the output of the script (a description of the collections it's created) to a text file for later perusal.
Just change "C:\Python27\" to the folder where you installed Python, "K:" to the drive letter of your Kindle and set the command line
options you prefer.


Some notes on how to use this script
====================================
Short version:

    1) Set up your tags, series, etc as you want them in Calibre. Add a dash (-) to the start of any tags you want to use for collections
    2) Make sure Calibre's "metadata management" setting is set to "automatic management" (the setting's in Preferences->Sending Books to Devices)
    3) Connect your kindle and allow calibre to detect it. Allow Calibre to send its metadata to your kindle.
    4) Run the script, with command-line options if you wish
    5) Restart your Kindle by pressing and holding the power button until it restarts or by going Home->Menu->Settings->Menu->Restart Kindle
    6) Once your Kindle restarts it will take a few minutes to work out where to put everything but once it's done you should have properly
        organised collections based on your metadata in calibre. Collections created by this script are prefixed with "- " (a dash and a space),
        so that if you view your home page sorted by title then the collections will show first, in alphabetical order, followed by your individual
        books, also in alphabetical order by title.

If you have wireless switched on then it's possible that your kindle will retrieve your old collections from Amazon instead of using the
new ones. If that happens, repeat the above steps but switch wireless off before you run the script and don't switch it back on again until
after the Kindle restarts.


Detailed version:

For the collections to be accurately built, it is essential that your kindle has the current metadata from Calibre. This will *ONLY* be the
case if Calibre's "metadata management" setting is set to "automatic management".

However, if that setting is in effect then Calibre is unable to read the collections data from the kindle for display in the "Device" page.
If this is a concern then the only solution is to:

    1) Set "metadata management" to "automatic management" in Calibre
    2) Disconnect your kindle if it's already connected. Or, alternatively, completely exit Calibre.
    3) Connect your kindle (or start up Calibre)
    4) Calibre should detect the Kindle and send it a sizeable package of metadata
    5) Once that metadata is sent then you can run this script and your collections will be created *based on the tags as they were when that
        packet was sent*
    6) Once the metadata has been sent, you can switch Calibre's metadata management option back to your preferred setting until the next time
        you wish to run this script

Once the script's run you need to restart your Kindle *immediately*. If you have SSH server enabled on your Kindle then rather than restarting
the kindle you can send the following command to your Kindle instead:

    ssh root@yo.ur.I.P 'dbus-send --system /default com.lab126.powerd.resuming int32:1' 

Search for the "usbnetwork" hack for details as to how to set up an SSH server on your kindle, but frankly I wouldn't recommend it just for this.

Known Issues
============
1) Some books have an embedded Amazon asin id somewhere in their metadata, and the Kindle uses that asin id in preference to a calculated one.
    Such books cannot be assigned to collections by this script (or, rather, they can be assigned but the kindle won't put them into collections).

    There are two possible workarounds. The simplest is to convert those books to another format, such as epub, and then convert them back to
    mobi, re-send them to the kindle and then rebuild your collections.

    If that's not possible, or desirable, then the alternative is to either manually move them to collections yourself on the kindle or to
    add such books, individually, as exceptions in the getAsin() function in this script, in the same way that the Kindle User Guide is handled
    by the code at present.

    Be aware, though, that to do this you will need to obtain the book's asin id AND that the collections-assignment of such books will not
    be visible in Calibre (they'll show up on the "device" page as not being in a collection).

    One way to obtain the book's internal asin is to, on the kindle, create a new collection and manually add that specific book *and no others*
    to it. Thenconnect the kindle to your computer and open its collections.json file. Look for the newly-created collection and the id number of
    the sole item inside that collection is the asin id which the kindle uses for that book.

    Unless/Until some method is found for this script to automatically retrieve such values from the eBook in such cases, there's no way to
    automate that process, unfortunately.

2) If a book is moved out of a collection (but is still on the kindle and otherwise unchanged) and the script is running in update mode then it
    relies on the kindle to detect that change rather than doing it itself. This won't be changed, most likely, as such detection would require
    this script to retain a snapshot of the previous state of the collections and it's just not worth it.

    Similarly, if you scripted to create collections for authors who have at least 3 books and then later change that limit to 4 books the
    previously-created collections will not be removed if you run the script in update mode. This is intentional: Update mode should never
    delete collections which have books inside them. Either manually delete those collections or run the script in rebuild mode.
'''

import sys,os.path,os,json,time
from hashlib import sha1
from optparse import OptionParser
from xml.sax.saxutils import escape


#
# Configurable values
#
# These values may be changed to suit your own preferences

# Name to use for "Miscellaneous" collection (dumping ground for everything not assigned to a collection)
miscRawName = "- Miscellany"

# Minimum number of books which an author/tag/series must have in order to have a collection created
minBooksForAuthor = 4
minBooksForTag = 1
minBooksForSeries = 2

# Prefix to use for author/tag/series collection names
prefixAuthor = "- "
prefixTag = "- "
prefixSeries = "- "

# Tag prefix indicating that a collection is to be created of all tagged items.
# Note that this prefix is stripped off the front of the tag name and replaced with
# the value of prefixTag to form the name of the collection
collectionTagPrefix = "-"

# Any authors for whom you do not wish to create collections should be listed below, separated by commas
#
# There's no need to put authors in here who have fewer than minBooksForAuthor books as
# those won't get their own collections anyway. I use this mostly for those authors who
# have contributed to series where I'm far more likely to want a series collection than
# several individual ones for each author.
#excludedAuthorsCSV = "H F Cary,Henry Wadsworth Longfellow,Eric Van Lustbader,W W Shols"
excludedAuthorsCSV = ""

# Any authors for whom you ALWAYS wish to create collections, even if they have fewer than
# minBooksForAuthor books (but at least 1), should be listed below (and NOT listed above).
#includedAuthorsCSV = "H Rider Haggard"
includedAuthorsCSV = ""

# Any series for which you do not wish to create collections should be listed below, separated by commas
#
# There's no need to put series in here who have fewer than minBooksForSeries books as
# those won't get their own collections anyway.
excludedSeriesCSV = ""

# Any series for which you ALWAYS wish to create collections, even if they have fewer than
# minBooksForSeries books (but at least 1), should be listed below (and NOT listed above).
includedSeriesCSV = ""


##################################################################
# Non-configurable values. Don't change anything below this line #
##################################################################

# Relative path to collection json file
COLLECTIONS = 'system/collections.json'
# Relative path to documents folder
DOCUMENTS = 'documents'
# Relative path to Calibre's on-kindle metadata database
CALIBRE = 'metadata.calibre'

# for the computation of the sha1 identifier
KINDEL_ABS = '/mnt/us'

# We don't want to put the kindle user guide into a collection as the kindle will
# just move it out again and we may well end up with an empty collection as a result.
#
# So these lines define how to recognise the Kindle user guide from its metadata
# (assuming that it's been previously detected by Calibre, which it probably has)
KINDEL_USER_GUIDE_PREFIX = "Kindle User"
KINDEL_USER_GUIDE_AUTHOR = "Amazon.com"

# Dictionary for storing collections
kindleC = {}
calibreC = {}

# Dictionary for command-line options
options = {}

def setup():
    ''' 
    In case the script is not run in the root folder of the Kindle device, the Kindle Mount Point has to be updated because relative paths won`t work anymore 
    '''
    global COLLECTIONS,CALIBRE,DOCUMENTS,options,miscRawName,minBooksForAuthor,minBooksForTag,minBooksForSeries,prefixSeries,prefixTag
    global prefixAuthor,collectionTagPrefix,KINDEL_USER_GUIDE_PREFIX,KINDEL_USER_GUIDE_EXT,KINDEL_USER_GUIDE_AUTHOR
    global excludedAuthors,excludedSeries,includedAuthors,includedSeries

    # parse commmand line for options
    parser = OptionParser()
    parser.add_option("--na","--noauthors", action="store_false", dest="useAuthors", default=True, help="Do not use authors to create collections")
    parser.add_option("--nc","--nocleanupdeadfiles", action="store_false", dest="cleanupDeadFiles", default=True, help="Do not clean collections to remove references to files which are no longer on the kindle")
    parser.add_option("--ne","--nocleanupemptycollections", action="store_false", dest="cleanupEmptyCollections", default=True, help="Do not delete empty collections from the kindle")
    parser.add_option("--nm","--nomiscellanycollection", action="store_false", dest="useMiscellanyCollection", default=True, help="Do not put all your uncollected books into the '%s' collection"%miscRawName )
    parser.add_option("--ns","--noseries", action="store_false", dest="useSeries", default=True, help="Do not use series to create collections")
    parser.add_option("--nt","--notags", action="store_false", dest="useTags", default=True, help="Do not use tags to create collections. By default, all tags which start with a dash (-) character are used to create collections")

    parser.add_option("--v","--verbose", action="store_true", dest="verboseMessages", default=False, help="Produces more messages for logging purposes")

    parser.add_option("--rb","--rebuild", "--rebuildcollections", action="store_false", dest="updateCollections", default=True, help="Rebuild Kindle collections completely from scratch")
    parser.add_option("--sc","--sort", "--sortcollections", action="store_true", dest="sortCollections", default=False, help="Sort ALL Kindle collections into alphabetical order by adjusting their lastAccess times")

    # change default for your use. use '' if u want to set it completely from command line
    parser.add_option("-m","--mnt", dest="mntPoint", default=".", help="Required if script is not run from the root folder of Kindle.")

    (options,args) = parser.parse_args()    

    # Use the mount point 
    COLLECTIONS = os.path.join(options.mntPoint,COLLECTIONS)
    CALIBRE = os.path.join(options.mntPoint,CALIBRE)
    DOCUMENTS = os.path.join(options.mntPoint,DOCUMENTS)

    # Parse inclusions and exclusions
    excludedAuthors = excludedAuthorsCSV.split(",")
    excludedSeries = excludedSeriesCSV.split(",")
    includedAuthors = includedAuthorsCSV.split(",")
    includedSeries = includedSeriesCSV.split(",")

    # When rebuilding the entire collections structure from scratch, automatically sort them so they show in alphabetical order in "collections" view
    if not options.updateCollections:
        options.sortCollections = True;

def getAsin(lpath):

    # Standardise lpath to kindle absolute path
    lpath = '%s/%s'%(KINDEL_ABS,lpath.replace('\\','/'));

    # Check for "rogue" asin values
    # These are cases where the getAsin() formula does not work but instead the kindle apparently extracts the asin id from the book's metadata
    #
    # The returned value here is correct for my Kindle, but I don't know if it's universally applicable. If not, change the two values
    # below to match those for the Kindle User guide on your Kindle
    if lpath == "/mnt/us/documents/Kindle User s Guide-asin_B003O86FMM-type_EBOK-v_0.azw":
        return "#B003O86FMM^EBOK"
    else:
        # No rogue value found, so use the generic formula to calculate the asin
        return '*%s'%sha1(lpath).hexdigest()

def loadCalibre():
    ''' Loads/Transform Calibre metadata in dictionary '''
    global calibreB,calibreC,options,UserGuideAsin

    UserGuideAsin = ""

    try:
        cf = open(CALIBRE,'r')
        calibreB = json.load(cf)
        cf.close()
      
        colls= {}
         
        for book in calibreB:

            title = book['title']
            authors = book['authors']
            lpath = book['lpath']

            # If we detect the kindle user guide, make a note of its asin so we can ignore it later on
            if title.startswith(KINDEL_USER_GUIDE_PREFIX) and len(authors) == 1 and authors[0] == KINDEL_USER_GUIDE_AUTHOR:
                UserGuideAsin = getAsin(lpath)
                if options.verboseMessages:
                    print "Detected Kindle User Guide with asin [%s]\n"%(UserGuideAsin)
            else:

                tags = book['tags']
                series = book['series']

                if len(authors) == 1 and authors[0].find(";") > -1:
                    authors = authors[0].split(";")

                if options.useSeries:
                     if series != None:

                         series = fixupNameString(series);

                         if series not in excludedSeries:
                             if series not in colls:
                                 colls[series] = { 'collName': prefixSeries + series, 'collType': 'series', 'authors': [], 'asins': [], 'titles': [] }

                             for author in authors:
                                 if author not in colls[series]['authors']:
                                     colls[series]['authors'].append(author)

                             colls[series]['asins'].append(getAsin(lpath))
                             colls[series]['titles'].append(title)

                             if options.verboseMessages:
                                 print '[%s]\n\tTITLE %s\n\tLPATH %s\n\tASIN [%s]\n'%("SERIES: " + series.encode('utf-8'),title.encode('utf-8'),lpath.encode('utf-8'),getAsin(lpath))

                if options.useAuthors:
                    for author in authors:
                        if author not in excludedAuthors:
                            if author not in colls:
                                colls[author] = { 'titles': [], 'collName': prefixAuthor + author,'collType': 'author', 'authors': [author], 'asins': [] }

                            colls[author]['asins'].append(getAsin(lpath))
                            colls[author]['titles'].append(title)

                            if options.verboseMessages:
                                print '[%s]\n\tTITLE %s\n\tLPATH %s\n\tASIN [%s]\n'%("AUTHOR: " + author.encode('utf-8'),title.encode('utf-8'),lpath.encode('utf-8'),getAsin(lpath))

                if options.useTags:
                     for tag in tags:

                         tag = fixupNameString(tag);
                         if tag.startswith(collectionTagPrefix):

                             if tag not in colls:
                                 colls[tag] = { 'collName': prefixTag + tag[len(collectionTagPrefix):], 'collType': 'tag', 'authors': [], 'asins': [], 'titles': [] }

                             for author in authors:
                                 if author not in colls[tag]['authors']:
                                     colls[tag]['authors'].append(author)

                             colls[tag]['asins'].append(getAsin(lpath))
                             colls[tag]['titles'].append(title)

                             if options.verboseMessages:
                                 print '[%s]\n\tTITLE %s\n\tLPATH %s\n\tASIN [%s]\n'%("TAG: " + tag.encode('utf-8'),title.encode('utf-8'),lpath.encode('utf-8'),getAsin(lpath))

        calibreC = colls.values()

    except Exception, err:
        sys.stderr.write('ERROR loadCalibre: %s\n'%str(err))
        exit()

def loadCollections():
    global kindleC
    if options.updateCollections:
        if options.verboseMessages:
            print 'Loading existing Kindle collections'

        ''' Loads Kindle collections in dictionary '''
        try:
            cf = open(COLLECTIONS,'r')
            kindleC = json.load(cf)
            cf.close()
        except:
            print 'loadCollections'
            print 'WARNING: %s could not be loaded. Creating a new version.'%(COLLECTIONS.encode('utf-8'))
    elif options.verboseMessages:
        print 'Not loading existing Kindle collections: Starting from a blank slate'

def saveCollections():
    ''' Dump kindle collections dictionary back into json file '''
    cf = open(COLLECTIONS,'wb')
    json.dump(kindleC,cf)
    cf.close()


# Gets a list of asins of all documents physically present on the kindle
def get_kindle_books_asins(root,
                    ignored_names=("metadata.db",),
                    ignore_extensions=(".jpg", ".gif", ".mbp")):
    asins = []

    def grab_file(arg, dirname, fnames):
        for short_name in fnames:
            lpath = os.path.join(dirname, short_name)
            if os.path.isfile(lpath):
                if any(short_name.endswith(ext) for ext in ignore_extensions):
                    continue
                lpath = os.path.relpath(lpath,options.mntPoint)
                asins.append(getAsin(lpath))
                if options.verboseMessages:
                    print '%s\n\t[%s]\n'%(('%s/%s'%(KINDEL_ABS,lpath.replace('\\','/'))).encode('utf-8'),getAsin(lpath))

    os.path.walk(DOCUMENTS, grab_file, None)
    return asins

def lastAccess():
    ''' Returns a lastaccess value in milliseconds '''
    return int(time.time()*1000)

def fixupCollname(str):
    return '%s@en-US'%(fixupNameString(str))

def fixupNameString(str):
    return escape(str.strip().replace('  ',' '))

def updateCollections():
    '''
    Calibre data is traversed each book encountered is added to a collection with the name of
    each author and series (if defined).
    '''
    global changeMade;

    asinslist= {}
    changeMade = False;

    for coll in calibreC:
        asins = coll['asins']
        authors = coll['authors']
        collName = coll['collName']
        collType = coll['collType']
        titles = coll['titles']

        minBooks = minBooksForAuthor

        # create/get collections names for the following:
        #   author with multiple books, 
        #   series with one author, 
        #   series with multiple authors
        if collType == 'series':

             if collName in includedSeries:
                minBooks = 1
             else:
                minBooks = minBooksForSeries

             if len(authors) == 1:
                cName = fixupCollname(collName + " - " + authors[0])
             else:
                for author in authors:
                    cName = fixupCollname(collName + " - " + author)
                    if cName in kindleC:
                        break;
        else:
            cName = fixupCollname(collName)
            if collType == 'tag':
                minBooks = minBooksForTag
            elif collType == 'author' and collName in includedAuthors:
                minBooks = 1

        # Add books if there are more than three books in the collection
        # OR if the collection already exists on the kindle
        # OR if it's one being specifically created from "- foo" tags
        if len(asins) >= minBooks:

            if cName in kindleC:
                collInKindle = True
            else:
                collInKindle = False

            # create the kindle collection if it does not exist and update access time
            if not collInKindle:
                kindleC[cName] = {'items':[], 'lastAccess':lastAccess()}
                changeMade = True
                if options.verboseMessages:
                    print 'CREATED COLLECTION "%s"'%(cName[0:cName.find('@en')])

            # add the paths to the books for the collection
            itemAdded = False
            for asin in asins:

                # if the document is not already in the collection we add it
                if asin not in kindleC[cName]['items']:
                    kindleC[cName]['items'].append(asin) 
                    itemAdded = True
                    if options.verboseMessages:
                       print '[%s] ADDED TO "%s"'%(asin,cName[0:cName.find('@en')])
                else:
                    if options.verboseMessages:
                        print '[%s] ALREADY IN "%s"'%(asin,cName[0:cName.find('@en')])

            # print a description of the collection if it's changed
            if itemAdded:
                changeMade = True;
                collDesc = '%s:\n%s\n'%(cName[0:cName.find('@en')],'\t'+'\n\t'.join(titles))
                print collDesc.encode('utf-8')
                if collInKindle:
                    kindleC[cName]['lastAccess'] = lastAccess()

            # Keep a running list of all books which are in collections
            if options.useMiscellanyCollection:
                for asin in kindleC[cName]['items']:
                    asinslist[asin] = { 'asin': asin }


    if options.useMiscellanyCollection:

        # Put anything not assigned to a collection into a "Miscellany" collection
        miscName = fixupCollname(miscRawName)
        numMiscTitles = 0
        miscTitles = '\n'

        for book in calibreB:

            asin = getAsin(book['lpath'])

            ## Don't put the kindle user guide into the Miscellany collection
            if asin != UserGuideAsin and asin not in asinslist:

                authors = book['authors']
                series = book['series']
                title = book['title']
                tags = book['tags']

                # create the kindle collection if it does not exist and update access time
                if miscName not in kindleC:
                    kindleC[miscName] = {'items':[], 'lastAccess':lastAccess()}
                    changeMade = True
                    if options.verboseMessages:
                        print 'CREATED COLLECTION "%s"'%(miscRawName.encode('utf-8'))

                # if the document is not already in the collection we add it
                if asin not in kindleC[miscName]['items']:
                    kindleC[miscName]['items'].append(asin)
                    numMiscTitles = numMiscTitles + 1
                    miscTitles = miscTitles + '\t' + title + '\n'
                    changeMade = True
                    if options.verboseMessages:
                        print '[%s] ADDED TO "%s" (%s)'%(asin,miscRawName.encode('utf-8'),title.encode('utf-8'))

        # print a description of the miscName collection if it exists
        if numMiscTitles > 0 and miscName in kindleC:
            collDesc = '%s:%s'%(miscName[0:miscName.find('@en')],miscTitles)
            print collDesc.encode('utf-8')
            changeMade = True
            kindleC[miscName]['lastAccess'] = lastAccess()


    # Now check for items in collections but no longer on the kindle
    collsToRemove = {}
    if options.cleanupDeadFiles:
        print 'Retrieving list of all books on kindle. This may take a little while.\n'
        booksOnKindle = get_kindle_books_asins(DOCUMENTS)

        print 'Checking for any books which are in collections but are no longer on the kindle.\n'
        for coll in kindleC:
            numBooks = 0
            for book in kindleC[coll]['items']:
                # Count books in collection and remove any books in the collection but no longer on the kindle
                if book in booksOnKindle:
                    numBooks = numBooks + 1
                else:
                    # Never delete the Kindle user guide
                    if book != UserGuideAsin:
                        kindleC[coll]['items'].remove(book)
                        print 'Removed book asin [%s] from collection `%s`'%(book,coll.encode('utf-8'))
                        changeMade = True

            # Build list of any empty collections
            if numBooks == 0:
                collsToRemove[coll] = { 'colls': coll }

    elif options.cleanupEmptyCollections:
        # Check for empty collections
        for coll in kindleC:
            if len(kindleC[coll]['items']) == 0:
                collsToRemove[coll] = { 'colls': coll }

    # Delete any empty collections
    if not options.cleanupEmptyCollections:
        for coll in collsToRemove:
            kindleC.pop(coll)
            print 'Deleted empty collection `%s`'%(coll[0:coll.find('@en')].encode('utf-8'))
            changeMade = True

    if options.sortCollections:
        # Fake collection lastAccess times so that they initially show up in
        # alphabetical order in the kindle's "collections" view
        fakeAccess = lastAccess()
        for coll in sorted(kindleC.iterkeys(), key=lambda x: x.lower()):
            kindleC[coll]['lastAccess'] = fakeAccess
            if options.verboseMessages:
                print 'SET lastAccess to %u for collection "%s"'%(fakeAccess,coll[0:coll.find('@en')].encode('utf-8'))
            fakeAccess = fakeAccess - 1000

if __name__ == '__main__':

    setup()
    
    # Check if path are correct
    if not (os.path.exists(CALIBRE)):
        print 'ERROR: unknown path to Kindle mounting point. Please set `%s` to correct path (e.g. /Volumes/Kindle)'%(options.mntPoint)
        sys.exit() 

    if options.verboseMessages:
        print 'CalibreKindleCollections.py log\n===============================\n'

    # Load collections
    loadCalibre()
    loadCollections()    
    # Add new files to the collection dictionary 
    updateCollections()    
    # Save updated collections back into json
    if changeMade or options.updateCollections == False:
        saveCollections()
    
        # display reminder to restart kindle 
        print '\nREMINDER:\nCOLLECTIONS WILL NOT BE UPDATED UNLESS YOU RESTART YOUR KINDLE NOW!\nHold the power switch for about 20 seconds and then release. Then wait 20 seconds.\nThe screen will flash and then the device will restart' 
    else:
        print "\nNo changes made to Kindle collections.\nNo need to restart your kindle (assuming you've restarted it since the last time you ran this script)."
