/*
 * File Name: text_model.h
 */

/*
 * This file is part of uds-plugin-plaintext.
 *
 * uds-plugin-plaintext is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * uds-plugin-plaintext is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#ifndef FB2_MODEL_H
#define FB2_MODEL_H

#include <string>
#include <vector>
#include <map>
#include <cassert>
#include <glib.h>
#include <expat.h>
#include "plugin_inc.h"
#include "fb2_base_types.h"
#include "signal_slot.h"


namespace fb2
{

#define DEFAULT_ENCODING "utf-8"
class FB2Model;
struct FB2XmlScannerParams;

struct Paragraph{
	public:
		enum FontType {
			RegularFont,
			ItalicFont,
			BoldFont,
			TitleFont,
			SubTitleFont,
			SuperScriptFont,
			EmphasisFont,
			EmphasisCiteFont,
			Image,
		};
		enum ParaAlign {
			LeftAlign,
			CenterAlign,
			RightAlign,
			FullPage,
		};

		size_t       start_file_pos;
		FontType	 font;
		ParaAlign	 align;
		std::string  text;

	public:
		Paragraph(size_t pos,FontType _f=RegularFont,ParaAlign _a=LeftAlign)
			: start_file_pos(pos),font(_f),align(_a)
			{
			}
		Paragraph(size_t pos, const std::string& _text,FontType _f=RegularFont,ParaAlign _a=LeftAlign)
			: start_file_pos(pos),font(_f),align(_a),text(_text)
			{
			}
};


class FB2Model
{
public:
	typedef std::map<std::string, std::string> DocAttrMap;
	typedef DocAttrMap::iterator               DocAttrMapIter;

	struct ImageDescr{
		public:
			std::string image_id,image_type,image_text;
            unsigned char* bits;
			Size size,client_orig;
			int row_stride;
			ImageDescr(const std::string& id, const std::string& type):image_id(id),image_type(type),bits(0),size(0,0),client_orig(0,0),row_stride(0){}
			~ImageDescr();


			bool build_image(int client_width, int client_height, bool outzoom);
			void render(unsigned char *bmp,const Rect& clientRect,int display_width, int display_height) const;
	};

	struct Author{
		std::string first_name,middle_name,last_name,
					nickname,home_page,email;
	};
	struct CoverPage{
		std::string image_name;
	};

	/// @brief text document
	typedef std::vector<Paragraph*> Paragraphs;
	typedef Paragraphs::iterator 	ParagraphsIter;
		
	class BookInfo{
		public:
			BookInfo(){}
			~BookInfo(){clear();}
			void clear();

			typedef std::vector<Author*> Authors;
			typedef Authors::iterator AuthorsIter;
			typedef std::vector<std::string> Strings;

			// book attributes
			Authors		authors;
			Strings 	genre;
			std::string book_title;
			std::string sequence;
			std::string date;
			Paragraphs	annotations;
			std::string language;
			CoverPage  	cover_page;
	};

    /// @brief Constructors and destructors
    FB2Model();
    ~FB2Model();

public:
    /// @brief Open specified document with default encoding.
    PluginStatus open(const std::string& path);

    /// @brief Open document with specified encoding
    PluginStatus open(const std::string& path, const std::string& encoding);

	/// @frief document is opened
	bool is_open()const { return doc.size() > 0; }

    /// @brief Close document
	void close();

	/// @brief Get current encoding
	const std::string& get_encoding() const
	{
		return encoding;
	}

	const std::string& get_path() const
	{
		return path;
	}

	/// @brief Get number of paragraphs
	unsigned int get_paragraph_count() const
	{
		return static_cast<unsigned int>(doc.size());
	}

    /// @brief Get 1 paragraph
    const Paragraph* get_paragraph(unsigned int index) const
    {
        assert(index < doc.size());
        return doc[index];
    }

	const ImageDescr* get_build_image(const std::string& name, int width, int height, bool outzoom=false);

	const BookInfo&	  get_book_info() const { return info;}
    /// @brief Search specified pattern given by search criteria.
    /// @param result_ranges Output range collection.
    /// @param search_context Search criteria.
    /// @return Return true if search is complete (i.e. we reach the start/end
    ///  of the document, or find an occurrence if SearchType is SEARCH_NEXT),
    ///  otherwise false is returned, user needs to call this function again to get
    ///  the search procedure complete.
    bool search(std::vector<Range>& result_ranges, SearchContext* sc);

    /// @brief Check if the document contains the anchor or not.
    bool has_anchor(const Position &pos);

    /// @brief Get absolute file position from anchor.
    bool get_file_pos_from_anchor(size_t& file_pos, const Position &pos);

    /// @brief Get a word from specified document position.
    bool get_word_from_anchor(const Position& pos,
                              Position& word_start_pos,
                              Position& word_end_pos);

    /// @brief Get the words from range, the range will be extended/shrinked to words boundary.
    bool get_words_from_range(const Position& range_start,
                              const Position& range_end,
                              Position& words_start,
                              Position& words_end);

    /// @brief Get the text between start_pos and end_pos.
    bool get_text_from_range(std::string& result,
                             const Position& start_pos,
                             const Position& end_pos);

    /// @brief Dump the content to disk file.
    void dump();

    /// @brief Abort specified search task.
    void set_aborting_search_task_id(unsigned int id)
    {
        aborting_search_task_id = id;
    }

    unsigned int get_aborting_search_task_id()
    {
        return aborting_search_task_id;
    }
	inline DocAttrMap& 	get_doc_attr_map() { return doc_attr_map;}

public:
    /// @brief Signals.
    utils::Signal<const std::vector<Range>&, const SearchContext*> search_done_signal;


	/// @brief xml handlers
	void xml_FictionBook(FB2XmlScannerParams* sp, const char **atts);
	void xml_description(FB2XmlScannerParams* sp, const char **atts);
	void xml_title_info(FB2XmlScannerParams* sp, const char **atts);
	void xml_genre(FB2XmlScannerParams* sp, const char **atts);
	void xml_author(FB2XmlScannerParams* sp, const char **atts);
	void xml_first_name(FB2XmlScannerParams* sp, const char **atts);
	void xml_middle_name(FB2XmlScannerParams* sp, const char **atts);
	void xml_last_name(FB2XmlScannerParams* sp, const char **atts);
	void xml_nickname(FB2XmlScannerParams* sp, const char **atts);
	void xml_home_page(FB2XmlScannerParams* sp, const char **atts);
	void xml_email(FB2XmlScannerParams* sp, const char **atts);
	void xml_book_title(FB2XmlScannerParams* sp, const char **atts);
	void xml_annotation(FB2XmlScannerParams* sp, const char **atts);
	void xml_date(FB2XmlScannerParams* sp, const char **atts);
	void xml_coverpage(FB2XmlScannerParams* sp, const char **atts);
	void xml_image(FB2XmlScannerParams* sp, const char **atts);
	void xml_lang(FB2XmlScannerParams* sp, const char **atts);
	void xml_document_info(FB2XmlScannerParams* sp, const char **atts);
	void xml_program_used(FB2XmlScannerParams* sp, const char **atts);
	void xml_src_ocr(FB2XmlScannerParams* sp, const char **atts);
	void xml_id(FB2XmlScannerParams* sp, const char **atts);
	void xml_version(FB2XmlScannerParams* sp, const char **atts);
	void xml_publish_info(FB2XmlScannerParams* sp, const char **atts);
	void xml_book_name(FB2XmlScannerParams* sp, const char **atts);
	void xml_publisher(FB2XmlScannerParams* sp, const char **atts);
	void xml_city(FB2XmlScannerParams* sp, const char **atts);
	void xml_year(FB2XmlScannerParams* sp, const char **atts);
	void xml_isbn(FB2XmlScannerParams* sp, const char **atts);
	void xml_custom_info(FB2XmlScannerParams* sp, const char **atts);
	void xml_body(FB2XmlScannerParams* sp, const char **atts);
	void xml_section(FB2XmlScannerParams* sp, const char **atts);
	void xml_title(FB2XmlScannerParams* sp, const char **atts);
	void xml_paragraph(FB2XmlScannerParams* sp, const char **atts);
	void xml_empty_line(FB2XmlScannerParams* sp, const char **atts);
	void xml_binary(FB2XmlScannerParams* sp, const char **atts);
    void xml_char_data_handler(FB2XmlScannerParams* sp,const XML_Char* s,int len);
    void xml_a(FB2XmlScannerParams* sp, const char **atts);
    void xml_epigraph(FB2XmlScannerParams* sp, const char **atts);
    void xml_text_author(FB2XmlScannerParams* sp, const char **atts);
    void xml_sequence(FB2XmlScannerParams* sp, const char **atts);
    void xml_sub(FB2XmlScannerParams* sp, const char **atts);
    void xml_poem(FB2XmlScannerParams* sp, const char **atts);
    void xml_stanza(FB2XmlScannerParams* sp, const char **atts);
    void xml_v(FB2XmlScannerParams* sp, const char **atts);
	void xml_emphasis(FB2XmlScannerParams* sp, const char **atts);
	void xml_subtitle(FB2XmlScannerParams* sp, const char **atts);
	void xml_cite(FB2XmlScannerParams* sp, const char **atts);

	int  xml_unknownEncodingHandler(const XML_Char *name, XML_Encoding *info);
private:
    /// @brief Clear content read from disk file
    void clear();

    /// @brief Save block with paragraphs
    PluginStatus parse_fb2_data(const char* test_data, size_t test_len, int file_p, size_t len);

    /// @brief Check the char is a word seperator
    bool is_seperator(const char* p);

    /// @brief Judge whether it is a new searhing or searching next again.
    bool is_new_search(SearchContext * sc);

private:
    /// @brief Current encoding string
    std::string encoding;

    /// @brief Document path in file system
    std::string path;

    /// @brief The id of the search task which needs to be aborted.
    unsigned int aborting_search_task_id;

    /// @brief Remember the last search result range.
    Range        last_search_result;

    /// @brief Remember the last search context.
    SearchContext last_sc;

	typedef std::map<std::string,ImageDescr*> Images;
	typedef Images::iterator 	ImagesIter;

    /// @brief Paragraph array, the paragraph is already in UTF-8 format
    Paragraphs 	doc;
	BookInfo	info;
	Images      images;
	DocAttrMap 	doc_attr_map;
};

};  // text

#endif // FB2_MODEL_H

