/*
 * File Name: text_view.cpp
 */

/*
 * This file is part of uds-plugin-plaintext.
 *
 * uds-plugin-plaintext is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * uds-plugin-plaintext is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#include "log.h"
#include "utils.h"
#include "font_cache.h"
#include "pango_renderer.h"

#include "fb2_config.h"
#include "fb2_view.h"
#include "fb2_tasks.h"
#include "plugin_render_result.h"

namespace fb2
{

FB2View::FB2View(FB2Model *m)
: model(m),
  ctrl(0),
  font_family(DEFAULT_ASCII_FONT),
  font_size(DEFAULT_FONT_SIZE),
  display_width(DEFAULT_SURFACE_WIDTH),
  display_height(DEFAULT_SURFACE_HEIGHT),
  dpi(DEFAULT_DPI),
  color_depth(DEFAULT_COLOR_DEPTH),
  left_margin(DEFAULT_LEFT_MARGIN),
  right_margin(DEFAULT_RIGHT_MARGIN),
  top_margin(DEFAULT_TOP_MARGIN),
  bottom_margin(DEFAULT_BOTTOM_MARGIN),
  client_width(display_width - left_margin - right_margin),
  client_height(display_height - top_margin - bottom_margin),
  pango_mutex(0),
  is_repagination_needed(false),
  blank_lines(0),
  blank_lines_offset(0)
{
    // Special handling for Chinese font.
    const char *enc = model->get_encoding().c_str();
#ifdef WIN32
    if (_stricmp(enc, "GB2312") == 0 || _stricmp(enc, "GBK") == 0)
#else
    if (strcasecmp(enc, "GB2312") == 0 || strcasecmp(enc, "GBK") == 0)
#endif
    {
        font_family = DEFAULT_CHINESE_FONT;
    }
	memset(panfo_info,0,sizeof(panfo_info));
    initialize();
}

FB2View::~FB2View()
{
    deinitialize();
}

void FB2View::initialize()
{
	TRACEFUNCTION();
    // Create mutex for synchronous access to pango stuff.
    pango_mutex = g_mutex_new();

    // Initialize pango context.
    pango_context_init();
}

void FB2View::deinitialize()
{
	TRACEFUNCTION();
    // Release pango stuff
    pango_context_final();
    g_mutex_free(pango_mutex);
}

void FB2View::pango_context_init()
{
//	TRACEFUNCTION();
	for(int font=(int)Paragraph::RegularFont; font < (int)Paragraph::Image; font++) {
		// Create PangoFT2FontMap
		PangoFT2FontMap *font_map = PANGO_FT2_FONT_MAP(pango_ft2_font_map_new());

		// Set dpi information.
		pango_ft2_font_map_set_resolution(font_map, dpi, dpi);

		// Create pango context
		panfo_info[font].pango_context = pango_ft2_font_map_create_context(font_map);

		// Create pango font description
		PangoFontDescription* font_desc = pango_font_description_new();
		pango_font_description_set_family(font_desc, font_family.c_str());

		switch((Paragraph::FontType)font) {
			case Paragraph::RegularFont:
				pango_font_description_set_size(font_desc, font_size * PANGO_SCALE);
				break;
			case Paragraph::ItalicFont:
				pango_font_description_set_size(font_desc, font_size * PANGO_SCALE);
				pango_font_description_set_style(font_desc, PANGO_STYLE_ITALIC);
				break;
			case Paragraph::BoldFont:
				pango_font_description_set_size(font_desc, font_size * PANGO_SCALE);
				pango_font_description_set_weight(font_desc,PANGO_WEIGHT_BOLD);
				break;
			case Paragraph::TitleFont:
				pango_font_description_set_size(font_desc, (3 * font_size * PANGO_SCALE)/2);
				pango_font_description_set_weight(font_desc,PANGO_WEIGHT_BOLD);
				break;
			case Paragraph::SubTitleFont:
				pango_font_description_set_size(font_desc, (5 * font_size * PANGO_SCALE) / 4);
				pango_font_description_set_weight(font_desc,PANGO_WEIGHT_BOLD);
				break;
			case Paragraph::SuperScriptFont:
				pango_font_description_set_size(font_desc, (2 * font_size * PANGO_SCALE) / 3 );
				break;
			case Paragraph::EmphasisFont:
				pango_font_description_set_size(font_desc, (4 * font_size * PANGO_SCALE) / 5 );
				//pango_font_description_set_weight(font_desc,PANGO_WEIGHT_SEMIBOLD);
				pango_font_description_set_style(font_desc, PANGO_STYLE_OBLIQUE);
				break;
			case Paragraph::EmphasisCiteFont:
				pango_font_description_set_size(font_desc, (4 * font_size * PANGO_SCALE) / 5 );
				//pango_font_description_set_weight(font_desc,PANGO_WEIGHT_SEMIBOLD);
				pango_font_description_set_style(font_desc, PANGO_STYLE_ITALIC);
				break;
			case Paragraph::Image:
				break;
		}
		pango_context_set_font_description(panfo_info[font].pango_context, font_desc);

		char *desc_str = pango_font_description_to_string(font_desc);
		panfo_info[font].font_hash_code = make_hash_code(desc_str, static_cast<unsigned int>(strlen(desc_str)));
		g_free(desc_str);

		// Since pango_context has copied the font description from us,
		// our font description can be freed safely.
		pango_font_description_free(font_desc);
	}
}

void FB2View::pango_context_final()
{
//	TRACEFUNCTION();
	for(int font=(int)Paragraph::RegularFont; font < (int)Paragraph::Image; font++) 
	{
		// Release font caches in font map.
		PangoFT2FontMap* font_map = PANGO_FT2_FONT_MAP(pango_context_get_font_map(panfo_info[font].pango_context));
		pango_ft2_font_map_substitute_changed(font_map);

		// 	Release pango context, this will release font map and font description as well.
		if (panfo_info[font].pango_context != NULL)
		{
			g_object_unref(panfo_info[font].pango_context);
			panfo_info[font].pango_context = NULL;
		}
	}
}

bool FB2View::paginate(const Position& start_pos)
{
	TRACEFUNCTION();
	LOGPRINTF("start_pos(%i,%i), page_count(%d)", start_pos.paragraph,start_pos.offset,pages.size());
    Position in_pos(0, 0);
    Position out_pos;
    bool forward = true;

    size_t page_count = pages.size();
    if (page_count == 0)
    {
        in_pos = start_pos;
    }
    else if (pages.back().end != Position(0, 0))
    {
        // The forward pagination is not finished yet.
        in_pos = pages.back().end;
    }
    else if (pages.front().start != Position(0, 0))
    {
        // The backward pagination is not finished yet.
        in_pos = pages.front().start;
        forward = false;
    }
    else
    {
		LOGPRINTF("completed");
        // The pagination is complete.
        return true;
    }

	bool finished = false;
    if (forward)
    {
        finished = !calculate_next_page_pos(in_pos, out_pos);
		pages.push_back(PageInfo(in_pos, out_pos));
    }
    else
    {
        finished = !calculate_prev_page_pos(in_pos, out_pos);
		pages.push_front(PageInfo(out_pos, in_pos));
    }
    return finished;
}

bool FB2View::get_anchor_by_page(unsigned int page_index, Position& pos)
{
	TRACEFUNCTION();
    if (pages.size() == 0){
        // Pagination hasn't been started yet.
        if (page_index == 0){
            // We can render page 0 despite pagination is not ready.
            pos = Position(0, 0);
            return true;
        }
        else
            return false;
    }
    else{
        // Pagination starts, but maybe partial is successful.
        if (page_index >= pages.size())
            return false;
        else{
            pos = pages[page_index].start;
            return true;
        }
    }
}

PangoLayout* FB2View::create_layout(const Paragraph* par, int len)
{
//	TRACEFUNCTION();
	PangoLayout *layout;
	if(par->font != Paragraph::Image){
		layout = pango_layout_new(panfo_info[par->font].pango_context);
		assert(layout != NULL);

		pango_layout_set_width(layout, client_width * PANGO_SCALE);
		pango_layout_set_spacing(layout, DEFAULT_LINE_SPACING * PANGO_SCALE);
		pango_layout_set_wrap(layout, PANGO_WRAP_WORD);
		pango_layout_set_justify(layout, FALSE);
        pango_layout_set_indent(layout, DEFAULT_PARA_SPACING * PANGO_SCALE);

		PangoAlignment align_local[]={PANGO_ALIGN_LEFT,PANGO_ALIGN_CENTER,PANGO_ALIGN_RIGHT};
        pango_layout_set_alignment(layout, align_local[par->align]);

		pango_layout_set_auto_dir(layout, TRUE);
		pango_layout_set_text(layout, par->text.c_str(), len==-1 ? par->text.length() : len);
		pango_layout_set_single_paragraph_mode(layout, TRUE);
	}else{
		ERRORPRINTF("Creating layout for Image !!!!");
		layout = pango_layout_new(panfo_info[0].pango_context);
		pango_layout_set_width(layout, client_width * PANGO_SCALE);
		pango_layout_set_spacing(layout, DEFAULT_LINE_SPACING * PANGO_SCALE);
		pango_layout_set_wrap(layout, PANGO_WRAP_WORD);
		pango_layout_set_justify(layout, TRUE);
		pango_layout_set_text(layout, " ", len==-1 ? 1 : len);
		pango_layout_set_single_paragraph_mode(layout, TRUE);
	}
    return layout;
}

int calculate_line_height(PangoLayout *layout)
{
	PangoRectangle  rect;
	PangoLayoutIter *iter = pango_layout_get_iter(layout);
	PangoLayoutLine *pango_line = pango_layout_iter_get_line_readonly(iter);
	pango_layout_line_get_pixel_extents(pango_line, NULL, &rect);
	pango_layout_iter_free(iter);
	return rect.height;
}

bool FB2View::calculate_prev_page_pos(const Position& in_pos, Position& out_pos)
{
	TRACEFUNCTION();
    // Check if we are already the first page.
    if (in_pos == Position(0, 0)){
        return false;
    }

    // Check if we can get next page anchor from page table.
    size_t page_count = pages.size();
    for (unsigned int i=0; i<page_count; i++){
        if (pages[i].end == in_pos){
            out_pos = pages[i].start;
            return true;
        }
    }

    g_mutex_lock(pango_mutex);
    unsigned int start_paragraph = in_pos.paragraph;
    unsigned int start_offset    = in_pos.offset;
    PangoLayout *layout = NULL;
    if (start_offset == 0)
        start_paragraph--;

    unsigned int total_lines_offset = 0;
    while (total_lines_offset < client_height){
        if (static_cast<int>(start_paragraph) < 0){
            if (in_pos > Position(0, 0)){
				blank_lines_offset = client_height - total_lines_offset;
            }
            break;
        }

        if (layout != NULL){
            g_object_unref(layout);
            layout = NULL;
        }
        const Paragraph* par = model->get_paragraph(start_paragraph);
		unsigned int total_offset = total_lines_offset;
		int line_count;
		unsigned int line_height;
		unsigned int line_height_with_spacing;
		if(par->font!=Paragraph::Image){
			layout = create_layout(par);
			line_height = calculate_line_height(layout);
			line_height_with_spacing = line_height + DEFAULT_LINE_SPACING;
	
			line_count = pango_layout_get_line_count(layout);
			if (start_offset != 0){
				// We are calculating from the second half part of a paragraph
				int line_offset, x_pos;
				pango_layout_index_to_line_x(layout, start_offset, FALSE, &line_offset, &x_pos);
				line_count = line_offset;
			}
	
			total_offset += line_count*line_height_with_spacing;
		}else{
			LOGPRINTF("model->get_build_image");
			const FB2Model::ImageDescr* image = model->get_build_image(par->text,
				   								client_width, client_height,
												par->align==Paragraph::FullPage?true:false);
			if(image)
				total_offset += image->size.cy;
		}
        if (total_offset < client_height){
            // We have enough space to load more paragraphs for current page
            total_lines_offset = total_offset;
            start_paragraph--;
            start_offset = 0;
        }
        else if (par->align==Paragraph::FullPage || total_offset == client_height){
            out_pos.paragraph = start_paragraph;
            out_pos.offset = 0;
            break;
        }
        else{
            // The remained space is not enough for current paragraph
			if(par->font!=Paragraph::Image) {
				PangoLayoutIter *iter = pango_layout_get_iter(layout);
				unsigned int line_index = line_count - (client_height - total_lines_offset)/line_height_with_spacing;
				for (unsigned int i=0; i<line_index; i++){
					pango_layout_iter_next_line(iter);
				}

				out_pos.paragraph = start_paragraph;
				out_pos.offset = pango_layout_iter_get_index(iter);
				pango_layout_iter_free(iter);
			}else{
				out_pos.paragraph = start_paragraph;
				out_pos.offset = 0;
			}
            break;
        }
    }

    if (layout != NULL){
        g_object_unref(layout);
        layout = NULL;
    }
    g_mutex_unlock(pango_mutex);
	LOGPRINTF("in_pos(%i,%i), out_pos(%i,%i),start_paragraph(%i)", in_pos.paragraph,in_pos.offset,out_pos.paragraph,out_pos.offset,start_paragraph);
    return (static_cast<int>(start_paragraph) >= 0);
}

bool FB2View::calculate_next_page_pos(const Position& in_pos, Position& out_pos)
{
	TRACEFUNCTION();
	LOGPRINTF("in_pos(%i,%i), out_pos(%i,%i)", in_pos.paragraph,in_pos.offset,out_pos.paragraph,out_pos.offset);

	/// Check if we can get next page anchor from page table.
    size_t page_count = pages.size();
    for (unsigned int i=0; i<page_count; i++){
        if (pages[i].start == in_pos){
            out_pos = pages[i].end;
            return out_pos != Position(0, 0);
        }
    }

    unsigned int start_paragraph = in_pos.paragraph;
    unsigned int start_offset    = in_pos.offset;
    unsigned int total_lines_offset = 0;

    g_mutex_lock(pango_mutex);
    PangoLayout *layout = NULL;
	LOGPRINTF("total_lines_offset(%i) < client_height(%i)", total_lines_offset, client_height);
	while (total_lines_offset < client_height){
        if (start_paragraph >= model->get_paragraph_count()){
			LOGPRINTF("break as start_paragraph(%d) >= model->get_paragraph_count(%d)", start_paragraph, model->get_paragraph_count());
            break;
		}

        if (layout != NULL){
            g_object_unref(layout);
            layout = NULL;
        }
        const Paragraph* par = model->get_paragraph(start_paragraph);
		unsigned int total_offset = total_lines_offset;
		unsigned int line_height;
		unsigned int line_height_with_spacing;
		int line_offset = 0;
		//LOGPRINTF("start_paragraph=>%i, total_lines_offset=> %i", start_paragraph, total_lines_offset);
		if(par->font!=Paragraph::Image){
			layout = create_layout(par);
			line_height = calculate_line_height(layout);
			line_height_with_spacing = line_height + DEFAULT_LINE_SPACING;
	
			int line_count = pango_layout_get_line_count(layout);
			if (start_offset != 0){
				// We are calculating from the second half part of a paragraph
				int x_pos;
				pango_layout_index_to_line_x(layout, start_offset, FALSE, &line_offset, &x_pos);
				LOGPRINTF("line_count=>%i,line_offset=>%i, new line_count=>%i", line_count,line_offset,(line_count-line_offset));
				line_count -= line_offset;
			}
			total_offset += line_count*line_height_with_spacing;
		}else{
			LOGPRINTF("model->get_build_image");
			const FB2Model::ImageDescr* image = model->get_build_image(par->text,
				   								client_width, client_height,
												par->align==Paragraph::FullPage?true:false);
			if(image)
				total_offset += image->size.cy;
		}
		if (total_offset < client_height){
            // We have enough space to load more paragraphs for current page
            total_lines_offset = total_offset;
            start_paragraph++;
            start_offset = 0;
			//LOGPRINTF("total_offset(%i) < client_height(%i),start_paragraph(%i)", total_offset,client_height,start_paragraph);
        }
        else if (par->align==Paragraph::FullPage || total_offset == client_height){
			start_paragraph++;
            if (start_paragraph < model->get_paragraph_count()){
                out_pos.paragraph = start_paragraph;
                out_pos.offset = 0;
            }
			//LOGPRINTF("total_offset(%i) == client_height,  start_paragraph(%i), out_pos.paragraph(%i)", total_offset, start_paragraph, out_pos.paragraph);
            break;
        }
        else{
            // The remained space is not enough for current paragraph
			if(par->font!=Paragraph::Image){
				PangoLayoutIter *iter = pango_layout_get_iter(layout);
				unsigned int line_index = line_offset + (client_height - total_lines_offset)/line_height_with_spacing;
				//LOGPRINTF("line_index=>%i,line_offset=>%i,client_height=>%i,total_lines_offset=>%i,line_height_with_spacing=>%i", line_index,line_offset,client_height,total_lines_offset,line_height_with_spacing);
				for (unsigned int i=0; i<line_index; i++){
					pango_layout_iter_next_line(iter);
				}
	
				out_pos.offset = pango_layout_iter_get_index(iter);
				out_pos.paragraph = start_paragraph;
				pango_layout_iter_free(iter);
				//LOGPRINTF("else remaining text paragraph for next: out_pos.offset(%d),out_pos.paragraph(%d)", out_pos.offset,out_pos.paragraph);
			}else{
				out_pos.paragraph = start_paragraph;
				out_pos.offset = 0;
				//LOGPRINTF("else remaining image paragraph for next: out_pos.offset(%d),out_pos.paragraph(%d)", out_pos.offset,out_pos.paragraph);
			}
            break;
        }
    }

    if (layout != NULL){
        g_object_unref(layout);
        layout = NULL;
    }
    g_mutex_unlock(pango_mutex);
	LOGPRINTF("in_pos(%i,%i), out_pos(%i,%i),start_paragraph(%d)<get_paragraph_count(%d)", in_pos.paragraph,in_pos.offset,out_pos.paragraph,out_pos.offset,start_paragraph,model->get_paragraph_count());
	return (start_paragraph < model->get_paragraph_count());
}

void FB2View::check_page_table(const Position& rendering_pos)
{
	TRACEFUNCTION();
    size_t page_count = pages.size();
    for (unsigned int i=0; i<page_count; i++){
        if (pages[i].start == rendering_pos)
            return;
    }

    // Can't find page boundaries for specified anchor.
    // Re-pagination is needed.
    is_repagination_needed = true;
}

unsigned int FB2View::get_page_index_by_anchor(const Position& anchor)
{
	TRACEFUNCTION();
	LOGPRINTF("anchor(%i,%i),page_count(%d)", anchor.paragraph,anchor.offset,pages.size());
    /// Check if we can get next page anchor from page table.
    size_t page_count = pages.size();

    // Always return 0 if pagination is not complete.
    if (page_count == 0 || pages[page_count-1].end != Position(0, 0))
        return 0;

    for (unsigned int i=0; i<page_count; i++){
        if (pages[i].end > anchor && pages[i].start <= anchor)
            return i;
    }

    return static_cast<unsigned int>(page_count-1);
}

Position FB2View::get_page_anchor_by_anchor(const Position& anchor)
{
	TRACEFUNCTION();
    /// Check if we can get next page anchor from page table.
    size_t page_count = pages.size();
    for (unsigned int i=0; i<page_count; i++){
        if (pages[i].end > anchor && pages[i].start <= anchor)
            return pages[i].start;
    }

    if (page_count>0 && pages[page_count-1].end == Position(0, 0))
        return pages[page_count-1].start;

    return Position(anchor.paragraph, 0);
}

unsigned int FB2View::get_current_page_index()
{
    return get_page_index_by_anchor(rendering_pos);
}

PluginStatus FB2View::render_cover_page(const int width, const int height, 
										PluginBitmapAttributes *cover_page)
{
	TRACEFUNCTION();
	LOGPRINTF("model->get_build_image");
	const FB2Model::ImageDescr* image = model->get_build_image(model->get_book_info().cover_page.image_name,
													 width, height, true);
	if(image) {
		memcpy((void*)cover_page->data,image->bits,image->row_stride*image->size.cy);
		return PLUGIN_OK;
	}
	return PLUGIN_FAIL;
}

Position FB2View::render(unsigned char *bmp, const Position& start_pos)
{
	TRACEFUNCTION();
	LOGPRINTF("start_pos(%i,%i)", start_pos.paragraph,start_pos.offset);
    if (is_repagination_needed){
        pages.clear();
        //blank_lines = 0;
		blank_lines_offset = 0;
		Position _start_pos(0,0);
        ctrl->paginate(_start_pos, true /* Send start notification */);
        is_repagination_needed = false;
    }

    g_mutex_lock(pango_mutex);
    unsigned int start_paragraph = start_pos.paragraph;
    unsigned int start_offset    = start_pos.offset;
    PangoLayout *layout = NULL;
	unsigned int line_height = 10;
	unsigned int line_height_with_spacing = 10;

    rendering_pos = start_pos;

    unsigned int total_lines_offset = 0;
    if (start_pos == Position(0, 0)){
        total_lines_offset = blank_lines_offset;
    }
    int y_offset = total_lines_offset*PANGO_SCALE;

    // If we don't start with a new paragraph, then we must render the
    // remained part of current paragraph first.
    if (start_offset != 0){
        if (layout != NULL){
            g_object_unref(layout);
            layout = NULL;
        }
        const Paragraph *par = model->get_paragraph(start_paragraph);
		if(par->font!=Paragraph::Image){

			layout = create_layout(par);
			line_height = calculate_line_height(layout);
			line_height_with_spacing = line_height + DEFAULT_LINE_SPACING;
	
			PangoLayoutIter *iter = pango_layout_get_iter(layout);
	
			int line_offset, x_pos;
			pango_layout_index_to_line_x(layout, start_offset, FALSE, &line_offset, &x_pos);
			y_offset = line_height_with_spacing * line_offset * PANGO_SCALE;
	
			int baseline = 0;
			for (int i=0; i<line_offset; i++){
				pango_layout_iter_next_line(iter);
			}
	
			//WARNPRINTF("Rendering text paragraph(%i): x=>%i,y=>%i,line_height=%i",start_paragraph,left_margin,(line_height_with_spacing*line_offset),line_height_with_spacing);
			do{
				baseline = pango_layout_iter_get_baseline(iter) - y_offset;
				PangoLayoutLine *pango_line = pango_layout_iter_get_line_readonly(iter);
				render_single_line(bmp,
								   pango_line,
								   panfo_info[par->font].font_hash_code,
								   left_margin * PANGO_SCALE,
								   baseline+top_margin*PANGO_SCALE,
								   (display_width+3)/4*4);

				total_lines_offset += line_height_with_spacing;
				if ((total_lines_offset+line_height) > client_height){
					// Calculate the offset of the rightmost character.
					int index, trailing;
					pango_layout_line_x_to_index(pango_line, client_width - 1, &index, &trailing);
					start_offset = index;
					pango_layout_iter_free(iter);
					goto RenderEnd;
				}
			} while (pango_layout_iter_next_line(iter));
	
			// Release iter since we finish with it.
			pango_layout_iter_free(iter);
	
			start_paragraph++;
			y_offset = total_lines_offset*PANGO_SCALE;
		}else{
			WARNPRINTF("Unexpected paragraph of image %s",par->text.c_str());
			//assert(par->font!=Paragraph::Image);
		}
    }

    // Render the remained paragraphs
    while (start_paragraph < model->get_paragraph_count()){
        // Render current paragraph
        if (layout != NULL){
            g_object_unref(layout);
            layout = NULL;
        }
        const Paragraph* par = model->get_paragraph(start_paragraph);
		if(par->font!=Paragraph::Image){
			layout = create_layout(par);
			line_height = calculate_line_height(layout);
			line_height_with_spacing = line_height + DEFAULT_LINE_SPACING;
			//WARNPRINTF("Rendering text paragraph(%i): x=>%i,y=>%i,line_height=%i",start_paragraph,left_margin,total_lines_offset,line_height_with_spacing);
	
			PangoLayoutIter *iter = pango_layout_get_iter(layout);
			int baseline = 0;
			do
			{
				baseline = pango_layout_iter_get_baseline(iter) + y_offset;
				PangoLayoutLine *pango_line = pango_layout_iter_get_line_readonly(iter);
				render_single_line(bmp,
								   pango_line,
								   panfo_info[par->font].font_hash_code,
								   left_margin * PANGO_SCALE,
								   baseline+top_margin*PANGO_SCALE,
								   (display_width+3)/4*4);
				total_lines_offset += line_height_with_spacing;
				if ((total_lines_offset + line_height) > client_height){
					int index, trailing;
					pango_layout_line_x_to_index(pango_line, client_width - 1, &index, &trailing);
					start_offset = index;
					pango_layout_iter_free(iter);
					goto RenderEnd;
				}
			} while (pango_layout_iter_next_line(iter));
			pango_layout_iter_free(iter);
	
			start_paragraph++;
			y_offset = total_lines_offset * PANGO_SCALE;
		}else{
			//LOGPRINTF("model->get_build_image");
			const FB2Model::ImageDescr* image = model->get_build_image(par->text,
				   								client_width, client_height,
											   par->align==Paragraph::FullPage?true:false);
			
			if(image) {
				if ((total_lines_offset + image->size.cy) > client_height){
					start_offset = 0;
					goto RenderEnd;
				}
				//WARNPRINTF("Rendering image paragraph(%i): x=>%i,y=>%i",start_paragraph,left_margin,total_lines_offset);
				Rect clientRect(left_margin, top_margin+total_lines_offset, client_width, client_height);
				image->render(bmp,clientRect,display_width, display_height);
				total_lines_offset += image->size.cy;
			}else{
				ERRORPRINTF("Fail to find image (%s)", par->text.c_str());
			}
			start_paragraph++;
			y_offset = total_lines_offset * PANGO_SCALE;
			if(par->align==Paragraph::FullPage || 
			   (total_lines_offset + line_height) > client_height){
				start_offset = 0;
				goto RenderEnd;
			}
		}
    }

RenderEnd:
    // Release pango layout generated by render.
    if (layout != NULL){
        g_object_unref(layout);
        layout = NULL;
    }
    g_mutex_unlock(pango_mutex);

    return Position(start_paragraph, start_offset);
}

bool FB2View::set_font_size(int _font_size)
{
	TRACEFUNCTION();
	LOGPRINTF("_font_size(%i)", _font_size);

	if (font_size == _font_size){
        return true;
    }

    font_size = _font_size;

    g_mutex_lock(pango_mutex);

	pango_context_final();
	pango_context_init();


    // Change the repagination flag to true.
    is_repagination_needed = true;

    // Clear font cache.
    FontCache& font_cache = FontCache::instance();
    font_cache.clear();

    g_mutex_unlock(pango_mutex);
    return true;
}

void FB2View::set_display_size(unsigned int width, unsigned int height)
{
	TRACEFUNCTION();
    LOGPRINTF("\n\nwidth=%d, height=%d\n\n", width, height);

	if (width == display_width && height == display_height)
        return;

	g_mutex_lock(pango_mutex);
    if (height != display_height){
        // Display height changed. The current layout can be reused.
        // But we need to re-calculate lines per page.
        display_height = height;
        client_height = display_height - top_margin - bottom_margin;
    }

    if (width != display_width){
        display_width = width;
        client_width = display_width - left_margin - right_margin;
    }

    // Change the repagination flag to true.
    is_repagination_needed = true;
    g_mutex_unlock(pango_mutex);
}

bool FB2View::map_doc_pos_to_view_pos(const Position& page_anchor, const Position& doc_pos,
                                      ViewPosition& view_pos, bool trailing)
{
	TRACEFUNCTION();
    if (doc_pos < page_anchor)
        return false;

    // Calculate the line offset and x offset inside layout corresponding to doc pos.
    const Paragraph* par = model->get_paragraph(doc_pos.paragraph);
	int line_offset_pos_offset = 0;
	PangoLayout *layout = NULL;
	unsigned int line_height_with_spacing = 0;
	int line_offset_pos = 0;
	int x_offset_pos = 0;
	if(par->font!=Paragraph::Image){
		layout = create_layout(par);
		line_height_with_spacing = calculate_line_height(layout) + DEFAULT_LINE_SPACING;
	
		pango_layout_index_to_line_x(layout, doc_pos.offset, trailing, &line_offset_pos, &x_offset_pos);
		line_offset_pos_offset = line_offset_pos * line_height_with_spacing;
	}else{
		LOGPRINTF("model->get_build_image");
		const FB2Model::ImageDescr* image = model->get_build_image(par->text,
				   								client_width, client_height,
												par->align==Paragraph::FullPage?true:false);
		if(image)
			line_offset_pos_offset += image->size.cy;
	}

    unsigned int total_lines_offset = 0;
    unsigned int start_paragraph = page_anchor.paragraph;
    unsigned int start_offset    = page_anchor.offset;
    if (start_paragraph == 0 && start_offset == 0)
        total_lines_offset += blank_lines_offset;

    if (start_offset != 0){
        // The page does not start with a new paragraph.
        if (layout != NULL){
            g_object_unref(layout);
            layout = NULL;
        }
        const Paragraph* par = model->get_paragraph(start_paragraph);
		if(par->font!=Paragraph::Image){
			layout = create_layout(par);
			line_height_with_spacing = calculate_line_height(layout) + DEFAULT_LINE_SPACING;
	
			int line_offset_start, x_pos;
			pango_layout_index_to_line_x(layout, start_offset, FALSE, &line_offset_start, &x_pos);
	
			if (doc_pos.paragraph == start_paragraph){
				total_lines_offset = (line_offset_pos - line_offset_start)*line_height_with_spacing;
				goto MapEnd;
			}
			else{
				total_lines_offset += (pango_layout_get_line_count(layout) - line_offset_start)*line_height_with_spacing;
				start_paragraph++;
			}
		}else{
			ERRORPRINTF("Image paragraph unexpected here name (%s)", par->text.c_str());
		}
    }

    for (; start_paragraph != doc_pos.paragraph; start_paragraph++){
        if (layout != NULL){
            g_object_unref(layout);
            layout = NULL;
        }
        const Paragraph* par = model->get_paragraph(start_paragraph);
		if(par->font!=Paragraph::Image){
			layout = create_layout(par);
			line_height_with_spacing = calculate_line_height(layout) + DEFAULT_LINE_SPACING;
			total_lines_offset += pango_layout_get_line_count(layout)*line_height_with_spacing;
		}
		else{
			LOGPRINTF("model->get_build_image");
			const FB2Model::ImageDescr* image = model->get_build_image(par->text,
				   								client_width, client_height,
												par->align==Paragraph::FullPage?true:false);
			if(image){
				total_lines_offset += image->size.cy;
			}
			else{
				ERRORPRINTF("Image not found. name (%s)", par->text.c_str());
			}
		}
    }

    total_lines_offset += line_offset_pos_offset;

MapEnd:
    if (layout != NULL){
        g_object_unref(layout);
        layout = NULL;
    }
    if (total_lines_offset < client_height){
        view_pos.x = PANGO_PIXELS(x_offset_pos) + left_margin;
        view_pos.y = top_margin + total_lines_offset;
        return true;
    }
    return false;
}

bool FB2View::map_view_pos_to_doc_pos(const Position&     page_anchor,
                                       const ViewPosition& view_pos,
                                       Position&           doc_pos)
{
	TRACEFUNCTION();
    int x_offset_pos = view_pos.x - left_margin;
    int y_offset_pos = view_pos.y - top_margin;

    unsigned int start_paragraph = page_anchor.paragraph;
    unsigned int start_offset    = page_anchor.offset;
    unsigned int line_height_with_spacing;

    if (start_paragraph == 0 && start_offset == 0)
        y_offset_pos -= blank_lines_offset;

    PangoLayout *layout = NULL;
    bool found = false;

	const Paragraph* par = 0;
    if (start_offset != 0){
        par = model->get_paragraph(start_paragraph);
		if(par->font!=Paragraph::Image){
			layout = create_layout(par);
			line_height_with_spacing = calculate_line_height(layout) + DEFAULT_LINE_SPACING;
	
			int line_offset_start, x_pos;
			pango_layout_index_to_line_x(layout, start_offset, FALSE, &line_offset_start, &x_pos);
	
			int lines_left = pango_layout_get_line_count(layout) - line_offset_start;
			if (static_cast<int>(lines_left * line_height_with_spacing) > y_offset_pos){
				// The corresponding doc pos is inside this paragraph.
				doc_pos.paragraph = start_paragraph;
	
				// Calculate offset.
				int bytes_index, trailing;
				pango_layout_xy_to_index(layout,
										 x_offset_pos * PANGO_SCALE,
										 (line_offset_start*line_height_with_spacing+y_offset_pos) * PANGO_SCALE,
										 &bytes_index, &trailing);
				doc_pos.offset = bytes_index;
				found = true;
				goto MapEnd;
			}
			else{
				y_offset_pos -= lines_left * line_height_with_spacing;
				start_paragraph++;
			}
		}else{
			ERRORPRINTF("Image paragraph unexpected here name (%s)", par->text.c_str());
		}
    }

    for (; start_paragraph < model->get_paragraph_count(); start_paragraph++){
        if (layout != NULL){
            g_object_unref(layout);
            layout = NULL;
        }
        par = model->get_paragraph(start_paragraph);
		if(par->font!=Paragraph::Image){
			layout = create_layout(par);
			line_height_with_spacing = calculate_line_height(layout) + DEFAULT_LINE_SPACING;
	
			int line_offset = pango_layout_get_line_count(layout) * line_height_with_spacing;
			if (line_offset > y_offset_pos)
				break;
			else
				y_offset_pos -= line_offset;
		}else{
			LOGPRINTF("model->get_build_image");
			const FB2Model::ImageDescr* image = model->get_build_image(par->text,
				   								client_width, client_height,
												par->align==Paragraph::FullPage?true:false);
			if(image){
				if (image->size.cy > y_offset_pos)
					break;
				else
					y_offset_pos -= image->size.cy;
			}
			else{
				ERRORPRINTF("Image not found. name (%s)", par->text.c_str());
			}
		}
    }

    if (start_paragraph == model->get_paragraph_count())
        goto MapEnd;

	if(par->font!=Paragraph::Image){
		int bytes_index, trailing;
		pango_layout_xy_to_index(layout, x_offset_pos * PANGO_SCALE, y_offset_pos * PANGO_SCALE, &bytes_index, &trailing);
		doc_pos.paragraph = start_paragraph;
		doc_pos.offset    = bytes_index;
	}else{
		doc_pos.paragraph = start_paragraph;
		doc_pos.offset    = 0;
	}
    found = true;

MapEnd:
    if (layout)
        g_object_unref(layout);
    return found;
}

// wrongly implemented must calculate bounding rectangle for every line
bool FB2View::get_bounding_rectangles(const Position& page_anchor,
                                       std::vector<Rect>& bounding_rect,
                                       const Range& range)
{
	TRACEFUNCTION();
    // A flag indicating that the start position and end position of specified
    // range is in different pages.
    bool span_pages = false;
    ViewPosition view_pos_start, view_pos_end;

    if (!map_doc_pos_to_view_pos(page_anchor, range.start, view_pos_start, false))
    {
        // The range's start position is not inside the page which starts with
        // page_anchor.
        return false;
    }

    if (!map_doc_pos_to_view_pos(page_anchor, range.end, view_pos_end, true))
    {
        // The range's end position is not inside the page which starts with
        // page_anchor.
        span_pages = true;
    }

    //unsigned int line_height_with_spacing = line_height + DEFAULT_LINE_SPACING;
	unsigned int line_height;
	const Paragraph* par = model->get_paragraph(page_anchor.paragraph);
	if(par->font!=Paragraph::Image){
		PangoLayout *layout = create_layout(par);
		line_height = calculate_line_height(layout);
		g_object_unref(layout);
	}else{
		LOGPRINTF("model->get_build_image");
		const FB2Model::ImageDescr* image = model->get_build_image(par->text,
			   								client_width, client_height,
											par->align==Paragraph::FullPage?true:false);
		if(image){
			line_height = image->size.cy;
		}
		else{
			ERRORPRINTF("Image not found. name (%s)", par->text.c_str());
		}
	}

    // Check if the start position and end position are in the same line.
    if (view_pos_start.y == view_pos_end.y){
        // The start position and end position are in the same line.
        bounding_rect.push_back(Rect(view_pos_start.x,
                                     view_pos_start.y + DEFAULT_LINE_SPACING,
                                     view_pos_end.x - view_pos_start.x,
                                     line_height));
    }
    else{
        // More lines.
        bounding_rect.push_back(Rect(view_pos_start.x,
                                     view_pos_start.y + DEFAULT_LINE_SPACING,
                                     display_width - view_pos_start.x - right_margin,
                                     line_height));

		if(par->font!=Paragraph::Image){
			unsigned int line_height_with_spacing = line_height + DEFAULT_LINE_SPACING;
			for (unsigned int y_offset = view_pos_start.y+line_height_with_spacing;
				 static_cast<int>(y_offset) < view_pos_end.y &&
				 y_offset < (top_margin + client_height);
				 y_offset += line_height_with_spacing)
			{
				bounding_rect.push_back(Rect(left_margin,
											 y_offset + DEFAULT_LINE_SPACING,
											 client_width,
											 line_height));
			}
		}else{
			ERRORPRINTF("Not implemented yet"); // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		}

        if (!span_pages){
            bounding_rect.push_back(Rect(left_margin,
                                         view_pos_end.y + DEFAULT_LINE_SPACING,
                                         view_pos_end.x - left_margin,
                                         line_height));
        }
    }
    return true;
}

}
