/*
 * File Name: text_controller.cpp
 */

/*
 * This file is part of uds-plugin-plaintext.
 *
 * uds-plugin-plaintext is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * uds-plugin-plaintext is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#include "log.h"
#include "fb2_tasks.h"
#include "fb2_view.h"
#include "fb2_controller.h"

namespace fb2
{

FB2Controller::FB2Controller(FB2Model *_model)
: model(_model)
, view(0)
, pagination_paused(false)
{
}

FB2Controller::~FB2Controller()
{
}

void FB2Controller::set_view(FB2View *_view)
{
	view = _view;
	if(view) view->set_controller(this);
}

void FB2Controller::start()
{
    // Launch worker thread.
    thread.start();
}

void FB2Controller::stop()
{
	TRACEFUNCTION();
    // Stop worker thread, wait until all tasks done.
    thread.stop();
}

void FB2Controller::render(unsigned int id, unsigned int page_index, void *data, unsigned char* bmp)
{
	TRACEFUNCTION();
	LOGPRINTF("id=>%d, page_index=>%d", id,page_index);
    Position start_pos;
    if (view->get_anchor_by_page(page_index, start_pos))
    {
        render(id, start_pos, data, bmp);
    }
    else
    {
        ERRORPRINTF("Can't render page %d because it does not exist.\n", page_index);
    }
}

void FB2Controller::render(unsigned int id, const Position& start_pos, void *data, unsigned char* bmp)
{
    LOGPRINTF("About to launch render task.");
    view->check_page_table(start_pos);
    thread.prepend_task(new RenderTask(id, view, start_pos, data, bmp), true);
}

void FB2Controller::paginate(const Position& start, bool send_start_notification)
{
	TRACEFUNCTION();
	LOGPRINTF("start(%i,%i), send_start_notification=>%i", start.paragraph,start.offset,send_start_notification?1:0);
    pagination_pending = true;
    PaginationTask *task = new PaginationTask(view, start, !send_start_notification);
    task->abort_signal.add_slot(this, &FB2Controller::on_pagination_aborted);
    view->pagination_end_signal.add_slot(this, &FB2Controller::on_pagination_finished);
    thread.append_task(task);
}

void FB2Controller::on_pagination_finished(unsigned int, unsigned int)
{
    if (!pagination_paused)
    {
        pagination_pending = false;
    }
}

void FB2Controller::on_pagination_aborted(const Position& start)
{
	TRACEFUNCTION();
	LOGPRINTF("start(%i,%i)", start.paragraph,start.offset);
    // If repagination is needed, we don't need to launch a same
    // pagination again. A new pagination task will be launched
    // by render task soon.
    //
    // Also when pagination is paused, we don't want to restart
    // the pagination immediately
    //
    if ((!pagination_paused) && (!view->need_repagination()))
    {
        paginate(start, false /* don't send pagination start notification */);
    }
}

void FB2Controller::pause_pagination()
{
    TRACEFUNCTION();

    if (pagination_pending)
    {
        pagination_paused = true;

        unsigned int total_pages = view->get_page_count();
        unsigned int current_page = view->get_current_page_index();

        // Reporting pagination done to UDS through our internal pagination end signal
        // UDS probably doesn't receive this event because the document window is
        // about to be deactivated (hence, UDS will unsubscribe from plugin events),
        // however I still think it is good practice to try to notify UDS anyway.
        // [MdB]
        view->pagination_end_signal.broadcast(current_page, total_pages);

        PaginationAbortTask *task = new PaginationAbortTask(view);
        thread.prepend_task(task, true /* Cancel running task */);
    }
}

void FB2Controller::restart_pagination()
{
	TRACEFUNCTION();
	LOGPRINTF("pagination_pending=>%i", pagination_pending);


    // Only restart pagination if there is a need
    if (pagination_pending)
    {
        pagination_paused = false;

        Position start_pos;
        view->check_page_table(start_pos);
		LOGPRINTF("start(%i,%i)", start_pos.paragraph,start_pos.offset);
        paginate(start_pos, false /* don't send pagination start notification */);
    }
}


void FB2Controller::search(SearchContext* search_context)
{
    SearchTask* task = new SearchTask(model, search_context);
    task->abort_signal.add_slot(this, &FB2Controller::on_search_aborted);
    thread.prepend_task(task, true);
}

void FB2Controller::on_search_aborted(SearchContext* sc)
{
    SearchTask* task = new SearchTask(model, sc);
    task->abort_signal.add_slot(this, &FB2Controller::on_search_aborted);
    thread.prepend_task(task, false);
}

}

