/*
 * File Name: text_tasks.cpp
 */

/*
 * This file is part of uds-plugin-plaintext.
 *
 * uds-plugin-plaintext is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * uds-plugin-plaintext is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#include "plugin_render_result.h"

#include "log.h"
#include "fb2_tasks.h"
#include "fb2_view.h"

namespace fb2
{

InitializationTask::InitializationTask(FB2View *_view)
: view(_view)
{
}

InitializationTask::~InitializationTask()
{
}

void InitializationTask::execute()
{
    view->initialize();
}

DeinitializationTask::DeinitializationTask(FB2View *_view)
: view(_view)
{
}

DeinitializationTask::~DeinitializationTask()
{
}

void DeinitializationTask::execute()
{
	TRACEFUNCTION();
    view->deinitialize();
}

PaginationTask::PaginationTask(FB2View *_view, const Position& _start, bool _is_child)
: view(_view), start_pos(_start), is_child(_is_child)
{
}

PaginationTask::~PaginationTask()
{
}

void PaginationTask::execute()
{
	TRACEFUNCTION();
    if (!is_child)
    {
        // Tell listener that pagination starts.
        LOGPRINTF("About to send pagination start signal.");
        view->pagination_start_signal.broadcast(1);
    }

    bool pagination_done = false;
    while (!is_aborted())
    {
        if (view->paginate(start_pos))
        {
            // Pagination is complete.
            pagination_done = true;
            break;
        }
    }

    if (pagination_done)
    {
        // Tell listener that pagination is done.
        unsigned int total_pages = view->get_page_count();
        unsigned int current_page = view->get_current_page_index();
        LOGPRINTF("About to send pagination done signal.");
        view->pagination_end_signal.broadcast(current_page, total_pages);
    }
    else
    {
        // We are aborted.
        LOGPRINTF("Pagination aborted...");
        abort_signal.broadcast(start_pos);
    }
}


PaginationAbortTask::PaginationAbortTask(FB2View *_view)
: view(_view)
{
    TRACEFUNCTION();
}

PaginationAbortTask::~PaginationAbortTask()
{
    TRACEFUNCTION();
}

void PaginationAbortTask::execute()
{
    TRACEFUNCTION();

    //
    // Dummy task to abort pagination
    //

    // Example usage:
    //   PaginationAbortTask *task = new PaginationAbortTask();
    //   thread.prepend_task(task, true /* Cancel running task */);
}

RenderTask::RenderTask(unsigned int     _id,
                       FB2View        *_view,
                       const Position&  _pos,
                       void            *_user_data,
                       unsigned char   *_bmp)
: id(_id)
, view(_view)
, start_pos(_pos)
, user_data(_user_data)
, bmp(_bmp)
{
}

RenderTask::~RenderTask()
{
}

void RenderTask::execute()
{
	TRACEFUNCTION();
    Position end_pos = view->render(bmp, start_pos);

    // Tell listener that render done.
    view->render_done_signal.broadcast(id, start_pos, end_pos, user_data);
}

SearchTask::SearchTask(FB2Model *_model, SearchContext* _sc)
: model(_model), sc(_sc)
{
}

SearchTask::~SearchTask()
{
}

void SearchTask::execute()
{
	TRACEFUNCTION();
    std::vector<Range> result_ranges;
    bool search_done = false;
    while (!is_aborted() &&
           model->get_aborting_search_task_id() != sc->search_id)
    {
        if (model->search(result_ranges, sc))
        {
            // We reached the start/end of the document,
            // or we find an occurrence if the SearchType is SEARCH_NEXT.
            search_done = true;
            break;
        }
    }

    if (is_aborted())
    {
        LOGPRINTF("Search task is aborted by another task...");
        abort_signal.broadcast(sc);
    }

    if (search_done)
    {
        // Tell listeners search task is done.
        model->search_done_signal.broadcast(result_ranges, sc);
    }
    else
    {
        // We are aborted, do nothing.
        LOGPRINTF("Search task is aborted by user...");
    }
}

}

