/*
 * File Name: text_model.cpp
 */

/*
 * This file is part of uds-plugin-plaintext.
 *
 * uds-plugin-plaintext is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * uds-plugin-plaintext is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <expat.h>
#include <memory>
//#include <iconv.h>
#include <gio/gio.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <libermetadb/ermetadb.h>

#include "fb2_encoding.h"
#include "../minizip/unzip.h"
#include "fb2_model.h"
#include "utils.h"
#include "log.h"

#ifdef XML_LARGE_SIZE
#if defined(XML_USE_MSC_EXTENSIONS) && _MSC_VER < 1400
#define XML_FMT_INT_MOD "I64"
#else
#define XML_FMT_INT_MOD "ll"
#endif
#else
#define XML_FMT_INT_MOD "l"
#endif


namespace fb2
{

/// You must feed enca with some text to get the possible encoding.
/// If the sample text is too short, you will get "Unknown" encoding.
static const int SAMPLE_LEN = 1024 * 4;
static const int BLOCK_SIZE = 4096;
static const std::string TARGET_CODESET = "utf-8";

FB2Model::ImageDescr::~ImageDescr()
{
	delete [] bits;
}
static int get_rowstride(int width, int bytes_per_pixel, int alignment)
{
    int rowstride, original_rowstride = width * bytes_per_pixel;
    if (original_rowstride % alignment)
    {
        int padding = alignment - (original_rowstride % alignment);
        rowstride = original_rowstride + padding;
    }else
        rowstride = original_rowstride;
    return rowstride;
}

#define WHITE_BACKGROUND 255

static unsigned char* dither_32bits_to_8bits(unsigned char* data, int w, int h, int rowstride)
{
    LOGPRINTF("entry %p, %d, %d, %d", data, w, h, rowstride);

    int n_channels = 4;

    // Get the memory.
    int rowstride_dst = get_rowstride(w, 1, 4);
    unsigned char * dst = new unsigned char[rowstride_dst * h];
    
    // Walk throuth the whole pixels buffer.
    unsigned char r, g, b, a;
    unsigned char * p_src, *p_dst;
 
    // For dithering.
    int c;
    int orig_grey = 0;
    int grey = 0;
    int error = 0;
    int *errors = new int[(rowstride + 2) * 4];
    memset(errors, 0, (rowstride + 2) * 4);
    int error_right = 0;
    int error_down_right = 0;

    for (int y = 0; y < h; y++)
    {
        p_src = (unsigned char*)(data + y * rowstride);
        p_dst = dst + y * rowstride_dst;

        for (int x = 0; x < w; x++)
        {
            r = p_src[0];
            g = p_src[1];
            b = p_src[2];
            a = p_src[3];
            
            // Convert RGBA to grayscale color; 
            // take care of the alpha channel, 
            // but always use WHITE as background color
            // *p_dst = (((r << 8) + (g << 9) + (b << 7)) >> 10);
            // *p_dst = ((r * 307 + g * 604 + b * 113) >> 10);
            c = ((((255 - a) * WHITE_BACKGROUND) 
                + (a * ((r * 307 + g * 604 + b * 113) >> 10))) >> 8);
            c = c & 0x00ff;

            // Get Mono 8 pixel and add error distribution
            orig_grey = c + error_right + errors[x+1];

            // Convert pixel to Mono 4, clip < 0 and > Mono 4
            if (orig_grey <= 0) 
            {
              grey = 0;
            } 
            else if (orig_grey  >= 0x00ff) 
            {
              grey = 0x00ff; // No error for white of 255
            } 
            else 
            {
              grey = (orig_grey & 0x00f0) + ((orig_grey & 0x00f0) >> 4);
            }

            // Put Mono 4 pixel with distributed error back
            *p_dst = grey;

            // Calculate error
            error = orig_grey - grey;

            // Distribute error
            error_right = (error * 7) >> 4;
            errors[x] += (error * 3) >> 4;
            errors[x+1] = ((error * 5) >> 4) + error_down_right;
            error_down_right = error >> 4;
             
            p_src += n_channels;
            p_dst++;
        }
    }

    delete [] errors;
    errors = 0;

    // LOGPRINTF("return %p", dst);
    
    return dst;
}

static unsigned char* dither_24bits_to_8bits(unsigned char* data, int w, int h, int rowstride)
{
    LOGPRINTF("entry %p, %d, %d, %d", data, w, h, rowstride);

    int n_channels = 3;

    // Get the memory.
    int rowstride_dst = get_rowstride(w, 1, 4);
    unsigned char * dst = new unsigned char[rowstride_dst * h];
    
    // Walk throuth the whole pixels buffer.
    unsigned char r, g, b;
    unsigned char * p_src, *p_dst;
 
    // For dithering.
    int c;
    int orig_grey = 0;
    int grey = 0;
    int error = 0;
    int *errors = new int[(rowstride + 2) * 4];
    memset(errors, 0, (rowstride + 2) * 4);
    int error_right = 0;
    int error_down_right = 0;

    for (int y = 0; y < h; y++)
    {
        p_src = (unsigned char*)(data + y * rowstride);
        p_dst = dst + y * rowstride_dst;

        for (int x = 0; x < w; x++)
        {
            r = p_src[0];
            g = p_src[1];
            b = p_src[2];

            // Convert RGB to grayscale color.
            // *p_dst = (((r << 8) + (g << 9) + (b << 7)) >> 10);
            c = ((r * 307 + g * 604 + b * 113) >> 10);
            c = c & 0x00ff;

            // Get Mono 8 pixel and add error distribution
            orig_grey = c + error_right + errors[x+1];

            // Convert pixel to Mono 4, clip < 0 and > Mono 4
            if (orig_grey <= 0) 
            {
              grey = 0;
            } 
            else if (orig_grey  >= 0x00ff) 
            {
              grey = 0x00ff; // No error for white of 255
            } 
            else 
            {
              grey = (orig_grey & 0x00f0) + ((orig_grey & 0x00f0) >> 4);
            }

            // Put Mono 4 pixel with distributed error back
            *p_dst = grey;

            // Calculate error
            error = orig_grey - grey;

            // Distribute error
            error_right = (error * 7) >> 4;
            errors[x] += (error * 3) >> 4;
            errors[x+1] = ((error * 5) >> 4) + error_down_right;
            error_down_right = error >> 4;

            p_src += n_channels;
            p_dst++;
        }
    }

    delete [] errors;
    errors = 0; 

    // LOGPRINTF("return %p", dst);
    return dst;
}

static Size calculate_dimensions(int image_width, int image_height,int client_width, int client_height)
{
	Size size(0,0);
	float width_rel,height_rel;
	width_rel = client_width / (float)image_width;
	height_rel = client_height / (float)image_height;
	if(width_rel > height_rel){
		// zoom on height
		size.cy = image_height * height_rel;
		size.cx = image_width * height_rel;
	}else{
		// zoom on width
		size.cy = image_height * width_rel;
		size.cx = image_width * width_rel;
	}
	return size;
}

bool FB2Model::ImageDescr::build_image(int client_width, int client_height, bool outzoom)
{
	TRACEFUNCTION();
	// 1 calculate 
	if(bits && client_orig.cx == client_width && client_orig.cy == client_height) {
		return true;
	}
	delete [] bits;
	bits = 0;
	GdkPixbufLoader* loader = gdk_pixbuf_loader_new();
	GdkPixbuf* pixbuf = 0;
	if(loader)
	{
		//LOGPRINTF("1a size %d",image_text.length());
		if(gdk_pixbuf_loader_write(loader,
								(const guchar*)image_text.c_str(),
            	                (gsize)image_text.length(),
                	            NULL))
		{
			//LOGPRINTF("2");
			if(gdk_pixbuf_loader_close(loader,NULL))
			{
				//LOGPRINTF("3");
				pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
				if(pixbuf)
					g_object_ref(pixbuf);
			}else{
				ERRORPRINTF("Fail to close loader!!!");
			}
		}else{
			ERRORPRINTF("Can't read file!!!");
		}
		g_object_unref(loader);
	}else{
		ERRORPRINTF("Can't create loader!!!");
	}
	if(!pixbuf)
		return false;

	// remember orig size
	client_orig.cx = client_width;
	client_orig.cy = client_height;

	int image_width = gdk_pixbuf_get_width(pixbuf);
	int image_height = gdk_pixbuf_get_height(pixbuf);
	LOGPRINTF("image_width(%d),image_height(%d)",image_width,image_height);
	Size new_size = calculate_dimensions(image_width, image_height,client_width, client_height);
	// currently I do not want zoom out
	if(!outzoom) {
		if(new_size.cx > image_width) 
			new_size.cx = image_width;
		if(new_size.cy > image_height) 
			new_size.cy = image_height;
	}
	if(new_size.cx != image_width || new_size.cy != image_height){
		GdkPixbuf* pixbuf1 = gdk_pixbuf_scale_simple(pixbuf,new_size.cx,new_size.cy,
													GDK_INTERP_BILINEAR);
		if(pixbuf1) {
			g_object_unref(pixbuf);
			pixbuf = pixbuf1;
		}
	}
	int original_color_depth = gdk_pixbuf_get_bits_per_sample(pixbuf) * gdk_pixbuf_get_n_channels(pixbuf);

    unsigned char* pix_bits = gdk_pixbuf_get_pixels(pixbuf);
    int bytes_per_pixel = original_color_depth / 8;
	int rowstride = gdk_pixbuf_get_rowstride(pixbuf);

	if(bytes_per_pixel == 4)
		bits = dither_32bits_to_8bits(pix_bits, new_size.cx, new_size.cy, rowstride);
	else
		bits = dither_24bits_to_8bits(pix_bits, new_size.cx, new_size.cy, rowstride);
	size = new_size;
	row_stride = get_rowstride(size.cx, 1, 4);
	g_object_unref(pixbuf);
	return true;
}

void FB2Model::ImageDescr::render(unsigned char *bmp, const Rect& clientRect, 
								  int display_width, int display_height) const
{
	int rowstride_dst = get_rowstride(size.cx, 1, 4);

	// set to middle of page
	unsigned int start_x = (display_width-clientRect.x-size.cx)/2;
	for (int y = 0; y < size.cy; y++)
	{
		unsigned char* dst = (unsigned char*)(bmp + (y+clientRect.y)*display_width + clientRect.x+start_x);
		unsigned char* src = bits + y * rowstride_dst;
		int x = size.cx;
		while(x--){
			*dst++ = *src++;
		}
	}
}

FB2Model::FB2Model()
: encoding("utf-8")
, path()
, aborting_search_task_id(0)
{
}

FB2Model::~FB2Model()
{
    if (is_open())
    {
        close();
    }
}

PluginStatus FB2Model::open(const std::string& doc_path)
{
    return open(doc_path, TARGET_CODESET);
}

PluginStatus FB2Model::open(const std::string& doc_path,
                             const std::string& encoding)
{
    PluginStatus result = PLUGIN_FAIL;

    close();

    // Try to open specified file.
    //int file_p = ::open(doc_path.c_str(), "rb");
    struct stat statbuf;
    stat(doc_path.c_str(), &statbuf);
	size_t file_size = statbuf.st_size;
	 
	int file_p = ::open(doc_path.c_str(), O_RDONLY);
    if (file_p < 0)
    {
		ERRORPRINTF("Fail to open file: %s",doc_path.c_str());
        return PLUGIN_ERROR_OPEN_FILE;
    }


    // Update document information.
    path = doc_path;


	//lseek(file_p, 0, SEEK_END);
	//size_t file_size = ftell(file_p);
	//fseek(file_p, 0, SEEK_SET);

	char test_data[10];
	if(::read(file_p, test_data, sizeof(test_data)) != sizeof(test_data)){
		::close(file_p);
		ERRORPRINTF("Fail to read file: %s",doc_path.c_str());
		return PLUGIN_ERROR_OPEN_FILE;
	}

    // Build up paragraphs.
    result = parse_fb2_data(test_data, sizeof(test_data), file_p, file_size);
	::close(file_p);
    if (result != PLUGIN_OK)
    {
        close();
		ERRORPRINTF("Fail to parse file: %s",doc_path.c_str());
        return result;
    }

    if (doc.empty())
    {
        doc.push_back(new Paragraph(0, std::string(" ")));
    }

    return result;
}

void FB2Model::close()
{
    clear();
    encoding.clear();
}

void FB2Model::clear()
{
    // Clear current paragraphs
    for (ParagraphsIter it = doc.begin(); it != doc.end(); ++it)
        delete *it;
    doc.clear();
	info.clear();
	for (ImagesIter it1 = images.begin(); it1 != images.end(); ++it1)
		delete (*it1).second;
	images.clear();
}

void FB2Model::BookInfo::clear()
{
	for (AuthorsIter it1 = authors.begin(); it1 != authors.end(); ++it1)
	{
		delete *it1;
	}
	authors.clear();
	for (ParagraphsIter it = annotations.begin(); it != annotations.end(); ++it)
	{
		delete *it;
	}
	annotations.clear();
}

//------------------------------------------------------------------------------------------------------
static std::string convert_to_image_data(const std::string& image_text)
{
   if (image_text.empty()) {
	  WARNPRINTF("imsge_text empty");
      return 0;
   }

   size_t dataLength = image_text.length();

   string myData;
   myData.reserve(dataLength / 4 * 3);
   for (size_t pos = 0, dataPos = 0; pos < dataLength; dataPos += 3) {
      unsigned int sum = 0;
      for (int i = 0; (i < 4) && (pos < dataLength); ++pos) {
         char encodedByte = image_text[pos];
         unsigned int number = 0;
         if (('A' <= encodedByte) && (encodedByte <= 'Z')) {
            number = encodedByte - 'A';
         } else if (('a' <= encodedByte) && (encodedByte <= 'z')) {
            number = encodedByte - 'a' + 26;
         } else if (('0' <= encodedByte) && (encodedByte <= '9')) {
            number = encodedByte - '0' + 52;
         } else if (encodedByte == '+') {
            number = 62;
         } else if (encodedByte == '/') {
            number = 63;
         } else if (encodedByte == '=') {
            number = 64;
         } else {
            continue;
         }
         sum += number << (6 * (3 - i));
         ++i;
      }
      char triple[3];
      for (int j = 2; j >= 0; --j) {
         triple[j] = sum & 0xff; 
         sum >>= 8;
      }
      myData.append(triple, 3);
   }
   return myData;
}
//------------------------------------------------------------------------------------------------------
enum XmlStatus{
/*0*/	xmlNone,
/*1*/	xmlStart,
/*2*/	xmlDescription,
/*3*/	xmlTitleInfo,
/*4*/	xmlGenre,
/*5*/	xmlAuthor,
/*6*/	xmlFirstName,
/*7*/	xmlMiddleName,
/*8*/	xmlLastName,
/*9*/	xmlNickname,
/*10*/	xmlHomePage,
/*11*/	xmlEmail,
/*12*/	xmlBookTitle,
/*13*/	xmlAnnotation,
/*14*/	xmlPar,
/*15*/	xmlDate,
/*16*/	xmlCoverPage,
/*17*/	xmlLang,
/*18*/	xmlBody,
/*19*/	xmlSection,
/*20*/	xmlTitle,
/*21*/	xmlBinary,
/*22*/	xmlLink,
/*23*/	xmlEpigraph,
/*24*/	xmlTextAuthor,
/*25*/	xmlSequence,
/*26*/	xmlSub,
/*27*/	xmlPoem,
/*28*/	xmlStanza,
/*29*/	xmlV,
/*30*/	xmlEmphasis,
/*31*/  xmlSubtitle,
/*32*/	xmlCite,
/*33*/	xmlDocumentInfo,
};
//------------------------------------------------------------------------------------------------------
struct FB2XmlScannerParams {
	FB2XmlScannerParams(FB2Model* _model):last_pos(0),param(0),model(_model),type_index(-1)
		{
			types[type_index] = xmlNone;
		}
	FB2Model*   get_model() { return model;}
	bool 		is_type(XmlStatus type)  {return types[type_index]==type;}
	int			get_type()  {return (int)types[type_index];}
	XmlStatus 	get_prev_type(){return types[type_index-1];}
	XmlStatus 	get_prev_prev_type(){return types[type_index-2];}
	void 		push_type(XmlStatus type){types[++type_index] = type;}
	void 		pop_type()  {type_index--;}

	int			last_pos;
	void*		param;

protected:
	FB2Model*   model;
#define TYPES_STACK 60
	int 		type_index;
	XmlStatus	types[TYPES_STACK];
};
typedef void (FB2Model::*XmlHandler)(FB2XmlScannerParams* params, const char **atts);
struct Xml2HandlerS{
	std::string 	name;
	XmlHandler    	handler;
};

static Xml2HandlerS xml2handlers[] = {
	{"FictionBook",&FB2Model::xml_FictionBook},
	{"description",&FB2Model::xml_description},
	{"title-info",&FB2Model::xml_title_info},
	{"genre",&FB2Model::xml_genre},
	{"author",&FB2Model::xml_author},
	{"first-name",&FB2Model::xml_first_name},
	{"middle-name",&FB2Model::xml_middle_name},
	{"last-name",&FB2Model::xml_last_name},
    {"nickname",&FB2Model::xml_nickname},
    {"home-page",&FB2Model::xml_home_page},
    {"email",&FB2Model::xml_email},
	{"book-title",&FB2Model::xml_book_title},
	{"annotation",&FB2Model::xml_annotation},
	{"date",&FB2Model::xml_date},
	{"coverpage",&FB2Model::xml_coverpage},
	{"image",&FB2Model::xml_image},
	{"lang",&FB2Model::xml_lang},
	{"document-info",&FB2Model::xml_document_info},
	{"program-used",&FB2Model::xml_program_used},
	{"src-ocr",&FB2Model::xml_src_ocr},
	{"id",&FB2Model::xml_id},
	{"version",&FB2Model::xml_version},
	{"publish-info",&FB2Model::xml_publish_info},
    {"book-name",&FB2Model::xml_book_name},
	{"publisher",&FB2Model::xml_publisher},
	{"city",&FB2Model::xml_city},
	{"year",&FB2Model::xml_year},
	{"isbn",&FB2Model::xml_isbn},
	{"custom-info",&FB2Model::xml_custom_info},
	{"body",&FB2Model::xml_body},
	{"section",&FB2Model::xml_section},
	{"title",&FB2Model::xml_title},
	{"p",&FB2Model::xml_paragraph},
	{"empty-line",&FB2Model::xml_empty_line},
	{"binary",&FB2Model::xml_binary},
	{"a",&FB2Model::xml_a},
	{"epigraph",&FB2Model::xml_epigraph},
	{"text-author",&FB2Model::xml_text_author},
	{"sequence",&FB2Model::xml_sequence},
	{"sub",&FB2Model::xml_sub},
	{"poem",&FB2Model::xml_poem},
	{"stanza",&FB2Model::xml_stanza},
	{"v",&FB2Model::xml_v},
	{"emphasis",&FB2Model::xml_emphasis},
    {"subtitle",&FB2Model::xml_subtitle},
	{"cite",&FB2Model::xml_cite},
};

static Xml2HandlerS* find_handler(const char* name)
{
	for(int i=0;i< int(sizeof(xml2handlers)/sizeof(xml2handlers[0])); i++){
		if(xml2handlers[i].name==name)
			return &xml2handlers[i];
	}
	return 0;
}
static void XMLCALL
startElement(void *userData, const char *name, const char **atts)
{
	FB2XmlScannerParams* sp = (FB2XmlScannerParams*)userData;
	Xml2HandlerS* h = find_handler(name);
	if(h) {
		(sp->get_model()->*h->handler)(sp,atts);
	}else{
		WARNPRINTF("Handler not found for %s", name);
	}
}

static void XMLCALL
endElement(void *userData, const char *name)
{
	FB2XmlScannerParams* sp = (FB2XmlScannerParams*)userData;
	Xml2HandlerS* h = find_handler(name);
	if(h) {
		(sp->get_model()->*h->handler)(sp,0);
	}else{
		WARNPRINTF("Handler not found for %s", name);
	}
}

static void XMLCALL
chardataHandler(void *userData,const XML_Char *s,int len)
{
	FB2XmlScannerParams* sp = (FB2XmlScannerParams*)userData;
	sp->get_model()->xml_char_data_handler(sp,s,len);
}
static int XMLCALL
unknownEncodingHandler(void *encodingHandlerData, const XML_Char *name, XML_Encoding *info)
{
	FB2XmlScannerParams* sp = (FB2XmlScannerParams*)encodingHandlerData;
	return 	sp->get_model()->xml_unknownEncodingHandler(name, info);
}

#define DEFINE_XML_HANDLER(fname,fromType,toType) \
void FB2Model::fname(FB2XmlScannerParams* sp, const char **atts)\
{\
	if(atts && sp->is_type(fromType)){\
		sp->push_type(toType);\
	}else if(atts && sp->is_type(toType)){\
		sp->push_type(toType);\
	}else if(!atts && sp->is_type(toType)){\
		sp->pop_type();\
	/*}else{*/\
	/*	WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());*/\
	}\
}

#define DEFINE_XML_HANDLER1(fname,toType) \
void FB2Model::fname(FB2XmlScannerParams* sp, const char **atts)\
{\
	if(atts){\
		sp->push_type(toType);\
	}else if(!atts && sp->is_type(toType)){\
		sp->pop_type();\
	}else{\
		WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());\
	}\
}

#define DEFINE_XML_HANDLER_NOT_IMPLEMENTED(fname) \
void FB2Model::fname(FB2XmlScannerParams* sp, const char **atts)\
{\
}

DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_program_used)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_src_ocr)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_id)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_version)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_publish_info)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_book_name)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_publisher)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_city)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_year)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_isbn)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_custom_info)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_a)
DEFINE_XML_HANDLER_NOT_IMPLEMENTED(xml_sub)

///////////////////////////////////////////////////////////////////////////////////////////
// 
DEFINE_XML_HANDLER(xml_FictionBook,xmlNone,xmlStart)

void FB2Model::xml_description(FB2XmlScannerParams* sp, const char **atts)
{
	if(sp->is_type(xmlStart)){
        sp->push_type(xmlDescription);
	}else if(sp->is_type(xmlDescription)){
		sp->pop_type();
		if(info.authors.size()>0){
			std::string author = info.authors[0]->last_name + std::string(" ") + info.authors[0]->first_name;
			get_doc_attr_map().insert(std::make_pair("author", author));
		}
		if(!info.book_title.empty())
			get_doc_attr_map().insert(std::make_pair("title", info.book_title));

		// prepare first and second pages
		sp->last_pos = 0;
		if(!info.cover_page.image_name.empty()) {
			// first image is a cover page
			doc.push_back(new Paragraph(sp->last_pos,info.cover_page.image_name,Paragraph::Image,Paragraph::FullPage));
			sp->last_pos++;
		}
		// second page
		if(!info.authors.empty()){
			std::string author = info.authors[0]->last_name + std::string(" ") + info.authors[0]->first_name;
			doc.push_back(new Paragraph(sp->last_pos,author,Paragraph::TitleFont,Paragraph::CenterAlign));
			sp->last_pos += author.size();
		}
		if(!info.book_title.empty()) {
			doc.push_back(new Paragraph(sp->last_pos,info.book_title,Paragraph::TitleFont,Paragraph::CenterAlign));
			sp->last_pos += info.sequence.size();
		}
		if(!info.sequence.empty()){
			doc.push_back(new Paragraph(sp->last_pos,info.sequence,Paragraph::TitleFont,Paragraph::CenterAlign));
			sp->last_pos += info.sequence.size();
		}
		if(!info.date.empty()) {
			std::string date = "\t\t";
			date+=info.date;
			doc.push_back(new Paragraph(sp->last_pos,date));
			sp->last_pos += date.size();
		}
		doc.push_back(new Paragraph(sp->last_pos++,std::string(" ")));
		// add to documents also annotations
		for (unsigned int i=0; i<info.annotations.size(); i++){
			Paragraph* par = info.annotations[i];
			doc.push_back(new Paragraph(sp->last_pos,par->text,Paragraph::ItalicFont));
			sp->last_pos += par->text.size();
		}
		doc.push_back(new Paragraph(sp->last_pos++,std::string(" ")));
	}else{
		WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());
	}
}


DEFINE_XML_HANDLER(xml_title_info,xmlDescription,xmlTitleInfo)
DEFINE_XML_HANDLER(xml_document_info,xmlDescription,xmlDocumentInfo)
DEFINE_XML_HANDLER(xml_genre,xmlTitleInfo,xmlGenre)

void FB2Model::xml_author(FB2XmlScannerParams* sp, const char **atts)
{
	if(sp->is_type(xmlTitleInfo)){
		sp->push_type(xmlAuthor);
		sp->param = new Author;
	}else if(sp->is_type(xmlAuthor)){
		sp->pop_type();
		info.authors.push_back(static_cast<Author*>(sp->param));
		sp->param = 0;
	//}else if(sp->is_type(xmlDocumentInfo)){
		// unused
	//}else{
	//	WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());
	}
}
DEFINE_XML_HANDLER(xml_first_name,xmlAuthor,xmlFirstName)
DEFINE_XML_HANDLER(xml_middle_name,xmlAuthor,xmlMiddleName)
DEFINE_XML_HANDLER(xml_last_name,xmlAuthor,xmlLastName)
DEFINE_XML_HANDLER(xml_nickname,xmlAuthor,xmlNickname)
DEFINE_XML_HANDLER(xml_home_page,xmlAuthor,xmlHomePage)
DEFINE_XML_HANDLER(xml_email,xmlAuthor,xmlEmail)
DEFINE_XML_HANDLER(xml_book_title,xmlTitleInfo,xmlBookTitle)
DEFINE_XML_HANDLER(xml_annotation,xmlTitleInfo,xmlAnnotation)
DEFINE_XML_HANDLER(xml_coverpage,xmlTitleInfo,xmlCoverPage)
DEFINE_XML_HANDLER(xml_lang,xmlTitleInfo,xmlLang)

//DEFINE_XML_HANDLER(xml_date,xmlTitleInfo,xmlDate)
void FB2Model::xml_date(FB2XmlScannerParams* sp, const char **atts)
{
	if(atts && sp->is_type(xmlTitleInfo)){
		sp->push_type(xmlDate);
		while(*atts){
			if(strcmp(atts[0],"value")==0){
				info.date=atts[1];
				break;
			}
			atts+=2;
		}
	}else if(!atts && sp->is_type(xmlDate)){
		sp->pop_type();
	//}else if(atts && sp->is_type(xmlDocumentInfo)){
		// unused
	//}else{
	//	WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());
	}
}
void FB2Model::xml_image(FB2XmlScannerParams* sp, const char **atts)
{
	if(atts && sp->is_type(xmlCoverPage)){
		info.cover_page.image_name = atts[1] + 1;
	}else if(atts && sp->is_type(xmlSection)){
		doc.push_back(new Paragraph(sp->last_pos,std::string(atts[1] + 1),Paragraph::Image,Paragraph::CenterAlign));
		sp->last_pos++;
	}else if(!atts && (sp->is_type(xmlCoverPage) || sp->is_type(xmlSection))){
		//
	}else{
		WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());
	}
}

void FB2Model::xml_sequence(FB2XmlScannerParams* sp, const char **atts)
{
	if(atts) {
		std::string number,name;
		while(*atts){
			if(strcmp(atts[0],"number")==0){
				number=atts[1];
			}else if(strcmp(atts[0],"name")==0){
				name=atts[1];
			}
			atts+=2;
		}
		info.sequence = name;
		info.sequence += " - ";
		info.sequence += number; 
    }
}

/////////////////////////////////////////////////////////////////////
// 
DEFINE_XML_HANDLER(xml_body,xmlStart,xmlBody)
DEFINE_XML_HANDLER1(xml_section,xmlSection)
DEFINE_XML_HANDLER1(xml_title,xmlTitle)
DEFINE_XML_HANDLER1(xml_epigraph,xmlEpigraph)

void FB2Model::xml_text_author(FB2XmlScannerParams* sp, const char **atts)
{
	if(atts){
		sp->push_type(xmlTextAuthor);
		sp->param = new Paragraph(sp->last_pos,std::string(""),Paragraph::RegularFont,Paragraph::RightAlign);
	}else if(!atts && sp->is_type(xmlTextAuthor)){
		sp->pop_type();
		sp->last_pos = static_cast<Paragraph*>(sp->param)->start_file_pos;
		doc.push_back(static_cast<Paragraph*>(sp->param));
		sp->param = 0;
	}else{
		WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());
	}
}

DEFINE_XML_HANDLER1(xml_poem,xmlPoem)
DEFINE_XML_HANDLER(xml_stanza,xmlPoem,xmlStanza)
DEFINE_XML_HANDLER(xml_cite,xmlSection,xmlCite)
//DEFINE_XML_HANDLER(xml_v,xmlStanza,xmlV)
void FB2Model::xml_v(FB2XmlScannerParams* sp, const char **atts)
{
	if(sp->is_type(xmlStanza)){
		sp->push_type(xmlV);
		sp->param = new Paragraph(sp->last_pos,std::string("\t\t"),Paragraph::ItalicFont,Paragraph::CenterAlign);
	}else if(!atts && sp->is_type(xmlV)){
		sp->pop_type();
		sp->last_pos = static_cast<Paragraph*>(sp->param)->start_file_pos;
		doc.push_back(static_cast<Paragraph*>(sp->param));
		sp->param = 0;
	}else{
		WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());
	}
}

void FB2Model::xml_emphasis(FB2XmlScannerParams* sp, const char **atts)
{
	if(sp->is_type(xmlPar) || sp->is_type(xmlV)){
		Paragraph* par = (Paragraph*)sp->param;
		// we will not change font if already start fill data
		// cose we still not handling change font insight paragraph
		if(par->text.empty() && !sp->is_type(xmlEmphasis)){
			if(((Paragraph*)sp->param)->font==Paragraph::ItalicFont)
				((Paragraph*)sp->param)->font = Paragraph::EmphasisCiteFont;
			else
				((Paragraph*)sp->param)->font = Paragraph::EmphasisFont;
		}
		sp->push_type(xmlEmphasis);
	}else if(atts && sp->is_type(xmlEmphasis)){
		sp->push_type(xmlEmphasis);
	}else if(!atts && sp->is_type(xmlEmphasis)){
		sp->pop_type();
	}else{
		WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());
	}
}

void FB2Model::xml_subtitle(FB2XmlScannerParams* sp, const char **atts)
{
	if(sp->is_type(xmlSection)){
		sp->push_type(xmlSubtitle);
		sp->param = new Paragraph(sp->last_pos,Paragraph::SubTitleFont,Paragraph::CenterAlign);
	}else if(!atts && sp->is_type(xmlSubtitle)){
		sp->pop_type();
		sp->last_pos = static_cast<Paragraph*>(sp->param)->start_file_pos;
		doc.push_back(static_cast<Paragraph*>(sp->param));
		sp->param = 0;
	}else{
		WARNPRINTF("Unexpected case in %s!!! type=%d.",__FUNCTION__,sp->get_type());
	}
}

void FB2Model::xml_paragraph(FB2XmlScannerParams* sp, const char **atts)
{
	switch (sp->get_type()){
		case xmlAnnotation:
			sp->push_type(xmlPar);
			sp->param = new Paragraph(sp->last_pos,std::string("\t"));
			break;
		case xmlTitle:
			if(sp->get_prev_prev_type()==xmlSection)
				sp->param = new Paragraph(sp->last_pos,Paragraph::SubTitleFont,Paragraph::CenterAlign);
			else
				sp->param = new Paragraph(sp->last_pos,Paragraph::TitleFont,Paragraph::CenterAlign);
			sp->push_type(xmlPar);
			break;
		case xmlCite:
			sp->push_type(xmlPar);
			sp->param = new Paragraph(sp->last_pos,std::string("\t"),Paragraph::ItalicFont);
			break;
		case xmlSection:
			sp->push_type(xmlPar);
			sp->param = new Paragraph(sp->last_pos,std::string("\t"));
			break;
		case xmlEpigraph:
			sp->push_type(xmlPar);
			sp->param = new Paragraph(sp->last_pos,std::string("\t"),Paragraph::ItalicFont,Paragraph::RightAlign);
			break;


		case xmlPar:
			sp->pop_type();
			sp->last_pos = static_cast<Paragraph*>(sp->param)->start_file_pos;
			if(sp->is_type(xmlAnnotation))
				info.annotations.push_back(static_cast<Paragraph*>(sp->param));
			else
				doc.push_back(static_cast<Paragraph*>(sp->param));
			sp->param = 0;
			break;
	}
	return;
}
void FB2Model::xml_empty_line(FB2XmlScannerParams* sp, const char **atts)
{
	if(sp->is_type(xmlAnnotation))
		info.annotations.push_back(new Paragraph(sp->last_pos,std::string(" ")));
	else
		doc.push_back(new Paragraph(sp->last_pos,std::string(" ")));
	sp->last_pos++;
}
void FB2Model::xml_binary(FB2XmlScannerParams* sp, const char **atts)
{
	if(sp->is_type(xmlStart)){
		sp->push_type(xmlBinary);
		std::string id,content_type;
		while(*atts){
			if(strcmp(atts[0],"id")==0){
				id=atts[1];
			}else if(strcmp(atts[0],"content-type")==0){
				content_type=atts[1];
			}
			atts+=2;
		}
		sp->param = new ImageDescr(id,content_type);
	}else if(!atts && sp->is_type(xmlBinary)){
		ImageDescr* descr = static_cast<ImageDescr*>(sp->param);
		descr->image_text = convert_to_image_data(descr->image_text);
		images.insert(Images::value_type(descr->image_id,descr));
		sp->pop_type();
	}else{
		WARNPRINTF("Unexpected case in xml_binary!!! type=%d.",sp->get_type());
	}
}

void FB2Model::xml_char_data_handler(FB2XmlScannerParams* sp,const XML_Char *s,int len)
{
	switch(sp->get_type()) {
		case xmlGenre:
			info.genre.push_back(std::string(s, len));
			break;
		case xmlFirstName:
			assert(sp->param);
			static_cast<Author*>(sp->param)->first_name.assign(s, len);
			break;
		case xmlMiddleName:
			assert(sp->param);
			static_cast<Author*>(sp->param)->middle_name.assign(s, len);
			break;
		case xmlLastName:
			assert(sp->param);
			static_cast<Author*>(sp->param)->last_name.assign(s, len);
			break;
		case xmlNickname:
			assert(sp->param);
			static_cast<Author*>(sp->param)->nickname.assign(s, len);
			break;
		case xmlHomePage:
			assert(sp->param);
			static_cast<Author*>(sp->param)->home_page.assign(s, len);
			break;
		case xmlEmail:
			assert(sp->param);
			static_cast<Author*>(sp->param)->email.assign(s, len);
			break;
		case xmlBookTitle:
			info.book_title.assign(s, len);
			break;
		case xmlLang:
			info.language.assign(s, len);
			break;
		case xmlDate:
			info.date.append(s, len);
			break;
		case xmlV:
		case xmlPar:
		case xmlTextAuthor:
		case xmlEmphasis:
		case xmlSubtitle:
			assert(sp->param);
			static_cast<Paragraph*>(sp->param)->text.append(s, len);
			static_cast<Paragraph*>(sp->param)->start_file_pos += len;
			break;
		case xmlBinary:
			assert(sp->param);
			static_cast<ImageDescr*>(sp->param)->image_text.append(s, len);
			break;
	}
}
static int XMLCALL
unknownEncodingConvert(void *data, const char *p)
{
	convert_funcs* iconv = (convert_funcs*)data;
	ucs4_t  wc;
	if(iconv->convert(iconv, &wc, (const unsigned char*)p, 1))
		return wc;
	return -1;
}


int FB2Model::xml_unknownEncodingHandler(const XML_Char *name, XML_Encoding *info)
{
	convert_funcs* iconv = find_converter(name);
	if(!iconv){
		WARNPRINTF("Decoder not found for charset %s", name);
		return 0;
	}
	
	info->convert = unknownEncodingConvert;
	info->release = 0;
	info->data = (void*)iconv;
	iconv->build_map(iconv, info->map,256);
	return 1;
}
//------------------------------------------------------------------------------------------------------------------------------
struct zlib_data {
	char* data;
	char* cur;
	size_t len;
};

voidpf ZCALLBACK z_open(voidpf opaque, const char* filename, int mode)
{
	zlib_data* z =  (zlib_data*)opaque;
	return z->cur;
}
uLong  ZCALLBACK z_read(voidpf opaque, voidpf stream, void* buf, uLong size)
{
	zlib_data* z =  (zlib_data*)opaque;
	uLong my_len = z->len - (z->cur - z->data);
	if(my_len<size)
		size = my_len;
	if(size>0){
		memcpy(buf,z->cur,size);
		z->cur += size;
	}
	return size;
}
uLong  ZCALLBACK z_write(voidpf opaque, voidpf stream, const void* buf, uLong size)
{
	return 0;
}
long   ZCALLBACK z_tell(voidpf opaque, voidpf stream)
{
	zlib_data* z =  (zlib_data*)opaque;
	return (long)(z->cur - z->data);
}
long ZCALLBACK z_seek(voidpf opaque, voidpf stream, uLong offset, int origin)
{
	zlib_data* z =  (zlib_data*)opaque;
	switch(origin)
	{
		case ZLIB_FILEFUNC_SEEK_CUR :
			z->cur += offset;
			break;
		case ZLIB_FILEFUNC_SEEK_END :
			z->cur = z->data+z->len + offset;
			break;
		case ZLIB_FILEFUNC_SEEK_SET :
			z->cur = z->data + offset;
			break;
	}
	return 0;
}
int ZCALLBACK z_close(voidpf opaque, voidpf stream)
{
	return 0;
}
int ZCALLBACK z_testerror(voidpf opaque, voidpf stream)
{
	return 0;
}

PluginStatus FB2Model::parse_fb2_data(const char* test_data, size_t test_len, int file_p, size_t len)
{
	XML_Parser parser = XML_ParserCreate(NULL);
	void* data = 0;
	if(test_data[0]=='P' && test_data[1]=='K' &&
	   test_data[2]==0x03 && test_data[3]==0x04){
		std::auto_ptr<char> ptr(new char[len]);
		lseek(file_p, 0, SEEK_SET);
		if(read(file_p, ptr.get(), len)!=(int)len){
			ERRORPRINTF("Fail to read file: %s",path.c_str());
			XML_ParserFree(parser);
			return PLUGIN_ERROR_OPEN_FILE;
		}

		zlib_data param = {ptr.get(),ptr.get(),len};
		zlib_filefunc_def my_fub_zip_func = {z_open,z_read,z_write,z_tell,z_seek,z_close,z_testerror,&param};

		unzFile zfile = unzOpen2(path.c_str(),&my_fub_zip_func);
		if(zfile){
			unz_file_info info;
			if(unzGetCurrentFileInfo(zfile, &info, NULL, 0, NULL, 0, NULL, 0)==UNZ_OK){
				if (unzOpenCurrentFile(zfile) == UNZ_OK){
					data = XML_GetBuffer(parser, info.uncompressed_size);
					if(unzReadCurrentFile(zfile, data, (unsigned)info.uncompressed_size) != (int)info.uncompressed_size){
						unzCloseCurrentFile(zfile);
						unzClose(zfile);
						XML_ParserFree(parser);
						ERRORPRINTF("Fail to read zip file");
						return PLUGIN_DOCUMENT_CORRUPT;
					}
					unzCloseCurrentFile(zfile);
					len = info.uncompressed_size;
				}else{
					ERRORPRINTF("Fail open current file: %s.",path.c_str());
					unzClose(zfile);
					XML_ParserFree(parser);
					return PLUGIN_ERROR_OPEN_FILE;
				}
			}else{
				ERRORPRINTF("Fail to get file info: %s.",path.c_str());
				unzClose(zfile);
				XML_ParserFree(parser);
				return PLUGIN_ERROR_OPEN_FILE;
			}
			unzClose(zfile);
		}else{
			ERRORPRINTF("Fail to open zip arhive: %s.",path.c_str());
			XML_ParserFree(parser);
			return PLUGIN_ERROR_OPEN_FILE;
		}
	}else{
		lseek(file_p, 0, SEEK_SET);
		data = XML_GetBuffer(parser, len);
		if(read(file_p, data, len)!=(int)len){
			ERRORPRINTF("Fail to read file: %s",path.c_str());
			XML_ParserFree(parser);
			return PLUGIN_ERROR_OPEN_FILE;
		}
	}

  	FB2XmlScannerParams s(this);
  	XML_SetUserData(parser, &s);
  	XML_SetElementHandler(parser, startElement, endElement);
	XML_SetCharacterDataHandler(parser,chardataHandler);
    XML_SetUnknownEncodingHandler(parser, unknownEncodingHandler, (void*)&s);

  	if (XML_ParseBuffer(parser, len, 1) == XML_STATUS_ERROR) {
	    ERRORPRINTF("%s at line %" XML_FMT_INT_MOD "u",XML_ErrorString(XML_GetErrorCode(parser)),
                    XML_GetCurrentLineNumber(parser));
		return PLUGIN_DOCUMENT_CORRUPT;
	}
  	XML_ParserFree(parser);
	//dump();
    return PLUGIN_OK;
}
bool FB2Model::is_new_search(SearchContext * sc)
{
    bool is_new = true;

    if (sc)
    {
        if (   sc->search_type      == last_sc.search_type
            && sc->pattern          == last_sc.pattern
            && sc->case_sensitive   == last_sc.case_sensitive
            && sc->forward          == last_sc.forward
            && sc->match_whole_word == last_sc.match_whole_word
            && sc->from             == last_search_result.end)
        {
            // If the end anchor of last search result is the same as the start position
            // of searching and the other search options are also the same that means
            // it's 'search next again' after finding the text.
            is_new = false;
        }
    }
    return is_new;
}

bool FB2Model::search(std::vector<Range>& result_ranges, SearchContext* sc)
{
    // Exact search type and search criteria from search context.
    SearchType search_type      = sc->search_type;
    Position   &from            = sc->from;
    const char *pattern         = sc->pattern.c_str();
    bool       case_sensitive   = sc->case_sensitive;
    bool       forward          = sc->forward;
    bool       match_whole_word = sc->match_whole_word;

    size_t pattern_len = strlen(pattern);
	while(doc[from.paragraph]->font==Paragraph::Image) 
		from.paragraph++;

	const char *paragraph_head = doc[from.paragraph]->text.c_str();
    bool new_search = is_new_search(sc);

    // Remember last search context.
    last_sc = *sc;

    // Clear the last search result range.
    Position s(0, 0), e(0,0);
    last_search_result.start = s;
    last_search_result.end = e;

    if (forward)
    {
        const char *p = paragraph_head + from.offset;
        if (p && !new_search)
        {
            // It's searching next again, move the start position of searching one character.
            // Otherwize, searching one letter results in deadlock.
            //
            // See more in  0003861: [UDS TXT plugin] TXT Search next
            // for single character word does not work.
            //
            // The solution is very dirty. But txt plugin has low priority
            // which is not specified for dr1000 and dr800.
            // And redefinition what is range takes much more time and has more effects.
            p++;
        }
        while (true)
        {
            const char* find = utf8_strstr(p, pattern, case_sensitive);
            if (find)
            {
                // See if matching whole word.
                if (!match_whole_word || is_whole_word(paragraph_head, find, pattern_len))
                {
                    // Pattern found.
                    Position start(from.paragraph, static_cast<unsigned int>(find - paragraph_head));
                    const char* last_char = g_utf8_prev_char(find + pattern_len);
                    Position end(from.paragraph, static_cast<unsigned int>(last_char - paragraph_head));
                    result_ranges.push_back(Range(start, end));

                    if (search_type == SEARCH_NEXT)
                    {
                        // Remember last search result.
                        last_search_result.start = start;
                        last_search_result.end = end;

                         // Search complete.
                        return true;
                    }

                    // If SEARCH_ALL we must continue with current paragraph.
                }

                p = find + pattern_len;
            }
            else
            {
                // Can't find any match in current paragraph.
                from.offset = 0;
                return ++(from.paragraph) == doc.size();
            }
        }
    }
    else
    {
        // Backward search.
        int len = static_cast<int>(from.offset);
        while (true)
        {
            const char *find = utf8_strrstr(paragraph_head, len, pattern, case_sensitive);
            if (find)
            {
                // See if matching whole word.
                if (!match_whole_word || is_whole_word(paragraph_head, find, pattern_len))
                {
                    // Pattern found.
                    Position start(from.paragraph, static_cast<unsigned int>(find - paragraph_head));
                    const char* last_char = g_utf8_prev_char(find + pattern_len);
                    Position end(from.paragraph, static_cast<unsigned int>(last_char - paragraph_head));
                    result_ranges.push_back(Range(start, end));

                    // Remember last search result.
                    last_search_result.start = start;
                    last_search_result.end = end;

                    return true;
                }

                len = static_cast<int>(find - paragraph_head);
            }
            else
            {
                // Can't find any match in current paragraph.
                if (from.paragraph == 0)
                {
                    return true;
                }
                else
                {
                    from.paragraph--;
                    from.offset = static_cast<unsigned int>(doc[from.paragraph]->text.size());
                    return false;
                }
            }
        }
    }
}

bool FB2Model::has_anchor(const Position &pos)
{
    // Sanity check.
    if (pos.paragraph >= doc.size() || pos.offset >= doc[pos.paragraph]->text.size())
    {
        return false;
    }

    return true;
}

bool FB2Model::get_file_pos_from_anchor(size_t& file_pos, const Position &pos)
{
    // Sanity check.
    if (pos.paragraph >= doc.size() || pos.offset >= doc[pos.paragraph]->text.size())
    {
        return false;
    }

    file_pos = doc[pos.paragraph]->start_file_pos + pos.offset;
    return true;
}

bool FB2Model::is_seperator(const char* p)
{
    gunichar ch = g_utf8_get_char(p);

    if (g_unichar_isspace(ch))
    {
        return true;
    }

    if (g_unichar_ispunct(ch))
    {
        // Punctuation.
        if (*p != '\'' && *p != '\"')
        {
            return true;
        }
    }

    return false;
}

bool FB2Model::get_word_from_anchor(const Position& pos,
                                     Position& word_start_pos,
                                     Position& word_end_pos)
{

    const char* paragraph = doc[pos.paragraph]->text.c_str();
    word_start_pos.paragraph = word_end_pos.paragraph = pos.paragraph;

    const char* p = paragraph + pos.offset;

    // Check if the character at pos is a seperator.
    if (is_seperator(p))
    {
        // Then there is no word at pos.
        word_start_pos.offset = word_end_pos.offset = pos.offset;
        return false;
    }

    // Find the first space before pos.
    for (; p > paragraph; p = g_utf8_prev_char(p))
    {
        if (is_seperator(p))
        {
            p = g_utf8_next_char(p);
            break;
        }
    }
    word_start_pos.offset = static_cast<int>(p - paragraph);

    // Find the first space after pos.
    for (p = paragraph + pos.offset; *p != 0; p = g_utf8_next_char(p))
    {
        if (is_seperator(p))
        {
            p = g_utf8_prev_char(p);
            break;
        }
    }
    word_end_pos.offset = static_cast<int>(p - paragraph);

    return true;
}

bool FB2Model::get_words_from_range(const Position& range_start,
                                     const Position& range_end,
                                     Position& words_start,
                                     Position& words_end)
{
    if (range_end < range_start)
    {
        ERRORPRINTF("Invalid range, range_start = %s, range_end = %s",
            range_start.to_string().c_str(),
            range_end.to_string().c_str());
        return false;
    }

    Position tmp;

    // Get the object range the range_start anchor points to.
    get_word_from_anchor(range_start, words_start, tmp);

    // Get the object range the range_end anchor points to.
    get_word_from_anchor(range_end, tmp, words_end);

    // Strip any leading seperators.
    const char* start_paragraph = 0;
    const char* p = 0;
    while (true)
    {
        start_paragraph = doc[words_start.paragraph]->text.c_str();
        for (p = start_paragraph + words_start.offset; *p != 0; p = g_utf8_next_char(p))
        {
            if (!is_seperator(p))
            {
                break;
            }
        }

        if (*p == 0)
        {
            words_start.paragraph++;
            words_start.offset = 0;
        }
        else
        {
            break;
        }
    }
    words_start.offset = static_cast<int>(p - start_paragraph);

    // Strip any trailing seperators.
    const char* end_paragraph = doc[words_end.paragraph]->text.c_str();
    for (p = end_paragraph + words_end.offset; p > end_paragraph; p = g_utf8_prev_char(p))
    {
        if (!is_seperator(p))
        {
            break;
        }
    }
    words_end.offset = static_cast<int>(p - end_paragraph);

    return words_end >= words_start;
}

bool FB2Model::get_text_from_range(std::string& result,
                                    const Position& start_pos,
                                    const Position& end_pos)
{
    unsigned int start_paragraph = start_pos.paragraph;
    unsigned int end_paragraph = end_pos.paragraph;

    for (unsigned int i = start_paragraph;
            (i <= end_paragraph) && (i < doc.size());
            i++)
    {
        if (doc[i]->font != Paragraph::Image && 
			doc[i]->text.length())
        {
            const char* start_p = doc[i]->text.c_str();
            if (i == start_paragraph)
            {
                start_p += start_pos.offset;
            }

            size_t len = doc[i]->text.length();
            if (i == end_paragraph)
            {
                const char* p = doc[i]->text.c_str() + end_pos.offset;
                len = g_utf8_next_char(p) - start_p;
            }

            result.append(start_p, len);
        }
    }

    return true;
}

const FB2Model::ImageDescr* 
FB2Model::get_build_image(const std::string& name, int width, int height, bool outzoom)
{
	TRACEFUNCTION();

	ImageDescr* image = images[name];
	if(image) {
		image->build_image(width, height,outzoom);
		if(image->bits)
			return image;
	}
	return 0;
}

void FB2Model::dump()
{
    // Generate the dump file.
    std::string dump_path = path + ".converted";
    FILE* fp = fopen(dump_path.c_str(), "w");

    if (fp != NULL)
    {
        for (unsigned int i=0; i<doc.size(); i++)
        {
            fputs(doc[i]->text.c_str(), fp);
        }

        fclose(fp);
    }
}

}
