#!/bin/bash

# read the file name from the commandline
file="$1"
# set the file name of the outgoing file as FILENAME-stretched.pdf
outfile="${file%.pdf}-stretched.pdf"
# create temporary files with crop information, and pre-stetched cropped pdf
bounding="tmp-bounding.txt"
echo -n "Looking for ebook-meta ..."
if which "ebook-meta" ; then
    emexec=$(which "ebook-meta")
else
    if locate "ebook-meta" ; then
        emexec=$(locate "ebook-meta" | head -n1)
    else
       echo ""
       echo "Could not find ebook-meta calibre subprogram. Skipping metadata steps."
       emexec="notfound"
    fi
fi
echo ""
echo -n "Looking for pdfmanipulate..."
if which "pdfmanipulate" ; then
    pdmexec=$(which "pdfmanipulate")
else
    if locate "pdfmanipulate" ; then
        pdmexec=$(locate "pdfmanipulate" | head -n1)
    else
       echo ""
       echo "Error, Could not find calibre's pdfmanipulate subprogram."
       echo "The script cannot work without it."
       exit 3 
    fi
fi
echo ""
echo -n "Looking for ghostscript..."
if which "gs" ; then
    gsexec=$(which "gs")
else
    if locate "gs" ; then
        gsexec=$(locate "gs" | head -n1)
    else
       echo ""
       echo "Error, Could not find ghostscript."
       echo "The script cannot work without it."
       exit 3 
    fi
fi

# see if the file exists
if [ -f "$file" ] ; then
    pages="$("${pdmexec}" info "${file}" 2>/dev/null | grep 'Pages' | sed 's/[^:]*: //')" 
    echo -n "Creating temporary folder..."
    wd="tmppdf3to4dirkk"
    mkdir -p "$wd" 
    cp "$file" "${wd}/${file}"
    cd "$wd"
    echo "Delete the temporary folder if the script fails!"
    if [ "$emexec" != "notfound" ] ; then
        echo ""
        echo "Reading metadata..."
        author="$("${emexec}" "${file}" | grep 'Author' | sed 's/[^:]*: //')"
        title="$("${emexec}" "${file}" | grep 'Title' | sed 's/[^:]*: //')"
        language="$(ebook-meta "${file}" | grep 'Language' | sed 's/[^:]*: //')"
        echo "Author(s) recognized as ${author}."
        echo "Title recognized as ${title}."
        echo "Language recognized as ${language}."
    fi
    echo ""
    echo -n "Splitting pages..."
    for p in $(eval echo {1..$pages}) ; do
         "${gsexec}" -sOutputFile="split${p}.pdf" -dBATCH -dSAFER -dNOPAUSE -sDEVICE=pdfwrite -dFirstPage=$p -dLastPage=$p "${file}" &> /dev/null 
    done
    echo " done."
    echo ""
    origfile="${file}"
    for splitpagenum in $(eval echo {1..$pages}) ; do
        echo "*********************************************"
        echo "Now working on the original page $splitpagenum"
        echo "**********************************************"
        file="split${splitpagenum}.pdf"
        echo -n "Analyzing page geometry..."
        "${gsexec}" -dSAFER -dNOPAUSE -dBATCH -sDEVICE=bbox "$file" 2> "${bounding}" > /dev/null
        sed /HiRes/d "${bounding}" > "tmp-newbounding.txt"
        sed "s/.*Box: //" "tmp-newbounding.txt" > "tmp-newnewbounding.txt"
        cat "tmp-newnewbounding.txt" | grep -v "[A-Za-z]" | grep -v "^$" > "${bounding}"
        toph=$(sed -n "s/\([0-9]*\) .*/\1/p" "${bounding}") 
        topv=$(sed -n "s/[0-9]* \([0-9]*\) .*/\1/p" "${bounding}") 
        both=$(sed -n "s/[0-9]* [0-9]* \([0-9]*\) .*/\1/p" "${bounding}") 
        botv=$(sed -n "s/[0-9]* [0-9]* [0-9]* \([0-9]*\).*/\1/p" "${bounding}") 
        maxh=$((both - toph))
        maxv=$((botv - topv))
        hscale="1"
        scaleamt=$(( $((maxh * 400)) / $((maxv * 3)) ))
        if [ "$scaleamt" -gt "100" ] ; then
           hscale="$((10000 / scaleamt))" 
           hscale="0$(echo "$hscale" | sed "s/\(.*\)\(..\)/\1\.\2/")"
           scaleamt="1"
        else
            scaleamt="0$(echo "$scaleamt" | sed "s/\(.*\)\(..\)/\1\.\2/")"
        fi   
        echo " done."
        echo -n "Now scaling page ${splitpagenum}..."
        skewedfile="skew${splitpagenum}.pdf"
        croppedfile="crop${splitpagenum}.pdf"
        pageout="out${splitpagenum}.pdf"
        "${gsexec}" -sOutputFile="${skewedfile}" -dBATCH -dNOPAUSE -sDEVICE=pdfwrite -c "<</BeginPage{${hscale} ${scaleamt} scale}>> setpagedevice" -f "${file}" &> /dev/null
        echo " done."
        echo -n "Now determining the crop region for page ${splitpagenum}..."
        "${gsexec}" -dSAFER -dNOPAUSE -dBATCH -sDEVICE=bbox "${skewedfile}" 2> "${bounding}" > /dev/null
        echo " done."
        echo -n "Now cropping page ${splitpagenum}..."
        "${pdmexec}" crop -o "${croppedfile}" -b "${bounding}" "${skewedfile}"
        echo " done."
        echo -n "Now forcing page ${splitpagenum} to ebook screen size..." 
        "${gsexec}" -sOutputFile="${pageout}" -dBATCH -dSAFER -dNOPAUSE -sDEVICE=pdfwrite -g1062x1416 -r300 -dPDFFitPage ${croppedfile} &> /dev/null
        echo " done."
        # rm "$skewedfile"
        # rm "$croppedfile"
    done
    echo ""
    echo "****** ****** ******"
    echo -n "Now re-merging the various pages..."
    "${pdmexec}" merge -o "output.pdf" $(eval echo out{1..$pages}.pdf)
    echo " done."
    if [ "$emexec" != "notfound" ] ; then
        echo ""
        echo -n "Reinstating metadata..."
        "${emexec}" "output.pdf" --authors "$author" --title "${title}" --language "${language}"
    fi
    echo " done."
    echo ""
    echo -n "Cleaning up..."
    cd ..
    mv "${wd}/output.pdf" "$outfile"
    rm -rf "$wd"
    echo " done."
    echo "Finished. Output saved as $outfile."
else
    # what happens if the file cannot be found
    echo 'The file was not found.'
    exit 1
fi
exit 0
