#!/usr/bin/python -u
# -*- coding: utf-8 -*-

# ePUB utilities python script, v. 1.2
# © 2010, Jellby
#
# Features:
# * Displays some basic metadata for the ePUB
# * Creates .thn thumbnails for the Cybooks (needs ImageMagick's convert)
# * Converts ePUB to PDF using Prince (www.princexml.com)
#
# Configuration:
# * If a file named .epub2pdf/default.css exists in your home directory,
#   it will override the "default" CSS stylesheet and be used in all PDF
#   conversions
# * The command lines used with convert and prince can currently only be
#   changed by modifying this script (search for "convert" and "prince",
#   with the quotes)
#
# Changelog:
#   1.2 Show only image files in the cover selection drop-down
#       Show also XHTML files in the CSS selection drop-down
#       Fixed PDF metadata creation
#       Fixed file path computation for non-UNIX filesystems
#       Fixed temporary CSS files in Windows
#   1.1 Fixed detection of Book CSS
#       Included default CSS (defaultprincecss)
#       Added tab to view CSS files in the ePUB
#       The generated PDF now includes correct title and author

import sys
import zipfile
import os
import os.path
import shutil
import math
import re
import codecs
import subprocess
import tempfile
import xml.dom.minidom
from PyQt4 import QtGui, QtCore

#===============================================================================
# Class for managing ePUB files

class ePUB():
  def __init__(self, epubfile):
    self.filename = epubfile
    self.basedir = ""
    self.OPF = ""
    self.title = []
    self.author = []
    self.files = {}
    self.types = {}
    self.spine = []
    self.pages = 0
    self.lpages = 0
    self.coverid = ""
    self.coverdata = ""
    self.thndata = ""
    self.princecss = ""
    self.princefile = ""
    self.isEpub = self.testEpub()
    if self.isEpub:
      self.readMetadata()

  # Check the file is a valid ePUB
  def testEpub(self):
    # If it is not ZIP, it is not ePUB
    if not zipfile.is_zipfile(self.filename):
      return False
    zfile = zipfile.ZipFile(self.filename, "r")
    # If it first file is not "mimetype" or not uncompressed, it is not ePUB
    if (zfile.infolist()[0].filename != "mimetype") or (zfile.infolist()[0].compress_type != 0):
      return False
    zfile.close()
    return True

  # Read the ePUB metadata and files
  def readMetadata(self):
    zfile = zipfile.ZipFile(self.filename, "r")
    # Get the OPF file name and directory
    container = xml.dom.minidom.parseString(zfile.read("META-INF/container.xml"))
    opffile = container.getElementsByTagName("rootfile")[0].getAttribute("full-path")
    self.basedir = os.path.dirname(opffile)
    self.OPF = xml.dom.minidom.parseString(zfile.read(opffile))
    # Read title(s) and author(s)
    for title in self.OPF.getElementsByTagNameNS("http://purl.org/dc/elements/1.1/", "title"):
      self.title.append(title.childNodes[0].nodeValue)
    for author in self.OPF.getElementsByTagNameNS("http://purl.org/dc/elements/1.1/", "creator"):
      string = author.childNodes[0].nodeValue
      role = author.getAttributeNS("http://www.idpf.org/2007/opf", "role")
      if role:
        string += " [%s]" % role
      self.author.append(string)
    # Read ids and files
    for item in self.OPF.getElementsByTagName("item"):
      id = item.getAttribute("id")
      href = item.getAttribute("href")
      type = item.getAttribute("media-type")
      self.files[id] = href
      self.types[id] = type
    # Read spine and calculate pages
    for itemref in self.OPF.getElementsByTagName("itemref"):
      idref = itemref.getAttribute("idref")
      linear = itemref.getAttribute("linear")
      file = self.file(idref)
      self.spine.append(file)
      pages = int(math.ceil(zfile.getinfo(file).compress_size/1024.0))
      self.pages += pages
      if linear != "no":
        self.lpages += pages
    # Read possible cover and Prince CSS
    for meta in self.OPF.getElementsByTagName("meta"):
      if meta.getAttribute("name") == "cover":
        self.coverid = meta.getAttribute("content")
        self.coverdata = zfile.read(self.file(self.coverid))
      if meta.getAttribute("name") == "prince-style":
        princeid = meta.getAttribute("content")
        self.princefile = self.file(princeid)
        self.princecss = unicode(zfile.read(self.princefile),"utf-8")
    # Find other common cover names
    other = ["content/resources/_cover_.jpg"]
    if not self.coverid:
      for file in other:
        try:
          self.coverdata = zfile.read(file)
          for id in self.files.keys():
            if self.file(id) == file:
              self.coverid = id
              break
          break
        except KeyError:
          pass
    zfile.close()

  def file(self, fileid):
    return os.path.join(self.basedir, self.files[fileid]).replace("\\", "/")

  def writeThumbnail(self):
    if not self.thndata:
      return
    #Write the tumbnail in the format expected by the Cybook
    thn = open(self.filename + ".thn", "wb")
    thn.write("t4bp")
    thumbdata = self.thndata.split()
    num = int(thumbdata[3])
    del thumbdata[0:4]
    for i in range(0,len(thumbdata),2):
      px1 = round(int(thumbdata[i])*15.0/num)
      px2 = round(int(thumbdata[i+1])*15.0/num)
      byte = 16*int(px1)+int(px2)
      thn.write(chr(byte))
    thn.close()

#===============================================================================
# Subroutine for creating the .thn thumbnail

def create_thumbnail(data, num, dither="none"):
  if not data:
    return ""
  palette = {}
  palette[4] = [32, 96, 160, 224]
  palette[8] = [16, 48, 80, 112, 144, 176, 208, 240]
  palette[16] = [8, 24, 40, 56, 72, 88, 104, 120, 136, 152, 168, 184, 200, 216, 232, 248]
  size = "96x144"
  white = "rgb(255,255,255)"
  # write the palette to a temporary file
  plt = tempfile.NamedTemporaryFile()
  plt.write("P2\n1 %d\n255\n" % len(palette[num]))
  plt.write(" ".join(map(str, palette[num])) + "\n")
  plt.flush()
  # perform the conversion with ImageMagick's convert
  command = ["convert", "-", "-resize", size, "-dither", dither, "-bordercolor", white, "-border", size, "-gravity", "center", "-crop", "%s+0+0" % size, "-map", plt.name, "-strip", "-compress", "none", "PGM:-"]
  try:
    convert = subprocess.Popen(command, stdin=subprocess.PIPE, stdout=subprocess.PIPE)
    thumbdata = convert.communicate(input=data)[0]
    plt.close()
  except OSError:
    plt.close()
    return -1
  # get a reverse palette
  revplt = {}
  for i in range(len(palette[num])):
    revplt[str(palette[num][i])] = " %d" % i
  # remap the output
  thumbnail = "P2\n96 144\n%d\n" % (num-1)
  thumbdata = thumbdata.split()
  del thumbdata[0:4]
  for i in thumbdata:
    thumbnail += revplt[i]
  thumbnail += "\n"
  return thumbnail

#===============================================================================
# Subroutine for setting the metadata of a file
# (Prince sets the PDF metadata from the first XHTML file in the command line)

def set_metadata(file, epub):
  # read the file and parse it
  openfile = open(file, "r+b")
  data = xml.dom.minidom.parseString(openfile.read())
  # create new title and author elements
  title = data.createElement("title")
  title.appendChild(data.createTextNode(" | ".join(epub.title)))
  author = data.createElement("meta")
  author.setAttribute("name", "author")
  author.setAttribute("content", "; ".join(epub.author))
  # replace existing title and author with the new ones
  for node in data.getElementsByTagName("head"):
    for child in node.getElementsByTagName("title"):
      node.removeChild(child)
    for child in node.getElementsByTagName("meta"):
      if child.getAttribute("name") == "author":
        node.removeChild(child)
    node.appendChild(title)
    node.appendChild(author)
  # write the modified file
  openfile.seek(0)
  openfile.write(data.toxml("utf-8"))
  openfile.close()

#===============================================================================
# Default CSS for conversion with Prince

defaultprincecss = """/*
@font-face {
    font-family: serif;
    src: local("Droid Serif")
}
*/

@page {
  size: 9cm 12cm;
  margin: 5mm 1mm 1mm 1mm;
  @top-left {
    font-size: 60%;
    font-style: italic;
    border-bottom: solid thin black;
    margin-bottom: 1mm;
    content: string(booktitle);
  }
  @top-center {
    font-size: 60%;
    font-style: italic;
    border-bottom: solid thin black;
    margin-bottom: 1mm;
    content: string(chaptertitle);
  }
  @top-right {
    font-size: 50%;
    border-bottom: solid thin black;
    margin-bottom: 1mm;
    content: counter(page) "/" counter(pages);
  }
}

@page:first {
  margin: 1mm 1mm 1mm 1mm;
  @top-left {
    border-width: 0;
    margin: 0;
    content: normal;
  }
  @top-center {
    border-width: 0;
    margin: 0;
    content: normal;
  }
  @top-right {
    border-width: 0;
    margin: 0;
    content: normal;
  }
}

body {
  font-size: 10pt;
  font-family: serif;
  text-align: justify;
  prince-image-resolution: 166dpi;
  hyphens: auto;
}"""

#===============================================================================
# Definition of the Qt GUI

class Main(QtGui.QMainWindow):
  def __init__(self, parent=None, file=None):
    QtGui.QMainWindow.__init__(self)

    self.setWindowTitle("ePUB utilities")
    self.menu = self.menuBar()
    self.status = self.statusBar()

    load = self.menu.addAction("&Load", self.loadFile)
    quit = self.menu.addAction("&Quit", self, QtCore.SLOT("close()"))
    load.setStatusTip("Select an ePUB file to load")
    quit.setStatusTip("Exit the application")

    self.label_title = QtGui.QLabel("Title:")
    self.label_author = QtGui.QLabel("Author(s):")
    self.label_pages = QtGui.QLabel("Pages:")
    self.label_files = QtGui.QLabel("Image files:")
    self.title = QtGui.QLineEdit()
    self.title.setReadOnly(True)
    self.author = QtGui.QLineEdit()
    self.author.setReadOnly(True)
    self.pages = QtGui.QLabel()

    self.tabs = QtGui.QTabWidget()

    self.files = QtGui.QComboBox()
    self.add = QtGui.QPushButton("Add file")
    self.add.setEnabled(False)
    self.create = QtGui.QPushButton("Create thumbnail")
    self.create.setEnabled(False)
    self.save = QtGui.QPushButton("Save thumbnail")
    self.save.setEnabled(False)
    self.cover = QtGui.QLabel()
    self.cover.setFixedSize(192, 288)
    self.cover.setAlignment(QtCore.Qt.AlignCenter)
    self.cover.setStyleSheet("QLabel { background-color: white }")
    self.thumbnail = QtGui.QLabel()
    self.thumbnail.setFixedSize(96, 144)
    self.thumbnail.setStyleSheet("QLabel { background-color: white }")
    self.colors = QtGui.QGroupBox("Gray levels")
    self.color4 = QtGui.QRadioButton("4", self.colors)
    self.color8 = QtGui.QRadioButton("8", self.colors)
    self.color8.setChecked(True)
    self.color16 = QtGui.QRadioButton("16", self.colors)
    self.dither = QtGui.QCheckBox("Dither")
    self.dither.setChecked(False)

    self.convert = QtGui.QPushButton("Convert to PDF")
    self.convert.setEnabled(False)
    self.addcss = QtGui.QPushButton("Add CSS")
    self.reset = QtGui.QPushButton("Reset")
    self.output = QtGui.QPushButton("...")
    self.filename = QtGui.QLineEdit()
    self.filename.setText("epub2pdf.pdf")
    self.cssstyle1 = QtGui.QTextEdit()
    self.cssstyle1.setLineWrapMode(QtGui.QTextEdit.NoWrap)
    self.cssstyle2 = QtGui.QTextEdit()
    self.cssstyle2.setLineWrapMode(QtGui.QTextEdit.NoWrap)
    self.cssstyle3 = QtGui.QTextEdit()
    self.cssstyle3.setLineWrapMode(QtGui.QTextEdit.NoWrap)
    self.cssstyle4 = QtGui.QTextEdit()
    self.cssstyle4.setLineWrapMode(QtGui.QTextEdit.NoWrap)
    self.cssstyle4.setReadOnly(True)
    self.cssstyle4.setAcceptRichText(False)
    self.princeoutput = QtGui.QTextEdit()
    self.princeoutput.setLineWrapMode(QtGui.QTextEdit.NoWrap)
    self.princeoutput.setReadOnly(True)
    self.label_css = QtGui.QLabel("CSS files:")
    self.cssfiles = QtGui.QComboBox()
    self.css = QtGui.QTabWidget()
    self.css.setMinimumWidth(500)

    info = QtGui.QGridLayout()
    info.addWidget(self.label_title, 0, 0)
    info.addWidget(self.title, 0, 1, 1, 2)
    info.addWidget(self.label_author, 1, 0)
    info.addWidget(self.author, 1, 1, 1, 2)
    info.addWidget(self.label_pages, 2, 0)
    info.addWidget(self.pages, 2, 1, 1, 2)
    info.setColumnStretch(1,1)
    info.setVerticalSpacing(0)

    group = QtGui.QVBoxLayout(self.colors)
    group.addWidget(self.color4)
    group.addWidget(self.color8)
    group.addWidget(self.color16)

    thn = QtGui.QVBoxLayout()
    thn.addWidget(self.colors)
    thn.addWidget(self.dither)
    thn.addStretch(1)
    thn.addWidget(self.thumbnail)

    cvr = QtGui.QVBoxLayout()
    cvr.addStretch(1)
    cvr.addWidget(self.cover)

    fls = QtGui.QHBoxLayout()
    fls.addWidget(self.label_files)
    fls.addWidget(self.files)
    fls.addWidget(self.add)
    fls.setStretch(1, 1)

    pixbtn = QtGui.QHBoxLayout()
    pixbtn.addStretch(1)
    pixbtn.addWidget(self.create)
    pixbtn.addSpacing(20)
    pixbtn.addWidget(self.save)
    pixbtn.addStretch(1)

    pix = QtGui.QHBoxLayout()
    pix.addStretch(1)
    pix.addLayout(cvr)
    pix.addSpacing(20)
    pix.addLayout(thn)
    pix.addStretch(1)

    vbox1 = QtGui.QVBoxLayout()
    vbox1.addLayout(fls)
    vbox1.addLayout(pixbtn)
    vbox1.addLayout(pix)
    self.thumbnails = QtGui.QWidget()
    self.thumbnails.setLayout(vbox1)

    buttons2 = QtGui.QHBoxLayout()
    buttons2.addWidget(self.label_css)
    buttons2.addWidget(self.cssfiles)
    buttons2.addWidget(self.addcss)
    buttons2.addWidget(self.reset)
    buttons2.setStretch(1, 1)

    outputfile = QtGui.QHBoxLayout()
    outputfile.addWidget(self.convert)
    outputfile.addWidget(self.filename)
    outputfile.addWidget(self.output)

    vbox2 = QtGui.QVBoxLayout()
    vbox2.addLayout(outputfile)
    vbox2.addLayout(buttons2)
    vbox2.addWidget(self.css)
    self.pdf = QtGui.QWidget()
    self.pdf.setLayout(vbox2)

    self.tabs.addTab(self.thumbnails, "Thumbnail")
    self.tabs.addTab(self.pdf, "PDF")

    vbox = QtGui.QVBoxLayout()
    vbox.addLayout(info)
    vbox.addWidget(self.tabs)
    vbox.addStretch(1)

    widget = QtGui.QWidget()
    widget.setLayout(vbox)
    self.setCentralWidget(widget)

    self.connect(self.files, QtCore.SIGNAL("activated(int)"), self.selectCover)
    self.connect(self.add, QtCore.SIGNAL("clicked()"), self.addFile)
    self.connect(self.create, QtCore.SIGNAL("clicked()"), self.createThumbnail)
    self.connect(self.save, QtCore.SIGNAL("clicked()"), self.saveThumbnail)
    self.connect(self.convert, QtCore.SIGNAL("clicked()"), self.toPDF)
    self.connect(self.addcss, QtCore.SIGNAL("clicked()"), self.loadCSS)
    self.connect(self.reset, QtCore.SIGNAL("clicked()"), self.resetCSS)
    self.connect(self.output, QtCore.SIGNAL("clicked()"), self.selectPDF)
    self.connect(self.cssfiles, QtCore.SIGNAL("activated(int)"), self.selectCSS)

    self.title.setToolTip("<qt>Title(s) of the book, multiple instances are separated with \"|\" (read-only)</qt>")
    self.author.setToolTip("<qt>Author(s) of the book, role in brackets, multiple instances are separated with \";\" (read-only)</qt>")
    self.pages.setToolTip("<qt>Number of pages, as calculated by ADE (not the pages in the PDF), pages of linear elements in parentheses if different</qt>")
    self.tabs.setTabToolTip(0, "<qt>Create a thumbnail for use in the Bookeen Cybooks</qt>")
    self.tabs.setTabToolTip(1, "<qt>Create a PDF file using Prince</qt>")
    self.files.setToolTip("<qt>Select a file from the book to use as cover image</qt>")
    self.add.setToolTip("<qt>Add an external file to the list on the left</qt>")
    self.create.setToolTip("<qt>Create a thumbnail from the selected cover image</qt>")
    self.save.setToolTip("<qt>Save the generated thumbnail (filename is the same as the loaded ePUB, with .thn appended)</qt>")
    self.colors.setToolTip("<qt>Select the number of grey levels to use in the thumbnail</qt>")
    self.dither.setToolTip("<qt>Enable or disable dithering when creating the thumbnail</qt>")
    self.cover.setToolTip("<qt>Selected cover image for generating the thumbnail (not real size)</qt>")
    self.thumbnail.setToolTip("<qt>Generated thumbnail (real size)</qt>")
    self.convert.setToolTip("<qt>Convert the loaded ePUB into PDF</qt>")
    self.addcss.setToolTip("<qt>Add an external file to the \"Additional CSS\" tab</qt>")
    self.reset.setToolTip("<qt>Undo changes in the CSS stylesheets</qt>")
    self.output.setToolTip("<qt>Select a name and location for the output PDF</qt>")
    self.filename.setToolTip("<qt>Name for the output PDF (default: \"epub2pdf.pdf\")</qt>")
    self.cssstyle1.setToolTip("<qt>The stylesheet can be modified before converting to PDF (the original file will not be modified)</qt>")
    self.cssstyle2.setToolTip("<qt>The stylesheet can be modified before converting to PDF (the original file will not be modified)</qt>")
    self.cssstyle3.setToolTip("<qt>The stylesheet can be modified before converting to PDF (the original file will not be modified)</qt>")
    self.cssstyle4.setToolTip("<qt>This stylesheet cannot be modified (but can be overridden by other stylesheets with \"!important\")</qt>")
    self.cssfiles.setToolTip("<qt>Select a file, which can contain CSS code, to view in the \"(View CSS)\" tab</qt>")

    if file:
      self.fileLoaded(file)
    else:
      self.epub = ePUB("")

    self.resetCSS()

  def loadFile(self):
    dialog = QtGui.QFileDialog()
    dialog.setFileMode(QtGui.QFileDialog.ExistingFile)
    filename = dialog.getOpenFileName(self, "Open an ePUB file", "", "ePUB (*.epub);;All files (*.*)")
    if filename:
      self.fileLoaded(unicode(filename))
      self.resetCSS()

  def fileLoaded(self, filename):
    self.epub = ePUB(filename)
    if not self.epub.isEpub:
      self.title.clear()
      self.author.clear()
      self.pages.clear()
      self.files.setEnabled(False)
      self.files.clear()
      self.files.setEnabled(True)
      self.add.setEnabled(False)
      self.create.setEnabled(False)
      self.save.setEnabled(False)
      self.cover.setPixmap(QtGui.QPixmap())
      self.thumbnail.setPixmap(QtGui.QPixmap())
      self.convert.setEnabled(False)
      self.status.showMessage("Not a valid ePUB file")
    else:
      self.title.setText(" | ".join(self.epub.title))
      self.title.home(True)
      self.author.setText("; ".join(self.epub.author))
      self.author.home(True)
      if self.epub.pages != self.epub.lpages:
        self.pages.setText("%d (%d)" % (self.epub.pages, self.epub.lpages))
      else:
        self.pages.setText("%d" %(self.epub.pages))
      self.files.clear()
      self.files.addItem("[None]")
      list = []
      for id in self.epub.files.keys():
        if (self.epub.types[id] == "image/jpeg") or \
           (self.epub.types[id] == "image/gif") or \
           (self.epub.types[id] == "image/png") or \
           (self.epub.types[id] == "image/svg+xml"):
          list.append(self.epub.files[id])
      self.files.addItems(sorted(list))
      if self.epub.coverid:
        i = self.files.findText(self.epub.files[self.epub.coverid], QtCore.Qt.MatchExactly)
        self.files.setCurrentIndex(i)
      self.add.setEnabled(True)
      self.status.showMessage("ePUB file loaded")
      self.cover.setPixmap(QtGui.QPixmap())
      self.thumbnail.setPixmap(QtGui.QPixmap())
      self.save.setEnabled(False)
      if self.epub.coverdata:
        pixmap = QtGui.QPixmap()
        pixmap.loadFromData(self.epub.coverdata)
        self.cover.setPixmap(pixmap.scaled(192, 288, QtCore.Qt.KeepAspectRatio))
        self.create.setEnabled(True)
        self.status.showMessage("Cover found")
      self.convert.setEnabled(True)

  def selectCover(self, index):
    file = unicode(self.files.currentText())
    self.epub.coverid = ""
    for id in self.epub.files.keys():
      if file == self.epub.files[id]:
        self.epub.coverid = id
    if self.epub.coverid:
      zfile = zipfile.ZipFile(self.epub.filename, "r")
      self.epub.coverdata = zfile.read(os.path.join(self.epub.basedir, file))
      zfile.close()
    elif index > 0:
      tmp = open(file, "rb")
      self.epub.coverdata = tmp.read()
      tmp.close()
    else:
      self.epub.coverdata = ""
    pixmap = QtGui.QPixmap()
    pixmap.loadFromData(self.epub.coverdata)
    if pixmap.isNull():
      self.cover.setPixmap(QtGui.QPixmap())
      self.create.setEnabled(False)
      self.status.showMessage("Not a valid cover")
    else:
      self.cover.setPixmap(pixmap.scaled(192, 288, QtCore.Qt.KeepAspectRatio))
      self.create.setEnabled(True)
      self.status.showMessage("Cover selected")
    self.thumbnail.setPixmap(QtGui.QPixmap())
    self.save.setEnabled(False)

  def addFile(self):
    dialog = QtGui.QFileDialog()
    dialog.setFileMode(QtGui.QFileDialog.ExistingFile)
    filename = dialog.getOpenFileName(self, "Select a cover file", "", "All files (*.*)")
    self.files.addItem(unicode(filename))
    self.files.setCurrentIndex(self.files.count()-1)

  def createThumbnail(self):
    if not self.epub.coverdata:
      return
    if self.color4.isChecked():
      num = 4
    if self.color8.isChecked():
      num = 8
    if self.color16.isChecked():
      num = 16
    if self.dither.isChecked():
      self.epub.thndata = create_thumbnail(self.epub.coverdata, num, "FloydSteinberg")
    else:
      self.epub.thndata = create_thumbnail(self.epub.coverdata, num)
    if self.epub.thndata == -1:
      QtGui.QMessageBox.critical(self,"Error","<qt>There was an error running \"convert\". Is it installed?</qt>")
      self.thumbnail.setPixmap(QtGui.QPixmap())
      self.status.showMessage("Error creating thumbnail")
    else:
      pixmap = QtGui.QPixmap()
      pixmap.loadFromData(self.epub.thndata)
      self.thumbnail.setPixmap(pixmap)
      self.save.setEnabled(True)
      self.status.showMessage("Thumbnail created")

  def saveThumbnail(self):
    self.epub.writeThumbnail()
    self.status.showMessage("Thumbnail saved: %s" % (self.epub.filename+".thn"))

  def resetCSS(self):
    self.css.clear()
    self.cssstyle1.clear()
    self.cssstyle2.clear()
    self.cssstyle3.clear()
    self.cssstyle4.clear()
    self.princeoutput.clear()
    i = self.css.addTab(self.cssstyle1, "Additional CSS")
    self.css.setTabToolTip(i, "<qt>An external file can be added as additional CSS stylesheet</qt>")
    if self.epub.princecss:
      self.cssstyle2.setPlainText(self.epub.princecss)
      i = self.css.addTab(self.cssstyle2, "Book CSS")
      self.css.setTabToolTip(i, "<qt>Book-specific CSS stylesheet included in the ePUB</qt>")
    i = self.css.addTab(self.cssstyle3, "Default CSS")
    defaultcss = os.path.expanduser(os.path.join("~",".epub2pdf","default.css"))
    try:
      cssfile = codecs.open(defaultcss, "rb", "utf-8")
      self.cssstyle3.setPlainText(cssfile.read())
      self.css.setTabToolTip(i, "<qt>Default CSS stylesheet (%s)</qt>" % defaultcss)
      cssfile.close()
    except IOError:
      self.cssstyle3.setPlainText(defaultprincecss)
      self.css.setTabToolTip(i, "<qt>Default CSS stylesheet (can be overwritten in %s)</qt>" % defaultcss)
    self.cssfiles.clear()
    list = []
    for id in self.epub.files.keys():
      if os.path.join(self.epub.basedir, self.epub.files[id]) == self.epub.princefile:
        continue
      if (self.epub.types[id] == "text/css") or \
         (self.epub.types[id] == "application/xhtml+xml") or \
         (self.epub.types[id] == "application/x-dtbook+xml") or \
         (self.epub.types[id] == "text/x-oeb1-document") or \
         (self.epub.types[id] == "text/x-oeb1-css"):
        list.append(self.epub.files[id])
    self.cssfiles.addItems(sorted(list))
    if self.cssfiles.count():
      i = self.css.addTab(self.cssstyle4, "(View CSS)")
      self.css.setTabToolTip(i, "<qt>View box for other CSS stylesheets included in the book</qt>")

  def selectCSS(self):
    file = unicode(self.cssfiles.currentText())
    zfile = zipfile.ZipFile(self.epub.filename, "r")
    self.cssstyle4.setPlainText(unicode(zfile.read(os.path.join(self.epub.basedir, file)),"utf-8"))
    zfile.close()
    self.css.setCurrentWidget(self.cssstyle4)

  def loadCSS(self):
    dialog = QtGui.QFileDialog()
    dialog.setFileMode(QtGui.QFileDialog.ExistingFile)
    filename = dialog.getOpenFileName(self, "Open a CSS stylesheet", "", "CSS (*.css);;All files (*.*)")
    if filename:
      cssfile = codecs.open(str(filename), "rb", "utf-8")
      content = cssfile.read()
      self.cssstyle1.setPlainText(content)
      cssfile.close()
      self.css.setCurrentWidget(self.cssstyle1)

  def selectPDF(self):
    dialog = QtGui.QFileDialog()
    dialog.setFileMode(QtGui.QFileDialog.AnyFile)
    filename = dialog.getOpenFileName(self, "Select a PDF output", "", "PDF (*.pdf);;All files (*.*)")
    if filename:
      self.filename.setPlainText(filename)

  def toPDF(self):
    css1 = unicode(self.cssstyle1.toPlainText())
    css2 = unicode(self.cssstyle2.toPlainText())
    css3 = unicode(self.cssstyle3.toPlainText())
    file1 = 0
    file2 = 0
    file3 = 0
    command = ["C:\\Program Files\\Prince\\Engine\\bin\\prince.exe", "-v"]
    tmpdir = tempfile.mkdtemp()
    if css1:
      file1 = tempfile.NamedTemporaryFile(dir=tmpdir, delete=False)
      file1.write(css1.encode("utf-8"))
      file1.close()
      command.append("-s")
      command.append(file1.name)
    if css2:
      file2 = tempfile.NamedTemporaryFile(dir=tmpdir, delete=False)
      file2.write(css2.encode("utf-8"))
      file2.close()
      command.append("-s")
      command.append(file2.name)
    if css3:
      file3 = tempfile.NamedTemporaryFile(dir=tmpdir, delete=False)
      file3.write(css3.encode("utf-8"))
      file3.close()
      command.append("-s")
      command.append(file3.name)
    command.append("-o")
    #vz command.append(str(self.filename.text()))
    command.append(os.path.join(os.getcwd(), str(self.filename.text())))
    zfile = zipfile.ZipFile(self.epub.filename, "r")
    zfile.extractall(tmpdir)
    zfile.close()
    for i in self.epub.spine:
      #vz command.append(os.path.join(tmpdir,i))
      command.append(i)
    # (this is a hack, but it works...)
    set_metadata(os.path.join(tmpdir,self.epub.spine[0]), self.epub)
    self.status.showMessage("Running Prince")
    try:
      #vz convert = subprocess.Popen(command, stderr=subprocess.PIPE)
      convert = subprocess.Popen(command, stderr=subprocess.PIPE, cwd=tmpdir)
      messages = ""
      messages = convert.communicate()[1]
      i = self.css.addTab(self.princeoutput, "Prince output")
      self.css.setTabToolTip(i, "<qt>Output from the Prince run</qt>")
      self.css.setCurrentIndex(i)
      self.princeoutput.setPlainText(messages)
      self.status.showMessage("Prince run finished")
    except OSError:
      QtGui.QMessageBox.critical(self,"Error","<qt>There was an error running \"prince\". Is it installed?</qt>")
      self.status.showMessage("Error creating PDF")
    shutil.rmtree(tmpdir)

#===============================================================================
# Main program

app = QtGui.QApplication(sys.argv)
try:
  mw = Main(file=sys.argv[1])
except IndexError:
  mw = Main()
mw.show()
sys.exit(app.exec_())
