/*
    Test script for PRS600 to interact with the screen via /dev/fb0 using
    a readline-based interface (requires libeditline).
    
    by Rafal Kolanski (xaph.net) 2010
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "editline.h"
#include "eink.h"

/* E-ink screen */
static eink_t* eink;

/* holds a line for rl_gets */
static char* line_read = (char*) NULL;

/* delimiters for command parsing */
char delim[] = " \t";

/* keep track of last rect of pixel change */
struct eink_update last_damage;


/* Strings and tokens */

/* read string, return a pointer to it, or NULL on EOF. */
char* rl_gets() {
    /* if the buffer has already been allocated,
       return the memory to the free pool. */
    if (line_read) {
        free(line_read);
        line_read = (char*) NULL;
    }
    /* get a line from the user. */
    line_read = readline("> ");
    /* if the line has any text in it, save it in the history. */
    if (line_read && *line_read) add_history (line_read);
    return line_read;
}

/* read long via strtok and return it, also return 0 on success via res */
long longtok(int* res) {

    char* tok = strtok(NULL, delim);

    if (!tok) {
        *res = -1;
        return 0;
    }

    char* endptr;
    long l = strtol(tok, &endptr, 0);

    if (tok[0] != '\0' && *endptr == '\0') {
        *res = 0;
    } else {
        *res = -1;
    }
    return l;
}

int no_next_tok() {
    char* tok = strtok(NULL, delim);
    return tok == NULL;
}


/* Drawing */

void draw_fill(unsigned char color) {
    unsigned char* buf = eink_buffer(eink);
    for (int i = 0 ; i < eink_width(eink) * eink_height(eink) ; ++i) {
        buf[i] = color;
    }
}

void draw_rect(int x, int y, int sx, int sy, int color) {
    /* yes, hardly optimal, I just want it to work */
    int w = eink_width(eink);
    int h = eink_height(eink);
    unsigned char* buf = eink_buffer(eink);
    for (int i = x ; i < x + sx ; ++i) {
        for (int j = y ; j < y + sy ; ++j) {
            if (i < 0 || i > w || j < 0 || j > h) continue;
            if (color == -1) {
                // use xor pattern
                buf[j * w + i] = i ^ j;
                continue;
            }
            buf[j * w + i] = color;
        }
    }
    last_damage.x = x;
    last_damage.y = y;
    last_damage.w = sx;
    last_damage.h = sy;
}

int update_rect(int x, int y, int sx, int sy, int mode) {
    struct eink_update damage;
    /* yes, hardly optimal, I just want it to work */
    damage.x = x;
    damage.y = y;
    damage.w = sx;
    damage.h = sy;
    damage.upmode = mode;
    damage.orientation = eink_get_orientation(eink, 0);
    return eink_update(eink, &damage);
}

int update_last(int mode) {
    last_damage.upmode = mode;
    last_damage.orientation = eink_get_orientation(eink, 0);
    return eink_update(eink, &last_damage);
}


/* Command handlers */

void handle_fill() {
    int res;
    unsigned char arg;
    arg = longtok(&res);
    if (res || !no_next_tok()) {
        printf("syntax: fill <color>\n");
        printf("fills entire framebuffer with <color>\n");
        return;
    }
    printf("Fill %u\n", arg);
    draw_fill(arg);
}

void handle_orientation() {
    int res;
    int arg;
    arg = longtok(&res);
    if (res) {
        printf("Orientation: %d\n", eink_get_orientation(eink, 1));
    } else {
        if (no_next_tok()) {
            printf("Setting orientation to %d: ", arg);
            res = eink_set_orientation(eink, arg);
            printf("%s", res ? "failed" : "ok");
            printf("\nOrientation: %d\n", eink_get_orientation(eink, 1));
        } else {
            printf("syntax: orientation [orientation]\n");
            printf("gets orientation or sets it if argument provided (0-3)\n");
        }
    }
}

void handle_update_last() {
    int res;
    int arg;
    arg = longtok(&res);
    if (res || !no_next_tok()) {
        printf("syntax: update_last <mode>\n");
        printf("update the last rectangle drawn. use <mode>\n");
        return;
    }
    res = update_last(arg);
    printf("Update %d %d (%d x %d) -> %d\n", 
            last_damage.x, last_damage.y, last_damage.w, last_damage.h, res);
}

void handle_update_full() {
    int res;
    int arg;
    arg = longtok(&res);
    if (res || !no_next_tok()) {
        printf("syntax: update_full <mode>\n");
        printf("update entire display using <mode>\n");
        return;
    }
    res = eink_update_full(eink, arg);
    printf("Update of entire screen with mode %d\n", arg);
}

void print_power_mode(int p) {
    printf("Power status: ");
    switch (p) {
        case POWER_MODE_NORMAL:
            printf("normal");
            break;
        case POWER_MODE_SLEEP:
            printf("sleep");
            break;
        case POWER_MODE_STANDBY:
            printf("standby");
            break;
        default:
            printf("%d", p);
    }
    printf("\n");
}

void handle_power() {
    if (!no_next_tok()) {
        printf("syntax: power\n");
        return;
    }
    int p = eink_get_power_mode(eink);
    print_power_mode(p);
}

void handle_sleep() {
    if (!no_next_tok()) {
        printf("syntax: sleep\n");
        return;
    }
    if (eink_sleep(eink)) {
        printf("Power setting failed\n");
    }
    int p = eink_get_power_mode(eink);
    print_power_mode(p);
}

void handle_wakeup() {
    if (!no_next_tok()) {
        printf("syntax: wakeup\n");
        return;
    }
    if (eink_wakeup(eink)) {
        printf("Power setting failed\n");
    }
    int p = eink_get_power_mode(eink);
    print_power_mode(p);
}

void handle_temp() {
    if (!no_next_tok()) {
        printf("syntax: temp\n");
        return;
    }
    printf("Temperature: %d\n", eink_get_temperature(eink));
}

void handle_size() {
    if (!no_next_tok()) {
        printf("syntax: wakeup\n");
        return;
    }
    printf("%d x %d\n", eink_width(eink), eink_height(eink));
}

void handle_update() {
    int x,y,sx,sy,mode;
    int res;
    
    x = longtok(&res);
    if (res) goto err;
    y = longtok(&res);    
    if (res) goto err;
    sx = longtok(&res);    
    if (res) goto err;
    sy = longtok(&res);    
    if (res) goto err;
    mode = longtok(&res);
    if (res) goto err;
    if (!no_next_tok()) goto err;

    printf("update %d %d (%d x %d) mode: %d\n", x,y,sx,sy, mode);
    update_rect(x,y,sx,sy,mode);
    return;
err:
    printf("syntax: update x y sx sy mode\n");
    printf("update rectangle at (x,y) of size (sx, sy) using given mode\n");
    return;
}

void handle_rect() {
    int x,y,sx,sy,color;
    int res;
    
    x = longtok(&res);
    if (res) goto err;
    y = longtok(&res);    
    if (res) goto err;
    sx = longtok(&res);    
    if (res) goto err;
    sy = longtok(&res);    
    if (res) goto err;
    color = longtok(&res);
    if (res) goto err;
    if (!no_next_tok()) goto err;

    printf("rect %d %d (%d x %d) %d\n", x,y,sx,sy,color);
    draw_rect(x,y,sx,sy,color);
    return;
err:
    printf("syntax: rect x y sx sy color\n");
    printf("draw rectangle at (x,y) of size (sx, sy) into frame buffer\n");
    return;
}

void handle_line(char* line) {
    /* get first token, the command */
    char* cmd = strtok(line, delim);
    if (!cmd || !cmd[0]) return;

    if (strcmp(cmd, "rect") == 0) return handle_rect();
    if (strcmp(cmd, "orientation") == 0) return handle_orientation();
    if (strcmp(cmd, "power") == 0) return handle_power();
    if (strcmp(cmd, "wakeup") == 0) return handle_wakeup();
    if (strcmp(cmd, "sleep") == 0) return handle_sleep();
    if (strcmp(cmd, "size") == 0) return handle_size();
    if (strcmp(cmd, "update") == 0) return handle_update();
    if (strcmp(cmd, "update_last") == 0) return handle_update_last();
    if (strcmp(cmd, "update_full") == 0) return handle_update_full();
    if (strcmp(cmd, "fill") == 0) return handle_fill();
    if (strcmp(cmd, "temp") == 0) return handle_temp();
    
    printf("Unknown command: %s\n", cmd);
}

int main(int argc, char* argv[]) {
    char* s;
    eink = eink_open();
    if (!eink) {
        printf("Could not initialise e-ink display.\n");
        return 1;
    }
    printf("Make sure you use wakeup before trying to update the display,"
           "or the system will freeze.\n");
    while ((s = rl_gets())) {
        if (!s[0]) continue;
        handle_line(s);
    }
    eink_close(eink);
    printf("\n");
    return 0;
}

