/*
    Interface to broadsheet E-ink display via ioctls.
    
    by Rafal Kolanski (xaph.net) 2010
*/
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>

#include "eink.h"

#define EINK_WIDTH 800
#define EINK_HEIGHT 600
#define VALID_ORIENTATION(x) (x >= 0 && x < 4)

struct eink {
    int fd;
    unsigned char* buf;
    size_t buf_size;
    struct fb_fix_screeninfo info;
    int orientation;
};

eink_t* eink_open(void) {
    eink_t* e = malloc(sizeof(*e));

    if (!e) {
        perror("eink_open alloc");
        return NULL;
    }
    e->fd = 0;
    e->buf = NULL;

    int fd = open("/dev/fb0", O_RDWR);
    if (fd < 0) {
        perror("eink_open open");
        goto cleanup_alloc;
    }
    e->fd = fd;

    if (ioctl(fd, FBIOGET_FSCREENINFO, &(e->info))) {
        perror("eink_open ioctl");
        goto cleanup_open;
    }

    if (eink_get_orientation(e, 1) < 0) {
        fprintf(stderr, "eink_open: bad internal orientation\n");
        goto cleanup_open;
    }

    e->buf_size = EINK_WIDTH * EINK_HEIGHT;
    e->buf = mmap(NULL, e->buf_size, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0);
    if (e->buf == MAP_FAILED) {
        perror("eink_open mmap");
        goto cleanup_open;
    }
    return e;

cleanup_open:
    if (fd > 0) close(fd);
cleanup_alloc:
    free(e);
    return NULL;
}

void eink_close(eink_t* e) {
    munmap(e->buf, e->buf_size);
    close(e->fd);
    free(e);
}

unsigned eink_width(eink_t* e) {
    /* Orientations 0 and 2 are landscape, 1 and 3 are portrait. */
    if (e->orientation & 1) {
        return EINK_HEIGHT;
    } else {
        return EINK_WIDTH;
    }
}

unsigned eink_height(eink_t* e) {
    /* Orientations 0 and 2 are landscape, 1 and 3 are portrait. */
    if (e->orientation & 1) {
        return EINK_WIDTH;
    } else {
        return EINK_HEIGHT;
    }
}

int eink_get_orientation(eink_t* e, int force) {
    if (force) {
        e->orientation = ioctl(e->fd, FBIO_EINK_GET_ORIENTATION, NULL);
        if (!VALID_ORIENTATION(e->orientation)) {
            return -1;
        }
    }
    return e->orientation;
}

int eink_set_orientation(eink_t* e, int orientation) {
    ioctl(e->fd, FBIO_EINK_SET_ORIENTATION, orientation);
    return eink_get_orientation(e, 1) != orientation;
}

int eink_get_power_mode(eink_t* e) {
    return ioctl(e->fd, FBIO_EINK_GET_POWER_MODE, NULL);
}

int eink_is_awake(eink_t* e) {
    return eink_get_power_mode(e) == POWER_MODE_NORMAL;
}

int set_power_mode(eink_t* e, int mode) {
    return ioctl(e->fd, FBIO_EINK_SET_POWER_MODE, mode);
}

int eink_sleep(eink_t* e) {
    set_power_mode(e, POWER_MODE_SLEEP);
    return eink_get_power_mode(e) != POWER_MODE_SLEEP;
}

int eink_wakeup(eink_t* e) {
    set_power_mode(e, POWER_MODE_NORMAL);
    return eink_get_power_mode(e) != POWER_MODE_NORMAL;
}

int eink_get_temperature(eink_t* e) {
    return ioctl(e->fd, FBIO_EINK_GET_TEMPERATURE, NULL);
}

unsigned char* eink_buffer(eink_t* e) {
    return e->buf;
}

int eink_update(eink_t* e, struct eink_update* ud) {
    return ioctl(e->fd, FBIO_EINK_UPDATE_PIC, (struct updata*) ud);
}

int eink_update_full(eink_t* e, int mode) {
    struct eink_update ud;
    ud.x = 0;
    ud.y = 0;
    ud.w = eink_width(e);
    ud.h = eink_height(e);
    ud.orientation = eink_get_orientation(e, 1);
    ud.upmode = mode;
    return eink_update(e, &ud);
}


