/*
 * dumpfile.c
 *
 *  Created on: Aug 18, 2010
 *      Author: choff
 */
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <ctype.h>
#include <stdbool.h>
#include "dumpfile.h"
#define BASE 16

bool file_read_offset(FILE *file, long int *offset);
bool file_read_assert_string(FILE *file, unsigned char *expected_data);
bool file_read_assert_char(FILE *file, unsigned char expected_character);
int read_number_from_file(FILE *file, long int *number, unsigned char base, unsigned char *terminator);

dumpfile_t *dumpfile_new(FILE *file) {
	dumpfile_t *dumpfile = malloc(sizeof(dumpfile_t));
	dumpfile->file = file;
	if (!file_read_offset(file, &dumpfile->start_addr)) {
		fprintf(stderr, "Error reading start address offset\n");
		return NULL;
	}
	dumpfile->curr_addr = dumpfile->start_addr;
	return dumpfile;
}

bool file_read_offset(FILE *file, long int *offset) {
	unsigned char addr_terminator;
	// Read the offset
	if (read_number_from_file(file, offset, BASE, &addr_terminator) <= 1) {
		fprintf(stderr, "Error reading address offset: Should be numeric hex number\n");
		return false;
	}

	// The character following the offset should be ": "
	if (addr_terminator != ':') {
		fprintf(stderr, "Abnormal address terminator encountered: \"%c\"(expecting \":\")\n", addr_terminator);
		return false;
	}
	if (!file_read_assert_char(file, ' ')) {
		fprintf(stderr, "Error: Expected \"<address>: \", but got \"<address>:[other character than space]\n");
		return false;
	}

	return true; // return offset
}

bool dumpfile_read_line(dumpfile_t *dumpfile, unsigned char *data, int *n_bytes) {
	int byte_idx;
	long int curr_byte;
	unsigned char terminator;

	for (byte_idx = 0; read_number_from_file(dumpfile->file, &curr_byte, BASE, &terminator) > 1; byte_idx++) {
		if (terminator != ' ')
			fprintf(stderr, "Unexpected data terminator: \"%c\", offset=%lx\n", terminator, dumpfile->curr_addr);
		if (curr_byte > UCHAR_MAX) {
			fprintf(stderr, "Number %ld is too large\n", curr_byte);
			return false;
		}

		data[byte_idx] = (unsigned char) curr_byte;
		// printf(" %lx\n", curr_byte);
		//printf("%c", data[byte_idx]);
	}
	*n_bytes = byte_idx;
	if (terminator != ' ')
		fprintf(stderr, "Unexpected data terminator at the end of data block: %c\n", terminator);

	int idx;
	for (idx = 0; idx < 2; idx++) {
		if (!file_read_assert_char(dumpfile->file, ' ')) // 3 spaces expected
			return false;
	}

	for (byte_idx = 0; byte_idx < *n_bytes; byte_idx++) {
		unsigned char expected_char;
		if (isprint(data[byte_idx]))
			expected_char = data[byte_idx];
		else
			expected_char = '.';

		if (!file_read_assert_char(dumpfile->file, expected_char))
			return false;
	}

	if(!file_read_assert_char(dumpfile->file, '\n')) {
		if (feof(dumpfile->file))
				return true;

		fprintf(stderr, "No newline found!\n");
		return false;
	}

	long int exp_offset = dumpfile->curr_addr + *n_bytes;
	long int actual_offset;
	if(!file_read_offset(dumpfile->file, &actual_offset)) {
		fprintf(stderr, "Error reading address offset from file\n");
		return false;
	}
	if (exp_offset != actual_offset) {
		fprintf(stderr, "Unexpected offset found for new data. Read %d bytes, old offset=%lx, expected new offset of %lx but found %lx\n", *n_bytes, dumpfile->curr_addr, exp_offset, actual_offset);
		return false;
	}

	//printf("\n");
	dumpfile->curr_addr = exp_offset;
	return true;
}

bool file_read_assert_string(FILE *file, unsigned char *expected_data) {
	unsigned char *expected_character = expected_data;
	while (*expected_character != 0) {
		if (!file_read_assert_char(file, *expected_character))
			return false;

		expected_character++;
	}

	return true;
}

bool file_read_assert_char(FILE *file, unsigned char expected_character) {
	int next_character = fgetc(file);
	if (next_character == EOF) {
		fprintf(stderr, "End of file!\n");
		return false;
	} else if ((unsigned char) next_character != expected_character) {
		fprintf(stderr, "Assertion \"%c\" = \"%c\" failed!\n", next_character, expected_character);
		return false;
	}

	return true;
}

/* Reads a number from "file" and stops as soon as a non-numeric character
 * is found.
 * The number is written into the address referenced by number.
 * Returns the total number of digits read in the file stream;
 * that is the number of digits plus 1 for the non-numeric terminator character
 */
int read_number_from_file(FILE *file, long int *number, unsigned char base, unsigned char *terminator) {
	int n_chars_read;
	int curr_digit_int;
	*number = 0;

	for (n_chars_read = 1; (curr_digit_int = fgetc(file)) != EOF; n_chars_read++) {
		unsigned char curr_char = (unsigned char) curr_digit_int;
		if (terminator)
			*terminator = curr_char;

		unsigned char curr_digit;

		int dist_to_0 = (int) (curr_char) - (int) '0';
		if (dist_to_0 < 0)
			return n_chars_read; // Everything below 0 in the ASCII table
		else if (dist_to_0 < 10)
			curr_digit = dist_to_0; // 0-9
		else {
			int dist_to_A = (int) (curr_char) - (int) 'a';
			if (dist_to_A < 0)
				return n_chars_read;
			curr_digit = (unsigned char) (dist_to_A + 10); // Hex: A-F
		}

		if (curr_digit >= base)
			return n_chars_read;

		*number = *number * base + curr_digit;
	}

	return n_chars_read;
}
