/*
 * File Name: menu.c
 */

/*
 * This file is part of hello-world.
 *
 * hello-world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello-world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <stdio.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// menus for application, must be unique
// TODO: replace "helloworld" with your application name
static const char *MENU_MAIN                = "helloworld_menu_main";
                                            
// menu groups, must be unique
// TODO: replace "helloworld" with your application name
static const char *GROUP_VIEWS              = "helloworld_view";
static const char *GROUP_ACTIONS            = "helloworld_actions";
                                            
// menu items
//   group "Views"                          
static const char *ITEM_VIEW_ICON           = "view_small";
static const char *ITEM_VIEW_LIST           = "view_detail";
//   group "Actions"                        
static const char *ITEM_ACTION_OPEN         = "open";
static const char *ITEM_ACTION_COPY         = "copy";
static const char *ITEM_ACTION_CUT          = "cut";
static const char *ITEM_ACTION_PASTE        = "paste";
static const char *ITEM_ACTION_CLOSE        = "close";
                                            
// item states                              
static const char *STATE_NORMAL             = "normal";
static const char *STATE_SELECTED           = "selected";
static const char *STATE_DISABLED           = "disabled";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------



//============================================================================
// Local Function Definitions
//============================================================================



//============================================================================
// Functions Implementation
//============================================================================

// initialise popup menu
void menu_init ( void ) 
{
    static gboolean firsttime = TRUE;
    const char      *group;

    LOGPRINTF("entry");

    // execute only once
    if ( !firsttime )
    {
        WARNPRINTF("function called twice");
        return;
    }
    firsttime = FALSE;

    // add groups
    group = GROUP_VIEWS;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_VIEW_ICON,         group, "view_small"      );
    ipc_menu_add_item ( ITEM_VIEW_LIST,         group, "view_detail"     );
    //                                                                                                                
    group = GROUP_ACTIONS;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_ACTION_OPEN,       group, "open"            );
    ipc_menu_add_item ( ITEM_ACTION_COPY,       group, "copy"            );
    ipc_menu_add_item ( ITEM_ACTION_CUT,        group, "cut"             );
    ipc_menu_add_item ( ITEM_ACTION_PASTE,      group, "paste"           );
    ipc_menu_add_item ( ITEM_ACTION_CLOSE,      group, "close"           );

    // add menus
    ipc_menu_add_menu( MENU_MAIN, GROUP_VIEWS, GROUP_ACTIONS, NULL );

    // set menu texts
    menu_set_text();

    // disable what is not yet implemented
    // TODO: implement all and remove this list
    group = GROUP_ACTIONS;
    ipc_menu_set_item_state ( ITEM_ACTION_COPY,  group, STATE_DISABLED );
    ipc_menu_set_item_state ( ITEM_ACTION_CUT,   group, STATE_DISABLED );
    ipc_menu_set_item_state ( ITEM_ACTION_PASTE, group, STATE_DISABLED );
    
    // set menu context
    menu_show();    
}


// remove the proper popup menu
void menu_destroy (void)
{
    LOGPRINTF("entry");

    // remove the main menu
    ipc_remove_menu( MENU_MAIN );
}


// show the proper popup menu
void menu_show (void)
{
    LOGPRINTF("entry");

    // show the main menu
    ipc_menu_show_menu( MENU_MAIN );
}


// set menu texts
void menu_set_text ( void ) 
{
    const char      *group;

    LOGPRINTF("entry");

    // set groups + items
    group = GROUP_VIEWS;
    ipc_menu_set_group_label( group,                    _("View")              );
    ipc_menu_set_item_label ( ITEM_VIEW_ICON,    group, _("Thumbnails")        );
    ipc_menu_set_item_label ( ITEM_VIEW_LIST,    group, _("Details")           );
    //                                                                                                                 
    group = GROUP_ACTIONS;                                                     
    ipc_menu_set_group_label( group,                    _("Actions")           );
    ipc_menu_set_item_label ( ITEM_ACTION_OPEN,  group, _("Open")              );
    ipc_menu_set_item_label ( ITEM_ACTION_COPY,  group, _("Copy")              );
    ipc_menu_set_item_label ( ITEM_ACTION_CUT,   group, _("Cut")               );
    ipc_menu_set_item_label ( ITEM_ACTION_PASTE, group, _("Paste")             );
    ipc_menu_set_item_label ( ITEM_ACTION_CLOSE, group, _("Close")             );

    // set menus
    ipc_menu_set_menu_label( MENU_MAIN,  _("Hello World Menu") );
}


// select one of the view types
void menu_select_view_type ( const viewtypes_t view_type )
{
    static const char   *item_old = "";
           const char   *item_new = "";

    LOGPRINTF("entry: view_type [%d]", view_type);
    g_assert(view_type < N_VIEWTYPES);

    switch (view_type)
    {
        case ICONVIEW:
            item_new = ITEM_VIEW_ICON;
            break;
        case LISTVIEW:
            item_new = ITEM_VIEW_LIST;
            break;
        default:
            ;  // ignore: handled by assert at function entry
    }

    if ( strcmp(item_old, item_new) != 0 )
    {
        if ( item_old[0] != '\0' )
        {
            ipc_menu_set_item_state ( item_old, GROUP_VIEWS, STATE_NORMAL   );
        }
        ipc_menu_set_item_state ( item_new, GROUP_VIEWS, STATE_SELECTED );

        item_old = item_new;
    }
}


//----------------------------------------------------------------------------
// Callbacks from popupmenu
//----------------------------------------------------------------------------

// user has pressed a menu button
void menu_on_item_activated ( const gchar *item,
                              const gchar *group,
                              const gchar *menu,
                              const gchar *state )
{
    gboolean    ok = TRUE;
    viewtypes_t view_type  = 0;
    gchar       *msg = NULL;

    LOGPRINTF("entry: item [%s] group [%s]", item, group);

    if ( strcmp(group, GROUP_VIEWS) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if (      strcmp(item, ITEM_VIEW_ICON) == 0 )
            {
                view_type = ICONVIEW;
            }
            else if ( strcmp(item, ITEM_VIEW_LIST) == 0 )
            {
                view_type = LISTVIEW;
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }

            if (ok)
            {
                main_set_view_type( view_type );
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_ACTIONS) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if (      strcmp(item, ITEM_ACTION_OPEN) == 0 )
            {
                // open a document in the viewer
                msg = g_strdup_printf("uds %s/%s", g_mountpoint, "test.pdf");
                ipc_sys_start_task(msg, g_mountpoint, "test.pdf", NULL, NULL);
                g_free(msg);
            }
            else if ( strcmp(item, ITEM_ACTION_CLOSE) == 0 )
            {
                // quit and exit application
                main_quit();
                _exit(0);
                return;
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }

            if (ok)
            {
                main_set_view_type( view_type );
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else
    {
        WARNPRINTF("unexpected menu group [%s]", group);
        ok = FALSE;
    }

    if (!ok)
    {
        msg = g_strdup_printf( _("Unhandled menu item.\n"
                                 "menu: %s\n"
                                 "group: %s\n"
                                 "item: %s\n"
                                 "state: %s"),
                               menu,
                               group,
                               item,
                               state );
        gtk_label_set_text( GTK_LABEL(g_action), msg);
        g_free(msg);
    }
}
