/*
 * File Name: db.c
 */

/*
 * This file is part of hello-world.
 *
 * hello-world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello-world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <unistd.h>
#include <gtk/gtk.h>

// local include files, between " "
#include <db.h>
#include "log.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

#define THUMBNAIL_MINI   0
#define THUMBNAIL_SMALL  1
#define THUMBNAIL_MEDIUM 2
#define THUMBNAIL_LARGE  3
#define THUMBNAIL_COUNT  4

// Column names for table file_metadata in metadb
static const char* FILE_TITLE          = "title";
static const char* FILE_AUTHOR         = "author";

// Columns in database table thumbnails.
static const char* THUMBNAIL_COLUMNS[THUMBNAIL_COUNT] =
{
    "thumb_data_mini",
    "thumb_data_small",
    "thumb_data_medium",
    "thumb_data_large"
};

static const int THUMBNAIL_SIZES[THUMBNAIL_COUNT] =
{
    30, 60, 120, 300
};


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------


//============================================================================
// Local Function Definitions
//============================================================================

static void  set_pixel_value(GdkPixbuf * pixbuf, int x, int y, gint color);


//============================================================================
// Functions Implementation
//============================================================================

erMetadb open_database(const char* IN dir)
{
    g_assert(dir != NULL);

    erMetadb db = ermetadb_global_open(dir, FALSE);
    if ( NULL == db) {
        ERRORPRINTF("Could not open database %s", dir );
    } 
    return db;
}


void close_database(erMetadb IN db) 
{
    g_assert ( NULL != db );
    
    ermetadb_close(db);
    // explicit sync since ermetadb_close_database does not flush to disk
    sync();
}


gboolean load_file_metadata(erMetadb        IN  db,
                            const gchar*    IN  filepath,
                            const gchar*    IN  filename,
                            GString*        OUT title,
                            GString*        OUT author,
                            thumbType*      OUT small,
                            thumbType*      OUT medium)
{
    g_assert(db != NULL);
    g_assert(filename != NULL);

    metadata_table* name_table = metadata_table_new();
    g_return_val_if_fail( NULL != name_table, FALSE);

    // add columns, caller is owner of OUT parameters
    if (title   ) { metadata_table_add_column(name_table, FILE_TITLE       ); }
    if (author  ) { metadata_table_add_column(name_table, FILE_AUTHOR      ); }
    if (small   ) { metadata_table_add_column(name_table, THUMBNAIL_COLUMNS[THUMBNAIL_SMALL]);  }
    if (medium  ) { metadata_table_add_column(name_table, THUMBNAIL_COLUMNS[THUMBNAIL_MEDIUM]); }

    // query
    metadata_table* results_table = NULL; // must be NULL, created by function below
    int rc = ermetadb_global_get_file(db, filepath, filename, name_table, &results_table);

    if (rc != ER_OK)
    {
        ERRORPRINTF("ermetadb_get_file_metadata() returns [%d]", rc);
        metadata_table_free(name_table);
        metadata_table_free(results_table);
        return FALSE;
    }

    if (title) 
    {
        // Get title.
        const char *column = FILE_TITLE;
        int index = metadata_table_find_column(results_table, column);
        if (index >= 0)
        {
            const metadata_cell *cell = metadata_table_get_cell(results_table, index);

            if (cell && cell->type == METADATA_TEXT)
            {
                g_string_assign(title, cell->value.v_text->str );
            }
        }
        else
        {
            ERRORPRINTF("Cannot find column %s", column);
        }
    }

    if (author)
    {
        // get author
        const char* column = FILE_AUTHOR;
        int index = metadata_table_find_column(results_table, column);
        if (index >= 0)
        {
            const metadata_cell *cell = metadata_table_get_cell(results_table, index);

            if (cell && cell->type == METADATA_TEXT)
            {
                g_string_assign(author, cell->value.v_text->str );
            }
        }
        else
        {
            ERRORPRINTF("Cannot find column %s", column);
        }
    }

    if (small)
    {
        // get file thumbnail (medium)
        const char* column = THUMBNAIL_COLUMNS[THUMBNAIL_SMALL];
        int index = metadata_table_find_column(results_table, column);
        if (index >= 0)
        {
            const metadata_cell *cell = metadata_table_get_cell(results_table, index);

            if (cell && cell->type != METADATA_BLOB)
            {
                if (cell->type != METADATA_NULL) {
                    ERRORPRINTF("illegal cell type [%d] for thumbnail", cell->type);
                }
                small->data = NULL ;
                small->size = 0;
            }
            else {
                small->size = cell->value.v_blob.len;
                memcpy((guchar*) small->data, (guchar*) cell->value.v_blob.data, small->size);
            }
        }
        else
        {
            ERRORPRINTF("Cannot find column %s", column);
        }
    }

    if (medium)
    {
        // get file thumbnail (medium)
        const char* column = THUMBNAIL_COLUMNS[THUMBNAIL_MEDIUM];
        int index = metadata_table_find_column(results_table, column);
        if (index >= 0)
        {
            const metadata_cell *cell = metadata_table_get_cell(results_table, index);

            if (cell && cell->type != METADATA_BLOB)
            {
                if (cell->type != METADATA_NULL) {
                    ERRORPRINTF("illegal cell type [%d] for thumbnail", cell->type);
                }
                medium->data = NULL ;
                medium->size = 0;
            }
            else {
                medium->size = cell->value.v_blob.len;
                memcpy((guchar*) medium->data, (guchar*) cell->value.v_blob.data, medium->size);
            }
        }
        else
        {
            ERRORPRINTF("Cannot find column %s", column);
        }
    }

    // clean up
    metadata_table_free(name_table);
    metadata_table_free(results_table);

    return TRUE;
}


gboolean save_file_metadata(erMetadb        IN db,
                            const gchar*    IN filepath,
                            const gchar*    IN filename,
                            GString*        IN title,
                            GString*        IN author,
                            thumbType*      IN small,
                            thumbType*      IN medium)
{
    g_assert(db != NULL);
    g_assert(filename != NULL);

    // create table
    metadata_table *value_table = metadata_table_new();
    g_return_val_if_fail( NULL != value_table, FALSE);

    int value_index = -1;

    int rc = ER_FAIL;
    gboolean save_needed = FALSE;

    // NOTE: This could be optimized; right now all information that is present is
    //       always written to the metadata library, even if nothing has changed
    if (title)
    {
        // set title
        gchar* tmp_title =   g_strdup(title->str);
        // get rid of some special characters
        g_strdelimit(tmp_title, "\t\n\r", ' ');
        // trim the heading and tailing spaces
        g_strstrip(tmp_title);

        metadata_table_add_column(value_table, FILE_TITLE);
        value_index++;
        rc = metadata_table_set_text(value_table, value_index, tmp_title);
        if (rc == ER_OK)
        {
            save_needed = TRUE;
        }
        else
        {
            ERRORPRINTF("Cannot write value %s to table.", tmp_title);
        }
        g_free(tmp_title);
    }

    // set author
    if (author)
    {
        gchar * tmp =   g_strdup(author->str);
        // get rid of some special characters
        g_strdelimit(tmp, "\t\n\r", ' ');
        // trim the heading and tailing spaces
        g_strstrip(tmp);

        metadata_table_add_column(value_table, FILE_AUTHOR);
        value_index++;
        rc = metadata_table_set_text(value_table, value_index, tmp);
        if (rc == ER_OK)
        {
            save_needed = TRUE;
        }
        else
        {
            ERRORPRINTF("Cannot write value %s to table.", tmp);
        }

        g_free(tmp);
    }

    if ( small )
    {
        gchar* blob = (gchar*) small->data;
        gsize  blob_size = small->size;

        // add column according to the size
        metadata_table_add_column(value_table, THUMBNAIL_COLUMNS[THUMBNAIL_SMALL]);
        value_index++;
        rc = metadata_table_set_blob(value_table, value_index, blob, blob_size);
        if (rc== ER_OK)
        {
            save_needed = TRUE;
        }
        else
        {
            ERRORPRINTF("Cannot write thumbnail %s to table.", "small");
        }
    }

    if ( medium )
    {
        gchar* blob = (gchar*) medium->data;
        gsize  blob_size = medium->size;

        // add column according to the size
        metadata_table_add_column(value_table, THUMBNAIL_COLUMNS[THUMBNAIL_MEDIUM]);
        value_index++;
        rc = metadata_table_set_blob(value_table, value_index, blob, blob_size);
        if (rc== ER_OK)
        {
            save_needed = TRUE;
        }
        else
        {
            ERRORPRINTF("Cannot write thumbnail %s to table.", "medium");
        }
    }

    // write values to database
    if (save_needed)
    {
        rc = ermetadb_global_change_file(db, filepath, filename, value_table);
    }
    else
    {
        ERRORPRINTF("No metadata saved, options empty?");
    }

    // clean up
    metadata_table_free(value_table);

    if (rc != ER_OK)  { return FALSE; }
    
    return TRUE;
}


gboolean pixbuf_to_blob(GdkPixbuf * pixbuf, gchar ** buffer, gsize * buf_size)
{
    return gdk_pixbuf_save_to_buffer(pixbuf,
            buffer,
            buf_size,
            "png",
            NULL,
            NULL);
}


// Set the pixel at (x, y) with color. Caller must check the input parameters
static void  set_pixel_value(GdkPixbuf * pixbuf, int x, int y, gint color)
{
    gint channel = gdk_pixbuf_get_n_channels(pixbuf);
    gint rowstride = gdk_pixbuf_get_rowstride(pixbuf);
    guchar * pixel = gdk_pixbuf_get_pixels(pixbuf);
    guchar * p = pixel + y * rowstride + x * channel;

    p[0] = (color & 0xFF000000) >> 24;
    p[1] = (color & 0x00FF0000) >> 16;
    p[2] = (color & 0x0000FF00) >> 8;
}


void pixbuf_draw_rectangle(GdkPixbuf* pixbuf, gint xx, gint yy, gint ww, gint hh, gint color)
{
    if (pixbuf == NULL)
    {
        return;
    }

    gint w = gdk_pixbuf_get_width (pixbuf);
    gint h = gdk_pixbuf_get_height (pixbuf);

    gint i, j, x, y;
    
    // draw top, buttom line
    for (i = 0; i < 2; ++i)
    {
        if (i == 0)
        {
            y = yy;
        }
        else
        {
            y = yy + hh - 1;
        }

        x = xx;
        for (j = 0; j < ww; ++j)
        {
            if ((x + j) < w && y < h)
                set_pixel_value(pixbuf, x + j, y, color);
        }
    }

    // draw left, right line
    for (i = 0; i < 2; ++i)
    {
        if (i == 0)
        {
            x = xx;
        }
        else
        {
            x = xx + ww -1;
        }

        y = yy;
        for (j = 0; j < hh; ++j)
        {
            if (x < w && (y + j) < h)
            {
                set_pixel_value(pixbuf, x, y + j, color);
            }
        }
    }
}
