/*
 * File Name: main.c
 */

/*
 * This file is part of hello-world-settings.
 *
 * hello-world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello-world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <glib.h>
#include <gtk/gtk.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>
#include <liberutils/gconf_utils.h>

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"
#include "screen_utils.h"
#include "views_order.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

static const char  *rc_filename = DATADIR "/" PACKAGE_NAME ".rc";

// registry access
#define GCONF_ROOT              "/apps/er/" PACKAGE
#define GCONF_TODO_ENTRY        GCONF_ROOT "/todo_entry"
#define GCONF_TODO_RADIOBUTTON  GCONF_ROOT "/todo_radiobutton"

// screen layout
static const int        WINDOW_TOP_PADDING      =  40;
static const int        WINDOW_BOTTOM_PADDING   =  20;
static const int        WINDOW_H_PADDING        =  20;
static const int        SCREENITEMS_V_SPACING   =  20;
static const int        SCREENITEMS_H_SPACING   =  20;
static const int        BUTTONS_H_SPACING       =  20;
static const int        ENTRY_LABEL_WIDTH       = 200;

#define WINDOW_MIN_WIDTH  700
#define WINDOW_MIN_HEIGHT 825

#if (WINDOW_MIN_WIDTH * WINDOW_MIN_HEIGHT) < (1024 * 1280 / 3)
#error  Window size must be at least 1/3rd of screen size
#error  this forces display mgr to use a 16 colour (flashing) waveform when the window appears
#endif

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget        *g_title         = NULL;  // screen title
static GtkWidget        *g_button_cancel = NULL;  // button "Cancel"
static GtkWidget        *g_button_save   = NULL;  // button "Save"

//============================================================================
// Local Function Definitions
//============================================================================

static void     load_settings           ( void );
static void     main_set_text           ( void );
static void     save_settings           ( void );

// signal handlers
static void     on_cancel_clicked       ( GtkButton *button, gpointer user_data );
//static gboolean on_focus_out            ( GtkWidget *widget, GdkEventFocus *event, gpointer user_data );
static void     on_save_clicked         ( GtkButton *button, gpointer user_data );
static gboolean on_startup_complete     ( gpointer data );

//============================================================================
// Functions Implementation
//============================================================================

// print usage text and quit
static void usage (const char *argv_0)
{
    static const char *usage_text = 
                        "\n"
                        "usage: %s [options]\n"
                        "\n"
                        "options:\n"
                        "    --help\n"
                        "        Print help text and quit\n";

    printf(usage_text, argv_0);

    exit(1);
}

// get command-line options
static void parse_arguments (int argc, char **argv)
{
    int  i;

    // parse options
    for (i = 1 ; i < argc ; i++)
    {
        if (strcmp(argv[i], "--help") == 0)
        {
            usage(argv[0]);
        }
        else
        {
            ERRORPRINTF("ignore unknown option [%s]", argv[i]);
            usage(argv[0]);
        }
    }
}


// set locale (language)
void main_set_locale (const char *locale)
{
    LOGPRINTF("entry: locale [%s]", locale);

    g_setenv("LANG", locale, TRUE);
    setlocale(LC_ALL, "");

    main_set_text();
    menu_set_text();
}


// create main screen layout
static GtkWidget *create_screen_layout (void)
{
    GtkWidget   *background = NULL;  // return value
    GtkWidget   *widget;
    GtkWidget   **p_widget;
    GtkWidget   *scrolled_window;
    GtkBox      *vbox;
    GtkBox      *hbox;

    LOGPRINTF("entry");

    // object hierarchy:
    //     background (alignment)
    //       |
    widget = gtk_alignment_new(0.0, 0.0, 1.0, 1.0);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget),
                               WINDOW_TOP_PADDING,
                               WINDOW_BOTTOM_PADDING,
                               WINDOW_H_PADDING,
                               WINDOW_H_PADDING  );
    gtk_widget_show(widget);
    background = widget;
    //       |
    //       |-- vbox
    //             |
    widget = gtk_vbox_new(FALSE, SCREENITEMS_V_SPACING);
    gtk_container_add(GTK_CONTAINER(background), widget);
    gtk_widget_show(widget);
    vbox = GTK_BOX(widget);
    //             |
    //             |-- g_title
    //             |
    widget = create_screen_title( &g_title );
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    //             |
    //             |-- tree
    //             |
    /* create scrolled window for the widget */
    widget = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(widget), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
    gtk_box_pack_start(vbox, widget, TRUE, TRUE, 0);
    gtk_widget_show(widget);
    scrolled_window = widget;

    widget = create_vieworder_items( );
    gtk_container_add(GTK_CONTAINER(scrolled_window), widget);
    //gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), widget);
    gtk_widget_show(widget);
    //             |
    //             |-- separator
    //             |
    widget = create_separator();
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    //             |
    //             |-- change
    //             |
    widget = create_changeorder_items( );
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    //             |
    //             |-- separator
    //             |
    widget = create_separator();
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    //             |
    //             |-- filler (event box)
    //             |     Note: filler takes no space itself, but puts buttons at double V spacing
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_size_request(widget, -1, -1);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    //             |
    //             |-- hbox
    //                   |-- g_button_cancel
    //                   |-- g_button_save
    //
    widget = gtk_hbox_new(FALSE, BUTTONS_H_SPACING);
    gtk_box_pack_end(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    hbox = GTK_BOX(widget);
    //
    p_widget = &g_button_cancel;
    widget = gtk_button_new_with_label(NULL);
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "clicked", G_CALLBACK(on_cancel_clicked),    NULL);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;
    //
    p_widget = &g_button_save;
    widget = gtk_button_new_with_label(NULL);
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "clicked", G_CALLBACK(on_save_clicked),      NULL);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;

    main_set_text();
    gtk_widget_grab_focus(g_button_cancel);
    
    return background;
}


// set screen texts
static void main_set_text (void)
{
    LOGPRINTF("entry");

    if (g_title)
    {
        gtk_label_set_text(GTK_LABEL(g_title), _("MackxPatch - Settings"));
    }

    if (g_button_cancel)
    {
        gtk_button_set_label(GTK_BUTTON(g_button_cancel), _("Cancel"));
    }

    if (g_button_save)
    {
        gtk_button_set_label(GTK_BUTTON(g_button_save), _("Save & Close"));
    }
}


// terminate application
void main_quit (void)
{
    WARNPRINTF("entry");
    
    // clean up modules
    menu_destroy();
    ergconf_finalize();

    // clean up widgets
    if (g_main_window)
    {
        gtk_widget_destroy(g_main_window);
        g_main_window = NULL;
    }    
    
    // clean up gtk
    if (gtk_main_level() > 0)
    {
        WARNPRINTF("quit main loop");
        gtk_main_quit();
    }
    else
    {
        WARNPRINTF("no main loop to quit, exit directly");
        _exit(0);
    }
}


// terminate signal
static void on_sigterm (int signo)
{
    WARNPRINTF("    -- entry " PACKAGE_NAME "-settings, my pid [%d]", getpid());

    // stop main process, prepare to quit application
    main_quit();

    WARNPRINTF("    -- leave " PACKAGE_NAME "-settings");
}


// report startup completed
static gboolean on_startup_complete (gpointer data)
{
    LOGPRINTF("entry");

    ipc_sys_startup_complete();

    return FALSE;  // don't call me again
}


// button "cancel" clicked
static void on_cancel_clicked (GtkButton *button, gpointer user_data)
{
    LOGPRINTF("entry");

    main_quit();
}


// button "save" clicked
static void on_save_clicked (GtkButton *button, gpointer user_data)
{
    LOGPRINTF("entry");

    save_settings();
    main_quit();
}

#if 0
// main window looses focus
static gboolean on_focus_out (GtkWidget *widget, GdkEventFocus *event, gpointer user_data)
{
    LOGPRINTF("entry");

    main_quit();

    return FALSE;
}
#endif

// main function
int main (int argc, char *argv[])
{
    GtkWidget        *widget;
    GtkWindow        *window;
    struct sigaction on_term;

    // parse command-line arguments
    parse_arguments(argc, argv);

    // catch the SIGTERM signal
    memset(&on_term, 0x00, sizeof(on_term));
    on_term.sa_handler = on_sigterm;
    sigaction(SIGTERM, &on_term, NULL);
#if LOGGING_ON
    sigaction(SIGINT,  &on_term, NULL);
#endif

    // init domain for translations
    textdomain(GETTEXT_PACKAGE);
    
    // init gtk, list the default rc files
    gtk_init(&argc, &argv);
    gchar** files = gtk_rc_get_default_files();
    while( *files )
    {
        WARNPRINTF("gtk_rc_get_default_files [%s]", *files);
        files++;
    }

    // init GConf library
    ergconf_initialize();

    // init modules
    ipc_set_services();
    menu_init();

    // open the RC file associated with this program
    WARNPRINTF("gtk_rc_parse [%s]", rc_filename);
    gtk_rc_parse(rc_filename);

    // load settings from gconf
    load_settings();

    // create the top level window 
    widget = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_size_request(widget, WINDOW_MIN_WIDTH, WINDOW_MIN_HEIGHT);
    window = GTK_WINDOW(widget);
    gtk_window_set_type_hint( window, GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_title(     window, "");
    gtk_window_set_keep_above(window, TRUE);
    gtk_widget_realize(widget);
    gdk_window_set_decorations(widget->window, GDK_DECOR_BORDER);
    //g_signal_connect(window, "focus-out-event", G_CALLBACK (on_focus_out), NULL);
    g_main_window = widget;

    // add screen details
    widget = create_screen_layout();
    gtk_container_add(GTK_CONTAINER(window), widget);

    // make sure everything is visible 
    gtk_widget_show(GTK_WIDGET(window));

    // tell system daemon we are ready to go
    g_idle_add(on_startup_complete, NULL);
    
    // run the main loop
    LOGPRINTF("before gtk_main");
    gtk_main();
    LOGPRINTF("after gtk_main");

    // clean up modules

    return 0;
}


// run error dialog
void run_error_dialog (const gchar *msg)
{
    GtkWidget   *dialog = NULL;

    ERRORPRINTF("entry: msg [%s]", msg);

    dialog = gtk_message_dialog_new( GTK_WINDOW(g_main_window),
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR,
                                     GTK_BUTTONS_OK,
                                     msg );

    gtk_dialog_run( GTK_DIALOG(dialog) );
    gtk_widget_destroy( dialog );
}

// load settings from registry
void load_settings (void)
{
    LOGPRINTF("entry");

    load_vieworder_settings();
    
}


// save settings to registry
// Note: no need to compare with old values, just write to registry,
//       GConf compares with current registry values and writes only when needed
void save_settings (void)
{
    
    LOGPRINTF("entry");

    save_vieworder_settings();
}

