/*
 * File Name: db.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#include "db.h"
#include "ctb_log.h"

// column names
//   index in this table is a mdb_column_idx_t
static  const char* MDB_COLUMN_NAMES[] =
{
        [COL_FILENAME]          = "filename"           ,
        [COL_DIRECTORY_PATH]    = "directory_path"     ,
        [COL_SORT_PRIORITY]     = "sort_priority"      ,
        [COL_IS_DIRECTORY]      = "is_directory"       ,
        [COL_FILETYPE]          = "file_type"          ,
        [COL_FILESIZE]          = "file_size"          ,
        [COL_FILETIME_MODIFIED] = "file_time_modified" ,
        [COL_FILETIME_LASTREAD] = "file_time_lastread" ,
        [COL_FILETIME_ADDED]    = "file_time_added"    ,
        [COL_TITLE]             = "title"              ,
        [COL_AUTHOR]            = "author"             ,
        [COL_THUMB_MINI]        = "thumb_data_mini"    ,
        [COL_THUMB_SMALL]       = "thumb_data_small"   ,
        [COL_THUMB_MEDIUM]      = "thumb_data_medium"  ,
        [COL_THUMB_LARGE]       = "thumb_data_large"   ,
        [N_METADATA_COLUMNS]    = NULL
};

//
// database columns allowed as sorting order
//   index in this table is a ctb_sort_order_t
//   data  in this table is a mdb_column_idx_t, an index in MDB_COLUMN_NAMES[]
static const int SORT_COLUMN_NAME_IDX[ N_CTB_SORT_ORDER ] =
{
        [CTB_SORT_BY_NAME]       = COL_TITLE,
        [CTB_SORT_BY_TYPE]       = COL_FILETYPE,
        [CTB_SORT_BY_SIZE]       = COL_FILESIZE,
        [CTB_SORT_BY_DATE_ADDED] = COL_FILETIME_ADDED,
        [CTB_SORT_BY_DATE_READ]  = COL_FILETIME_LASTREAD,
        [CTB_SORT_BY_AUTHOR]     = COL_AUTHOR,
};


// maximum time to wait for a database transaction (seconds)
static const int        MAX_WAIT_DATABASE = 30;

static erMetadb g_metadb = NULL;
metadata_table *g_query_names = NULL;
gboolean g_query_in_progress = FALSE;


erMetadb get_database()
{
    return g_metadb;
}


int open_global_database(const gchar *directory)
{
    if (g_metadb && strcmp(directory, ermetadb_get_dir(g_metadb)) == 0 )
    {
        return ER_OK;
    }

    close_database();
    g_metadb = ermetadb_global_open(directory, FALSE);
    if (!g_metadb) {
        ERRORPRINTF("error opening global database in at [%s]", directory);
        return ER_FAIL;
    }
    return ER_OK;
}


void close_database()
{
    if (g_metadb)
    {
        ermetadb_close(g_metadb);
        g_metadb = NULL;
    }
}


int db_query_create( int column1, ... )
{
    va_list ap;
    int col;

    if (g_query_in_progress)
    {
        WARNPRINTF("trying to start a new query mid-query");
        if (g_query_names)
            metadata_table_free(g_query_names);
    }

    g_query_in_progress = TRUE;
    g_query_names = metadata_table_new();
    if (!g_query_names)
        return ER_FAIL;

    va_start(ap, column1);
    for (col = column1; col!=-1; col = va_arg(ap, int))
    {
        int rc = metadata_table_add_column(g_query_names, MDB_COLUMN_NAMES[col]);
        if (rc!=ER_OK)
        {
            va_end(ap);
            return rc;
        }
    }
    va_end(ap);
    return ER_OK;
}


static const gchar *get_sort_order_name( int sort_order, gboolean show_filenames )
{
    int sort_column = SORT_COLUMN_NAME_IDX[sort_order];
    if (show_filenames && sort_order == CTB_SORT_BY_NAME) sort_column = COL_FILENAME;
    return MDB_COLUMN_NAMES[sort_column];
}


int db_query_execute(int sort_order,
                     gboolean sort_asc,
                     metadata_table **values,
                     const gchar* tag_filter)   // can be NULL
{
    int rc = ermetadb_global_select_files( g_metadb,
                                           get_sort_order_name(sort_order, FALSE),
                                           sort_asc,
                                           g_query_names,
                                           values,
                                           tag_filter );
    metadata_table_free(g_query_names);
    g_query_in_progress = FALSE;
    return rc;
}


int db_query_execute_recent(metadata_table **values, int limit)
{
    int rc = ermetadb_global_select_recent( g_metadb,
                                            //MH get_sort_order_name(CTB_SORT_BY_DATE_ADDED, FALSE),
                                            get_sort_order_name(CTB_SORT_BY_DATE_READ, FALSE),
                                            limit,
                                            g_query_names,
                                            values);
    metadata_table_free(g_query_names);
    g_query_in_progress = FALSE;
    return rc;
}


int db_query_execute_search_filter(int            sort_order,
                                   gboolean       sort_asc,
                                   metadata_table **values,
                                   const gchar    *search_filter)
{
    int rc = ermetadb_global_select_search( g_metadb,
                                            get_sort_order_name(sort_order, FALSE),
                                            sort_asc,
                                            g_query_names,
                                            values,
                                            search_filter);
    metadata_table_free(g_query_names);
    g_query_in_progress = FALSE;
    return rc;
}


int db_query_execute_path_filter(int sort_order,
                                 gboolean sort_asc,
                                 metadata_table **values,
                                 const gchar* path_filter,
                                 gboolean show_filenames)
{
    int rc = ermetadb_global_select_subdir( g_metadb,
                                            get_sort_order_name(sort_order, show_filenames),
                                            sort_asc,
                                            g_query_names,
                                            values,
                                            path_filter);
    metadata_table_free(g_query_names);
    g_query_in_progress = FALSE;
    return rc;
}


int db_query_get_metadata(const gchar* filename,
                          const gchar* dirpath,
                          metadata_table **values)
{
    int rc = ermetadb_global_get_file( g_metadb,
                                       dirpath,
                                       filename,
                                       g_query_names,
                                       values);
    metadata_table_free(g_query_names);
    g_query_in_progress = FALSE;
    return rc;
}


int db_query_update_lastread(const GString *filename, const GString *directory, int value)
{
    metadata_table *query = metadata_table_new();

    int rc = metadata_table_add_column(query, MDB_COLUMN_NAMES[COL_FILETIME_LASTREAD]);
    if (rc == ER_OK) rc = metadata_table_set_int64(query, 0, value);
    if (rc == ER_OK) rc = ermetadb_global_change_file(g_metadb,
                                                      directory->str,
                                                      filename->str,
                                                      query);
    metadata_table_free(query);
    return rc;
}

