/*
 * File Name: actions.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

#include <stdlib.h>

#include "ctb_log.h"
#include "ctb_actions.h"
#include "filetypes.h"
#include "i18n.h"
#include "ipc.h"
#include "shortcut.h"
#include "fileview.h"

static void set_special_viewmode(const filelist_entry_t *item,
                                 ViewMode mode)
{
    fileview_stop_update_display();
    filemodel_set_viewmode2(mode);
    fileview_dir_down(item->directory_path->str, item->filename->str);
}


static void handle_special_item(const filelist_entry_t *item)
{
    const gchar *filename = item->filename->str;
    if (strcmp(filename, SPECIAL_BOOKS) == 0) {
        set_special_viewmode(item, BOOKS_VIEW);
    } else if (strcmp(filename, SPECIAL_IMAGES) == 0) {
        set_special_viewmode(item, IMAGES_VIEW);
    } else if (strcmp(filename, SPECIAL_NEWS) == 0) {
        set_special_viewmode(item, NEWS_VIEW);
    } else if (strcmp(filename, SPECIAL_DIR) == 0) {
        set_special_viewmode(item, DIR_VIEW);
    } else if (strcmp(filename, SPECIAL_SHORTCUTS) == 0) {
        set_special_viewmode(item, SHORTCUT_VIEW);
    } else if (strcmp(filename, SPECIAL_NOTES) == 0) {
        set_special_viewmode(item, NOTES_VIEW);
    } else if (strcmp(filename, SPECIAL_SETTINGS) == 0) {
        set_special_viewmode(item, SETTINGS_VIEW);
    } else if (strcmp(filename, SPECIAL_HELP) == 0) {
        set_special_viewmode(item, HELP_VIEW);
    } else if (strcmp(filename, SPECIAL_PERSONAL) == 0) {
        set_special_viewmode(item, PERSONAL_VIEW);
    } else if (strcmp(filename, SPECIAL_RECENT) == 0) {
        set_special_viewmode(item, RECENT_VIEW);
    } else if (strcmp(filename, SPECIAL_UPDIR) == 0) {
        fileview_stop_update_display();
        fileview_dir_up();
    } else if (strcmp(filename, SPECIAL_SEARCH) == 0) {
        fileview_show_search_dialog();
    } else {
        ERRORPRINTF("unknown special entry '%s'", filename);
        g_assert(0);
    }
}


// activate: launch URL as specified in link shortcut
static int activate_shortcut_to_web_location ( const filelist_entry_t  *fileinfo,
                                               const char* url)
{
    g_assert(url && url[0]);

    LOGPRINTF( "entry: filename [%s] url [%s]", fileinfo->filename->str, url);
    
    const gchar *filename_display = fileinfo->filename_display->str;
    gchar *ret_message = NULL;
    gint errcode = ipc_sys_open_url(url, filename_display, NULL, &ret_message);
    
    if ( errcode > 0 )
    {
        gchar *error_msg = NULL;
        ERRORPRINTF("cannot open url, url [%s], errcode %d [%s]", url, errcode, ret_message);
        switch (errcode)
        {
        case 1: // url cannot be opened
        default:
            error_msg = g_strdup_printf(
                _("The webpage '%s' cannot be opened."), url);
            show_error_dialog(error_msg);
            break;
        case 4: // browser failed to open url (may be handled above with custom message)
            error_msg = g_strdup_printf(
                _("The webpage '%s' cannot be opened."), url);
            show_error_dialog(error_msg);
            break;
        case 2: // timeout waiting for application window
        case 3: // application has exited before creating a window
            break;
        }
        g_free(error_msg);
    }

    g_free(ret_message);
    return ER_OK;
}


static int activate_shortcut_to_application(const filelist_entry_t *fileinfo, const char* cmdline)
{
    const gchar *filename_display = fileinfo->filename_display->str;
    const gchar *directory        = fileinfo->directory_path->str;

    gchar *ret_message = NULL;
    gint errcode = ipc_sys_start_task(cmdline, directory, filename_display, NULL, &ret_message);
    if (errcode > 0) {
        if (ret_message && ret_message[0] != '\0') {
            ERRORPRINTF("cannot launch viewer, cmd_line [%s], errcode %d [%s]", cmdline, errcode, ret_message);
            show_error_dialog(ret_message);
        } else {
            gchar* error_msg = NULL;
            switch (errcode)
            {
            case 1: // application cannot be started
            default:
                error_msg = g_strdup_printf(
                    _("The '%s' application cannot be opened.\n"
                    "Possibly the application is already open. Close it first and try again."),
                    filename_display );
                show_error_dialog(error_msg);
                break;
            case 4: // application failed to open file (may be handled above with custom message)
                error_msg = g_strdup_printf(_("'%s' cannot be opened."), filename_display);
                show_error_dialog(error_msg);
                break;
            case 2: // timeout waiting for application window
            case 3: // application has exited before creating a window
                break;
            }
            g_free(error_msg);
        }
        return ER_FAIL;
    }
    return ER_OK;
}


static int activate_shortcut (const filelist_entry_t *fileinfo)
{
    shortcut_t *shortcut = NULL;
    int ret = parse_shortcut_file(fileinfo->directory_path->str,
                                  fileinfo->filename->str, &shortcut);
    if (ret == ER_NOT_FOUND) {
        gchar* error_msg = g_strdup_printf(
            _("%s appears to no longer exist.\n"
              "Try connecting and ejecting the device from the computer."),
              fileinfo->filename_display->str);
        show_error_dialog(error_msg);
        g_free(error_msg);
        return ret;
    }
    if (ret != ER_OK) return ret;

    switch (shortcut->type)
    {
        case SHORTCUT_TO_FILE:
        {
            filelist_entry_t *fileinfo_target = filelist_entry_new();
            fileinfo_target->filename         = g_string_new(shortcut->details.file.filename);
            fileinfo_target->filename_display = g_string_new( shortcut->name );
            fileinfo_target->directory_path   = g_string_new(shortcut->details.file.directory);
            fileinfo_target->filetype         = g_string_new( g_extension_pointer(shortcut->details.file.filename));
            fileinfo_target->is_directory     = FALSE;
            ret = activate_item(fileinfo_target);
            filelist_entry_free(fileinfo_target);
            break;
        }
        case SHORTCUT_TO_FOLDER:
        {
            if (shortcut->details.folder.directory == NULL) {
                ret = ER_NOT_FOUND;
                break;
            }
            fileview_stop_update_display();
            char fullname[512];
            sprintf(fullname, "%s/%s", shortcut->details.folder.directory,
                                       shortcut->details.folder.filename);
            fileview_dir_down(fullname, fileinfo->filename->str);
            ret = ER_OK;
            break;
        }
        case SHORTCUT_TO_APPLICATION:
            ret = activate_shortcut_to_application(fileinfo, shortcut->details.application.command_line);
            break;
        
        case SHORTCUT_TO_WEB_LOCATION:
            ret = activate_shortcut_to_web_location(fileinfo, shortcut->details.web.url);
            break;
        default:
            // not implemented
            ret = ER_FAIL;
    }

    shortcut_free(shortcut);
    return ret;
}


// activate: launch viewer, goto directory, ...
int activate_item (const filelist_entry_t *item)
{
    g_assert(item);
    g_assert(item->directory_path   && item->directory_path->str);
    g_assert(item->filename         && item->filename->str        );

    const gchar *filetype = item->filetype->str;
    if (strcmp(filetype, SPECIAL_ITEM_NAME) == 0) {
        handle_special_item(item);
        return ER_OK;
    }

    const gchar *filename  = item->filename->str;
    const gchar *directory = item->directory_path->str;
    if (item->is_directory)
    {
        fileview_stop_update_display();
        char fullname[512];
        sprintf(fullname, "%s/%s", directory, filename);
        fileview_dir_down(fullname, filename);
        return ER_OK;
    }

    if (is_shortcut_file_extension(filetype))
    {
        return activate_shortcut(item);
    }

    // normal file, start viewer
    const gchar *viewer_cmd = get_viewer_from_file_extension(filetype);
    if (viewer_cmd == NULL)
    {
        gchar *error_msg = g_strdup_printf( _("'%s' cannot be opened; the format is unknown."),
                                     filename);
        show_error_dialog(error_msg);
        g_free(error_msg);
        return ER_FAIL;
    }

    gchar *cmd_line = g_strdup_printf("%s \"%s/%s\"", viewer_cmd, directory, filename);
    gchar *ret_message = NULL;
    gint errcode = ipc_sys_start_task(cmd_line, directory,
                                      item->filename_display->str, NULL, &ret_message);
    if (errcode > 0 && ret_message && ret_message[0] != '\0')
    {
        ERRORPRINTF("cannot launch viewer, cmd_line [%s], errcode %d [%s]", cmd_line, errcode, ret_message);
        show_error_dialog(ret_message);
    }
//MH: bug in original code, (sorting on last read date is not working)
//#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
    else if (errcode == 0) filemodel_update_last_read(item);
//#endif

    g_free(ret_message);
    g_free(cmd_line);
    return ER_OK;
}


static void delete_item_from_db(erMetadb db, const filelist_entry_t *item)
{
    g_assert(db);
    if (item->is_directory) {
        ermetadb_global_remove_folder(db, item->directory_path->str, item->filename->str);
    } else {
        ermetadb_global_remove_file(db, item->directory_path->str, item->filename->str);
    }
}


int delete_item(const filelist_entry_t  *item)
{
    g_assert(item);
    g_assert(item->filename          && item->filename->str        );
    g_assert(item->filetype          && item->filetype->str        );
    g_assert(item->directory_path    && item->directory_path->str  );

    const gchar *filename = item->filename->str;
    const gchar *filetype = item->filetype->str;

    LOGPRINTF( "entry: filename [%s] dirpath [%s]", filename, item->directory_path->str);

    // tell sysd to close document, then wait for a reply from sysd
    gchar *filepath = g_strdup_printf("%s/%s", item->directory_path->str, filename);

    const gchar *viewer_cmd = get_viewer_from_file_extension(filetype);
    if (viewer_cmd)
    {
        gchar *cmd_line = g_strdup_printf("%s \"%s\"", viewer_cmd, filepath);
        ipc_sys_stop_task(cmd_line);
        g_free(cmd_line);
    }

    // remove file or directory from FS
    const char *argv[10];
    unsigned int argc = 0;
    argv[argc++] = "rm";
    argv[argc++] = "-fr";
    argv[argc++] = filepath;
    argv[argc++] = NULL;
    g_assert(argc < sizeof(argv)/sizeof(argv[0]));
    gint stat;
    GError *err = NULL;
    gboolean ok = g_spawn_sync( NULL,          // working directory: inherit
                       (char**)argv,           // child's argument vector
                       NULL,                   // environment: inherit
                       G_SPAWN_SEARCH_PATH,    // flags
                       NULL,                   // child_setup: none
                       NULL,                   // child setup data: none
                       NULL,                   // stdout: not interested
                       NULL,                   // stderr: not interested
                       &stat,                  // exit status
                       &err                );  // error return
    g_free(filepath);
    if (!ok)
    {
        ERRORPRINTF("g_spawn_sync error [%s] on delete [%s/%s]",
                err->message, item->directory_path->str, filename);
        g_clear_error(&err);
        return ER_FAIL;
    }

    delete_item_from_db(filemodel_get_database(), item);
    return ER_OK;
}


static gboolean shortcut_allowed(const filelist_entry_t *fileinfo)
{
    // Special entries
    if (strcmp(fileinfo->filetype->str, SPECIAL_ITEM_NAME) == 0) return FALSE;

    // Shortcuts
    if (is_shortcut_file_extension(fileinfo->filetype->str)) return FALSE;

    // New Note entry
    if (strcmp(fileinfo->filename->str, "new") == 0) return FALSE;

    return TRUE;
}


int create_shortcut_item (const filelist_entry_t *item)
{
    g_assert(item);
    g_assert(item->filename          && item->filename->str        );
    g_assert(item->filename_display  && item->filename_display->str);
    g_assert(item->filetype          && item->filetype->str        );
    g_assert(item->directory_path    && item->directory_path->str  );

    if (!shortcut_allowed(item)) {
        gchar *error_msg = g_strdup_printf(_("A shortcut cannot be created for this item"));
        show_error_dialog(error_msg);
        g_free(error_msg);
        return ER_FORBIDDEN;
    }

    const gchar *mountpoint = ipc_get_media();
    char shortcut_dir[256];
    sprintf(shortcut_dir, "%s/%s", mountpoint, DIR_SHORTCUTS);

    // create shortcut dir if it doesn't exist
    g_mkdir_with_parents (shortcut_dir, 644);

    // create shortcut file on disk
    const gchar *target_dir = item->directory_path->str;
    const gchar *target_file = item->filename->str;
    const gchar *target_display = item->filename_display->str;
    GString *shortcut_file = g_string_new("");
    int ret = create_shortcut_file(target_dir, target_file, target_display, shortcut_dir, shortcut_file);
    if (ret != ER_OK) {
        ERRORPRINTF("cannot create shortcut for %s/%s", target_dir, target_file);
        g_string_free(shortcut_file, TRUE);
        return ret;
    }

    // add to metadb
    erMetadb db = filemodel_get_database();
    gint64 now = time(NULL);
    char subtitle[512];
    // NOTE: assumes shortcut is created on mountpoint, skip
    snprintf(subtitle, 512, "%s/%s", target_dir + strlen(mountpoint), target_file);
    subtitle[sizeof(subtitle)-1] = 0;
    char* cp = subtitle;
    if (subtitle[0] == '/') cp++;
    ret = ermetadb_global_add_file(db, shortcut_dir, shortcut_file->str, 0, now, target_display, cp, "");
    if (ret != ER_OK) {
        ERRORPRINTF("cannot add %s/%s to metadb", shortcut_dir, shortcut_file->str);
    }

    g_string_free(shortcut_file, TRUE);
    return ret;
}

