#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <dbus/dbus.h>
#include <libermetadb/ermetadb.h>

#include "dbus.h"
#include "db.h"
#include "log.h"


// Those labels are visible in the reading menu.
#define PAGE_FLOW_LABEL "Page flow :)"
#define FULL_SCREEN_LABEL "Fullscreen :)"


/* Add new menu entries to reading menu.
 *
 * @param [in] c - DBus connection to use
 */
void
add_menu_items(DBusConnection *c) {
  add_item(c, "trickdPageFlow", "uds_navigation", PAGE_FLOW_LABEL, "mode_continuous");
  add_item(c, "trickdFullScreen", "uds_navigation", FULL_SCREEN_LABEL, "mode_full_screen");
};


/* Modify state of both menu entries.
 *
 * @param [in] c - DBus connection to use
 * @param [in] pageflow - state of pageflow menu entry
 * @param [in] fullscreen - state of fullscreen menu entry
 *
 * Values for both states are as follows:
 *   1 - selected (bold)
 *   0 - normal
 *  -1 - disabled (not visible in menu)
 *   anything else - don't toggle that option
 */
void
set_menu_items(DBusConnection *c, const int pageflow, const int fullscreen) {
  // TODO: Surely there's more terse way of saying that.
  switch (pageflow) {
  case 0:
    set_item_state(c, "trickdPageFlow", "uds_navigation", "normal");
    break;
  case 1:
    set_item_state(c, "trickdPageFlow", "uds_navigation", "selected");
    break;
  case -1:
    set_item_state(c, "trickdPageFlow", "uds_navigation", "disabled");
    break;
  default:
    break;
  };

  switch (fullscreen) {
  case 0:
    set_item_state(c, "trickdFullScreen", "uds_navigation", "normal");
    break;
  case 1:
    set_item_state(c, "trickdFullScreen", "uds_navigation", "selected");
    break;
  case -1:
    set_item_state(c, "trickdFullScreen", "uds_navigation", "disabled");
    break;
  default:
    break;
  };
};


/* Handle opening of a new file. This function is called when uds
 * opens a new file.
 *
 * @param [in]  c - DBus connection to use
 * @param [out] old_filename - pointer to string, filename of currently
 *              opened file
 * @param [in]  new_filename - pointer to string, filename of file
 *              being opened
 */
void
opened_file(DBusConnection *c, char **old_filename, char **new_filename) {
  LOG("Current file: %s", *old_filename);
  LOG("Potential new file: %s", *new_filename);

  // Check whether we're opening a PDF file.
  gchar *lowercasename = g_ascii_strdown(*new_filename, -1);
  gchar *suffix = ".pdf";
  if (g_str_has_suffix(lowercasename, suffix)) {
    LOG("Yay, we have new PDF file: %s", *new_filename);

    // If we had an old filename noted, free it.
    if (*old_filename) {
      free(*old_filename);
    };
    // Save new filename. 
    *old_filename = strdup(*new_filename);

    // Read settings for new file, hilight menu options accordingly.
    int pageflow, fullscreen;
    get_db_fields(*new_filename, &pageflow, &fullscreen);
    LOG("Current settings for %s: pf:[%d], fs:[%d]",
	*new_filename, pageflow, fullscreen);
    set_menu_items(c, pageflow, fullscreen);
  } else {
    // Not a PDF file, disable menu entries.
    LOG("opened_file: It's not a PDF file, disabling options.");
    set_menu_items(c, -1, -1);
  };
  // Cleanup.
  g_free(lowercasename);
};


/* Handle menu item selections. This function is called when user
 * clicks on one of the new menu items.
 *
 * @param [in] c - DBus connection to use
 * @param [in] filename - file for which a flag should be flipped
 * @param [in] menuentry - name of clicked menu entry
 * @param [in] old_state - current state of clicked menu entry
 */
void
toggle_state(DBusConnection *c, const char *filename, 
	     const char *name, const char *old_state) {
  // New state of db field (-2 = "Don't change that field")
  int ns = -2;
  // New state of menu item (selected|normal).
  char *new_state;

  // Flip the state.
  if (strcmp("normal", old_state) == 0) {
    new_state = "selected";
    ns = 1;
  } else {
    new_state = "normal";
    ns = 0;
  };
  
  // Prepare new state for both options.
  int pageflow = -2;
  int fullscreen = -2;
  if (strcmp("trickdFullScreen", name) == 0) {
    fullscreen = ns;
  } else if (strcmp("trickdPageFlow", name) == 0) {
    pageflow = ns;
  };
  LOG("Clicked on %s - as a result, I'll ask for: pageflow: [%d], fullscreen: [%d]",
      name, pageflow, fullscreen);

  /* Uds caches information about currently opened file. As a result,
     we need a way of telling uds to sync its cached data to db, and
     then reading that data back. Current (sloppy) way to do it is to
     simulate SD card unmount, by sending sysPrepareUnmount. Once the
     data is flushed, we flip our bits and send sysVolumeMounted. We
     also need to reopen the file, as uds has closed it before fake
     umount */
  prepare_unmount(c);
  set_db_fields(filename, pageflow, fullscreen);
  volume_mounted(c);
  reopen_file(c, filename);
};


/* DBus message handler. Actually, it's hooked as a filter function,
 * but we're not picky.
 *
 * @param [in] c - DBus connection to use
 * @param [in] msg - DBus message to handle
 * @param [in] user_data - user data for the handler, unused
 */
DBusHandlerResult
dbus_handler (DBusConnection *c, DBusMessage *msg, void *user_data)
{
  const char *method;
  DBusError error;
  // This pointer should retain its value between dbus_handler() calls.
  static char *filename;
  // Fake that we're using user_data, so gcc shuts up :)
  (void ) user_data;

  // Get the name of DBus method we've just heard.
  method = dbus_message_get_member(msg);

  // If it was a reply or signal, 'method' will be empty. We ignore
  // those types of messages.
  if (method) {
    LOG("Handling dbus method call: %s", method);


   if (strcmp("openFile", method) == 0) {
     // Handle new file being opened.
     char *new_file;
     dbus_message_get_args(msg, &error, DBUS_TYPE_STRING, &new_file, DBUS_TYPE_INVALID);
     opened_file(c, &filename, &new_file);
    } else if (strcmp("menuItemActivated", method) == 0) {
     // Handle menu entry being clicked
     char *name, *parent, *menu, *state;
     dbus_message_get_args(msg, &error, DBUS_TYPE_STRING, &name,
			   DBUS_TYPE_STRING, &parent,
			   DBUS_TYPE_STRING, &menu,
			   DBUS_TYPE_STRING, &state,
			   DBUS_TYPE_INVALID);
     toggle_state(c, filename, name, state);
    };
  };

  // We've "handled" the message, so tell DBus not to worry.
  // TODO: should we be free()ing msg here?
  return DBUS_HANDLER_RESULT_HANDLED;
};


/* Main program.
 */
int 
main () {
  DBusConnection *connection;

  // Setup dbus.
  setup_dbus(&connection, dbus_handler);

  // Add menu entries.
  add_menu_items(connection);
  set_menu_items(connection, -1, -1);

  // Start processing dbus messages.
  while (dbus_connection_read_write_dispatch(connection, -1));

  // To make gcc happy... :)
  exit(0);
};
