/*
 * File Name: images_pages_cache.h
 */

/*
 * This file is part of uds-plugin-images.
 *
 * uds-plugin-images is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * uds-plugin-images is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#ifndef PAGES_CACHE_H_
#define PAGES_CACHE_H_

#include <map>
#include <string>
#include <glib.h>

#ifdef _WIN32
#include <unordered_map>
#else
#include <tr1/unordered_map>
#endif

#include "image_page.h" 

namespace images
{

#define DEFAULT_MEMORY_LIMIT 16777216 // 16M 

///@brief The pages cache can be used to cache ImagePages.
class PagesCache
{
public:
    PagesCache(void);
    ~PagesCache(void);

    /// @brief Reset the size of pages cache
    bool set_memory_limit(const int bytes);

    /// Remove the old pages to make sure the memory is enough
    /// NOTE: length might be less than 0
    bool make_enough_memory(const int length);

    /// @brief Add a new page
    bool add_page(ImagePage * p);

    /// @brief Get a page
    ImagePage * get_page(const size_t idx);

    // TODO Maybe should add a lock for a page 
    //      instead of using the lock of PagesCache?
    //
    /// @brief Lock the whole PagesCache.
    /// Notes, Don't call these two functions around APIs of PagesCache.
    ///        For inside of these APIs, already use lock() and unlock ()
    ///        to guarantee they are thread-safe.
    void lock(void);
    
    /// @brief Unlock the whole PagesCache.
    /// Notes, See lock().
    void unlock(void);

private:
    /// @brief Clear the pages cache
    void clear(void);

    // remove a page: return true when a page is sucessfully deleted;
    // otherwise there is only one page in cache and it is locked.
    bool remove_page();

private:
    typedef std::tr1::unordered_map<size_t, ImagePage *> Pages;
    typedef Pages::iterator PagesIter;

private:
    // the size limit
    unsigned int total_length;

    // the memory cost of current cached pages
    int used_length;

    // the pages list
    Pages pages;

    // the mutext of the cached pages
    GMutex *cache_mutex;
};

};//namespace common

#endif //PAGES_CACHE_H_


