/*
 * File Name: log.h
 */

/*
 * This file is part of uds-plugin-common.
 *
 * uds-plugin-common is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * uds-plugin-common is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#ifndef PLUGIN_LOG_H
#define PLUGIN_LOG_H

#include <stdio.h>
#include <pthread.h>
#include "config.h"

#ifndef LOGGING_ON
#define LOGGING_ON       0
#endif
#ifndef WARNING_ON
#define WARNING_ON       1
#endif
#ifndef ERROR_ON
#define ERROR_ON         1
#endif
#ifndef TRACE_ON 
#define TRACE_ON         0
#endif

#define LOG_PREFIX  "UDS_" PACKAGE_NAME

#if (LOGGING_ON)
#define LOGPRINTF(x, ...) fprintf(stderr, "(" LOG_PREFIX "_L)[%d]" __FILE__ ":%d,%s() " x "\n", (unsigned int)pthread_self(), __LINE__, __FUNCTION__ , ##__VA_ARGS__)
#else
#define LOGPRINTF(x, ...) do {} while (0)
#endif

#if (WARNING_ON)
#define WARNPRINTF(x, ...) fprintf(stderr, "(" LOG_PREFIX "_W)[%d]" __FILE__ ":%d,%s() " x "\n", (unsigned int)pthread_self(), __LINE__, __FUNCTION__ , ##__VA_ARGS__)
#else
#define WARNPRINTF(x, ...) do {} while (0)
#endif

#if (ERROR_ON)
#define ERRORPRINTF(x, ...) fprintf(stderr, "(" LOG_PREFIX "_E)[%d]" __FILE__ ":%d,%s() " x "\n", (unsigned int)pthread_self(), __LINE__, __FUNCTION__, ##__VA_ARGS__)
#else
#define ERRORPRINTF(x, ...) do {} while (0)
#endif

#if (TRACE_ON)
#ifdef PLAIN_C
#define TRACEFUNCTION() do {} while (0)
#else
namespace {
    class TraceHelper
    {
        public:
            explicit TraceHelper (
                    const char* file,
                    const int line,
                    const char* func
                ) :
                m_file(file), m_line(line), m_func(func), m_threadid((unsigned int)pthread_self())
                { fprintf(stderr, "(" LOG_PREFIX "_F)[%d]%s:%d,%s() entered -->\n", m_threadid, file, line, func); }
                
            virtual ~TraceHelper() { fprintf(stderr, "(" LOG_PREFIX "_F)[%d]%s:%d,%s() exited  <--\n", m_threadid, m_file, m_line, m_func); }
            
        private:
            const char* m_file;
            const int m_line;
            const char* m_func;
            const unsigned int m_threadid;
    };
};

// TODO: [MDB] Change into FUNCPRINTF("") macro to allow tracing of input parameters of functions
#define TRACEFUNCTION() TraceHelper th(__FILE__, __LINE__, __FUNCTION__); 
#endif // PLAIN_C

#else
#define TRACEFUNCTION() do {} while (0)
#endif

#endif // PLUGIN_LOG_H
