/* 
 * Copyright (C) 2006, iRex Technologies B.V.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "Settings.h"
#include <libermanifest/ermanifest.h>
#include <liberipc/eripctoolbar.h>

HistoryItem::HistoryItem()
: zoom(InvalidZoom)
, x(0)
, y(0)
{
}

LinkItem::LinkItem()
: zoom(InvalidZoom)
, page(InvalidPage)
, x(0)
, y(0)
{
}
LinkItem::LinkItem(double zz, int pp, int xx, int yy)
: zoom(zz)
, page(pp)
, x(xx)
, y(yy)
{
}

//////////////////////////////////////////////////////////////////////////
// Settings:
// screenx and screeny (view x and view y)
// page mode: page image position in screen
// continous mode: scroll position. 
CSettings::CSettings(void)
{
    zoom = ZoomPage; rotate = 0; screenX = screenY = 0;
    currentPage = -1; drawAreaWidth = SCREEN_WIDTH;
    drawAreaHeight = CLIENT_AREA; renderDir = RenderNext; 
    mode = ModePage; modeFullscreen = ModeFullscreenOn;
    modeToolbar = ModeToolbarHidden; zbState = iconState_grey;
    lbIndex = 0;

    // Pen defaults 
    penSize = penColor = penLineStyle = penAttributes = -1;

    enableScb = gTrue;
}

CSettings::~CSettings(void)
{
}

void CSettings::setNewZoom(const double z)
{
    record.zoom = zoom;
    zoom = z; 
    record.x = screenX; 
    record.y = screenY;
    zbState = iconState_normal; 
}

void CSettings::setNewLink()
{
    // erase everything after the current link index 
    if (lbIndex < (int)linkrec.size()-1) {
        linkrec.erase( linkrec.begin()+lbIndex+1, linkrec.end() );
    }
    LinkItem link(zoom, currentPage, screenX, screenY);
    linkrec.push_back(link);
    //linkrec.zoom = zoom;
    //linkrec.page = currentPage;
    //linkrec.x = screenX;
    //linkrec.y = screenY;
    lbIndex = linkrec.size()-1;
}

GBool CSettings::linkBack()
{
    if (linkrec.size() == 0)
    {
        return gFalse;
    }

    if (lbIndex == (int)linkrec.size()-1) {
        LinkItem link(zoom, currentPage, screenX, screenY);
        linkrec.push_back(link);
    }
    if (mode == ModePage)
    {
        if ((lbIndex >= 0) && (lbIndex < (int)linkrec.size())) {
            zoom = linkrec[lbIndex].zoom;
            currentPage = linkrec[lbIndex].page;
            screenX = linkrec[lbIndex].x;
            screenY = linkrec[lbIndex].y;
            lbIndex--;
        } else return gFalse;
        //swap(zoom, linkrec.zoom);
        //swap(currentPage, linkrec.page);
        //swap(screenX, linkrec.x);
        //swap(screenY, linkrec.y);
    }
    else 
    {
        if ((lbIndex >= 0) && (lbIndex < (int)linkrec.size())) {
            double ratio = linkrec[lbIndex].zoom / zoom;
            zoom = linkrec[lbIndex].zoom;
            currentPage = linkrec[lbIndex].page;

        //swap(zoom, linkrec.zoom);
        //swap(currentPage, linkrec.page);
            if (rotate == 270) linkrec[lbIndex].x = (int)(screenX * ratio);
            else if (rotate == 0) linkrec[lbIndex].y = (int)(screenY * ratio);
            screenX = linkrec[lbIndex].x;
            screenY = linkrec[lbIndex].y;
            lbIndex--;
            //swap(screenY, linkrec.y);
            //swap(screenX, linkrec.x);
        } else return gFalse;
    }
    return gTrue;
}

GBool CSettings::linkForward()
{
    if ((linkrec.size() == 0) || (lbIndex >= (int)linkrec.size()-2))
    {
        return gFalse;
    }

    if (mode == ModePage)
    {
        if (lbIndex < (int)linkrec.size()-2) {
            lbIndex+=2;
            zoom = linkrec[lbIndex].zoom;
            currentPage = linkrec[lbIndex].page;
            screenX = linkrec[lbIndex].x;
            screenY = linkrec[lbIndex].y;
            lbIndex--;
        } else return gFalse; 
        //swap(zoom, linkrec.zoom);
        //swap(currentPage, linkrec.page);
        //swap(screenX, linkrec.x);
        //swap(screenY, linkrec.y);
    }
    else 
    {
        if (lbIndex < (int)linkrec.size()-2) {
            lbIndex+=2;
            double ratio = linkrec[lbIndex].zoom / zoom;
            zoom = linkrec[lbIndex].zoom;
            currentPage = linkrec[lbIndex].page;

        //swap(zoom, linkrec.zoom);
        //swap(currentPage, linkrec.page);
            if (rotate == 270) linkrec[lbIndex].x = (int)(screenX * ratio);
            else if (rotate == 0) linkrec[lbIndex].y = (int)(screenY * ratio);
            screenX = linkrec[lbIndex].x;
            screenY = linkrec[lbIndex].y;
            lbIndex--;
            //swap(screenY, linkrec.y);
            //swap(screenX, linkrec.x);
        } else return gFalse;
    }
    return gTrue;
}

GBool CSettings::zoomBack()
{
    if (InvalidZoom == record.zoom)
    {
        return gFalse;
    }

    if (mode == ModePage)
    {
        swap(zoom, record.zoom);
        swap(screenX, record.x);
        swap(screenY, record.y);
    }
    else 
    {
        double ratio = record.zoom / zoom;
        swap(zoom, record.zoom);
        if (rotate == 270)
        {
            record.x = (int)(screenX * ratio);
            swap(screenX, record.x);
            swap(screenY, record.y);
        }
        else if (rotate == 0)
        {          
            record.y = (int)(screenY * ratio);
            swap(screenY, record.y);
            swap(screenX, record.x);
        }            
    }
    if (iconState_normal == zbState)
    {
        zbState = iconState_selected;
    }
    else if (iconState_selected == zbState)
    {
        zbState = iconState_normal;
    }
    return gTrue;
}

void CSettings::disableZoomBack()
{
    record.zoom = InvalidZoom;
    zbState = iconState_grey;
}

void CSettings::disableLinkBack()
{
    linkrec.clear();
    lbIndex = 0;
}

GBool CSettings::canZoomBack()
{
    return (InvalidZoom != record.zoom);
}

GBool CSettings::canLinkBack()
{
    return (linkrec.size() > 0);
}

void CSettings::load(const char * pathName)
{
    erManifest manifest;
    if (RET_OK == ermXmlOpenManifest(pathName, &manifest))
    {
        // usually the manifest file is existing
        ermXmlGetInt(&manifest, "/package/last-location/pagenumber", &currentPage);
    
        // zoom
        static const int MAX = 20;
        char tmp[MAX] = {0};
        ermXmlGetString(&manifest, "/package/viewer-settings/zoomfactor", tmp, MAX);
        
        if (0 == strcasecmp(tmp, "zoomPage"))
        {
            zoom = ZoomPage;
        }
        else if (0 == strcasecmp(tmp, "zoomWidth"))
        {
            zoom = ZoomWidth;
        }
        else
        {
            sscanf(tmp, "%lf", &zoom);
        }
        
        // rotation , position and mode
        ermXmlGetInt(&manifest, "/package/viewer-settings/rotation", &rotate);
        ermXmlGetInt(&manifest, "/package/viewer-settings/positionx", &screenX);
        ermXmlGetInt(&manifest, "/package/viewer-settings/positiony", &screenY);
        ermXmlGetInt(&manifest, "/package/viewer-settings/modefullscreen", &modeFullscreen);
        ermXmlGetInt(&manifest, "/package/viewer-settings/modetoolbar", &modeToolbar);
        ermXmlGetString(&manifest, "/package/viewer-settings/mode", tmp, MAX);
        
        if (0 == strcasecmp(tmp, "continous"))
        {
            mode = ModeContinous;
        }
        else if (0 == strcasecmp(tmp, "page"))
        {
            mode = ModePage;
        }

	// Load pen style
	if (RET_OK == ermXmlExist(&manifest, "/package/pen-settings/size")) 
	{
	  ermXmlGetInt(&manifest, "/package/pen-settings/size", &penSize);
	  ermXmlGetInt(&manifest, "/package/pen-settings/color", &penColor);
	  ermXmlGetInt(&manifest, "/package/pen-settings/linestyle", &penLineStyle);
	  ermXmlGetInt(&manifest, "/package/pen-settings/attributes", &penAttributes);
	}

        // check scribble state at first.
        // if the string contains any invalid characters, just disable scribble.
        enableScb = gTrue;
        if (RET_OK == ermXmlGetString(&manifest, "/package/metadata/y-metadata/modify-enable/scribble-enable", tmp, MAX))
        {
            if (strncmp(tmp, "true", MAX) != 0)
            {
                enableScb = gFalse;
            }
        }
        // if children nodes are not available, check parent node.
        else if (RET_OK == ermXmlGetAttributeString(&manifest, 
                        "/package/metadata/y-metadata/modify-enable",
                        "default", tmp, MAX))
        {
            if (strncmp(tmp, "true", MAX) != 0)
            {
                enableScb = gFalse;
            }
        }
    }    
    ermXmlClose(&manifest);    
}

void CSettings::save(const char *pathName)
{
    erManifest manifest;
    if (RET_OK == ermXmlOpenManifest(pathName, &manifest))
    {
        // usually the manifest file is existing
        
        // dump for debug
        // s.dump();
        
        // page
        if (RET_OK != ermXmlExist(&manifest, "/package/last-location/pagenumber"))
        {
            ermXmlNewString(&manifest, "/package", "last-location", "");            
            ermXmlNewString(&manifest, "/package/last-location", "pagenumber", "");            
        }
        ermXmlSetInt(&manifest, "/package/last-location/pagenumber", currentPage);                    

        // zoom 
        if (RET_OK != ermXmlExist(&manifest, "/package/viewer-settings/zoomfactor"))
        {
            ermXmlNewString(&manifest, "/package", "viewer-settings", "");            
            ermXmlNewString(&manifest, "/package/viewer-settings", "zoomfactor", "");            
        }
        if (zoom == ZoomPage)
        {
            ermXmlSetString(&manifest, "/package/viewer-settings/zoomfactor", "zoomPage");                    
        }
        else if (zoom == ZoomWidth)
        {
            ermXmlSetString(&manifest, "/package/viewer-settings/zoomfactor", "zoomWidth");                    
        }            
        else 
        {
            char tmp[20] = {0};
            snprintf(tmp, 20, "%lf", zoom);
            ermXmlSetString(&manifest, "/package/viewer-settings/zoomfactor", tmp);                    
        }
        
        // rotation
        if (RET_OK != ermXmlExist(&manifest, "/package/viewer-settings/rotation"))
        {
            ermXmlNewString(&manifest, "/package/viewer-settings", "rotation", "");            
        }
        ermXmlSetInt(&manifest, "/package/viewer-settings/rotation", rotate);                    

        // position x
        if (RET_OK != ermXmlExist(&manifest, "/package/viewer-settings/positionx"))
        {
            ermXmlNewString(&manifest, "/package/viewer-settings", "positionx", "");            
        }
        ermXmlSetInt(&manifest, "/package/viewer-settings/positionx", screenX);                    

        // position y
        if (RET_OK != ermXmlExist(&manifest, "/package/viewer-settings/positiony"))
        {
            ermXmlNewString(&manifest, "/package/viewer-settings", "positiony", "");            
        }
        ermXmlSetInt(&manifest, "/package/viewer-settings/positiony", screenY);                    

        // mode
        if (RET_OK != ermXmlExist(&manifest, "/package/viewer-settings/mode"))
        {
            ermXmlNewString(&manifest, "/package/viewer-settings", "mode", "");            
        }
        if (mode == ModeContinous)
        {
            ermXmlSetString(&manifest, "/package/viewer-settings/mode", "continous");
        }
        else 
        {
            ermXmlSetString(&manifest, "/package/viewer-settings/mode", "page");
        }

        // fullscreen
        if (RET_OK != ermXmlExist(&manifest, "/package/viewer-settings/modefullscreen"))
        {
            ermXmlNewString(&manifest, "/package/viewer-settings", "modefullscreen", ""); 
        }
        ermXmlSetInt(&manifest, "/package/viewer-settings/modefullscreen", modeFullscreen);      

        if (RET_OK != ermXmlExist(&manifest, "/package/viewer-settings/modetoolbar"))
        {
            ermXmlNewString(&manifest, "/package/viewer-settings", "modetoolbar", "");    
        }
        ermXmlSetInt(&manifest, "/package/viewer-settings/modetoolbar", modeToolbar);         
         
	// Pen settings
        if (RET_OK != ermXmlExist(&manifest, "/package/pen-settings"))
        {
            ermXmlNewString(&manifest, "/package", "pen-settings", "");            
	}

	// Save pen settings
        if (RET_OK != ermXmlExist(&manifest, "/package/pen-settings/size"))
        {
            ermXmlNewString(&manifest, "/package/pen-settings", "size", "");            
        }
        ermXmlSetInt(&manifest, "/package/pen-settings/size", penSize);

        if (RET_OK != ermXmlExist(&manifest, "/package/pen-settings/color"))
        {
            ermXmlNewString(&manifest, "/package/pen-settings", "color", "");            
        }
        ermXmlSetInt(&manifest, "/package/pen-settings/color", penColor);

        if (RET_OK != ermXmlExist(&manifest, "/package/pen-settings/linestyle"))
        {
            ermXmlNewString(&manifest, "/package/pen-settings", "linestyle", "");            
        }
        ermXmlSetInt(&manifest, "/package/pen-settings/linesteyle", penLineStyle);

        if (RET_OK != ermXmlExist(&manifest, "/package/pen-settings/attributes"))
        {
            ermXmlNewString(&manifest, "/package/pen-settings", "attributes", "");            
        }
        ermXmlSetInt(&manifest, "/package/pen-settings/attributes", penAttributes);

	// Save
        ermXmlSaveAndClose(&manifest);
    }     
}

void CSettings::getScreenRect(rectangle & rect)
{
    if (getRotate() == 0 || getRotate() == 180)
    {
        rect.left = screenX; rect.top = screenY;
        rect.right = screenX + drawAreaWidth;
        rect.bottom = screenY + drawAreaHeight;
    }
    else if (getRotate() == 90 || getRotate() == 270)
    {
        rect.left = screenY; rect.top = screenX; 
        rect.right = screenY + drawAreaHeight;
        rect.bottom = screenX + drawAreaWidth;
    }        
}

void CSettings::setScreenPosition(const int nX, const int nY)
{
    screenX = nX; screenY = nY;
}

void CSettings::setScreenSize(const int width, const int height)
{
    drawAreaWidth = width; drawAreaHeight = height;
}

void CSettings::setScreenRect(const rectangle & rect)
{
    screenX = rect.left; screenY = rect.top;
    drawAreaWidth = rect.right - rect.left; 
    drawAreaHeight = rect.bottom - rect.top;
}
