/*
 * File Name: main.c
 */

/*
 * This file is part of popupmenu.
 *
 * popupmenu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * popupmenu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gtk/gtk.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>

// local include files, between " "
#include "log.h"
#include "dialog.h"
#include "i18n.h"
#include "ipc.h"
#include "menustore.h"
#include "statusbar.h"
#include "taskbar.h"


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

static const char  *rc_filename = DATADIR "/" PACKAGE_NAME ".rc";
static guint wait_low_battery_sec = 10;

 
//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static gboolean g_has_stylus    = FALSE;
static gboolean g_has_wifi      = FALSE;
static gboolean g_has_bluetooth = FALSE;
static gboolean g_has_3g        = FALSE;

//============================================================================
// Functions Implementation
//============================================================================


static void on_sigterm(int signo)
{
    WARNPRINTF("    -- entry " PACKAGE_NAME ", my pid [%d]", getpid());

    // stop main process, prepare to quit application
    gtk_main_quit();

    WARNPRINTF("    -- leave " PACKAGE_NAME);
}


static gboolean on_startup_complete (gpointer data)
{
    ipc_send_startup_complete();
    return FALSE; // remove timer source
}


static gboolean on_check_low_battery(gpointer data)
{
    gint level;
    gchar *state = NULL;
    
    if (ipc_get_battery_state(&level, &state, NULL))
    {
        statusbar_update_battery_state(level, state);
        if ((state != NULL) && g_ascii_strcasecmp(state, "low") == 0)
        {
            dialog_splash_show("batterylow");
        }
        g_free(state);
    }
    
    // remove timeout
    return FALSE;
}


int main (int argc, char **argv)
{
    struct sigaction on_term;
    
    // catch the SIGTERM signal
    memset(&on_term, 0x00, sizeof(on_term));
    on_term.sa_handler = on_sigterm;
    sigaction(SIGTERM, &on_term, NULL);
#if LOGGING_ON
    sigaction(SIGINT,  &on_term, NULL);
#endif
 
    // init domain for translations
    textdomain(GETTEXT_PACKAGE);

    // init gtk
    gtk_init(&argc, &argv);
    
    // open the RC file associated with this program
    WARNPRINTF("gtk_rc_parse [%s]", rc_filename);
    gtk_rc_parse(rc_filename);
    
    // start IPC services first so it is ready to receive
    menustore_create(taskbar_select_folder);
    ipc_set_services();

    // init global
    dialog_create();
    
    ipc_get_device_capabilities(&g_has_stylus, &g_has_wifi, &g_has_bluetooth, &g_has_3g);

    statusbar_create(dialog_message_info);

    // set battery status
    gint level;
    gchar *state = NULL;
    if (ipc_get_battery_state(&level, &state, NULL))
    {
        statusbar_update_battery_state(level, state);
        g_free(state);
    }

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    taskbar_create();
#endif

    // report startup has completed
    g_idle_add(on_startup_complete, NULL);

    // wait for applications to be started and check battery again
    g_timeout_add_seconds(wait_low_battery_sec, on_check_low_battery, NULL);
    
    // enter GTK main loop
    WARNPRINTF("-- before gtk_main");
    gtk_main();
    WARNPRINTF("-- after gtk_main");

    return 0;
}
