/*
 * File Name: popup.c
 */

/*
 * This file is part of popupmenu.
 *
 * popupmenu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * popupmenu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>
#include <liberutils/display_utils.h>

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "menustore.h"
#include "pixlist.h"
#include "popup.h"

//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

const gint MENU_OFFSET_LEFT             = 12;
const gint MENU_OFFSET_BOTTOM           = 45;


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget    *g_menu             = NULL;
static gboolean      is_active          = FALSE;
static gboolean      g_dont_show        = FALSE;


//============================================================================
// Local Function Definitions
//============================================================================

#if (TIMING_ON)
#include <stdlib.h>
#include <time.h>
static u_int64_t get_time_now()
{
    struct timespec now;
    clock_gettime(CLOCK_MONOTONIC, &now);
    u_int64_t now64 = now.tv_sec;
    now64 *= 1000000;
    now64 += (now.tv_nsec/1000);
    return now64;
}

static u_int64_t t1 = 0;
static u_int64_t t2 = 0;

static void start_timer()
{
    LOGPRINTF("");
    if (t1 == 0) t1 = get_time_now();
}


static void stop_timer(const char* text)
{
    if (t1 != 0) {
        t2 = get_time_now();
        float duration = t2 - t1;
        printf("%s DURATION = %4.1lf ms\n\n", text, duration / 1000);
        t1 = 0;
    }
}
#endif

static void set_active          (gboolean do_active);
static void idle_update_display (gint type);


//============================================================================
// Functions Implementation
//============================================================================


gboolean popup_set_popup_show(const char* statestr)
{
    gboolean last_active = is_active;

    if (g_dont_show)
    {
        return FALSE;
    }
    
    if (g_ascii_strcasecmp(statestr, "show") == 0)
    {
        set_active(TRUE);
    }
    else if (g_ascii_strcasecmp(statestr, "hide") == 0)
    {
        set_active(FALSE);
    }
    else if (g_ascii_strcasecmp(statestr, "toggle") == 0)
    {
        set_active(!is_active);
    }
    else if (g_ascii_strcasecmp(statestr, "block") == 0)
    {
        // close and block popup
        set_active(FALSE);
        ipc_set_enabled(FALSE);
    }
    else if (g_ascii_strcasecmp(statestr, "unblock") == 0)
    {
        set_active(FALSE);
        ipc_set_enabled(TRUE);
    }
    else
    {
        ERRORPRINTF("state unknown: %s", statestr);
    }

    return (last_active != is_active);
}


void popup_set_popup_block(gboolean block)
{
    g_dont_show = block;
}


gboolean popup_get_popup_block()
{
    return g_dont_show;
}


static void set_menuitem_markup(GtkWidget* widget, const char* text, int state)
{
    LOGPRINTF("text='%s'  state=%d", text, state);
    GtkWidget *itemlabel = gtk_bin_get_child(GTK_BIN(widget));

    // update text
    gchar *markup = NULL;
    if (state == MENU_STATE_SELECTED) {
        markup = g_markup_printf_escaped("<span weight=\"bold\">%s</span>", text);
    }
    else
    {
        markup = g_markup_escape_text(text, -1);
    }
    gtk_label_set_markup(GTK_LABEL(itemlabel), markup);
    g_free(markup);

    // update visibility
    if (state == MENU_STATE_DISABLED || text[0] == 0)
    {
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        if (text[0] != 0) {
            gtk_widget_set_sensitive(widget, FALSE);
            gtk_widget_show(widget);
        } else {
            gtk_widget_hide(widget);

        }
#else
        gtk_widget_hide(widget);
#endif
    }
    else
    {
        gtk_widget_show(widget);
    }
}


//============================================================================
// Local Functions Implementation
//============================================================================

static void on_menu_deactivate(GtkMenuShell *menu, gpointer user_data)
{
    set_active(FALSE);
}


static gboolean on_menu_key_press(GtkMenuShell *menu, GdkEventKey *event, gpointer user_data)
{
    LOGPRINTF("entry: type [%d] keyval [%d]", event->type, event->keyval);
    
    if (event && (event->type == GDK_KEY_PRESS))
    {
        switch (event->keyval)
        {
        case GDK_F1:
            set_active(FALSE);
            break;
        case GDK_F10:
            ipc_sys_standby();
            break;
        default:
            break;
        }
    }
    return FALSE;
}


static void on_item_back_activate(GtkMenuItem *item, gpointer user_data)
{
    GtkWidget *menu_shell = GTK_WIDGET(user_data);

    // close submenu and go back to parent
    g_signal_emit_by_name(menu_shell, "move-current", GTK_MENU_DIR_PARENT, NULL);
}


static void on_item_activate(GtkMenuItem *item, gpointer menu_data)
{
    gboolean ok = menustore_activate_item_iter(menu_data, ipc_send_item_activated);
    if (ok) set_active(FALSE);
}


static void on_menu_move_scroll(GtkMenu      *menu,
                                GtkScrollType arg1,
                                gpointer      user_data)  
{
    display_gain_control();
    idle_update_display(DM_HINT_CURSOR);
}


static gboolean on_menu_move_selected(GtkMenuShell *menu_shell,
                                      gint          distance,
                                      gpointer      user_data)
{
    display_gain_control();
    idle_update_display(DM_HINT_CURSOR);
    return FALSE;
}


static void menu_set_position(GtkMenu *menu,
                              gint *x,
                              gint *y,
                              gboolean *push_in,
                              gpointer user_data)
{
    g_assert(g_menu);
    if (menu == GTK_MENU(g_menu))
    {
        gint w, h;
        GtkRequisition req;

        gdk_window_get_geometry(gdk_get_default_root_window(), NULL, NULL, &w, &h, NULL);
        gtk_widget_size_request(g_menu, &req);
        
        *x = MENU_OFFSET_LEFT; 
        *y = h - MENU_OFFSET_BOTTOM - req.height; 
        //LOGPRINTF("screen w [%d] h [%d], requested w [%d] h [%d], x [%d] y [%d]", w, h, req.width, req.height, *x, *y);
    }
}


static GtkWidget *create_item(const gchar *text, GdkPixbuf *img)
{
    GtkWidget *widget = gtk_image_menu_item_new_with_label(text);
    GtkWidget *item_image = gtk_image_new_from_pixbuf(img);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(widget),item_image);
    gtk_widget_show(item_image);
    gtk_widget_show(widget);
    
    return widget;
}


static gpointer add_popup_item(const char* name,
                           const char* text,
                           enum menu_state state,
                           GdkPixbuf *img,
                           gpointer menu_data,
                           gpointer user_data)
{
    GtkWidget *widget = create_item(text, img);
    set_menuitem_markup(widget, text, state);

    // add item to menu
    gtk_menu_shell_append(GTK_MENU_SHELL(user_data), widget);

    g_signal_connect(G_OBJECT(widget), "activate", G_CALLBACK(on_item_activate), menu_data);
    return widget;
}


static gpointer add_popup_submenu(const char* name,
                                  const char* text,
                                  enum menu_state state,
                                  GdkPixbuf *img,
                                  gpointer menu_data,
                                  gpointer user_data)
{
    GtkWidget* widget = add_popup_item(name, text, state, img, menu_data, user_data);

    // add submenu to item
    GtkWidget *submenu = gtk_menu_new();
    gtk_signal_connect(GTK_OBJECT(submenu), "key-press-event", GTK_SIGNAL_FUNC(on_menu_key_press), NULL);
    g_signal_connect(G_OBJECT(submenu), "move-scroll", G_CALLBACK(on_menu_move_scroll), NULL);
    g_signal_connect(G_OBJECT(submenu), "move-selected", G_CALLBACK(on_menu_move_selected), NULL);
    g_signal_connect(G_OBJECT(submenu), "deactivate", G_CALLBACK(on_menu_deactivate), NULL);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(widget), submenu);
    gtk_widget_show_all(submenu);

    // add Back button
    GtkWidget *back_item = create_item(_("Back"), pixlist_icon_state("back", "normal"));
    g_signal_connect(G_OBJECT(back_item), "activate", G_CALLBACK(on_item_back_activate), submenu);
    gtk_menu_item_set_keep_active(GTK_MENU_ITEM(back_item), TRUE);
    gtk_menu_shell_append(GTK_MENU_SHELL(submenu), back_item);

    return submenu;
}


static void add_popup_separator(gpointer user_data)
{
    GtkWidget* menu = (GtkWidget*)user_data;
    GtkWidget *widget = gtk_separator_menu_item_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), widget);
    gtk_widget_show(widget);
}


static void create_menu()
{
    if (g_menu && !menustore_popup_has_changed()) {
        // use cached menu
        return;
    }    
    menustore_clear_popup_changed();

    if (g_menu)
    {
        // remove existing menu
        gtk_widget_destroy(g_menu);
        g_menu = NULL;
    }

    // create main menu
    GtkWidget *widget = gtk_menu_new();
    gtk_signal_connect(GTK_OBJECT(widget), "key-press-event", GTK_SIGNAL_FUNC(on_menu_key_press), NULL);
    g_signal_connect(G_OBJECT(widget), "deactivate", G_CALLBACK(on_menu_deactivate), NULL);
    g_signal_connect(G_OBJECT(widget), "move-scroll", G_CALLBACK(on_menu_move_scroll), NULL);
    g_signal_connect(G_OBJECT(widget), "move-selected", G_CALLBACK(on_menu_move_selected), NULL);
    g_menu = widget;
    
    menustore_fill_menu(add_popup_item, add_popup_submenu, add_popup_separator, g_menu);

    gtk_widget_show(g_menu);
    // gtk_widget_realize(g_menu);
}


static gboolean update_display(gpointer data)
{
#if (TIMING_ON)
    stop_timer(__func__);
#endif    
    display_update_return_control( (gint) data);
    return FALSE; // don't call again
}


static void idle_update_display(gint type)
{
    g_idle_add(update_display, (gpointer) type);
}


static void set_active(gboolean do_active)
{
    if (do_active == is_active) return;

    if (do_active)
    {
#if (TIMING_ON)
        start_timer();
#endif        
        display_gain_control();
        create_menu();
        gtk_menu_popup(GTK_MENU(g_menu), NULL, NULL, menu_set_position, NULL, 0, 0);
        idle_update_display(DM_HINT_PARTIAL);
    }
    else
    {
        gtk_menu_popdown(GTK_MENU(g_menu));
    }
    
    is_active = do_active;
}

