/*
 * File Name: ipc.c
 */

/*
 * This file is part of popupmenu.
 *
 * popupmenu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * popupmenu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */
 
//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"
 
// system include files, between < >
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>
#include <liberipc/eripc.h>
#include <liberipc/eripc_support.h>

// local include files, between " "
#include "log.h"
#include "dialog.h"
#include "ipc.h"
#include "i18n.h"
#include "menustore.h"
#include "popup.h"
#include "statusbar.h"
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
#include "taskbar.h"
#endif


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

#define CHECK_ARG_STRING(x) \
    if (arg_array[x].type != ERIPC_TYPE_STRING || arg_array[x].value.s == NULL) { \
        ERRORPRINTF("arg %d is not a string", x); \
        return; \
    }

#define CHECK_ARG_INT(x) \
    if (arg_array[x].type != ERIPC_TYPE_INT) { \
        ERRORPRINTF("arg %d is not an integer", x); \
        return; \
    }

//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

 
//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static eripc_client_context_t *eripcClient = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static void add_group_cb            (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void add_item_cb             (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void add_menu_cb             (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void add_task_cb             (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void remove_group_cb         (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void remove_item_cb          (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void remove_menu_cb          (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void remove_task_cb          (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void rename_task_cb          (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_task_to_top_cb      (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_group_state_cb      (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_group_label_cb      (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_item_state_cb       (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_item_label_cb       (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_popup_show_cb       (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void show_menu_cb            (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_busy_show_cb        (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_splash_show_cb      (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_message_show_cb     (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_statusitem_state_cb (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void set_statusitem_show_cb  (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void update_page_counter_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void confirm_usbconnect_cb   (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void confirm_install_drz_cb  (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void confirm_install_update_cb (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void updates_finished_cb (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);

static void cb_sys_battery_state    (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_usb_state        (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_changed_locale   (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_changed_orientation(eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static void cb_sys_volume_mounted   (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_volume_unmounted (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
#endif
static void cb_sys_signal_strength  (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);

static void testing_list_menu_items_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void testing_route_menu_item_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data);

// Exported DBUS API list
static eripc_callback_function_t service_functions[] = {
        // message handlers (method calls to this service)
        { add_group_cb,                     "addGroup",                     NULL,              0 },
        { add_item_cb,                      "addItem",                      NULL,              0 },
        { add_menu_cb,                      "addMenu",                      NULL,              0 },
        { add_task_cb,                      "addTask",                      NULL,              0 },
        { remove_group_cb,                  "removeGroup",                  NULL,              0 },
        { remove_item_cb,                   "removeItem",                   NULL,              0 },
        { remove_menu_cb,                   "removeMenu",                   NULL,              0 },
        { remove_task_cb,                   "removeTask",                   NULL,              0 },
        { rename_task_cb,                   "renameTask",                   NULL,              0 },
        { set_task_to_top_cb,               "setTaskToTop",                 NULL,              0 },
        { set_group_state_cb,               "setGroupState",                NULL,              0 },
        { set_group_label_cb,               "setGroupLabel",                NULL,              0 },
        { set_item_state_cb,                "setItemState",                 NULL,              0 },
        { set_item_label_cb,                "setItemLabel",                 NULL,              0 },
        { set_popup_show_cb,                "setPopupShow",                 NULL,              0 },
        { show_menu_cb,                     "showMenu",                     NULL,              0 },
        { set_busy_show_cb,                 "setBusyShow",                  NULL,              0 },
        { set_splash_show_cb,               "setSplashShow",                NULL,              0 },
        { set_message_show_cb,              "setMessageShow",               NULL,              0 },
        { set_statusitem_state_cb,          "setStatusItemState",           NULL,              0 },
        { set_statusitem_show_cb,           "setStatusItemShow",            NULL,              0 },
        { confirm_usbconnect_cb,            "confirmUsbConnect",            NULL,              0 },
        { update_page_counter_cb,           "updatePageCounter",            NULL,              0 },
        { confirm_install_update_cb,        "confirmInstallUpdate",         NULL,              0 },
        { confirm_install_drz_cb,           "confirmInstallDrz",            NULL,              0 },
        { updates_finished_cb,              "updatesFinished",              NULL,              0 },
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        { cb_sys_volume_mounted,   "        sysVolumeMounted",              NULL,              0 },
#endif
        // signal handlers (broadcasted from given service)
        { cb_sys_battery_state,             "sysBatteryState",              DBUS_SERVICE_SYSTEM_CONTROL, 0 },
        { cb_sys_usb_state,                 "sysUsbState",                  DBUS_SERVICE_SYSTEM_CONTROL, 0 },
        { cb_sys_changed_locale,            "sysChangedLocale",             DBUS_SERVICE_SYSTEM_CONTROL, 0 },
        { cb_sys_changed_orientation,       "sysChangedOrientation",        DBUS_SERVICE_SYSTEM_CONTROL, 0 },
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        { cb_sys_volume_mounted,            "sysVolumeMounted",             DBUS_SERVICE_SYSTEM_CONTROL, 0 },
        { cb_sys_volume_unmounted,          "sysVolumeUnmounted",           DBUS_SERVICE_SYSTEM_CONTROL, 0 },
#endif    
        { cb_sys_signal_strength,           "sysSignalStrength",            DBUS_SERVICE_SYSTEM_CONTROL, 0 },
        // for testing
        { testing_list_menu_items_cb,       "testingListMenuItems",         NULL,                        0 },
        { testing_route_menu_item_cb,       "testingRouteToMenuItem",       NULL,                        0 },
        { NULL, NULL, NULL, 0 }
    };

static gboolean g_enabled = TRUE;

    
//============================================================================
// Functions Implementation
//============================================================================

void ipc_set_services(void)
{
    eripcClient = eripc_client_context_new(
                    DBUS_APPL_NAME,
                    "1.0",
                    DBUS_SERVICE,
                    DBUS_PATH,
                    DBUS_INTERFACE,
                    service_functions);    
    
    // allow two seconds for responses of synchronous (blocking) calls
    eripc_set_timeout(eripcClient->context, 2000);
}


void ipc_unset_services(void)
{
    eripc_client_context_free(eripcClient, service_functions);
}


void ipc_send_task_activated(int xid)
{
    eripc_error_t result = eripc_send_int(eripcClient->context, NULL, NULL,
            ERIPC_BUS_SESSION, DBUS_SERVICE_SYSTEM_CONTROL,
            "activateTask", xid);
    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching the eripc handler (%s)", eripc_error_string(result));
    }
}


void ipc_send_item_activated(const char *iname, const char *pname, const char *mname, const char *state, const char *service)
{
    LOGPRINTF("Send menuItemActivated message to %s: %s, %s %s, state %s", 
              service, iname, pname, mname, state);
    
    eripc_error_t result = eripc_send_varargs(eripcClient->context, 
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                service,
                                "menuItemActivated",
                                ERIPC_TYPE_STRING, iname,
                                ERIPC_TYPE_STRING, pname,
                                ERIPC_TYPE_STRING, mname,
                                ERIPC_TYPE_STRING, state,
                                ERIPC_TYPE_INVALID);
    
    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching the eripc handler (%s)", eripc_error_string(result));
    }
}


// start task (application)
gboolean ipc_sys_start_task ( const gchar  *cmd_line,
                              const gchar  *work_dir,
                              const gchar  *label,
                              const gchar  *thumbnail_path )
{
    gboolean            ok = TRUE;    // return value
    eripc_error_t       result;
    eripc_event_info_t  *reply = NULL;
    const eripc_arg_t   *arg_array = NULL;
    int                 old_timeout = 0;
    gboolean            timeout_set = FALSE;
    
    WARNPRINTF("entry: cmd_line [%s] work_dir [%s] label [%s] thumbnail_path [%s]",
                       cmd_line,     work_dir,     label,     thumbnail_path       );
    g_assert( eripcClient->context );
    g_assert( cmd_line && *cmd_line );
    
    // set ipc timeout to "long"
    eripc_error_t eripc_get_timeout (eripc_context_t *context, int *timeout);
    result = eripc_get_timeout(eripcClient->context, &old_timeout);
    if (result == ERIPC_ERROR_SUCCESS)
    {
        result = eripc_set_timeout(eripcClient->context, 60*1000);
        if (result == ERIPC_ERROR_SUCCESS)
        {
            timeout_set = TRUE;
        }
    }
    
    result = eripc_send_varargs_and_wait( eripcClient->context,
                                          &reply,                // reply structure
                                          ERIPC_BUS_SESSION,
                                          DBUS_SERVICE_SYSTEM_CONTROL,
                                          "startTask",
                                          ERIPC_TYPE_STRING, cmd_line,
                                          ERIPC_TYPE_STRING, work_dir,
                                          ERIPC_TYPE_STRING, label,
                                          ERIPC_TYPE_STRING, thumbnail_path,
                                          ERIPC_TYPE_INVALID );
    
    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs_and_wait returns [%d]", result);
    }
    else if (reply == NULL || reply->args == NULL)
    {
        ERRORPRINTF("sysd returns OK but no reply structure");
    }
    else
    {
        // parse the reply strcuture
        arg_array = reply->args;
        if (arg_array[0].type == ERIPC_TYPE_INT)
        {
            ok = arg_array[0].value.i;
        }
        else
        {
            ERRORPRINTF("unexpected argument: type [%d]", arg_array[0].type);
        }
    }    
    
    // restore ipc timeout
    if (timeout_set)
    {
        eripc_set_timeout(eripcClient->context, old_timeout);
    }
  
    LOGPRINTF("leave: ok [%d]", ok);
    return ok;
}


void ipc_sys_standby()
{
    LOGPRINTF("entry");
    eripc_send_varargs(eripcClient->context,
                       NULL,
                       NULL,
                       ERIPC_BUS_SESSION,
                       DBUS_SERVICE_SYSTEM_CONTROL,
                       "sysStandby", 
                       ERIPC_TYPE_INVALID);
}


void ipc_send_request_popup(const char *state)
{
    LOGPRINTF("entry state [%s]", state);
    eripc_sysd_set_menu_state(eripcClient, state);
}


void ipc_send_status_item_activated(const char *iname, const char *state, const char *service)
{
    eripc_error_t result;

    LOGPRINTF("entry");
    
    result = eripc_send_varargs(eripcClient->context, 
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                service,
                                "statusItemActivated",
                                ERIPC_TYPE_STRING, iname,
                                ERIPC_TYPE_STRING, state,
                                ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent statusItemActivated message to %s: %s, state %s", service, iname, state);
    
    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching the eripc handler (%s)", eripc_error_string(result));
    }
}


void ipc_send_startup_complete(void)
{
    LOGPRINTF("entry");
    eripc_sysd_startup_complete(eripcClient, getpid(), FALSE, 0);
}


gboolean ipc_get_battery_state(gint *level, gchar **state, gint *timeleft)
{
    eripc_error_t retval;
    eripc_event_info_t *info = NULL;
    gboolean result = FALSE;
    
    LOGPRINTF("entry");
    
    retval = eripc_send_varargs_and_wait(eripcClient->context,
                                         &info,
                                         ERIPC_BUS_SESSION,
                                         DBUS_SERVICE_SYSTEM_CONTROL,
                                         "sysGetBatteryState",
                                         ERIPC_TYPE_INVALID);

    if (retval == ERIPC_ERROR_SUCCESS) 
    {
        const eripc_arg_t *arg_array = info->args;
        
        if ((arg_array[0].type == ERIPC_TYPE_INT) && 
            (arg_array[1].type == ERIPC_TYPE_STRING) &&
            (arg_array[2].type == ERIPC_TYPE_INT))
        {
            *level = arg_array[0].value.i;
            *state = g_strdup(arg_array[1].value.s);
            if (timeleft != NULL)
            {
                *timeleft = arg_array[2].value.i;
                LOGPRINTF("Reply received: battery at %d%% %s, time left: %d", *level, *state, *timeleft);
            }
            else
            {
                LOGPRINTF("Reply received: battery at %d%% %s", *level, *state);
            }
            
            result = TRUE;
        }
    }
    else
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
    
    eripc_event_info_free(eripcClient->context, info);
    return result;
}


gchar *ipc_get_orientation(void)
{
    eripc_error_t retval;
    eripc_event_info_t *info = NULL;
    gchar *result = NULL;
    
    LOGPRINTF("entry");
    
    retval = eripc_send_varargs_and_wait(eripcClient->context,
                                         &info,
                                         ERIPC_BUS_SESSION,
                                         DBUS_SERVICE_SYSTEM_CONTROL,
                                         "sysGetOrientation",
                                         ERIPC_TYPE_INVALID);

    if (retval == ERIPC_ERROR_SUCCESS) 
    {
        const eripc_arg_t *arg_array = info->args;
        if (arg_array[0].type == ERIPC_TYPE_STRING)
        {
            result = g_strdup(arg_array[0].value.s);
        }
    }
    
    eripc_event_info_free(eripcClient->context, info);
    return result;
}


void ipc_set_orientation(const gchar *orientation)
{
    LOGPRINTF("entry");
    
    eripc_error_t result = eripc_send_varargs(eripcClient->context,
                                            NULL,
                                            NULL,
                                            ERIPC_BUS_SESSION,
                                            DBUS_SERVICE_SYSTEM_CONTROL,
                                            "sysRotate",
                                            ERIPC_TYPE_STRING, orientation,
                                            ERIPC_TYPE_INVALID);

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching the eripc handler (%s)", eripc_error_string(result));
    }
}


void ipc_send_page_change(const gchar* dir)
{
    LOGPRINTF("entry");
    
    const gchar *service = menustore_get_current_service();
    if (service && (service[0]!='\0'))
    {
        eripc_error_t result = eripc_send_varargs(eripcClient->context,
                                                NULL,
                                                NULL,
                                                ERIPC_BUS_SESSION,
                                                service,
                                                "pageChange",
                                                ERIPC_TYPE_STRING, dir,
                                                ERIPC_TYPE_INVALID);

        if (result != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error launching the eripc handler (%s)", eripc_error_string(result));
        }
    }
    else
    {
        LOGPRINTF("Current IPC service unknown");
    }
}


gboolean ipc_get_device_capabilities(gboolean *has_stylus, gboolean *has_wifi, gboolean *has_bluetooth, gboolean *has_3g) 
{
    eripc_device_caps_t er_dev_caps;
    
    eripc_error_t result = eripc_sysd_get_device_capabilities(eripcClient, &er_dev_caps);

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        return FALSE;
    }
    
    if (has_stylus)
    {
        *has_stylus = er_dev_caps.has_stylus;
    }
    
    if (has_wifi)
    {
        *has_wifi = er_dev_caps.has_wifi;
    }
    
    if (has_bluetooth)
    {
        *has_bluetooth = er_dev_caps.has_bluetooth;
    }
        
    if (has_3g)
    {
        *has_3g = er_dev_caps.has_3g;
    }
        
    return TRUE;    
}


void ipc_send_reply(eripc_context_t *context, const char *message_id, gboolean result)
{
    LOGPRINTF("entry, reply %d", result);
    
    if (message_id)
    {
        LOGPRINTF("Sending reply %s to: %s, context %p", (result == TRUE ? "TRUE":"FALSE"), message_id, context);

        eripc_error_t retval = eripc_reply_bool(context, message_id, result);
        if (retval != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(retval));
        }
    }
    else
    {
        LOGPRINTF("Result is %s but no reply was requested", (result == TRUE ? "TRUE":"FALSE"));
    }
}


//============================================================================
// Local Function Implementation
//============================================================================

static void add_group_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");
    
    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    CHECK_ARG_STRING(2);
    CHECK_ARG_STRING(3);
    
    gboolean result = menustore_add_group(arg_array[0].value.s, 
                                     arg_array[1].value.s, 
                                     arg_array[2].value.s, 
                                     arg_array[3].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void add_item_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    CHECK_ARG_STRING(2);
    CHECK_ARG_STRING(3);
    gboolean result = menustore_add_item(arg_array[0].value.s, 
                                    arg_array[1].value.s, 
                                    arg_array[2].value.s, 
                                    arg_array[3].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void add_menu_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");
    
    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    CHECK_ARG_STRING(2);
    CHECK_ARG_STRING(3);
    CHECK_ARG_STRING(4);
    CHECK_ARG_STRING(5);
    CHECK_ARG_STRING(6);
    gboolean result = menustore_add_menu(arg_array[0].value.s, 
                                    arg_array[1].value.s, 
                                    arg_array[2].value.s, 
                                    arg_array[3].value.s, 
                                    arg_array[4].value.s, 
                                    arg_array[5].value.s, 
                                    arg_array[6].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void add_task_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_INT(0);
    CHECK_ARG_STRING(1);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    int xid = arg_array[0].value.i;
    const char* label =  arg_array[1].value.s;
    taskbar_add_task(xid, label);
#endif
    ipc_send_reply(context, info->message_id, TRUE);
}


static void remove_group_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    gboolean result = menustore_remove_group(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void remove_item_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");
    
    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    gboolean result = menustore_remove_item(arg_array[0].value.s,
                                     arg_array[1].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void remove_menu_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");
    
    CHECK_ARG_STRING(0);
    gboolean result = menustore_remove_menu(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void remove_task_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");
    
    CHECK_ARG_INT(0);   // xid
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    int xid = arg_array[0].value.i;
    taskbar_remove_task(xid);
#endif
    ipc_send_reply(context, info->message_id, TRUE);
}


static void rename_task_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_INT(0);       // xid
    CHECK_ARG_STRING(1);    // label
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    taskbar_rename_task(arg_array[0].value.i, arg_array[1].value.s);
#endif
}


static void set_task_to_top_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");
    
    CHECK_ARG_INT(0);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    int xid = arg_array[0].value.i;
    taskbar_select_task(xid);
#endif
    ipc_send_reply(context, info->message_id, TRUE);
}


static void set_group_state_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    gboolean result = menustore_set_group_state(arg_array[0].value.s, 
                                           arg_array[1].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void set_group_label_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    gboolean result = menustore_set_group_label(arg_array[0].value.s, 
                                           arg_array[1].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void set_item_state_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    CHECK_ARG_STRING(2);
    gboolean result = menustore_set_item_state(arg_array[0].value.s, 
                                          arg_array[1].value.s,
                                          arg_array[2].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void set_item_label_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    CHECK_ARG_STRING(2);
    gboolean result = menustore_set_item_label(arg_array[0].value.s, 
                                          arg_array[1].value.s,
                                          arg_array[2].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void set_popup_show_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    gboolean result = popup_set_popup_show(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void set_busy_show_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    gboolean retval = FALSE;
    static gboolean should_unblock = FALSE;
    LOGPRINTF("entry");
    
    CHECK_ARG_STRING(0);
    if (g_ascii_strcasecmp(arg_array[0].value.s, "show") == 0) 
    {
        gchar *busy_msg = _("Please wait...");

        if ((arg_array[1].type == ERIPC_TYPE_STRING) 
         && (arg_array[1].value.s != NULL) 
         && (arg_array[1].value.s[0] != '\0'))
        {
            busy_msg = arg_array[1].value.s;
        }

        if (!popup_get_popup_block())
        {
            popup_set_popup_block(TRUE);
            should_unblock = TRUE;
        }
        popup_set_popup_show("hide");
        dialog_wait_show(busy_msg);
        retval = TRUE;
    }
    else if (g_ascii_strcasecmp(arg_array[0].value.s, "hide") == 0)
    {
        dialog_wait_close();
        if (should_unblock)
        {
            popup_set_popup_block(FALSE);
            should_unblock = FALSE;
        }
        retval = TRUE;
    }
    else
    {
        ERRORPRINTF("state unknown: %s", arg_array[0].value.s);
    }        
    ipc_send_reply(context, info->message_id, retval);
}


static void set_splash_show_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");
    gboolean retval = FALSE;
    
    CHECK_ARG_STRING(0);
    if (strcmp(arg_array[0].value.s,"hide") == 0)
    {
        dialog_splash_remove();
    }
    else
    {
        retval = dialog_splash_show(arg_array[0].value.s);
    }

    ipc_send_reply(context, info->message_id, retval);
}


static void set_message_show_cb(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    gchar *prompt = arg_array[0].value.s;
    GtkWidget *retval = NULL;
    if (g_ascii_strcasecmp(prompt, "udserror") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_ERROR, NULL, _("An unexpected error has occurred; your documents must be closed."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "indexerror") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_ERROR, NULL, _("Thumbnails for one or more documents could not be generated; thumbnail generation has been aborted."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "browsererror") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_ERROR, NULL, _("An unexpected error has occurred; the web browser has been closed."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "sdfullwarn") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_WARNING, NULL, _("The SD card is almost full. You might want to delete files to free space."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "flightmode") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_WARNING, NULL, 
        _("Airplane mode is on and the device will not connect to networks.\n"
          "Turn airplane mode off by tapping on the icon at the bottom of the screen."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "3glowconnect") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_WARNING, NULL, 
        _("There is not sufficient battery power to access the network.\n"
          "Connect to a power source to charge."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "3glowwarning") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_WARNING, NULL, 
        _("The battery power is running low.\n"
          "Connect to a power source to continue using the network."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "3glowdisconnected") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_WARNING, NULL, 
        _("There is no longer sufficient battery power to continue using the network.\n"
          "The device is disconnected from the network."),
                                     context, info->message_id);
    }
    else if (g_ascii_strncasecmp(prompt, "3gcooloff_", strlen("3gcooloff_")) == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_WARNING, NULL, 
        _("Unable to connect to the network.\n"
          "Please try again in a couple of minutes."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "3gsarlimit") == 0)
    {
        retval = dialog_message_info(GTK_MESSAGE_WARNING, NULL, 
        _("The network connection has been lost.\n"
          "Please try again in a couple of minutes."),
                                     context, info->message_id);
    }
    else if (g_ascii_strcasecmp(prompt, "hide") == 0)
    {
        dialog_message_info_close();
        ipc_send_reply(context, info->message_id, TRUE);
        return;
    }
    else
    {
        WARNPRINTF("unknown dialog request: %s", prompt);
    }

    // delay user response unless an error has occurred but return an error immediately
    if (retval == NULL)
    {
        ipc_send_reply(context, info->message_id, TRUE);
    }
}


static void show_menu_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    menustore_set_current_menu(arg_array[0].value.s);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    statusbar_update_toolbar();
#endif
    statusbar_hide_pagecounter();
    ipc_send_reply(context, info->message_id, TRUE);
}


static void set_statusitem_state_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    gboolean result = statusbar_item_set_state(arg_array[0].value.s,
                                          arg_array[1].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void set_statusitem_show_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    gboolean result = statusbar_item_show(arg_array[0].value.s, arg_array[1].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void update_page_counter_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_INT(0);
    CHECK_ARG_INT(1);

    gint cur_page  = arg_array[0].value.i;
    gint num_pages = arg_array[1].value.i;

    // Boundary check is disabled temporary for reflowable documents, meanings
    // that the navigation arrows are not disabled on the first and last page,
    // because previous or next screens can still be available.
    gboolean boundary_check = TRUE;
    if (arg_array[2].type == ERIPC_TYPE_BOOL)
    {
        boundary_check = arg_array[2].value.b;
    }

    statusbar_update_pagecounter(cur_page, num_pages, boundary_check);
    ipc_send_reply(context, info->message_id, TRUE);
}


static void confirm_usbconnect_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    static gboolean result = FALSE;
    if (g_ascii_strcasecmp(arg_array[0].value.s, "show") == 0) 
    {
        gchar *message = g_strconcat(_("Now Charging...\n"
                                       "\n"
                                       "Are you connecting to a computer in order to manage your device?\n"
                                       "(All open files and applications will be saved and closed when "
                                       "connecting).\n"),
                                    NULL);
        
        result = dialog_message_confirm(NULL, message, context, info->message_id, _("Cancel"), _("Connect"));
        g_free(message);
    }
    else if (g_ascii_strcasecmp(arg_array[0].value.s, "hide") == 0)
    {
        dialog_message_confirm_close();
        result = TRUE;
    }
    else
    {
        ERRORPRINTF("state unknown: %s", arg_array[0].value.s);
    }        

    // delay user response unless an error has occurred but return an error immediately
    if (result == FALSE)
    {
        ipc_send_reply(context, info->message_id, FALSE);
    }
}


static void confirm_install_drz_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    static gboolean result = FALSE;
    
    LOGPRINTF("entry");
    
    if ( (arg_array[0].type == ERIPC_TYPE_STRING) && (arg_array[0].value.s != NULL) )
    {
        if (g_ascii_strcasecmp(arg_array[0].value.s, "show") == 0) 
        {
            gchar *message = g_strconcat(_("New installable file(s) found.\n\n"),
                                         _("Do you wish to install now?"),
                                        NULL);
            
            result = dialog_message_confirm(NULL, message, context, info->message_id, _("Cancel"), _("Install"));
            g_free(message);
        }
        else if (g_ascii_strcasecmp(arg_array[0].value.s, "hide") == 0)
        {
            dialog_message_confirm_close();
            result = TRUE;
        }
        else
        {
            ERRORPRINTF("state unknown: %s", arg_array[0].value.s);
        }        
    }

    // delay user response unless an error has occurred but return an error immediately
    //
    if (result == FALSE)
    {
        ipc_send_reply(context, info->message_id, FALSE);
    }
}


static void confirm_install_update_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    static gboolean result = FALSE;
    if (g_ascii_strcasecmp(arg_array[0].value.s, "show") == 0) 
    {
        gchar *message = g_strconcat(_("A new firmware update has been found. To install it, the device must be restarted.\n\n"),
                                     _("Do you wish to restart the device now?"),
                                    NULL);
        
        result = dialog_message_confirm(NULL, message, context, info->message_id, _("Cancel"), _("Restart"));
        g_free(message);
    }
    else if (g_ascii_strcasecmp(arg_array[0].value.s, "hide") == 0)
    {
        dialog_message_confirm_close();
        result = TRUE;
    }
    else
    {
        ERRORPRINTF("state unknown: %s", arg_array[0].value.s);
    }        

    // delay user response unless an error has occurred but return an error immediately
    //
    if (result == FALSE)
    {
        ipc_send_reply(context, info->message_id, FALSE);
    }
}


static void updates_finished_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    statusbar_update_toolbar();
#endif
    ipc_send_reply(context, info->message_id, TRUE);
}


static void cb_sys_battery_state(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_INT(0);
    CHECK_ARG_STRING(1);
    gint  level  = arg_array[0].value.i;
    gchar *state = arg_array[1].value.s;

    LOGPRINTF("Battery level %d, state %s", level, state);

    statusbar_update_battery_state(level, state);
    ipc_send_reply(context, info->message_id, FALSE);
}


static void cb_sys_signal_strength(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_INT(0);
    CHECK_ARG_STRING(1);
    gint  strength  = arg_array[0].value.i;
    gchar *medium = arg_array[1].value.s;

    LOGPRINTF("Signal strength on %s is %d", medium, strength);

    statusbar_update_signal_strength(medium, strength);
    ipc_send_reply(context, info->message_id, FALSE);
}


static void cb_sys_usb_state(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    gchar *state = arg_array[0].value.s;
    LOGPRINTF("USB state %s", state);

    if (g_ascii_strcasecmp(state, "mounted") == 0) 
    {
        popup_set_popup_block(TRUE);
    }
    else
    {
        popup_set_popup_block(FALSE);
    }
}


static void cb_sys_changed_locale(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);

    const char *locale = arg_array[0].value.s;
    const char *old_locale = g_getenv("LANG");
    if (!old_locale || (strcmp(old_locale, locale) != 0))
    {
        LOGPRINTF("Locale has changed to %s", locale);

        // set locale in environment 
        g_setenv("LANG", locale, TRUE);
        setlocale(LC_ALL, "");

        // update texts
        menustore_set_text();
        statusbar_set_text();
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        taskbar_update_locale();
#endif
    }
}


static void cb_sys_changed_orientation(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);

    const char* orientation = arg_array[0].value.s;
    if (orientation)
    {
        dialog_rotated(orientation);
    }
}


#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static void cb_sys_volume_mounted(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    char *volume = arg_array[0].value.s;
    LOGPRINTF("Volume mounted: %s", volume);

    if ((volume != NULL) && (strcmp(volume, "/media/mmcblk0p1") == 0))
    {
        menustore_set_item_state("eject_card", "system_bottom", "normal");
    }
}


static void cb_sys_volume_unmounted(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const eripc_arg_t *arg_array = info->args;
    LOGPRINTF("entry");

    CHECK_ARG_STRING(0);
    char *volume = arg_array[0].value.s;

    LOGPRINTF("Volume unmounted: %s", volume);
    if ((volume != NULL) && (strcmp(volume, "/media/mmcblk0p1") == 0))
    {
        menustore_set_item_state("eject_card", "system_bottom", "disabled");
    }
}
#endif


static void testing_list_menu_items_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const char* list = "no debug";
#if (LOGGING_ON)
    testing_taskbar_print();
#endif
#if (TESTING_ON)
    list = testing_menustore_print();
#endif
    eripc_reply_string(context, info->message_id, list);
}


static void testing_route_menu_item_cb(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    const char* route = "no debug";
#if (TESTING_ON)
    const eripc_arg_t *arg_array = info->args;
    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    route = testing_menustore_get_route(arg_array[0].value.s, arg_array[1].value.s);
#endif
    eripc_reply_string(context, info->message_id, route);
}


void ipc_set_enabled(gboolean enabled)
{
    g_enabled = enabled;
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    taskbar_enable(enabled);
#endif
}


gboolean ipc_is_enabled(void)
{
    return g_enabled;
}

