/*
 * File Name: statusbar.c
 */

/*
 * This file is part of popupmenu.
 *
 * popupmenu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * popupmenu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gtk/gtk.h>
#include <string.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "pixlist.h"
#include "statusbar.h"
#include "menustore.h"


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

static const int  STATUSBAR_HEIGHT     = 43;
static const int  TOOLBAR_MARGIN_LEFT  = 4;
static const int  TOOLBAR_MARGIN_RIGHT = 6;
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static const int  MENUBUTTON_MARGIN = 30;
#endif


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// for status items
typedef struct  
{
    gchar     *itemname;        // used to determine item
    GtkWidget *item;            // GTK toolitem, for show/hide functionality
    GtkWidget *eventbox;        // GTK eventbox, for click/activate functionality
    GtkWidget *image;           // GTK image, image from pixbuf
    gint       state;           // used to store the current state, index to stateArray
    GArray    *stateArray;      // used for the various images
} statusEntry_t;

typedef struct 
{
    gchar     *statename;
    GtkWidget *image;
} stateEntry_t;

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget   *statusbar        = NULL;
static GtkWidget   *g_menu_button    = NULL;
static GtkWidget   *g_right_spacer   = NULL;
static int         g_status_width = 0;
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static GtkWidget   *g_toolbar_img[MENUSTORE_NUM_TOOLS];
#endif
static GArray      *statusArray      = NULL;
static GtkWidget   *g_pagecounter    = NULL;
static GtkWidget   *g_leftbutton     = NULL;
static GtkWidget   *g_rightbutton    = NULL;
static gint        g_curpage         = 1;
static gint        g_numpages        = 1;
static gboolean    g_pagecounter_shown = FALSE;

static GtkWidget   *g_icon_left_enabled   = NULL;
static GtkWidget   *g_icon_left_disabled  = NULL;
static GtkWidget   *g_icon_right_enabled  = NULL;
static GtkWidget   *g_icon_right_disabled = NULL;
static gboolean    g_has_left             = FALSE;
static gboolean    g_has_right            = FALSE;
static dialog_cb_t dialog_cb = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static stateEntry_t    *get_status_item_state   (statusEntry_t    *pStatusEntry, 
                                                 const char       *statename,
                                                 gint             *entry);

static statusEntry_t   *get_status_item         (const char       *item);

static gboolean         status_pressed_event    (GtkWidget        *widget,
                                                 GdkEventButton   *event,
                                                 gpointer          user_data);

static gboolean         on_popup_button_press   (GtkWidget        *widget,
                                                 GdkEventButton   *event,
                                                 gpointer          data);

static gboolean         on_leftbutton_press     (GtkWidget        *widget,
                                                 GdkEventButton   *event,
                                                 gpointer          data);

static gboolean         on_rightbutton_press    (GtkWidget        *widget,
                                                 GdkEventButton   *event,
                                                 gpointer          data);

static void             handle_status_item      (const char       *item,
                                                 const char       *state);

static void             add_system_status_items (void);

static void             update_pagecounter      (gint curpage, gint numpages);


//============================================================================
// Functions Implementation
//============================================================================

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static gboolean on_toolitem_press(GtkWidget* button, GdkEventButton* event, gpointer data)
{
    if (ipc_is_enabled()) {
        menustore_activate_toolitem((int)data, ipc_send_item_activated);
    }
    return FALSE;
}


static void add_toolbar_item(int index)
{
    if (index > 0) {
        GtkWidget *widget = (GtkWidget*) gtk_separator_tool_item_new();
        gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(widget), FALSE);
        gtk_tool_item_set_expand(GTK_TOOL_ITEM(widget), FALSE); 
        gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(widget), -1);
        gtk_widget_set_size_request(widget, 4, -1);
        gtk_widget_show(widget);
    }
    GtkWidget *toolitem = (GtkWidget*)gtk_tool_item_new();

    GtkWidget *eventbox = (GtkWidget *) gtk_event_box_new();
    gtk_container_add(GTK_CONTAINER(toolitem), eventbox);
    // gtk_widget_set_name(eventbox, "irex-popupmenu-toolitem");
    gtk_widget_set_events(eventbox, GDK_BUTTON_PRESS_MASK);
    g_signal_connect(eventbox, "button-press-event" , G_CALLBACK(on_toolitem_press), (gpointer)index);
    gtk_widget_show(eventbox);

    GtkWidget *image = gtk_image_new();
    gtk_container_add(GTK_CONTAINER(eventbox), image);
    gtk_widget_show(image);
    g_toolbar_img[index] = image;

    if ( index <= 7 ) {
      gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(toolitem), -1);
      gtk_widget_show(toolitem);
    }
}


void statusbar_update_toolbar()
{
    if (!menustore_toolbar_has_changed()) return;

    LOGPRINTF("entry");
    int limit = menustore_get_tool_limit();
    if (limit == MENUSTORE_NUM_TOOLS) {
        gtk_widget_hide(g_right_spacer);
    } else {
        gtk_widget_show(g_right_spacer);
    }
    int i;
    for (i=0; i<MENUSTORE_NUM_TOOLS; i++) {
        if (i < limit) {
            gtk_image_set_from_pixbuf(GTK_IMAGE(g_toolbar_img[i]), menustore_get_tool_icon(i));
            gtk_widget_show(g_toolbar_img[i]);
        } else {
            gtk_widget_hide(g_toolbar_img[i]);
        }
    }
    menustore_clear_toolbar_changed();
}
#endif


static void fix_center_pagebar(int width)
{
    LOGPRINTF("entry");
    int spacing = width / 2;
	// substract size of right hand icons
    spacing -= g_status_width;  
    GtkRequisition requisition;
    gtk_widget_size_request(g_pagecounter, &requisition);
    // substract half size of pagecounter widgets: button, text, button, margin
    spacing -= (32+requisition.width+32+40)/2; 
    if (spacing < 0) spacing = 0;
    gtk_widget_set_size_request(g_right_spacer, spacing, -1);
}


static void on_size_allocate(GtkWidget *widget, GtkAllocation *allocation, gpointer data)
{
    fix_center_pagebar(allocation->width);
}


void statusbar_create(dialog_cb_t cb)
{
    LOGPRINTF("entry");
    dialog_cb = cb;
    
    GdkColor color;
    gdk_color_parse ("#aaaaaa", &color);
    
    // window 
    //   |
    GtkWidget *widget = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_name(widget, "irex-statusbar");
    GtkWindow *window = GTK_WINDOW(widget);
    g_statusbar_window = window;

    // use hint DOCK so it is hidden for fullscreen windows
    gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_DOCK);
    gtk_window_set_decorated(window, FALSE);
    gtk_window_set_accept_focus(window, FALSE);
    gtk_widget_modify_bg(widget, GTK_STATE_NORMAL, &color);

    // resize and move to dock at the botton of the screen
    gtk_window_set_default_size(window, -1, STATUSBAR_HEIGHT);
    int bar_width  = 0;
    int bar_height = 0;
    gtk_window_get_size(window, &bar_width, &bar_height);
    gtk_widget_set_size_request(widget, bar_width, bar_height);
    gtk_window_resize(window, bar_width, bar_height);   
    gtk_window_move(window, 0, gdk_screen_height());
    
    gtk_window_set_resizable(window, FALSE);
    gtk_widget_show(widget);

    //   |
    //   +-- alignment
    //         |
    GtkWidget *alignment = gtk_alignment_new(0, 0.0, 1.0, 1.0); 
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 0, 5, 
                              TOOLBAR_MARGIN_LEFT, TOOLBAR_MARGIN_RIGHT);
    gtk_container_add(GTK_CONTAINER(window), alignment);    
    gtk_widget_show(alignment);

    //         |
    //         +-- gtktoolbar
    //               |
    widget = gtk_toolbar_new();
    gtk_toolbar_set_orientation(GTK_TOOLBAR(widget), GTK_ORIENTATION_HORIZONTAL);
    
    gtk_toolbar_set_tooltips(GTK_TOOLBAR(widget), FALSE);
    gtk_toolbar_set_show_arrow(GTK_TOOLBAR(widget), FALSE);
    gtk_toolbar_set_style(GTK_TOOLBAR(widget), GTK_TOOLBAR_ICONS);
    gtk_container_add(GTK_CONTAINER(alignment), widget);
    gtk_widget_show(widget);
    statusbar = widget;    

    // Add menu button
    //               |
    //               +-- toolitem 
    //               |     |
    widget = (GtkWidget *) gtk_tool_item_new();
    
    //               |     |
    //               |     +-- eventbox
    //               |
    GtkWidget *eventbox = (GtkWidget *) gtk_event_box_new();
    gtk_widget_modify_fg (eventbox, GTK_STATE_NORMAL, &color);
    gtk_container_add(GTK_CONTAINER(widget), eventbox);
    
    //               |           |  
    //               |           +-- icon
    //               |
    GtkWidget *icon = gtk_image_new_from_file(PATH_IMG"statusbar_menu_button.png");
    gtk_container_add(GTK_CONTAINER(eventbox), icon);
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(widget), -1);
    g_signal_connect(G_OBJECT(eventbox), "button-release-event", G_CALLBACK(on_popup_button_press), NULL);
    gtk_widget_show(icon);
    gtk_widget_show(widget);
    gtk_widget_show(eventbox);
    
    //               |
    //               +-- toolitem
    //               |     |
    //               |     +-- button with label
    //               |
    widget = (GtkWidget *) gtk_tool_item_new();
    GtkWidget *button = gtk_button_new_with_label("");
    gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
    gtk_container_add(GTK_CONTAINER(widget), button);
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(widget), -1);
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(on_popup_button_press), NULL);
    gtk_widget_show(button);
    gtk_widget_show(widget);

    g_menu_button = button;

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    widget = (GtkWidget*) gtk_separator_tool_item_new();
    gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(widget), FALSE);
    gtk_tool_item_set_expand(GTK_TOOL_ITEM(widget), FALSE); 
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(widget), -1);
    gtk_widget_set_size_request(widget, MENUBUTTON_MARGIN, -1);
    gtk_widget_show(widget);

    int i;
    for (i=0; i<MENUSTORE_NUM_TOOLS; i++)
    {
        add_toolbar_item(i);
    }
#endif

    // Add left separator
    //               |
    //               +-- separator
    //               |
    widget = (GtkWidget*) gtk_separator_tool_item_new();
    gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(widget), FALSE);
    gtk_tool_item_set_expand(GTK_TOOL_ITEM(widget), TRUE); 
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(widget), -1);
    gtk_widget_show(widget);

    // Add left button
    //               |
    //               +-- tool button
    //               |
    g_icon_left_enabled = gtk_image_new_from_file(PATH_IMG"statusbar_arrow_left.png");
    g_object_ref(G_OBJECT(g_icon_left_enabled));
    gtk_widget_show(g_icon_left_enabled);

    g_icon_left_disabled = gtk_image_new_from_file(PATH_IMG"statusbar_arrow_left_disabled.png");
    g_object_ref(G_OBJECT(g_icon_left_disabled));
    gtk_widget_show(g_icon_left_disabled);

    GtkToolItem *leftbutton = gtk_tool_button_new(g_icon_left_disabled, "");
    gtk_widget_show(GTK_WIDGET(leftbutton));
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), leftbutton, -1);
    g_signal_connect(G_OBJECT(leftbutton), "clicked", G_CALLBACK(on_leftbutton_press), NULL);
    g_leftbutton = GTK_WIDGET(leftbutton);

    // Add page counter
    //               |
    //               +-- toolitem
    //               |     |  
    //               |     +-- label
    //               |
    widget = (GtkWidget *) gtk_tool_item_new();
    GtkWidget* pageLabel = gtk_label_new("");
    gtk_widget_set_name(pageLabel, "irex-page-label");
    gtk_widget_show(pageLabel);
    gtk_container_add(GTK_CONTAINER(widget), pageLabel);
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(widget), -1);
    gtk_widget_show(widget);
    g_pagecounter = pageLabel;

    // Add right button
    //               |
    //               +-- tool button
    //               |
    g_icon_right_enabled = gtk_image_new_from_file(PATH_IMG"statusbar_arrow_right.png");
    g_object_ref(G_OBJECT(g_icon_right_enabled));
    gtk_widget_show(g_icon_right_enabled);
    
    g_icon_right_disabled = gtk_image_new_from_file(PATH_IMG"statusbar_arrow_right_disabled.png");
    g_object_ref(G_OBJECT(g_icon_right_disabled));
    gtk_widget_show(g_icon_right_disabled);

    GtkToolItem *rightbutton = gtk_tool_button_new(g_icon_right_disabled, "");
    gtk_widget_show(GTK_WIDGET(rightbutton));
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), rightbutton, -1);
    g_signal_connect(G_OBJECT(rightbutton), "clicked", G_CALLBACK(on_rightbutton_press), NULL);
    g_rightbutton = GTK_WIDGET(rightbutton);

    // Fill out with blank space
    //               |
    //               +-- separator
    //
    widget = (GtkWidget*) gtk_separator_tool_item_new();
    gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(widget), FALSE);
    gtk_tool_item_set_expand(GTK_TOOL_ITEM(widget), FALSE); 
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(widget), -1);
    gtk_widget_show(widget);
    g_right_spacer = widget;
    g_signal_connect(statusbar, "size-allocate", G_CALLBACK(on_size_allocate), NULL);

    // Add system icons here
    add_system_status_items();

    // Set defaults
    statusbar_set_text();
}


/**---------------------------------------------------------------------------
 *
 * Name :  statusbar_add_item
 *
 * @brief  Add an item to the status bar
 *
 * @param  item          Item name
 * @param  statename     Name of default state
 * @param  is_visible    Set to TRUE to show the item, FALSE to hide
 * @param  is_clickable  Set to TRUE to emit a signal when clicked, FALSE otherwise
 *
 * NOTE:   this function expects an icon image with the name <item>_<statename>.png 
 *         in the global icon directory; when <statename> is "normal", it loads <item>.png 
 *
 * @return Returns TRUE on success, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
static gboolean statusbar_add_item(const char *item, const char *statename, gboolean is_visible, gboolean is_clickable)
{
    LOGPRINTF("entry");
    
    GtkWidget     *icon         = NULL;
    GtkWidget     *toolitem     = NULL;
    GtkWidget     *eventbox     = NULL;
    statusEntry_t *pStatusEntry = NULL;

    g_return_val_if_fail(item != NULL, FALSE);
    
    if (statusArray == NULL)
    {
        // first item, create array
        statusArray = g_array_new(FALSE, TRUE, sizeof(statusEntry_t*));
    }
    else 
    {
        // lookup item (if exist, error)
        guint i = 0;
        for (i=0; i < statusArray->len; i++)
        {
            pStatusEntry = g_array_index(statusArray, statusEntry_t*, i);
            if (strcmp(pStatusEntry->itemname, item) == 0)
            {
                WARNPRINTF("`%s` already defined", item);
                return FALSE;
            }
        }
    }

    //  (toolbar) from statusbar_create
    //    |
    //    +-- toolitem
    //          |
    toolitem = (GtkWidget *) gtk_tool_item_new();
    
    //          |
    //          +-- eventbox
    //                |
    eventbox = (GtkWidget *) gtk_event_box_new();
    gtk_container_add(GTK_CONTAINER(toolitem), eventbox);
    gtk_widget_show(eventbox);
    
    //                |  
    //                +-- icon
    //
    icon = gtk_image_new_from_pixbuf(pixlist_icon_state(item, statename));
    
    // increase reference to avoid it being removed on state change
    g_object_ref(icon); 
    
    gtk_container_add(GTK_CONTAINER(eventbox), icon);
    gtk_widget_show(icon);
    
    if (is_clickable)
    {
        gtk_widget_set_events(eventbox, GDK_BUTTON_PRESS_MASK);
        g_signal_connect(G_OBJECT(eventbox), "button_press_event",
                         G_CALLBACK(status_pressed_event), NULL);
    }

    // add to item to the end of the toolbar
    gtk_toolbar_insert(GTK_TOOLBAR(statusbar), GTK_TOOL_ITEM(toolitem), -1);
    
    if (is_visible)
    {
        gtk_widget_show(toolitem);
    }

    // create, prepare and add state entry
    statusEntry_t  *statusEntry = NULL;
    statusEntry = g_new0(statusEntry_t, 1);
    statusEntry->stateArray = g_array_new(FALSE, TRUE, sizeof(stateEntry_t*));
    
    stateEntry_t *stateEntry = NULL;
    stateEntry = g_new0(stateEntry_t, 1);
    stateEntry->image = icon;
    stateEntry->statename = g_strdup(statename);
    g_array_append_val(statusEntry->stateArray, stateEntry);

    // prepare rest of tool entry and add
    statusEntry->itemname = g_strdup(item);
    statusEntry->image = icon;
    statusEntry->item = toolitem;
    statusEntry->eventbox = eventbox;
    g_array_append_val(statusArray, statusEntry);

    return TRUE;    
}


/**---------------------------------------------------------------------------
 *
 * Name :  statusbar_add_item_state
 *
 * @brief  Add a state to a status bar item
 *
 * @param  item          Item name
 * @param  statename     Name of new state
 *
 * NOTE:   this function expects an icon image with the name <item>_<statename>.png 
 *         in the global icon directory; when <statename> is "normal", it loads <item>.png 
 *
 * @return Returns TRUE on success, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
static gboolean statusbar_add_item_state(const char *item, const char *statename)
{
    LOGPRINTF("entry");
    
    GtkWidget     *icon         = NULL;
    statusEntry_t *pStatusEntry = NULL;
    stateEntry_t  *pStateEntry  = NULL;
    
    pStatusEntry = get_status_item(item);
    if (pStatusEntry == NULL)
    {
        WARNPRINTF("item `%s` not defined", item);
        return FALSE;
    }
    
    pStateEntry = get_status_item_state(pStatusEntry, statename, NULL);
    if (pStateEntry != NULL)
    {
        WARNPRINTF("state `%s.%s` already defined", item, statename);
        return FALSE;
    }
    
    // create image widget
    icon = gtk_image_new_from_pixbuf(pixlist_icon_state(item, statename));
    
    // increase reference to avoid it being removed on state change
    g_object_ref(icon); 
    
    // add icon/state
    stateEntry_t *stateEntry = NULL;
    stateEntry = g_new0(stateEntry_t, 1);
    stateEntry->image = icon;
    stateEntry->statename = g_strdup(statename);

    // store array
    g_array_append_val(pStatusEntry->stateArray, stateEntry);
    
    return TRUE;    
}


gboolean statusbar_item_set_state(const char *item, const char *statename)
{
    TRACE("%s() item=%s  statename=%s\n", __func__, item, statename);

    statusEntry_t *pStatusEntry = get_status_item(item);
    if (pStatusEntry == NULL)
    {
        WARNPRINTF("item `%s` not defined", item);
        return FALSE;
    }
 
    gint state_index;
    stateEntry_t  *pStateEntry = get_status_item_state(pStatusEntry, statename, &state_index);
    if (pStateEntry == NULL)
    {
        WARNPRINTF("state `%s.%s` not defined", item, statename);
        return FALSE;
    }

    // remove old icon
    gtk_container_remove(GTK_CONTAINER(pStatusEntry->eventbox), pStatusEntry->image);
    
    // set new icon
    pStatusEntry->state = state_index;
    pStatusEntry->image = pStateEntry->image;
    gtk_widget_show(pStatusEntry->image);
    gtk_container_add(GTK_CONTAINER(pStatusEntry->eventbox), pStatusEntry->image);
    
    return TRUE;    
}


gboolean statusbar_item_show(const char *item, const char *mode)
{
    TRACE("%s() item=%s  mode=%s\n", __func__, item, mode);

    statusEntry_t *pStatusEntry = get_status_item(item);
    if (pStatusEntry == NULL)
    {
        WARNPRINTF("item `%s` not defined", item);
        return FALSE;
    }
    
    if (g_ascii_strcasecmp(mode, "show") == 0)
    {
        gtk_widget_show(pStatusEntry->item);
    }
    else if (g_ascii_strcasecmp(mode, "hide") == 0)
    {
        gtk_widget_hide(pStatusEntry->item);
    }
    else
    {
        ERRORPRINTF("mode unknown: %s", mode);
        return FALSE;
    }
    
    return TRUE;
}


gboolean statusbar_remove_item(const char *item)
{
    TRACE("%s() item=%s\n", __func__, item);
    
    statusEntry_t *pStatusEntry = get_status_item(item);
    if (pStatusEntry == NULL)
    {
        WARNPRINTF("item `%s` not defined", item);
        return FALSE;
    }

    // free state names
    guint i = 0;
    for (i=0; i < pStatusEntry->stateArray->len; i++)
    {
        g_free(g_array_index(pStatusEntry->stateArray, stateEntry_t*, i)->statename);
    }
    
    // free tool entry
    // icon will be freed automatically as it is toolitem container
    gtk_widget_destroy(pStatusEntry->item);
    g_array_free(pStatusEntry->stateArray, TRUE);
    g_free(pStatusEntry->itemname);
    
    return TRUE;
}

   
void statusbar_update_battery_state(gint level, const char *state)
{
    g_return_if_fail(state!=NULL);

    gchar *item  = "statusbar_battery";

#if MACHINE_IS_DR800SG || MACHINE_IS_DR800S || MACHINE_IS_DR800SW
    const gint four_bars_limit  = 88;
    const gint three_bars_limit = 63;
    const gint two_bars_limit   = 38;
    const gint one_bar_limit    = 13;
#elif MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    const gint four_bars_limit  = 88; //75;
    const gint three_bars_limit = 63; //50;
    const gint two_bars_limit   = 38; //25;
    const gint one_bar_limit    = 13; //10;
#else
#error "Unhandled machine type"
#endif
    
    static const gchar *last_icon = NULL;
    const gchar *icon = NULL;
    
    if (strcmp(state, "charging") == 0)
    {
        // charging, show level
        if (level >= four_bars_limit)
        {
            icon = "100_charging";
        }
        else if (level >= three_bars_limit)
        {
            icon = "75_charging";
        }
        else if (level >= two_bars_limit)
        {
            icon = "50_charging";
        }
        else if (level >= one_bar_limit)
        {
            icon = "25_charging";
        }
        else 
        {
            icon = "charging";
        }
    }
    else if (strcmp(state, "discharging") == 0)
    {
        // discharging, show level
        if (level > four_bars_limit)
        {
            icon = "100";
        }
        else if (level > three_bars_limit)
        {
            icon = "75";
        }
        else if (level > two_bars_limit)
        {
            icon = "50";
        }
        else if (level > one_bar_limit)
        {
            icon = "25";
        }
        else
        {
            icon = "normal";
        }
    }
    else if (strcmp(state, "low") == 0)
    {
        icon = "verylow";
    }
    else if (strcmp(state, "full") == 0)
    {
        icon = "100";
    }
    else
    {
        ERRORPRINTF("unhandled state [%s] level [%d]", state, level);
    }
        
    if (icon && (icon != last_icon))
    {
        // update icon to reflect new state
        statusbar_item_set_state(item, icon);
        last_icon = icon;
    }
}


void statusbar_update_signal_strength(const char *medium, gint strength)
{
    gchar *item = NULL;
    const gchar *icon = NULL;
    if (strcmp(medium,"3g")==0) {
        item  = "statusbar_3g"; 
    } else {
        WARNPRINTF("Unhandled signal strength update on medium %s", medium);
        return;
    }

    if (strength>=90)
        icon = "connected_5b";
    else if (strength>=70)
        icon = "connected_4b";
    else if (strength>=50)
        icon = "connected_3b";
    else if (strength>=30)
        icon = "connected_2b";
    else if (strength>=10)
        icon = "connected_1b";
    else
        icon = "connected";
        
    statusbar_item_set_state(item, icon);
}
static void enable_left(gboolean enabled)
{
    if (g_has_left == enabled) return;
    g_has_left = enabled;
    if (enabled) {
        gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(g_leftbutton), g_icon_left_enabled);
    } else {
        gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(g_leftbutton), g_icon_left_disabled);
    }
}


static void enable_right(gboolean enabled)
{
    if (g_has_right == enabled) return;
    g_has_right = enabled;
    if (enabled) {
        gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(g_rightbutton), g_icon_right_enabled);
    } else {
        gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(g_rightbutton), g_icon_right_disabled);
    }
}


void statusbar_update_pagecounter(gint curpage, gint numpages, gboolean boundary_check)
{
    LOGPRINTF("entry curpage [%d] numpages [%d] boundary_check [%d] has_left [%d] has_right [%d] g_pagecounter_shown [%d]", curpage, numpages, boundary_check, g_has_left, g_has_right, g_pagecounter_shown);

    if ((curpage == 0) && (numpages == 0))
    {
        if (g_pagecounter_shown)
        {
            statusbar_hide_pagecounter();
        }
    }
    else if ((curpage == -1) && (numpages == -1))   // magic value for Counting pages...
    {
        update_pagecounter(curpage, numpages);
        gtk_widget_show(g_pagecounter);

        // Ignore parameter boundary_check.
        // Enable previous-/next-page arrows.
        enable_left(TRUE);
        enable_right(TRUE);
        gtk_widget_show(g_leftbutton);
        gtk_widget_show(g_rightbutton);
    }
    else
    {
        update_pagecounter(curpage, numpages);
        
        if (curpage == 1 && boundary_check)
        {
            enable_left(FALSE);
        } else {
            enable_left(TRUE);
        }
        
        if (curpage == numpages && boundary_check)
        {
            enable_right(FALSE);
        } else {
            enable_right(TRUE);
        }

        if (!g_pagecounter_shown)
        {
            gtk_widget_show(g_pagecounter);
            gtk_widget_show(g_leftbutton);
            gtk_widget_show(g_rightbutton);
            g_pagecounter_shown = TRUE;
        }
    }
        
    // save for possible update on locale change
    g_curpage = curpage;
    g_numpages = numpages;
    
    LOGPRINTF("leave has_left [%d] has_right [%d] g_pagecounter_shown [%d]", g_has_left, g_has_right, g_pagecounter_shown);
}


void statusbar_hide_pagecounter()
{
    LOGPRINTF("entry");

    gtk_widget_hide(g_pagecounter);
    gtk_widget_hide(g_leftbutton);
    gtk_widget_hide(g_rightbutton);
    g_pagecounter_shown = FALSE;
}


void statusbar_set_text(void)
{
    update_pagecounter(g_curpage, g_numpages);
    gtk_button_set_label(GTK_BUTTON(g_menu_button), _("Menu"));
    
    GtkRequisition requisition;
    gtk_widget_size_request(statusbar, &requisition);
    fix_center_pagebar(requisition.width);
}


//============================================================================
// Local Functions Implementation
//============================================================================

static void add_system_status_items()
{
    LOGPRINTF("entry");
    
    const char *item = NULL;
    
    // NOTE: add items left to right
    // notifications go first to so they are leftmost
    
    static gboolean has_stylus    = FALSE;
    static gboolean has_wifi      = FALSE;
    static gboolean has_bluetooth = FALSE;
    static gboolean has_3g        = FALSE;
    ipc_get_device_capabilities(&has_stylus, &has_wifi, &has_bluetooth, &has_3g);

    int width = 0;

    // stylus selection mode - show only when stylus is available
    item = "statusbar_stylus";
    statusbar_add_item(      item, "none", has_stylus, TRUE);
    statusbar_add_item_state(item, "pointer");
    statusbar_add_item_state(item, "eraser");
    statusbar_add_item_state(item, "pan");
    statusbar_add_item_state(item, "pen");
    statusbar_add_item_state(item, "zoom");
    statusbar_add_item_state(item, "lookup");
    width += 50;    // NOTE hardcoded from icon

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    // sensor lock status
    item = "statusbar_lock";
    statusbar_add_item(      item, "normal", TRUE, TRUE);
    statusbar_add_item_state(item, "locked");
    width += 50;    // NOTE hardcoded from icon
#endif

    if (has_3g || has_wifi || has_bluetooth)
    {
        // download manager status
        item = "statusbar_downloadmgr";
        statusbar_add_item(      item, "normal", TRUE, TRUE);
        statusbar_add_item_state(item, "downloading");
        statusbar_add_item_state(item, "notify");
        statusbar_add_item_state(item, "error");
        width += 50;    // NOTE hardcoded from icon
    }

    if (has_3g)
    {
        // 3g connection status
        item = "statusbar_3g";
        statusbar_add_item(      item, "disconnected", TRUE, TRUE);
        statusbar_add_item_state(item, "connected");
        statusbar_add_item_state(item, "flightmode");
        statusbar_add_item_state(item, "disabled");
        statusbar_add_item_state(item, "connected_1b");
        statusbar_add_item_state(item, "connected_2b");
        statusbar_add_item_state(item, "connected_3b");
        statusbar_add_item_state(item, "connected_4b");
        statusbar_add_item_state(item, "connected_5b");
        width += 50;    // NOTE hardcoded from icon
    }

    // battery status
    item = "statusbar_battery";
    statusbar_add_item(      item, "normal", TRUE, TRUE);
    statusbar_add_item_state(item, "100_charging");
    statusbar_add_item_state(item, "75_charging");
    statusbar_add_item_state(item, "50_charging");
    statusbar_add_item_state(item, "25_charging");
    statusbar_add_item_state(item, "100");
    statusbar_add_item_state(item, "75");
    statusbar_add_item_state(item, "50");
    statusbar_add_item_state(item, "25");
    statusbar_add_item_state(item, "charging");
    statusbar_add_item_state(item, "verylow");
    width += 50;    // NOTE hardcoded from icon

    g_status_width = width;
}


static statusEntry_t *get_status_item(const char *item)
{
    if ((item == NULL) || (statusArray == NULL))
    {
        return NULL;
    }
    
    // lookup item
    guint i = 0;
    for (i=0; i < statusArray->len; i++)
    {
        statusEntry_t *pStatusEntry = g_array_index(statusArray, statusEntry_t*, i);
        if (strcmp(pStatusEntry->itemname, item) == 0)
        {
            return pStatusEntry;
        }
    }
    return NULL;
}


static stateEntry_t *get_status_item_state(statusEntry_t *pStatusEntry, const char *statename, gint *entry)
{
    guint i = 0;
    for (i=0; i < pStatusEntry->stateArray->len; i++)
    {
        stateEntry_t *pStateEntry = g_array_index(pStatusEntry->stateArray, stateEntry_t*, i);
        if (strcmp(pStateEntry->statename, statename) == 0)
        {
            if (entry != NULL)
            {
                *entry = i;
            }
            return pStateEntry;
        }
    }
    return NULL;
}


static gboolean status_pressed_event(GtkWidget      *widget,
                                     GdkEventButton *event,
                                     gpointer        user_data)
{
    LOGPRINTF("entry");

    if (!ipc_is_enabled()) return FALSE;

    statusEntry_t *pStatusEntry = NULL;
    
    if (statusArray != NULL)
    {
        guint i;
        for (i=0; i < statusArray->len; i++)
        {
            pStatusEntry = g_array_index(statusArray, statusEntry_t*, i);
             
            if (pStatusEntry->eventbox == widget)
            {
                stateEntry_t *pStateEntry = NULL;
                pStateEntry = g_array_index(pStatusEntry->stateArray, stateEntry_t*, pStatusEntry->state);
                LOGPRINTF("pressed %s, state %d (%s)", pStatusEntry->itemname, pStatusEntry->state, pStateEntry->statename);
                
                handle_status_item(pStatusEntry->itemname, pStateEntry->statename);
                  
                // don't propagate 
                return TRUE;
            }
        }
    }

    // propagate further
    return FALSE;
}


static gboolean on_popup_button_press(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    LOGPRINTF("entry");
    if (ipc_is_enabled()) ipc_send_request_popup("toggle");
    LOGPRINTF("leave");
    // event handled
    return TRUE;
}


static gboolean on_leftbutton_press(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    LOGPRINTF("entry");
    
    if (ipc_is_enabled() && g_has_left)
    {
        ipc_send_page_change("prev");
    }
    // event handled
    return TRUE;
}


static gboolean on_rightbutton_press(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    LOGPRINTF("entry");
    
    if (ipc_is_enabled() && g_has_right)
    {
        ipc_send_page_change("next");
    }
    // event handled
    return TRUE;
}


static void update_pagecounter(int curpage, int numpages)
{
    if (curpage == -1 && numpages == -1) {
        // magic numbers for counting pages
        gtk_label_set_text(GTK_LABEL(g_pagecounter), _("Counting pages..."));
    } else {
        /* TRANSLATORS: %1$d is replaced by the current page
         *              %2$d is replaced by the total number of pages
         *              Example: 'page 9 of 40' */
        gchar *cp = g_strdup_printf( _("page %1$d of %2$d"), curpage, numpages);
        gtk_label_set_text(GTK_LABEL(g_pagecounter), cp);
        g_free(cp);
    }
}


static void handle_status_item(const char *item, const char *state)
{
    LOGPRINTF("entry: item [%s] state [%s]", item, state);

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    if (strcmp(item, "statusbar_lock") == 0)
    {
        // toggle sensor lock
        //
        if (strcmp(state, "normal") == 0)
        {
          ipc_send_item_activated("lock", "general", menustore_get_current_menu(), "normal", DBUS_SERVICE_SYSTEM_CONTROL);
        }
        else if (strcmp(state, "locked") == 0)
        {
          ipc_send_item_activated("lock", "general", menustore_get_current_menu(), "selected", DBUS_SERVICE_SYSTEM_CONTROL);
        }
        else
        {
            LOGPRINTF("unknown state `%s` for %s", state, item);
        }
    } else
#endif
    if (strcmp(item, "statusbar_stylus") == 0)
    {
        // send statusItemActivated message to application
        //
        const gchar *service = menustore_get_current_service();
        if (service && (service[0]!='\0'))
        {
            ipc_send_status_item_activated(item, state, service);
        }
    }        
    else if ( (strcmp(item, "statusbar_downloadmgr") == 0)
           || (strcmp(item, "statusbar_3g") == 0) )
    {
        // send statusItemActivated message to System Daemon
        //
        ipc_send_status_item_activated(item, state, DBUS_SERVICE_SYSTEM_CONTROL);
    }
    else if (strcmp(item, "statusbar_battery") == 0)
    {
        // get battery info and show dialog
        //
        gchar *charge_state = NULL;
        gchar *message      = NULL;
        gint  batt_level    = 0;
        
        if (ipc_get_battery_state(&batt_level, &charge_state, NULL))
        {
            if ((strcmp(charge_state, "charging")==0) || (strcmp(charge_state, "discharging")==0))
            {
                if (batt_level == 100)
                {
                    message = g_strdup_printf(_("Fully charged (100%%%%)"));
                }
                else
                {
                    message = g_strdup_printf(_("%d%%%% available"), batt_level);
                }
            }
            else if (strcmp(charge_state, "full")==0)
            {
                message = g_strdup_printf(_("Fully charged (100%%%%)"));
            }
            else if (strcmp(charge_state, "low")==0)
            {
                message = g_strdup_printf(_("Almost empty, %d%%%% remaining"), batt_level);
            }
            
            if (message)
            {
                dialog_cb(GTK_MESSAGE_INFO, _("Battery"), message, NULL, NULL);
            }
            
            g_free(charge_state);
            g_free(message);
        }
    }
}

