# Test iTunes control under Windows

import os, traceback
import win32com.client
from calibre.ebooks.BeautifulSoup import BeautifulSoup

class iTunes(object):

    Sources = [
                'Unknown',
                'Library',
                'iPod',
                'AudioCD',
                'MP3CD',
                'Device',
                'RadioTuner',
                'SharedLibrary']

    PlayerState = [
                'Stopped',
                'Playing',
                'FastForward',
                'Rewind']

    PlaylistKind = [
                'Unknown',
                'Library',
                'User',
                'CD',
                'Device',
                'Radio Tuner'
                ]

    PlaylistSpecialKind = [
                'Unknown',
                'Purchased Music',
                'Party Shuffle',
                'Podcasts',
                'Folder',
                'Video',
                'Music',
                'Movies',
                'TV Shows',
                'Books',
                ]

    TrackKind = [
                'Unknown',
                'File track',
                'CD track',
                'URL track',
                'Device track',
                'Shared library track'
                ]

    SearchField = [
                'FieldAll',
                'FieldVisible',
                'FieldArtists',
                'FieldAlbums',
                'FieldComposers',
                'FieldSongNames'
                ]

    ArtworkFormat = [
                'Unknown',
                'JPEG',
                'PNG',
                'BMP'
                ]

    lib_location = None
    name = None
    version = None
    def __init__(self):
        self.iTunes = it = win32com.client.Dispatch("iTunes.Application")
        self.iTunes_media = self._discover_iTunes_media()
        self.name = it.Windows[0].name
        self.sources = self.get_sources()
        self.version = it.Version

    def get_sources(self, dump=False):
        names = [s.name for s in self.iTunes.sources]
        kinds = [self.Sources[s.kind] for s in self.iTunes.sources]
        sources = dict(zip(kinds,names))
        if dump:
            print "\nget_sources():"
            for source in sources.keys():
                print " kind: %-10.10s '%s'" % (source,sources[source])
            print
        return sources

    def find_device_book(self,title,author):
        print "\nfind_dev_books():"
        if not 'iPod' in self.sources:
            print " no connected device"
            return
        connected_device = self.sources['iPod']
        dev = self.iTunes.sources.ItemByName(connected_device)
        dev_playlists = [pl.Name for pl in dev.Playlists]

        if False:
            # Brute force
            if 'Books' in dev_playlists:
                dev_books = self.iTunes.sources.ItemByName(connected_device).Playlists.ItemByName('Books').Tracks
                print " looking for '%s' by %s" % (title, author)
                for book in dev_books:
                    print " evaluating '%s' by %s" % (book.Name, book.Artist)
                    if book.Name == title and book.Artist == author:
                        print " found it using brute force"
        else:
            if 'Books' in dev_playlists:
                dev_books_pl = self.iTunes.sources.ItemByName(connected_device).Playlists.ItemByName('Books')
                hits = dev_books_pl.Search( title, self.SearchField.index('FieldAlbums'))
                if hits:
                    for hit in hits:
                        if hit.Artist == author:
                            print " hit: '%s' by %s (kind: %s)" % (hit.Name, hit.Artist, self.TrackKind[hit.Kind])
                            break
                    else:
                        print " could not find book"

    def find_library_book(self,title,author):
        print "\nfind_lib_books():"
        lib = self.iTunes.sources.ItemByName('Library')
        lib_playlists = [pl.Name for pl in lib.Playlists]

        if 'Books' in lib_playlists:
            lib_books_pl = self.iTunes.sources.ItemByName('Library').Playlists.ItemByName('Books')
            hits = lib_books_pl.Search( title, self.SearchField.index('FieldAlbums'))
            if hits:
                for hit in hits:
                    if hit.Artist == author:
                        #print " '%s' by %s at %s" % (hit.Name, hit.Artist, hit.Location)
                        return hit
                else:
                    print " could not find book"
        else:
            print " no Library|Books in iTunes"

    def show_artwork(self,title,author):
        print "\nshow_artwork():"
        lib = self.iTunes.sources.ItemByName('Library')
        lib_playlists = [pl.Name for pl in lib.Playlists]

        if 'Books' in lib_playlists:
            lib_books_pl = self.iTunes.sources.ItemByName('Library').Playlists.ItemByName('Books')
            hits = lib_books_pl.Search( title, self.SearchField.index('FieldAlbums'))
            if hits:
                for hit in hits:
                    if hit.Artist == author:
                        print " '%s' by %s at %s" % (hit.Name, hit.Artist, hit.Location)
                        break
                else:
                    print " could not find book"
        else:
            print " no Library|Books in iTunes"

        print "len: %d" % hit.Artwork.Count
        print "%s" % self.ArtworkFormat[hit.Artwork.Item(1).Format]

    def show_dev_books(self):
        print "\nshow_dev_books()"

        if not 'iPod' in self.sources:
            print " no connected device"
            return
        connected_device = self.sources['iPod']
        device = self.iTunes.sources.ItemByName(connected_device)

        dev_books = None
        for pl in device.Playlists:
            #print "pl.Name: %s pl.Kind: %d" % (pl.Name, pl.Kind)
            #print "pl.Name: %s pl.Kind: %s" % (pl.Name, self.PlaylistKind[pl.Kind])
            if self.PlaylistKind[pl.Kind] == 'User' and self.PlaylistSpecialKind[pl.SpecialKind] == 'Books':
                print " Device|Books playlist: '%s' SpecialKind: '%s'" % (pl.Name, self.PlaylistSpecialKind[pl.SpecialKind])
                dev_books = pl.Tracks
                break
        else:
            print " no Books playlist found"

        for book in dev_books:
            if book.KindAsString in ['MPEG audio file']:
                print " ignoring '%s' of type %s" % (book.Name, book.KindAsString)
            else:
                print " %-30.30s %-20.20s  '%s'" % (book.Name, self.TrackKind[book.Kind], book.KindAsString)

    def show_free_space(self):
        if 'iPod' in self.sources:
            connected_device = self.sources['iPod']
            free_space = self.iTunes.sources.ItemByName(self.sources['iPod']).FreeSpace
            print "\nFreeSpace: %d\n" % free_space

    def show_lib_books(self):
        print "\nshow_lib_books()"

        lib = None
        for source in self.iTunes.sources:
            #print "'%s'   kind: %s" % (source.Name, self.Sources[source.Kind])
            if source.Kind == self.Sources.index('Library'):
                lib = source
                print " Library source: '%s'  kind: %s" % (lib.Name, self.Sources[lib.Kind])
                break
        else:
            print " Library source not found"

        if lib is not None:
            lib_books = None
            for pl in lib.Playlists:
                try:
                    print " Name: %-20.20s kind: %s" % (pl.Name, self.PlaylistKind[pl.Kind])
                    if self.PlaylistKind[pl.Kind] == 'User' and \
                       self.PlaylistSpecialKind[pl.SpecialKind] == 'Books':
                        books_pl = pl
                except:
                    print

            if books_pl:
                print "\n Library|Books playlist:\n '%s' SpecialKind: '%s'" % (books_pl.Name, self.PlaylistSpecialKind[books_pl.SpecialKind])
                for book in books_pl.Tracks:
                    # exclude 'MPEG audio file'
                    # print " %-30.30s Kind: %d  KindAsString: %s" % (book.Name, book.Kind, book.KindAsString)
                    if book.KindAsString in ['MPEG audio file']:
                        print " ignoring '%s' of type %s" % (book.Name, book.KindAsString)
                    else:
                        print " %-30.30s %-20.20s  '%s'" % (book.Name, self.TrackKind[book.Kind], book.KindAsString)

    def _discover_iTunes_media(self):

        with open(self.iTunes.LibraryXMLPath, 'r') as xml:
            soup = BeautifulSoup(xml.read().decode('utf-8'))
            mf = soup.find('key',text="Music Folder").parent
            string = mf.findNext('string').renderContents()
            media_dir = os.path.abspath(string[len('file://localhost/'):].replace('%20',' '))
            if os.path.exists(media_dir):
                print(" iTunes_media: %s" % media_dir)
                return media_dir
            else:
                print(" could not extract valid iTunes.media_dir from %s" % self.iTunes.LibraryXMLPath)
                print(" %s" % string.parent.prettify())
                print(" '%s' not found" % media_dir)

    def show_player_state(self):
        print "show_player_state(): %s" % self.PlayerState[self.iTunes.PlayerState]

    def show_sources(self):
        sources = self.iTunes.sources
        print "show_sources():"
        for x in range(1,sources.Count+1):
            item = sources.Item(x)
            print "Source %d: kind: %s " % (x, self.Sources[item.Kind])
            if item.Kind == self.Sources.index('Library'):
                playlists = item.Playlists
                for y in range(1, playlists.Count):
                    playlist = playlists.Item(y)
                    print " %-20.20s PlaylistKind: %s" % \
                     (playlist.name.encode("windows-1252"),
                      self.PlaylistKind[playlist.Kind])

            if item.Kind == self.Sources.index('iPod'):
                playlists = item.Playlists
                for y in range(1, playlists.Count):
                    playlist = playlists.Item(y)
                    try:
                        print " %-20.20s Kind: %s SpecialKind: %s" % \
                         (playlist.name.encode("windows-1252"),
                          self.PlaylistKind[playlist.Kind],
                          self.PlaylistSpecialKind[playlist.SpecialKind] if self.PlaylistKind[playlist.Kind] == 'User' else None)
                    except:
                        print

    def show_tracks(self):

        mainLibrary = self.iTunes.LibraryPlaylist
        tracks = mainLibrary.Tracks
        numTracks = tracks.Count
        print "numTracks: %d" % numTracks

        while numTracks:
            currTrack=tracks.Item(numTracks)
            if self.TrackKind[currTrack.Kind] == 'Track file':
                name_artist=currTrack.Artist
                name_song=currTrack.Name
                print("[%s] [%s]\n"%(name_artist.encode("utf-8"),name_song.encode("utf-8")))
            numTracks-=1

    def show_version(self):
        print "show_version():\n %s %s\n" % (self.name, self.version)

    def _hexdump(self, src, length=16):
        '''
        '''
        FILTER=''.join([(len(repr(chr(x)))==3) and chr(x) or '.' for x in range(256)])
        N=0; result=''
        while src:
           s,src = src[:length],src[length:]
           hexa = ' '.join(["%02X"%ord(x) for x in s])
           s = s.translate(FILTER)
           result += "%04X   %-*s   %s\n" % (N, length*3, hexa, s)
           N+=length
        print result



if __name__ == '__main__':
    try:
        print
        myTunes = iTunes()

        myTunes.show_version()
        myTunes.show_sources()
        myTunes.show_lib_books()
        myTunes.show_dev_books()

    except:
        traceback.print_exc()


