/*
    This script flips the orientation of the screen. This means 90<->270 and
    and 0<->180 degrees.
    
    by Rafal Kolanski (xaph.net) 2010

    Contains headers from broadsheet.h (Broadsheet MXC Frame buffer driver),
    part of the linux kernel (2.6.23).
*/
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <linux/fb.h>

#include <unistd.h>
#include <fcntl.h>

#include <stdio.h>

/* Headers from broadsheet.h */
#define FBIO_EINK_SET_LUT           0x4694
#define FBIO_EINK_GET_LUT           0x4695
#define FBIO_EINK_GET_LUT_VERSION       0x4698
#define FBIO_EINK_GET_TEMPERATURE       0x46A1  //Returns temperature in degree Celsius
#define FBIO_EINK_INIT              0x46A3  //Initializes the controller
#define FBIO_EINK_GET_STATUS            0x46A6  //Gets controller status
#define FBIO_EINK_GET_VEROFCONT         0x46B0  //Gets controller hardware version
#define FBIO_EINK_GET_DISPSTATUS        0x46B4  //Gets display status (BUSY=1, NOTBUSY=0)
#define FBIO_EINK_SET_ORIENTATION       0x46F2  //Sets picture orientation angle (0 for 0, 1 for 90, 2 for 180, 3 for 270 degrees)
#define FBIO_EINK_GET_ORIENTATION       0x46F3  //Gets picture orientation angle (0 for 0, 1 for 90, 2 for 180, 3 for 270 degrees)
#define FBIO_EINK_SET_POWER_MODE                0x46F6  //Sets broadsheet power mode (1:normal 2:sleep 3:standby)
#define FBIO_EINK_GET_POWER_MODE        0x46F7  //Gets broadsheet power mode
#define FBIO_EINK_UPDATE_PIC            0x4700  //Displays picture
#define FBIO_EINK_SEND_PIC          0x4701  //Send screen image
#define FBIO_EINK_UPDATE_SCREEN         0x4702  //Updater screen
#define FBIO_EINK_SHUTDOWN          0xFE06

#define OP_MODE_INIT 0 
#define OP_MODE_DU 1
#define OP_MODE_GU 2
#define OP_MODE_GC 3
#define OP_MODE_AUTO_DUGU 4

#define POWER_MODE_NORMAL  1
#define POWER_MODE_SLEEP   2
#define POWER_MODE_STANDBY 3

struct updata {
    unsigned int upmode;
    unsigned int orientation;
    unsigned int x;
    unsigned int y;
    unsigned int w;
    unsigned int h;
};
/* end headers from broadsheet.h */

int main(int argc, char** argv) {
    int fd;
    int out = 0, orientation;

    printf("Attempting to access the framebuffer device\n");
    fd = open("/dev/fb0", O_RDWR);
    if (fd < 0) {
        perror("open");
        goto cleanup;
    }

    /* Setting power mode to 1 (normal), otherwise screen update locks up the
       system completely (needs reset). */
    out = ioctl(fd, FBIO_EINK_SET_POWER_MODE, POWER_MODE_NORMAL);
    if (out) {
        perror("setting power mode");
    }
    printf("setting power mode result: %d\n", out);
    out = ioctl(fd, FBIO_EINK_GET_POWER_MODE, NULL);
    printf("Power mode: %d\n", out);
    
    /* Getting orientation - works */
    orientation = ioctl(fd, FBIO_EINK_GET_ORIENTATION, NULL);
    printf("Orientation: %d\n", orientation);

    if (orientation < 0 || orientation > 3) {
        printf("Got invalid orientation %d\n", orientation);
    }

    /* Setting orientation - doesn't work yet */
    orientation = (orientation + 2) % 4; // 0->2->0, 1->3->1
    out = ioctl(fd, FBIO_EINK_SET_ORIENTATION, orientation);
    if (out) {
        perror("set orientation");
    }

    orientation = ioctl(fd, FBIO_EINK_GET_ORIENTATION, NULL);
    printf("Orientation: %d\n", orientation);

    /* Clean up */
cleanup:
    if (fd >= 0)
        close(fd);
    printf("Exiting.\n");    
    return 0;
}

