--========================
--== Platform Specifics ==
--========================

------------
-- colors --
------------

if (eblua.platform == "windows") or (eblua.platform == "linux") or (eblua.platform == "mac") or (eblua.platform == "hanlin") then

  function color(r,g,b)
    return eblua.window_image:colorAllocate(r,g,b);
  end

elseif (eblua.platform == "pocketbook") then

  for i = 0, 255 do 
    eblua.window_image:colorAllocate(i, i, i);
  end

  function color(r,g,b)
    return (r * 0.30) + (g * 0.59) + (b * 0.11);
  end

end

-----------
-- fonts --
-----------

if (eblua.platform == "windows") then
  eblua.default_font = "Verdana";
elseif (eblua.platform == "mac") then
  eblua.default_font = "Verdana";
elseif (eblua.platform == "linux") then
  eblua.default_font = "ttf-liberation/LiberationSans-Regular.ttf";
elseif (eblua.platform == "pocketbook") then
  eblua.default_font = "LiberationSans";
elseif (eblua.platform == "hanlin") then
  eblua.default_font = "arial";
end  

--=============
--== Generic ==
--=============

-----------
-- tools --
-----------

function shallowCopy(array)
  local newArray = {};
  for name, value in pairs(array) do
    newArray[name] = value;
  end
  return newArray;
end

function escape(s)
  return string.gsub(s, "([^A-Za-z0-9_])", function(c)
    return string.format("%%%02x", string.byte(c))
  end)
end

function unescape(s)
  return string.gsub(s, "%%(%x%x)", function(hex)
    return string.char(base.tonumber(hex, 16))
  end)
end

------------
-- colors --
------------

white  = color(255, 255, 255);
gray   = color(128, 128, 128);
black  = color(  0,   0,   0);
red    = color(255,   0,   0);
green  = color(  0, 255,   0);
blue   = color(  0,   0, 255);
cyan   = color(  0, 255, 255);
orange = color(255, 128,  64);
yellow = color(255, 255,   0);
purple = color(128,   0, 128);

--------------------
-- event handling --
--------------------

eblua.event_handlers = {};

function eblua.event_handlers.char (key, p4)
end

function eblua.event_handlers.mouse_up (x, y)
end

function eblua.event_handlers.about_to_terminate (p3, p4)
end

function eblua.event_handlers.timer (p3, p4)
end

eblua.event_codes = {};

eblua.event_codes.char                  = 123;
eblua.event_codes.mouse_up              = 124;
eblua.event_codes.about_to_terminate    = 125;
eblua.event_codes.timer                 = 126;
eblua.event_codes.socket                = 127;

function eblua_handle_event (p1, p2, p3, p4)
  if (p2 == eblua.event_codes.char) then
    eblua.event_handlers.char (p3, p4);
  end
  if (p2 == eblua.event_codes.mouse_up) then
    eblua.event_handlers.mouse_up (p3, p4);
  end
  if (p2 == eblua.event_codes.about_to_terminate) then
    eblua.event_handlers.about_to_terminate (p3, p4);
  end
  if (p2 == eblua.event_codes.timer) then
    eblua.event_handlers.timer (p3, p4);
  end
  if (p2 == eblua.event_codes.socket) then
    eblua.event_handlers.socket(p3, p4);
  end
end

-------------
-- sockets --
-------------

coroutineOfSocket = {};

socket_event_code_READ    = 1;
socket_event_code_WRITE   = 2;
socket_event_code_ACCEPT  = 8;
socket_event_code_CONNECT = 16;
socket_event_code_CLOSE   = 32;

function waitForSocket(sock)
  local fd = sock:getfd();
  coroutineOfSocket[fd] = coroutine.running();
  eblua.sent_socket_events(sock);
  coroutine.yield();
  coroutineOfSocket[fd] = nil;
end

function waitForSocketWrite(sock)
  local fd = sock:getfd();
  coroutineOfSocket[fd] = coroutine.running();
  eblua.sent_socket_write_events(sock);
  coroutine.yield();
  coroutineOfSocket[fd] = nil;
end

function waitForSocketRead(sock)
  local fd = sock:getfd();
  coroutineOfSocket[fd] = coroutine.running();
  eblua.sent_socket_read_events(sock);
  coroutine.yield();
  coroutineOfSocket[fd] = nil;
end

function noEventsForSocket(sock)
  local fd = sock:getfd();
  eblua.stop_socket_events(sock);
end

-------------------
-- socket events --
-------------------

function eblua.event_handlers.socket(code, sock)
  local co = coroutineOfSocket[sock];
  if co then
    coroutine.resume(co);
  end
end


-----------------------------------------------------------------------------
-- TeniliSocket
-- non blocking co-routine aware tcp connections
--   wraps 'real' tcp connections
-----------------------------------------------------------------------------

TeniliSocket = {
  realSocket = nil
};

function TeniliSocket:New(realSocket)
  local instance = shallowCopy(self);
  instance.realSocket = realSocket;
  instance.realSocket:settimeout(0, 't');
  return instance;
end

function TeniliSocket:waitToReceive()
  waitForSocketRead(self.realSocket);
end

function TeniliSocket:waitToSend()
  waitForSocketWrite(self.realSocket);
end

function TeniliSocket:connect(host, port)
  self.realSocket:connect(host, port);
  self:waitToSend();
end

function TeniliSocket:receive(count)
  local result = "";
  local rest_count = count;
  while 1 do
    local l, e, p = self.realSocket:receive(rest_count);
    if e then
      result = result .. p;
      if rest_count then
        rest_count = rest_count - string.len(p);
      end
      if e == "timeout" then
        self:waitToReceive();
      else
        return nil, e, result;
      end
    else
      result = result .. l;
      return result, nil, nil;
    end
  end;
end

function TeniliSocket:receiveLine()
  local result = "";
  while 1 do
    local str = self:receive(1);
    if (str == "\n") then
      return result;
    end
    if (str ~= "\r") then
      result = result .. str;
    end
  end
end

function TeniliSocket:send(str)
  local i = 1;
  while 1 do
    c, e, l = self.realSocket:send(str, i);
    if e then
      i = l + 1;
      if e == "timeout" then
        self:waitToSend();
      else
        return nil, e, l;
      end
    else
      return (i - 1 + c), nil, nil;
    end
  end
end

function TeniliSocket:close()
  noEventsForSocket(self.realSocket);
  return self.realSocket:close();
end

--------------------
-- HttpConnection --
--------------------


HttpConnection = {};

HttpConnection.proxy = {};
HttpConnection.proxy.host = "webproxy.imec.be";
HttpConnection.proxy.port = 8123;

function HttpConnection:OpenInternet()
  if (eblua.platform == "pocketbook") then
    eblua.os_netConnect();
  end
end

function HttpConnection:CloseInternet()
end

function HttpConnection:New(host, port, uri)
  local instance = shallowCopy(self);
  instance.host = host;
  instance.port = port;
  instance.uri = uri;
  return instance;
end;

function HttpConnection:get()
  local sock = TeniliSocket:New(socket.tcp());
  if (HttpConnection.proxy) then
    sock:connect(HttpConnection.proxy.host, HttpConnection.proxy.port);
  else
    sock:connect(self.host, self.port);
  end
  
  sock:send("GET " .. self.uri .. " HTTP/1.1\r\n");
  sock:send("Host: " .. self.host .. "\r\n");
  sock:send("\r\n");
        
  local response_header, err = sock:receiveLine();

  self.headerlines = {};
  repeat
    local line, err = sock:receiveLine();
    if (not err) and (line ~= "") then
      local name, value = string.match(line, "(.-):%s*(.*)");
      self.headerlines[string.lower(name)] = value;
    end
  until (line == "") or err;
  
  local len_str = self.headerlines["content-length"]
  local len = tonumber(len_str);
  local resp, errMsg = sock:receive(len);
  sock:close();
  return resp;
end;

function HttpConnection:getToFile(filename)
  local resp, errMsg = self:get();
  if resp then
    local file = io.open (filename, "wb");
    file:write(resp);
    file:close();
  end
end

-----------------------------------------------------------------------------
-- XML Parsing
-----------------------------------------------------------------------------

XmlSaxParser = {
  nextCh      = nil,
  handler     = nil
};

--function nextCh ()
--handler:pitag (tag, attrs)
--handler:xmlcomment (txt)
--handler:starttag (tag, attrs)
--handler:endtag (tag)
--handler:startendtag (tag, attrs)
--handler:text (txt)
function XmlSaxParser:New(nextCh, handler)
  local instance = shallowCopy(self);
  instance.nextCh = nextCh;
  instance.handler = handler;
  return instance;
end

function XmlSaxParser:parse()
  local str_ar = {};
  local ch = ' ';
  repeat
    ch = self.nextCh();
    if ch then
      if ch == '<' then
        self.handler:text(table.concat(str_ar));
        str_ar = {};
        self:parseTag();
      else
        table.insert(str_ar, ch);
      end
    end
  until not ch;
end

function XmlSaxParser:parseTag()
  local ch = self.nextCh();
  if ch == '/' then
    self:parseEndTag();
  elseif ch == '?' then
    self:parsePITag();
  elseif ch == '!' then
    self:parseXmlCommentOrDocType();
  else
    self:parseStartTag(ch);
  end
end

function XmlSaxParser:isOkIdChar(ch)
  return (ch ~= '"') and (ch ~= '\'') and (ch ~= '=') and (ch ~= '>') and (ch ~= '/')
     and (ch ~= ' ') and (ch ~= '\t') and (ch ~= '\r') and (ch ~= '\n');
end

function XmlSaxParser:isWhiteSpaceChar(ch)
  return (ch == ' ') or (ch == '\t') or (ch == '\r') or (ch == '\n');
end

function XmlSaxParser:skipWhiteSpace(ch)
  if not self:isWhiteSpaceChar(ch) then
    return ch;
  end
  local ch2 = self.nextCh();
  while self:isWhiteSpaceChar(ch2) do
    ch2 = self.nextCh();
  end
  return ch2;
end

function XmlSaxParser:parseTagAttribute(ch)
  local attr_name = {};
  local attr_value = {};
  local ch = ch;
  ch = self:skipWhiteSpace(ch);
  if (ch == '>') or (ch == '/') or (ch == '?') then
    return nil, nil, ch;
  end
  while self:isOkIdChar(ch) do
    table.insert(attr_name, ch);
    ch = self.nextCh();
  end
  ch = self:skipWhiteSpace(ch);
  assert(ch == '=');
  local ch = self.nextCh();
  ch = self:skipWhiteSpace(ch);
  local quoteCh = ch;
  assert((ch == '\'') or (ch == '"'));
  local ch = self.nextCh();
  while (ch ~= quoteCh) do
    table.insert(attr_value, ch);
    ch = self.nextCh();
  end
  return table.concat(attr_name), table.concat(attr_value), self.nextCh();
end

function XmlSaxParser:parseTagAttributes(ch)
  local ch = ch
  local attrs = {};
  local attr_name = "";
  local attr_value = "";
  repeat
    attr_name, attr_value, ch = self:parseTagAttribute(ch);
    if attr_name then
      attrs[attr_name] = attr_value;
    end
  until not attr_name;
  return attrs, ch;
end

function XmlSaxParser:parseStartTag(ch)
  local tag = {};
  local ch = ch;
  while self:isOkIdChar(ch) do
    table.insert(tag, ch);
    ch = self.nextCh();
  end
  local attrs, ch = self:parseTagAttributes(ch);
  if ch == '/' then
    self.nextCh();
    self.handler:startendtag(table.concat(tag), attrs);
  else
    self.handler:starttag(table.concat(tag), attrs);
  end
end

function XmlSaxParser:parsePITag()
  local tag = "";
  local ch = self.nextCh();
  while self:isOkIdChar(ch) do
    tag = tag .. ch;
    ch = self.nextCh();
  end
  local attrs, ch = self:parseTagAttributes(ch);
  assert(ch == '?');
  assert(self.nextCh() == '>');
  self.handler:pitag(tag, attrs);
end


function XmlSaxParser:parseXmlCommentOrDocType()
  if (self.nextCh() == '-') then
    self:parseXmlComment();
  else
    self:parseXmlDocType();
  end
end

function XmlSaxParser:parseXmlDocType()
  local ch = self.nextCh();
  while ch ~= '>' do
    if ch == "<" then
      self:parseXmlDocType();
    end
    ch = self.nextCh();
  end
end

function XmlSaxParser:parseXmlComment()
  local comment = "";
  assert(self.nextCh() == '-');
  local ch = self.nextCh();
  while comment:sub(-3) ~= "-->"  do
    comment = comment .. ch;
    ch = self.nextCh();
  end
  self.handler:xmlcomment(comment:sub(1,-4));
end

function XmlSaxParser:parseEndTag()
  local tag = "";
  local ch = self.nextCh();
  while ch and ch ~= '>' do
    tag = tag .. ch;
    ch = self.nextCh();
  end
  self.handler:endtag(tag);
end

