/*
    Test script for PRS600 to query buttons and touchscreen.

    The touchscreen and buttons are plugged into the second serial port of
    the iMX.31 system-on-chip, which shows up at /dev/ttymxc1

    The firmware seems to refer to this touchscreen/button controller on the
    second serial port as "subcpu" or "subcpu2". For lack of a better name, 
    I'll stick with that convention.

    Hypothesis 1: tinyhttp is listening to /dev/ttymxc1, so kill it and
        tinyhttp.sh too
    Hypothesis 2: just like the e-ink device, there are likely sleep modes for
        the touch screen and buttons. 
        They are definitely active if we obtain a shell through tinyhttp and
        then kill it. Unknown at boot before tinyhttp runs.

    by Rafal Kolanski (xaph.net) 2010
*/
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <unistd.h>
#include <stdint.h>
#include <fcntl.h>
#include <string.h>

#include <stdio.h>

/* Communication occurs in 8-byte blocks */
struct subcpu_msg {
    uint8_t b[8];
};

void subcpu_msg_clear(struct subcpu_msg* m) {
    memset(m->b, 0, 8);
}

/* Compare two messages, Return zero if not equal. */
int subcpu_msg_equal(struct subcpu_msg* m1, struct subcpu_msg* m2) {
    for (int i = 0 ; i < 8 ; ++i) {
        if (m1->b[i] != m2->b[i]) return 0;
    }
    return 1;
}

/* Packet decoding, reverse-engineered.
   Decode the packet in place, return zero if packet is valid.
   Return -1 if MSB check failed.
   Return -2 if XOR checksum invalid.
*/
int subcpu_msg_decode(struct subcpu_msg* m) {
    int i;
    int valid = 1;

    /* Bit 7 must be set in first byte, not set in the rest */
    valid = m->b[0] & 0x80;
    for (i = 1 ; i < 7 && valid ; ++i) {
        if (m->b[i] & 0x80) valid = 0;
    }

    if (!valid) return -1;

    /* Bizarre decoding. Looks like error correction, but I'm not sure. */
    m->b[0] = (m->b[0] << 1) + (m->b[1] >> 6);
    m->b[1] = (m->b[1] << 2) + (m->b[2] >> 5);
    m->b[2] = (m->b[2] << 3) + (m->b[3] >> 4);
    m->b[3] = (m->b[3] << 4) + (m->b[4] >> 3);
    m->b[4] = (m->b[4] << 5) + (m->b[5] >> 2);
    m->b[5] = (m->b[5] << 6) + (m->b[6] >> 1);
    m->b[6] = (m->b[6] << 7) +  m->b[7];
    uint8_t checksum = m->b[0] ^ m->b[1] ^ m->b[2] ^ m->b[3] ^
                       m->b[4] ^ m->b[5];
    if (checksum != m->b[6]) return -2;

    return 0;
}

/* Packet encoding (in place). Reverse-engineered. */
void subcpu_msg_encode(struct subcpu_msg* m) {

    uint8_t checksum = m->b[0] ^ m->b[1] ^ m->b[2] ^ m->b[3] ^ 
                       m->b[4] ^ m->b[5];

    m->b[6] = checksum;
    m->b[7] = checksum & 0x7f; /* clear bit 8 */
    
    /* Bizarre encoding. Looks like error correction, but I'm not sure. */
    m->b[6] = ((m->b[5] << 1) + (m->b[6] >> 7)) & 0x7f;
    m->b[5] = ((m->b[4] << 2) + (m->b[5] >> 6)) & 0x7f;
    m->b[4] = ((m->b[3] << 3) + (m->b[4] >> 5)) & 0x7f;
    m->b[3] = ((m->b[2] << 4) + (m->b[3] >> 4)) & 0x7f;
    m->b[2] = ((m->b[1] << 5) + (m->b[2] >> 3)) & 0x7f;
    m->b[1] = ((m->b[0] << 6) + (m->b[1] >> 2)) & 0x7f;
    m->b[0] = 0x80 | (m->b[0] >> 1);
}

void subcpu_msg_dump(struct subcpu_msg* msg) {
    for (int i = 0 ; i < 8 ; ++i) {
        printf("%02x", msg->b[i]);
        if (i < 7) printf(" ");
    }
}

/* Read a message from fd into m, ensuring that a whole packet is read.
   Return -1 on read error.
   Return -2 on packet unexpectedly cut short.
*/
int subcpu_msg_read_raw(FILE* f, struct subcpu_msg* m) {

    subcpu_msg_clear(m);

    /* Hunt for the start of a packet (byte with bit 8 set) */
    while (!(m->b[0] & 0x80)) {
        if (fread(m->b, 1, 1, f) != 1) return -1;
    }

    /* Read the rest of the packet */
    if (fread(m->b + 1, 1, 7, f) != 7) return -2;

    return 0;
}

/* Writes a subcpu message directly to a file stream.
   Returns 0 on success, non-zero on fail.
*/
int subcpu_msg_write_raw(FILE* f, struct subcpu_msg* m) {
    return fwrite(m->b, 1, 8, f) != 8;
}

void message_loop(FILE* f) {
    struct subcpu_msg old_msg, msg;
    int res;
    int duplicate = 0;

    subcpu_msg_clear(&old_msg);
    subcpu_msg_clear(&msg);

    while (1) {
        res = subcpu_msg_read_raw(f, &msg);
        if (res) {
            printf("read_raw error: %d\n", res);
            continue;
        }

        /* Ignore duplicates (prevent flood if ACK doesn't work) */
        if (subcpu_msg_equal(&msg, &old_msg)) {
            if (!duplicate) {
                printf("Duplicate\n");
            }
            duplicate = 1;
            continue;
        } else {
            duplicate = 0;
        }
        old_msg = msg;

/*        printf("RAW    : ");*/
/*        subcpu_msg_dump(&msg);*/
/*        printf(" %d\n", res);*/
        res = subcpu_msg_decode(&msg);
        printf("MESSAGE: ");
        subcpu_msg_dump(&msg);
        printf(" %d\n", res);

        /* Apparently bit 8 in byte 0 of decoded message means we need to ACK */
        if (msg.b[0] & 0x80) {
            struct subcpu_msg m = msg;
            
            /* This is what their code does, but it also seems to work if
             * you just send back the original message. */
            m.b[0] = m.b[0] & 0x7f; /* clear bit 8 */
            m.b[0] |= 0x40;         /* set bit 3 */

            subcpu_msg_encode(&m);
            res = subcpu_msg_write_raw(f, &m);
            if (res) {
                printf("ACK write error\n");
            } else {
                printf("ACK\n");
            }
        }
    }
}

int main(int argc, char** argv) {

    FILE* mxc1;

    if (!(mxc1 = fopen("/dev/ttymxc1", "rb+"))) {
        printf("Could not open /dev/ttymxc1\n");
        return -1;
    }

    message_loop(mxc1);

    fclose(mxc1);

    return 0;
}

