/*
    Test script for PRS600 to query the screen via /dev/fb0, draw on it, and
    successfully display a test pattern on the physical screen.
    
    by Rafal Kolanski (xaph.net) 2010

    Contains headers from broadsheet.h (Broadsheet MXC Frame buffer driver),
    part of the linux kernel (2.6.23).
*/
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <linux/fb.h>

#include <unistd.h>
#include <fcntl.h>

#include <stdio.h>

/* Headers from broadsheet.h */
#define FBIO_EINK_SET_LUT           0x4694
#define FBIO_EINK_GET_LUT           0x4695
#define FBIO_EINK_GET_LUT_VERSION       0x4698
#define FBIO_EINK_GET_TEMPERATURE       0x46A1  //Returns temperature in degree Celsius
#define FBIO_EINK_INIT              0x46A3  //Initializes the controller
#define FBIO_EINK_GET_STATUS            0x46A6  //Gets controller status
#define FBIO_EINK_GET_VEROFCONT         0x46B0  //Gets controller hardware version
#define FBIO_EINK_GET_DISPSTATUS        0x46B4  //Gets display status (BUSY=1, NOTBUSY=0)
#define FBIO_EINK_SET_ORIENTATION       0x46F2  //Sets picture orientation angle (0 for 0, 1 for 90, 2 for 180, 3 for 270 degrees)
#define FBIO_EINK_GET_ORIENTATION       0x46F3  //Gets picture orientation angle (0 for 0, 1 for 90, 2 for 180, 3 for 270 degrees)
#define FBIO_EINK_SET_POWER_MODE                0x46F6  //Sets broadsheet power mode (1:normal 2:sleep 3:standby)
#define FBIO_EINK_GET_POWER_MODE        0x46F7  //Gets broadsheet power mode
#define FBIO_EINK_UPDATE_PIC            0x4700  //Displays picture
#define FBIO_EINK_SEND_PIC          0x4701  //Send screen image
#define FBIO_EINK_UPDATE_SCREEN         0x4702  //Updater screen
#define FBIO_EINK_SHUTDOWN          0xFE06

#define OP_MODE_INIT 0 
#define OP_MODE_DU 1
#define OP_MODE_GU 2
#define OP_MODE_GC 3
#define OP_MODE_AUTO_DUGU 4

#define POWER_MODE_NORMAL  1
#define POWER_MODE_SLEEP   2
#define POWER_MODE_STANDBY 3

struct updata {
    unsigned int upmode;
    unsigned int orientation;
    unsigned int x;
    unsigned int y;
    unsigned int w;
    unsigned int h;
};
/* end headers from broadsheet.h */

/* draw test pattern, currently just pixels of steadily rising intensity */
void draw(char* buf, struct fb_var_screeninfo screen_info) {
/*    unsigned i;*/
/*    for (i = 0 ; i < 600*800 ; i++) buf[i] = i % 0xFF;*/
    /* Sierpinsky carpet */
    unsigned x,y;
    for (y = 0 ; y < /*screen_info.yres*/ 800 ; ++y) {
        for (x = 0 ; x < /*screen_info.xres*/ 600 ; ++x) {
            buf[y * 600 /*screen_info.xres*/ + x] = x ^ y;
        }
    }
}

int main(int argc, char** argv) {
    struct fb_var_screeninfo screen_info; /* virtual screen */
    struct fb_fix_screeninfo fixed_info;  /* physical screen */
    char *buffer = NULL;
    size_t buflen;
    int fd;

    printf("Attempting to access the framebuffer device\n");
    fd = open("/dev/fb0", O_RDWR);
    if (fd < 0) {
        perror("open");
        goto cleanup;
    }
    if (ioctl(fd, FBIOGET_VSCREENINFO, &screen_info) ||
            ioctl(fd, FBIOGET_FSCREENINFO, &fixed_info)) {
        perror("ioctl");
        goto cleanup;
    }
    screen_info.activate = FB_ACTIVATE_NOW|FB_ACTIVATE_FORCE;
    if (ioctl(fd, FBIOPUT_VSCREENINFO, &screen_info)) {
        printf("Trying to set FB_ACTIVATE_FORCE failed.\n");
    }

    buflen = screen_info.yres_virtual * fixed_info.line_length;
    buffer = mmap(NULL, buflen, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0);

    if (buffer == MAP_FAILED) {
        perror("mmap");
        goto cleanup;
    }

    printf("Screen mapped: %d x %d, %dbpp, offset %d\n", 
            screen_info.xres, 
            screen_info.yres, 
            screen_info.bits_per_pixel,
            screen_info.yoffset);
    printf("Type: %d, linelen: %d, smemlen: %d, visual: %d, xpanstep: %d, ypanstep: %d, ywrapstep: %d, mmio: %lu (%d), id=%s\n", 
            fixed_info.type, 
            fixed_info.line_length, 
            fixed_info.smem_len,
            fixed_info.visual,
            fixed_info.xpanstep,
            fixed_info.ypanstep,
            fixed_info.ywrapstep,
            fixed_info.mmio_start,
            fixed_info.mmio_len,
            fixed_info.id
            );
    printf("grey: %d, rotate: %d\n",
            screen_info.grayscale,
            screen_info.rotate
          );
    printf("res: %d x %d, virtres: %d x %d, offsets: %d x %d, activ: %x, size: %d x %d\n",
            screen_info.xres,
            screen_info.yres,
            screen_info.xres_virtual,
            screen_info.yres_virtual,
            screen_info.xoffset,
            screen_info.yoffset,
            screen_info.activate,
            screen_info.height,
            screen_info.width
          );

    /* make a test pattern */
    draw(buffer, screen_info);

    int out = 0;
    /* Getting temperature - works */
    out = ioctl(fd, FBIO_EINK_GET_TEMPERATURE, NULL);
    printf("Temp: %d\n", out);
    /* Getting orientation - works */
    out = ioctl(fd, FBIO_EINK_GET_ORIENTATION, NULL);
    printf("Orientation: %d\n", out);
    /* Getting power mode - works */
    out = ioctl(fd, FBIO_EINK_GET_POWER_MODE, NULL);
    printf("Power mode: %d\n", out);

    /* Setting power mode to 1 (normal), otherwise screen update locks up the
       system completely (needs reset). */
    out = ioctl(fd, FBIO_EINK_SET_POWER_MODE, POWER_MODE_NORMAL);
    if (out) {
        perror("setting power mode");
    }
    printf("setting power mode result: %d\n", out);
    out = ioctl(fd, FBIO_EINK_GET_POWER_MODE, NULL);
    printf("Power mode: %d\n", out);

    /* Setting orientation - doesn't work yet */
/*    out = ioctl(fd, FBIO_EINK_SET_ORIENTATION, 2);*/
/*    if (out) {*/
/*        perror("set orientation");*/
/*    }*/
/*    printf("setting orientation result: %d\n", out);*/

    /* Try updating the picture */
    printf("Attempting to update picture\n");
    sleep(1);
    struct updata ud;
    ud.x = 0;
    ud.y = 0;
    ud.w = 600;
    ud.h = 800;
    ud.orientation = 1;
    ud.upmode = 3;
    if (ioctl(fd, FBIO_EINK_UPDATE_PIC, &ud)) {
        perror("Updating picture");
        goto cleanup;
    }

    /*
        Success in power mode 1 (normal) (need to set, starts in 2 (sleep)):
        FBIO_EINK_UPDATE_PIC upmode 0,1,2,3

        in update mode 1, only some shades got through, in 3 all did
        in 2, looks like they all did too, if 3 is complete screen update,
            then 2 is partial with full colors
        in 0, screen flashed multiple times, bars of about 8-16 wide show up
            maybe that's used for clearing the screen first? I don't know.
        so still not sure what modes 0 and 1 are supposed to do exactly
    */

    /* Clean up */
cleanup:
    if (buffer && buffer != MAP_FAILED)
        munmap(buffer, buflen);
    if (fd >= 0)
        close(fd);

    printf("Sleeping for 2 seconds.\n");
    sleep(2);
    printf("Exiting.\n");    
    return 0;
}

