/***-----
|
|	crFormat.c
|
|	Windows version of cr_fmt.
|
|	For "Procopius Seingalt" <pseingalt@hotmail.com>
|	from newsgroup. A program that will accept
|	a filename from user input,
|	load a text file, and reformat the file according
|	to the following specification:
|
|	change single carriage returns into spaces 
|	(but leave 2 or more carriage returns alone)
|	change carriage return followed by tab
|	 into 2 carriage returns
|	(but leave other tabs alone)
|	change 3 spaces or more into two carraige returns
|	(but leave 2 or 1 spaces alone)
|
|	The idea is to format raw text into a paragraph
|	formatting so that the text may become easier to deal
|	with in a wordprocessor type setting.
|
|	To compile, use Makefile
|
|
-----***/


/*** preprocessor commands ----------------------------------***/

#include	<windows.h>
#include	<stdlib.h>
#include	<stdio.h>
#include	<stdarg.h>
#include	<string.h>
#include	"bfr/bfr.h"
#include	"eio/eio.h"
#include	"crRef.h"

#define	CRF_OK		0
#define	CRF_ERROR		1


/*** global variables ---------------------------------------***/

HWND		hdialog;


/*** function prototypes ------------------------------------***/

LRESULT CALLBACK	WndProc		(HWND, UINT, WPARAM, LPARAM);
char		*		crf_fmttext	(char *text);
void				crf_init		();
void				crf_setctrl	(int ctrl_id, char *text);
void				crf_getctrl	(int ctrl_id, char *bfr);
int				crf_browse	(char *filename);
void				crf_printf	(char *fmt, ...);


/*** program functions --------------------------------------***/

/*>>>_____ FUNCTION: main _____ */

	/***-----
	|
	|	Accept input from
	|	cmd line, else get
	|	it from the user.
	|
	-----***/

int WINAPI WinMain (HINSTANCE hInst, HINSTANCE hPrev, PSTR szCmdLine, int iCmdShow)
{
	MSG			msg;
	WNDCLASS		wndclass;

	wndclass.style			= CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc	= WndProc;
	wndclass.cbClsExtra		= 0;
	wndclass.cbWndExtra		= DLGWINDOWEXTRA;
	wndclass.hInstance		= hInst;
	wndclass.hIcon			= LoadIcon (hInst, MAKEINTRESOURCE (IDI_PAGE));
	wndclass.hCursor		= LoadCursor (NULL, IDC_ARROW);
	wndclass.hbrBackground	= (HBRUSH) (COLOR_BTNFACE + 1);
	wndclass.lpszMenuName	= NULL;
	wndclass.lpszClassName	= "crf";
	if (! RegisterClass (&wndclass))
	{
		MessageBox (NULL, "Can't register window class.",
			"crFormat", MB_ICONERROR);
		return 0;
	}
	hdialog = CreateDialog (hInst, "CRFBOX", 0, NULL);
	crf_init ();
	ShowWindow (hdialog, iCmdShow);
	while (GetMessage (&msg, NULL, 0, 0))
	{
		TranslateMessage (&msg);
		DispatchMessage (&msg);
	}
	return msg.wParam;
}

/*>>>_____ FUNCTION: WndProc _____ */

	/***-----
	|
	|
	-----***/

LRESULT CALLBACK WndProc (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_COMMAND:
			switch (LOWORD (wParam))
			{
				case ID_BROWSE_OPEN:
				{
					char filename [FILENAME_MAX];
					char current_savefile [FILENAME_MAX];

					crf_getctrl (ID_EDIT_OPEN, filename);
					if (crf_browse (filename))
					{
						crf_setctrl (ID_EDIT_OPEN,
							filename);
						crf_getctrl (ID_EDIT_SAVE,
							current_savefile);
						if (! *current_savefile)
							crf_setctrl (ID_EDIT_SAVE,
								filename);
					}
					return 0;
				}
				case ID_BROWSE_SAVE:
				{
					char filename [FILENAME_MAX];

					crf_getctrl (ID_EDIT_SAVE, filename);
					if (crf_browse (filename))
						crf_setctrl (ID_EDIT_SAVE,
							filename);
					return 0;
				}
				case IDOK:
				{
					crf_perform ();
					return 0;
				}
				case IDCANCEL:
					SendMessage (hdialog, WM_DESTROY, 0, 0);
					return 0;
			}
			return 0;

		case WM_DESTROY:
			PostQuitMessage (0);
			return 0;
	}
	return DefWindowProc (hwnd, message, wParam, lParam);
}


/*>>>_____ FUNCTION: cr_fmttext _____ */

	/***-----
	|
	|	Formats C/Rs in a buffer
	|	according to the
	|	given specification.
	|
	-----***/

char *crf_fmttext (char *text)
{
	int cr_pos, cr_count, space_count;
	register int idx;

	idx = cr_count = space_count = 0;
	bfr_init ();
	while (text [idx])
	{
		switch (text [idx])
		{
			case '\n':
				cr_count ++;
				break;
			case ' ':
				space_count ++;
				break;
			case '\t':
				if (cr_count)
				{
					cr_count ++;
					break;
				}
				/* else fall through to default */
			default:
			{
				int loop;

				if (cr_count == 1)
					bfr_addch (' ');
				else if (cr_count)
					for (loop = 0; loop < cr_count; loop ++)
						bfr_addch ('\n');
				if (space_count >= 3)
					bfr_addstr ("\n\n");
				else if (space_count)
					for (loop = 0; loop < space_count; loop ++)
						bfr_addch (' ');
				bfr_addch (text [idx]);
				space_count = 0;
				cr_count = 0;
				break;
			}
		}
		idx ++;
	}
	return bfr_retrieve ();
}


/*>>>_____ FUNCTION: crf_init _____ */

	/***-----
	|
	|	Empty edit boxes
	|
	-----***/

void crf_init ()
{
	crf_setctrl (ID_EDIT_OPEN, "\0");
	crf_setctrl (ID_EDIT_SAVE, "\0");

	return;
}


/*>>>_____ FUNCTION: crf_setctrl _____ */

	/***-----
	|
	|	change window text
	|	in dialog ctrl
	|
	-----***/

void crf_setctrl (int ctrl_id, char *text)
{
	HWND hctrl;

	hctrl = GetDlgItem (hdialog, ctrl_id);
	SetWindowText (hctrl, text);

	return;
}

	
/*>>>_____ FUNCTION: crf_getctrl _____ */

	/***-----
	|
	|	retrieve window text
	|	in dialog ctrl
	|
	-----***/

void crf_getctrl (int ctrl_id, char *bfr)
{
	HWND hctrl;

	hctrl = GetDlgItem (hdialog, ctrl_id);
	GetWindowText (hctrl, bfr, FILENAME_MAX);

	return;
}


/*>>>_____ FUNCTION: crf_browse _____ */

	/***-----
	|
	|	Show browse dialog box
	|	to get a filename.
	|
	-----***/

int crf_browse (char *filename)
{
	OPENFILENAME ofn_select;
	char just_name [FILENAME_MAX];
	char initial_dir [FILENAME_MAX];

	GetCurrentDirectory (MAX_PATH, initial_dir);

	ofn_select.lStructSize		= sizeof (OPENFILENAME);
	ofn_select.hwndOwner		= hdialog;
	ofn_select.hInstance		= NULL;
	ofn_select.lpstrFilter		= NULL;
	ofn_select.lpstrCustomFilter	= NULL;
	ofn_select.nMaxCustFilter	= 0;
	ofn_select.nFilterIndex		= 0;
	ofn_select.lpstrFile		= filename;
	ofn_select.nMaxFile			= FILENAME_MAX;
	ofn_select.lpstrFileTitle	= just_name;
	ofn_select.nMaxFileTitle		= FILENAME_MAX;
	ofn_select.lpstrInitialDir	= initial_dir;
	ofn_select.lpstrTitle		= "Browse";
	ofn_select.nFileOffset		= 0;
	ofn_select.nFileExtension	= 0;
	ofn_select.lpstrDefExt		= NULL;
	ofn_select.lCustData		= 0L;
	ofn_select.lpfnHook			= NULL;
	ofn_select.lpTemplateName	= NULL;
	ofn_select.Flags = OFN_HIDEREADONLY | OFN_CREATEPROMPT;

	return GetOpenFileName (&ofn_select);
}


/*>>>_____ FUNCTION: crf_perform _____ */

	/***-----
	|
	|	Perform the cr format
	|	operation.
	|
	-----***/

int crf_perform ()
{
	char *bfr, *new_bfr,
		openfile [FILENAME_MAX], savefile [FILENAME_MAX];
	int length;

	crf_getctrl (ID_EDIT_OPEN, openfile);
	crf_getctrl (ID_EDIT_SAVE, savefile);
	if (! *openfile)
	{
		crf_printf ("You must specify a file to open.");
		return CRF_ERROR;
	}
	if (! (bfr = eio_loadfile (openfile, &length)))
	{
		crf_printf ("Error opening file named %s.",
			openfile);
		return CRF_ERROR;
	}
	new_bfr = crf_fmttext (bfr);
	if (! eio_savefile (savefile, new_bfr))
	{
		crf_printf ("Error saving file named %s.",
			savefile);
		return CRF_ERROR;
	}
	crf_printf ("File named %s formatted and saved as %s.\n"
		"You may perform the operation again.",
		openfile, savefile);
	crf_init ();
	return CRF_OK;
}


/*>>>_____ FUNCTION: crf_printf _____ */

	/***-----
	|
	|	write formatted line
	|	to message box
	|
	-----***/

void crf_printf (char *fmt, ...)
{
	char bfr [1000];
	va_list varg;

	va_start (varg, fmt);
	vsprintf (bfr, fmt, varg);
	va_end (varg);
	MessageBeep (MB_ICONHAND);
	MessageBox (NULL, bfr,
		"crFormat", MB_ICONINFORMATION);

	return;
}

