// Name: Core
// Description: controls PRS+ initialization 
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-17 kartu - Moved module specific global vars into local functions context
//	2010-04-21 kartu - Localized, removed call to lang.init
var log;

// initialized in lang
var coreL;

// dummy function, to avoid introducing global vars
var tmp = function() {
	Core.addons = [];
	Core.actions = [];
	
	// Calls given method for all array objects, passing arg as argument
	// Arguments:
	//	objArray - array of objects to call
	//	methodName - name of the method to call
	//	arg - argument to pass to <methodName> function
	//
	var callMethodForAll = function (objArray, methodName) {
		if (!objArray) {
			return;
		}
		for (var i = 0, n = objArray.length; i < n; i++) {
			var obj = objArray[i];
			var func = obj[methodName];
			if (typeof func === "function") {
				try {
					func.call(obj);
				} catch (e) {
					try {
						if (typeof obj != "undefined" && obj.hasOwnProperty("name")) {
							log.error("error when calling method " + methodName + " on " + obj.name + ": " + e);
						} else {
							log.error("error when calling method " + methodName + " on object [" + i + "]: " + e);
						}
					} catch (ignore) {
					}
				}
			}
		}
	};
	
	// Adds all addons actions to the Core.actions array
	var addActions = function(addon) {
		if(addon && addon.actions) {
			for(var i = 0, n = addon.actions.length; i < n; i++) {
				addon.actions[i].addon = addon;
				Core.actions.push(addon.actions[i]);
			}
		}
	};
	
	// Adds addon nodes, calls onPreInit & onInit
	//
	Core.addAddon = function(addon) {
		this.addons.push(addon);
		addActions(addon);
	};
	
	// Creates addon related nodes
	// 
	Core.init = function () {
		try {
			// Root settings node, located "Settings" => "Addon Settings"
			Core.ui.nodes.addonSettingsNode = Core.ui.createContainerNode({
				parent: Core.ui.nodes.settings,
				title: coreL("NODE_PRSP_SETTINGS"),
				kind: Core.ui.NodeKinds.SETTINGS,
				comment: ""
			});
			Core.ui.nodes.settings.nodes.splice(0, 0, Core.ui.nodes.addonSettingsNode);
	
			var addons = this.addons;
			
			// All addon's are loaded, call preInit (some addon's might change option defs)
			callMethodForAll(addons, "onPreInit");
			
			// Load options 
			for (var i = 0, n = addons.length; i < n; i++) {
				try {
					Core.settings.loadOptions(addons[i]);
				} catch (e0) {
					log.warn("error loading settings for addon " + addons[i].name + ": " + e0);
				}
			}
			
			// Addons and options are loaded, call init
			callMethodForAll(addons, "onInit");
			
			// Create addon nodes and addon option nodes
			// FIXME: shouldn't it be done by "settings.init()"???
			for (i = 0, n = addons.length; i < n; i++) {
				Core.settings.createAddonNodes(addons[i]);
				Core.settings.createAddonSettings(addons[i]);
			}
		} catch (e) {
			log.error("in core init", e);
		}
	};
};

// TODO onTerminate, onSleep, onResume
try {
	tmp();
} catch (ignore) {
	// logging system is not yet initialized
}// Name: Log
// Description: logging related methods 
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-03-17 kartu - Fixed date format to always have the same length
//	2010-04-21 kartu - Added "exception" parameter to log.error
//	2010-04-25 kartu - Marked setLevel as constructor for closure compiler to shut up
Core.log = {
	loggers: {},
	createLogger: function (cls, level) {
		if (typeof level === "undefined") {
			level = Core.config.defaultLogLevel;
		}
		var result = {};
		result.name = cls;
		result.log = this.log;
		result.setLevel = this.setLevel;
		result.setLevel(level);
		return result;
	},
	getLogger: function (cls, level) {
		var loggers = this.loggers;
		if (loggers.hasOwnProperty(cls)) {
			return loggers[cls];
		} else {
			var logger = this.createLogger(cls, level);
			loggers[cls] = logger;
			return logger;
		}
	},		
	log : function (msg, level) {
		try {
			if (typeof level === "undefined") {
				level = "";
			} else {
				level = " " + level;
			}
			var stream = new Stream.File(Core.config.logFile, 1, 0);
			try {
				// double digit
				var dd = function (n) {
					if (n < 10) {
						return "0" + n;
					} else {
						return n;
					}
		                };			
				
				stream.seek(stream.bytesAvailable);
				var d = new Date();
				var year, month, day, hour, minute, sec;
				year = dd(d.getFullYear());
				month = dd(d.getMonth() + 1);
				day = dd(d.getDate());
				hour = dd(d.getHours());
				minute = dd(d.getMinutes());
				sec = dd(d.getSeconds());				
				var dateStr = year + "-" + month + "-" + day + " " +  hour +
					":" + minute + ":" + sec + "." + d.getMilliseconds();
				stream.writeLine(dateStr + level + " " + this.name  + "\t" + msg);
			} catch (ignore) {
			} finally {
			    stream.close();
			}
		} catch (ignore2) {
		}
	},
	/**
	 * @constructor
	 */	
	setLevel: function (level) {
		this.trace = this.info = this.warn = this.error = Core.log.dummy;
		switch (level) {
			case "trace":
				this.trace = Core.log.trace;// fallthrough
			case "info":
				this.info = Core.log.info;// fallthrough
			case "warn":
				this.warn = Core.log.warn; // fallthrough
			case "error":
				this.error = Core.log.error;// fallthrough
		}
	},
	trace: function (msg) { this.log(msg, "T"); },
	info: function (msg) { this.log(msg, "I"); },
	warn: function (msg) { this.log(msg, "W"); },
	error: function (msg, e) { 
		if (e !== undefined) {
			this.log(msg + ": " + e, "E");
		} else {
			this.log(msg, "E");
		}	
	},
	dummy: function () {}
};

log = Core.log.getLogger("core");
// Name: Debug
// Description: Debugging related functions
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-21 kartu - Reformatted

try {
	Core.debug = {
		// Dumps properties of an object
		//
		dump: function (obj, log) {
			for (var p in obj) {
				log.trace(p + " => " + obj);
			}
		},
		
		dumpToString: function (o, prefix, depth) {
			var typeofo = typeof o;
			if (typeofo == "string" || typeofo == "boolean" || typeofo == "number") {
				return "'" + o + "'(" + typeofo + ")";
			}
			// Default depth is 1
			if (typeof depth == "undefined") {
				depth = 1;
			}
			// we show prefix if depth is 
			if (typeofo == "undefined") {
				return "undefined";
			}
			if (o === null) {
				return "null";
			}
			if (typeofo == "function") {
				return "a function";
			}
			if (o.constructor == Array) {
				var s = "Array(" + o.length + ")";
				if (depth > 0) {
					s += " dumping\n";
					for (var i = 0, n = o.length; i < n; i++) {
						s += prefix + "[" + i + "] => " + this.dumpToString(o[i], prefix + "\t", depth - 1) + "\n";
					}
				}
				// remove trailing "\n"
				if (s.charAt(s.length - 1) == "\n") {
					s = s.substring(0, s.length - 1);
				}
				return s;
			}
			if (typeofo != "object") {
				return "unknown entitiy of type (" + (typeof o) + ")";
			}
			
			// if depth is less than 1, return just "an object" string
			if (depth < 1) {
				return "an object";
			}
			if (typeof prefix == "undefined") {
				prefix = "";
			}
		
			// at this point, o is not null, and is an object
			var str = "dumping\n";
			var hasProps = false;
			for (var prop in o) {
				hasProps = true;
				var oprop = o[prop];
				try {
					str += prefix + prop + " => " + this.dumpToString(oprop, prefix + "\t", depth - 1) + "\n";
				} catch (ee) {
					str += prefix + prop + " => " + "failed to tostring: " + ee + "\n";
				}
			}
			if (!hasProps) {
				return "an object with no properties";
			}
			// remove trailing "\n"
			if (str.charAt(str.length - 1) == "\n") {
				str = str.substring(0, str.length - 1);
			}
			return str;
		}
	};
} catch (e) {
	log.error("initializing core-debug", e);
}	// Name: Hook
// Description: Hooking related methods.
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-17 kartu - Moved global vars into local functions context

try {
	// dummy function, to avoid introducing global vars
	tmp = function() {
		Core.hook = {};
		
		// Constants
		var CORE_HOOK_BEFORE = 0;
		var CORE_HOOK_INSTEAD = 1;
		var CORE_HOOK_AFTER = 2;
		
		var core_hook_doHook = function (where, what, oldFunc, newFunc, hookType, tag) {
			if(typeof where[what] !== "function" && hookType !== CORE_HOOK_INSTEAD) {
				log.error("cannot hook before/after non-existing function: " + what);
				return;
			}
			switch(hookType) {
			case CORE_HOOK_BEFORE:
				where[what] = function() {
					try {
						newFunc.call(this, arguments, oldFunc, tag);
					} catch(ignore) {
					}
					oldFunc.apply(this, arguments);
				};
				break;
			case CORE_HOOK_AFTER:
				where[what] = function() {
					oldFunc.apply(this, arguments);
					try {
						newFunc.call(this, arguments, oldFunc, tag);
					} catch (ignore) {
					}
				};
				break;
			case CORE_HOOK_INSTEAD:
				where[what] = function() {
					newFunc.call(this, arguments, oldFunc, tag);
				};
				break;
			default:
				log.error("unknown hook type: " + hookType);
			}
			
		};
		Core.hook.hook = function(where, what, newFunction, tag) {
			core_hook_doHook(where, what, where[what], newFunction, CORE_HOOK_INSTEAD, tag);
		};
		Core.hook.hookBefore = function(where, what, newFunction, tag) {
			core_hook_doHook(where, what, where[what], newFunction, CORE_HOOK_BEFORE, tag);
		};
		Core.hook.hookAfter = function(where, what, newFunction, tag) {
			core_hook_doHook(where, what, where[what], newFunction, CORE_HOOK_AFTER, tag);
		};
	};
	tmp();
} catch (e) {
	log.error("initializing core-hook", e);
}
// Name: Hook2
// Description: improved hook functions
// Author: kartu
//
// History:
//	2010-03-17 kartu - Initial version
//	2010-04-21 kartu - refactored

try {
	Core.hook2 = {
		_hooks: [],
		_doCall: function (type, obj, args, hookHandle, result) {
			if (hookHandle[type] !== undefined) {
				var funcs = hookHandle[type];
				var tags = hookHandle[type + "Tags"];
				for (var i = 0, n = funcs.length; i < n; i++) {
					try {
						funcs[i].call(obj, args, hookHandle, tags[i], result);
					} catch (e) {
						log.warn("error when calling hooked (" + type + ") function: "  + e);
					}
				}				
			}
		}, 
		_doHook: function (where, what, hookHandle) {
			hookHandle.where = where;
			hookHandle.what = what;
			hookHandle.func = where[what];
			hookHandle.hookFunc = function() {
				// Call before
				Core.hook2._doCall("before", this, arguments, hookHandle);
				
				var result;
				// Call instead
				if (hookHandle.instead !== undefined && hookHandle.instead.length > 0) {
					var idx = hookHandle.instead.length - 1;
					var tag = hookHandle.insteadTags[idx];
					// Warning: .call returns "this" object, if called function (unless it explicitly returns undefined)
					// Warning: if argument list is null, parents arguments are used
					result = hookHandle.instead[idx].call(this, arguments, hookHandle, tag, undefined, idx - 1);
				} else {
					// Call hooked function
					// Warning: .call returns "this" object, if called function (unless it explicitly returns undefined)
					// Warning: if argument list is null, parents arguments are used
					result = hookHandle.func.apply(this, arguments);
				}
				
				// Call after
				Core.hook2._doCall("after", this, arguments, hookHandle, result);
				
				return result;
			};
			where[what] = hookHandle.hookFunc; 
		},
		_getHookHandle: function (where, what, createIfMissing) {
			var h, f = where[what];
			for (var i = 0, n = this._hooks.length; i < n; i++) {
				h = this._hooks[i];
				// TODO is hookFunc === f needed?
				if (h.where === where && h.what === what && h.hookFunc === f) {
					return h;
				}
			}
			if (createIfMissing) {
				h = {};
				this._doHook(where, what, h);
				this._hooks.push(h);
				return h;
			} else {
				return null;
			}
		},
		_ensureArray: function (obj, prop) {
			if (obj[prop] === undefined) {
				obj[prop] = [];
				obj[prop + "Tags"] = [];
			}
		},
		_hook: function (type, where, what, func, tag) {
			var hd = this._getHookHandle(where, what, true);
			this._ensureArray(hd, type);
			hd[type].push(func);
			hd[type + "Tags"].push(tag);
			return hd;
		},
		_unhook: function (type, where, what, func, tag) {
			var hh = this._getHookHandle(where, what, false);
			if (hh === null) {
				return;
			}
			var funcs = hh[type];
			var tags = hh[type + "Tags"];
			for (var i = 0, n = funcs.length; i < n; i++) {
				if (funcs[i] === func && tags[i] === tag) {
					// TODO test
					funcs.splice(i, 1);
					tags.splice(i, 1);
					break;
				}
			}
		},
		isHooked: function (where, what) {
			return this._getHookHandle(where, what, false) !== null;
		},
		hookBefore: function (where, what, newFunction, tag) {
			return this._hook("before", where, what, newFunction, tag);
		},
		hookAfter: function (where, what, newFunction, tag) {
			return this._hook("after", where, what, newFunction, tag);
		},
		hook: function (where, what, newFunction, tag) {
			return this._hook("instead", where, what, newFunction, tag);
		},
		unhookBefore: function (where, what, newFunction, tag) {
			return this._unhook("before", where, what, newFunction, tag);
		},
		unhookAfter: function (where, what, newFunction, tag) {
			return this._unhook("after", where, what, newFunction, tag);
		},
		unhook: function (where, what, newFunction, tag) {
			return this._unhook("instead", where, what, newFunction, tag);
		}
	};
} catch (e) {
	log.error("initializing core-hook2", e);
}// Name: IO
// Description: File IO related methods
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-21 kartu - Reformatted

try {
	Core.io = {
		// Returns content of the file <path> as a string.
		// If any kind of error happens (file doesn't exist, or is not readable etc) returns <defVal>
		//
		getFileContent: function (path, defVal) {
			var stream;
			try {
				stream = new Stream.File(path);
				return stream.toString();
			} catch (whatever) {
			} finally {
				try {
					stream.close();
				} catch (ignore) {
				}
			}
			return defVal;
		},
		
		// Sets content of the file <path> to <content>. If file exists it will be overwritten.
		//
		setFileContent: function (path, content) {
			var stream;
			try {
				if (FileSystem.getFileInfo(path)) {
					FileSystem.deleteFile(path);
				}
				stream = new Stream.File(path, 1);
				stream.writeString(content);
				stream.flush();
			} catch (e) {
				throw "in setFileContent: " + e;
			} finally {
				try {
					stream.close();
				} catch (ignore) {
				}
			}
		},
		
		// Copies file from <from> to <to>, deleting the target file first
		//
		// Arguments:
		//	from - source file
		//	to - target file
		//
		// Throws exceptions on errors. 
		copyFile: function (from, to) {
			if (FileSystem.getFileInfo(to)) {
				FileSystem.deleteFile(to);
			}
			//FileSystem.copyFile(from, to);
			// Copy/paste from FileSystem.copyFile, slightly modified (removed progress)
			var s, d, c, len, totalLen, copied;
			try {
				s = new Stream.File(from, 2);
				d = new Stream.File(to, 3);
				len = 128 * 1024;
				copied = 0;
				totalLen = s.bytesAvailable;
				c = new Chunk(len);
				while (s.readChunk(len, c)) {
					copied += c.length;
					d.writeChunk(c);
				}
				if (copied !== totalLen) {
					throw "Error copying " + from + " to " + to;
				}
			} finally {
				if (c) {
					c.free();
				}
				if (s) {
					s.close();
				}
				if (d) {
					d.close();
				}
			}
		}
	};
} catch (e) {
	log.error("initializing core-io", e);
}// Name: PRS+ Settings
// Description: PRS+ Settings engine
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-17 kartu - Moved global vars into local functions context
//	2010-04-25 kartu - Marked lazyCreateSettings as constructor for closure compiler to shut up

try {
	// dummy function, to avoid introducing global vars
	tmp = function() {
		// Returns option title for given addon/option definition
		//
		var core_setting_translateValue = function(optionDef, value) {
			if(optionDef.hasOwnProperty("valueTitles") && optionDef.valueTitles.hasOwnProperty(value)) {
				return optionDef.valueTitles[value];
			}
			return value;
		};
		// Returns closure that retrieves given value from a given option object
		//
		var core_setting_getValueTranslator = function(options, optionDef) {
			return function() {
				return core_setting_translateValue(optionDef, options[optionDef.name]);
			};
		};
		
		// Creates "value" node (used in settings).
		// Arguments:
		//	arg, in addition to fields from createContainerNode, can have the following fields
		//		optionDef - option definition
		//		value - option value
		//		object - target object, to set option to (typically addon.options)
		//		addon - addon object
		//
		var core_setting_createValueNode = function(arg) {
			var node = Core.ui.createContainerNode(arg);
			node.enter = function() {
				try {
					var optionDef = arg.optionDef;
					var propertyName = optionDef.name;
		
					var oldValue = arg.object[propertyName];
					arg.object[propertyName] = arg.value;
					
					if(arg.addon && arg.addon.onSettingsChanged) {
						arg.addon.onSettingsChanged(propertyName, oldValue, arg.value, arg.object);
					}
			
					// Save changes
					Core.settings.saveOptions(arg.addon);
					
					// Goto parent node
					this.parent.parent.enter(kbook.model);
				} catch (e) {
					log.error("in valuenode.enter for option " + arg.optionDef.name + ": " + e);
				}
			};
			return node;
		};
		
		// Creates value nodes (used in addon settings) for given option definition and addon.
		//  
		var core_setting_createValueNodes = function(parent, optionDef, addon, options) {
			try {
				var values = optionDef.values;
				for (var i = 0, n = values.length; i < n; i++) {
					var v = values[i];
					var node = core_setting_createValueNode({
						parent: parent,
						title: core_setting_translateValue(optionDef, v),
						optionDef: optionDef,
						value: v,
						object: options,
						addon: addon,
						kind: Core.ui.NodeKinds.CROSSED_BOX,
						comment: ""
					});
					if(v === options[optionDef.name]) {
						node.selected = true;
					}
					parent.nodes.push(node);
				}
			} catch (e) {
				log.error("in core_setting_createValueNodes for addon " + addon.name + " option " + optionDef.name + ": " + e);
			}
		};
		
		
		var doCreateAddonSettings;
		/**
		 * @constructor
		 */		
		var lazyCreateSettings = function(parent, optionDefs, addon) {
			Core.hook.hookBefore(parent, "enter", function(args, oldFunc) {
				if(!this.hasOwnProperty("prspInitialized")) {
					this.prspInitialized = true;
					doCreateAddonSettings(parent, optionDefs, addon, true);
				}
			});
		};
		
		var doCreateSingleSetting;
		doCreateAddonSettings = function(parent, optionDefs, addon, ignoreLazy) {
			if(ignoreLazy !== true) {
				lazyCreateSettings(parent, optionDefs, addon);
				return;
			}
		
			for (var i = 0, n = optionDefs.length; i < n; i++) {
				doCreateSingleSetting(parent, optionDefs[i], addon);
			}
		};
		
		// Recursively creates setting nodes
		//
		doCreateSingleSetting = function(parent, optionDef, addon) {
			var node;
			if(optionDef.hasOwnProperty("groupTitle")) {
				// Group
				node = Core.ui.createContainerNode({
						parent: parent,
						title: optionDef.groupTitle,
						comment: optionDef.groupComment ? optionDef.groupComment : "",
						kind: Core.ui.NodeKinds.getIcon(optionDef.groupIcon)
				});
				parent.nodes.push(node);
		
				doCreateAddonSettings(node, optionDef.optionDefs, addon, false);
			} else {
				// If target is defined, use it, else create "options"
				var options;
				if(optionDef.hasOwnProperty("target")) {
					options = addon.options[optionDef.target];
				} else {
					options = addon.options;
				}
		
				// Create parent node
				node = Core.ui.createContainerNode({
						parent: parent,
						title: optionDef.title,
						kind: Core.ui.NodeKinds.getIcon(optionDef.icon)
				});
				parent.nodes.push(node);
				parent = node;
		
				parent._mycomment = core_setting_getValueTranslator(options, optionDef);
				core_setting_createValueNodes(parent, optionDef, addon, options);
			}
		};
	
		Core.settings = {};
		
		// Creates entry under "Games & Utilities" corresponding to the addon.
		// Arguments:
		//	addon - addon variable
		Core.settings.createAddonNodes = function(addon) {
			if(addon && addon.activate) {
				var kind = Core.ui.NodeKinds.getIcon(addon.icon);
				var title = addon.hasOwnProperty("title") ? addon.title : addon.name;
				var node = Core.ui.createContainerNode({
						parent: Core.ui.nodes.gamesAndUtils,
						title: title,
						kind: kind,
						comment: addon.comment ? addon.comment : ""
				});
				node.enter = function() {
					addon.activate();
				};
				if(!Core.ui.nodes.gamesAndUtils.hasOwnProperty("nodes")) {
					Core.ui.nodes.gamesAndUtils.nodes = [];
				}
				Core.ui.nodes.gamesAndUtils.nodes.push(node);
			}
		};
		
		
		// Creates entry under "Settings => Addon Settings" corresponding to the addon.
		// Arguments:
		//	addon - addon variable
		Core.settings.createAddonSettings = function(addon) {
			try {
				// Addon
				if(addon && addon.optionDefs && addon.optionDefs.length > 0) {
					var optionDefs = addon.optionDefs;
					var settingsNode = Core.ui.nodes.addonSettingsNode;
		
					// Settings node for this addon
					var title = addon.hasOwnProperty("title") ? addon.title : addon.name;
					var thisSettingsNode = Core.ui.createContainerNode({
							parent: settingsNode,
							title: title,
							kind: Core.ui.NodeKinds.getIcon(addon.icon),
							comment: addon.comment ? addon.comment : ""
					});
					settingsNode.nodes.push(thisSettingsNode);
		
					doCreateAddonSettings(thisSettingsNode, optionDefs, addon, false);
				}
			} catch (e) {
				log.error("failed to create addon settings: " + addon.name + " " + e);
			}
		};
		
		
		// Saves addon's non-default options as JSON object.
		// WARNING: no escaping is done!
		// Arguments:
		//	addon - addon who's settings must be saved
		//
		Core.settings.saveOptions = function(addon) {
			try {
				FileSystem.ensureDirectory(Core.config.settingsPath);
				var od;
				var name;
				
				// Find out which options need to be saved (do not save devault values)
				var options = addon.options;
				var optionDefs = addon.optionDefs;
				var optionDefsToSave = []; // option defs
				var gotSomethingToSave = false;
				for (var i = 0; i < optionDefs.length; i++) {
					od = optionDefs[i];
					
					// Add group suboptions
					if(od.hasOwnProperty("groupTitle")) {
						optionDefs = optionDefs.concat(od.optionDefs);
						continue;
					}
		
					name = od.name;
					var defValue = od.defaultValue;
					var target = od.hasOwnProperty("target") ? od.target : false;
		
					if(target) {
						if(options.hasOwnProperty(target) && options[target].hasOwnProperty(name) && options[target][name] !== defValue) {
							if (!optionDefsToSave.hasOwnProperty(target)) {
								optionDefsToSave[target] = {
									isGroup: true,
									target: target,
									optionDefs: []
								};
							}
							gotSomethingToSave = true;
							optionDefsToSave[target].optionDefs.push(od);
						}
					} else if(options.hasOwnProperty(name) && options[name] !== defValue) {
						gotSomethingToSave = true;
						optionDefsToSave.push(od);
					}
				}
				
				// If there is anything to save - save, if not, delete settings file
				var settingsFile = Core.config.settingsPath + addon.name + ".config";
				if (gotSomethingToSave) {
					var stream = new Stream.File(settingsFile, 1, 0);
					try {
						var globalStr = "";
						for (var ii in optionDefsToSave) {
							od = optionDefsToSave[ii];
							name = od.name;
							
							var str = null;
							if (od.isGroup) {
								str = "\t\"" + od.target + "\": {\n"; 
								for (var j = 0, m = od.optionDefs.length; j < m; j++) {
									var od2 = od.optionDefs[j];
									str = str +  "\t\t\"" + od2.name + "\":\"" + options[od2.target][od2.name] + "\",\n";
								}
								// remove trailing ,\n
								str = str.substring(0, str.length -2);
								str = str + "\n\t}";
							} else if (od.hasOwnProperty("name")) {
								str = "\t\"" + name + "\":\"" + options[name] + "\"";
							}
							globalStr = globalStr + str + ",\n";
						}
						// remove trailing ,\n
						globalStr = globalStr.substring(0, globalStr.length - 2);
						stream.writeLine("return {\n" + globalStr + "\n}");
					} finally {
						stream.close();
					}
				} else {
					// Remove settings file, since all settings have default values
					FileSystem.deleteFile(settingsFile);
				}
			} catch (e) {
				log.error("saving options for addon: " + addon.name);
			}
		};
		
		
		// Loads addon's options, using default option values, if settings file or value is not present.
		//
		Core.settings.loadOptions = function(addon) {
			try {
				if(addon.optionDefs) {
					// load settings from settings file
					var options;
					try {
						var settingsFile = Core.config.settingsPath + addon.name + ".config";
						options = Core.system.callScript(settingsFile, log);
					} catch (e0) {
						log.warn("Failed loading settings file for addon " + addon.name + ": " + e0);
					}
					if(!options) {
						options = {};
					}
					
					var optionDefs = addon.optionDefs;
					for (var i = 0; i < optionDefs.length; i++) {
						var od = optionDefs[i];
						if(od.hasOwnProperty("groupTitle")) {
							optionDefs = optionDefs.concat(od.optionDefs);
						} else {
							if(od.hasOwnProperty("target")) {
								if(!options.hasOwnProperty(od.target)) {
									options[od.target] = {};
								}
								
								if(!options[od.target].hasOwnProperty(od.name)) {
									options[od.target][od.name] = od.defaultValue;
								}
							} else {
								if(!options.hasOwnProperty(od.name)) {
									options[od.name] = od.defaultValue;
								}
							}
						}
					}
					
					addon.options = options;
				}
			} catch (e) {
				log.error("Loading settings of " + addon.name);
			}
		};
	};
	tmp();
} catch (e) {
	log.error("initializing core-settings", e);
}
// Name: Shell
// Description: Linux shell related methods
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-17 kartu - Moved global vars into local functions context

try {
	// dummy function, to avoid introducing global vars
	tmp = function() {
		Core.shell = {};
		
		// CONSTANTS
		var MOUNT_PATH = "/opt/mnt";
		var MS_MOUNT_PATH = MOUNT_PATH + "/ms";
		var SD_MOUNT_PATH = MOUNT_PATH + "/sd";
		var CMD_MOUNT_SD = "mount -t vfat -o utf8 -o shortname=mixed /dev/sdmscard/r5c807a1 " + SD_MOUNT_PATH;
		var CMD_MOUNT_MS = "mount -t vfat -o utf8 -o shortname=mixed /dev/sdmscard/r5c807b1 " + MS_MOUNT_PATH;
		var CMD_UMOUNT_SD = "umount " + SD_MOUNT_PATH;
		var CMD_UMOUNT_MS = "umount " + MS_MOUNT_PATH;
		var SCRIPT_HEADER = "#!/bin/sh\n"+
			"PATH=\"/usr/local/bin:/usr/bin:/sbin:/bin:/usr/bin/X11:/usr/games:/usr/local/sony/bin:/usr/sbin\"\n" +
			"LD_LIBRARY_PATH=\"/opt/sony/ebook/application:/lib:/usr/lib:/usr/local/sony/lib:/opt/sony/ebook/lib\"\n" +
			"export PATH LD_LIBRARY_PATH\n";
		var VM_FILE = "/opt/sony/ebook/application/prspVM.xml";	
		var RESULT_FILE = "/tmp/__result__";
		
		Core.shell.SD = 0;
		Core.shell.MS = 1;
		Core.shell.MOUNT_PATH = MOUNT_PATH;
		Core.shell.MS_MOUNT_PATH = MS_MOUNT_PATH;
		Core.shell.SD_MOUNT_PATH = SD_MOUNT_PATH;
		
		
		// Executes shell command
		// Arguments:
		//	cmd - linux command to execute
		// Throws exception, if command results with result other than zero
		Core.shell.exec = function (cmd) {
			try {
				FileSystem.deleteFile(RESULT_FILE);
			} catch (ignore) {
			}
		
			// Create script file
			Core.io.setFileContent("/tmp/script.sh", SCRIPT_HEADER + cmd + "\necho -n $?>" + RESULT_FILE);
		
			// Call script
			var myvm = FskInclude.load(VM_FILE);
			try {
				myvm.load();
			} catch(e) {
				throw "vm load error: " + e;
			}
		
			var result = Core.io.getFileContent(RESULT_FILE, "222");
			if(result !== "0") {
				throw "Failed to execute " + cmd + "\n" + result;
			}
		};
		
		// Mounts SD or MS card
		// Arguments:
		//	card - "MS" or "SD"
		Core.shell.mount = function (card) {
			if (card === this.MS) {
				this.exec(CMD_MOUNT_MS);
			} else if (card === this.SD) {
				this.exec(CMD_MOUNT_SD);
			}
		};
		
		// Mounts SD or MS card
		// Arguments:
		//	card - "MS" or "SD"
		Core.shell.umount = function (card) {
			if (card === this.MS) {
				this.exec(CMD_UMOUNT_MS);
			} else if (card === this.SD) {
				this.exec(CMD_UMOUNT_SD);
			}
		};
	};
	tmp();
} catch (e) {
	log.error("initializing core-shell", e);
}
// Name: String
// Description: String related methods
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-21 kartu - Reformatted
try {
	Core.string = {
		compareStrings: function(a, b) {
			return a.localeCompare(b);
		},
		
		startsWith: function(str, prefix) {
			return str.indexOf(prefix) === 0;
		},
		
		endsWith: function(str, postfix) {
			return str.lastIndexOf(postfix) === str.length - postfix.length;
		}
	};
} catch (e) {
	log.error("initializing core-string", e);
}	
// Name: System
// Description: Fsk system methods 
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-03-27 kartu - Added setSoValue, rootObj
//	2010-03-27 kartu - Modified getSoValue to support single parameter call
//	2010-04-03 kartu - Addec "compile" function
//	2010-04-04 kartu - Added "getFastSoValue" function stub
//	2010-04-21 kartu - Reformatted
//	2010-04-25 kartu - Fixed minor syntax glitch: removed trailing comma

try {
	Core.system = {
		// Calls script located in "path", using "log" to log errors
		// Arguments:
		//	path - path to the script
		//	log - logger
		// Throws exceptions if script fails or file cannot be found.
		callScript: function (path, log) {
			try {		
				if (FileSystem.getFileInfo(path)) {
					var f = new Stream.File(path);
					try {
						var fn = new Function("Core", f.toString(), path, 1);
						var result = fn(Core);
						delete fn;
						return result;
					} finally {
						f.close();
					}
				}
			} catch (e) {
				var msg = "Error calling " + path + ": " + e;
				if (log) {
					log.error(msg);
				}
				throw msg;
				
			}
		},
		
		// A bit weird way to clone an object. There might be a better function or FSK specific operator to do the same
		// Arguments:
		//      obj - object to clone
		// Returns:
		//      "copy" of an object (linked objects as well as functions aren't cloned)
		//
		cloneObj: function (obj) {
			var temp = FskCache.playlistResult;
			var dummy = {};
			try {
				FskCache.playlistResult = obj;
				var result = FskCache.playlist.browse(dummy);
				delete result.db;
				delete result.playlist;
				return result;
			} catch (e) {
				log.error("error cloning: " + e);
				return undefined;
			} finally {
				FskCache.playlistResult = temp;
			}
		},
		
		// Getting values of properties of objects created by .so bytecode isn't always possible for custom functions.
		// However they are visible to .xb code
		// Arguments:
		//      obj - object to get value from or full object path (propName should be undefined in this case)
		//      propName - property name, could also look like "prop1.prop2.prop3"
		// Returns:
		//      property value or undefined, if property is not defined
		//
		// Note: 
		//	Supports 2 ways of calling:
		//		getSoValue(obj, "someProperty.somOtherProperty...");
		//		getSoValue("SomeObject.someProperty.someOtherProperty...");
		getSoValue: function (obj, propName) {
			if (propName === undefined) {
				return FskCache.mediaMaster.getInstance.call(Core.system.rootObj, obj);
			} else {
				return FskCache.mediaMaster.getInstance.call(obj, propName);
			}
		},
		
		// Same as setSoValue but slightly faster, doesn't support nested properties
		//
		getFastSoValue: function (obj, propName) {
			// TODO replace with custom code from prsp.xsb
			return Core.system.getSoValue(obj, propName);
		}
	};

	// Reference to the root object
	//
	Core.system.rootObj =  this;
	
	// Similiar as getSoValue but could be used to set settings
	// Arguments:
	//	obj - object to set value
	//	field - field name
	//	value - new value
	Core.system.setSoValue = Core.system.getSoValue(this, "prsp.setSoValue");

	// Compiles code in the scope of FskCache.
	// Arguments:
	//	args - comma delimited list of arguments
	//	code - function code
	Core.system.compile = Core.system.getSoValue(this, "prsp.compile");
} catch (e) {
	log.error("initializing core-system", e);
}
// Name: UI
// Description: User interface related methods and constants
// Author: kartu
//
// Externals:
//	kbook.root
//	kbook.root.children.settings
//	kbook.root.nodes
//	kbook.root.nodes[9].nodes
//	kbook.tableData
//	function kbook.tableData.getValue
//	function kbook.tableData.getKind
//	
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-05 kartu - Removed stale code, added logging to getValue
//	2010-04-10 kartu - Improved error reporting
//	2010-04-17 kartu - Removed global var
//	2010-04-25 kartu - Marked setLevel Core.ui.ContainerNode as constructor

try {
	Core.ui = {
		// Node icons
		NodeKinds: {
			BOOK: 2,
			FILE: 2,
			AUDIO: 3,
			PICTURE: 4,
			SETTINGS: 5,
			AUTHOR: 6,
			PREVIOUS_PAGE: 8,
			NEXT_PAGE: 9,
			BOOKMARK: 10,
			LIST: 11,
			CLOCK: 12,
			PAUSE: 13,
			PLAY: 14,
			INFO: 15,
			LOCK: 16,
			BOOKS: 17,
			PICTURES: 18,
			CROSSED_BOX: 19,
			DATE: 22,
			ABOUT: 25,
			BACK: 26,
			ABC: 27,
			DATETIME: 28,
			DB: 29,
			SHUTDOWN: 31,
			FOLDER: 37,
			MS: 34,
			SD: 35,
			INTERNAL_MEM: 36,
			GAME: 38,
			DEFAULT: 37,
			getIcon: function (strKind) {
				var kind = this[strKind];
				if (typeof kind === "undefined") {
					kind = Core.ui.NodeKinds.DEFAULT;
				}
				return kind;
			}
		},
		
		// Small icons on the right side of books
		NodeSourceKinds: {
			NONE: 0,
			MS: 2, // memory stick
			SD: 3 // SD card
		},
		
		// Book MIME types
		BookMIMEs: {
			"application/x-sony-bbeb": "BBeB Book",
			"text/plain": "Plain Text",
			"application/rtf": "Rich Text Format",
			"application/pdf": "Adobe PDF",
			"application/epub+zip": "EPUB Document"
		},
		
		// Reference to nodes
		nodes: {
			root: kbook.root,
			"continue": kbook.root.nodes[0],
			booksByTitle: kbook.root.nodes[1],
			booksByAuthor: kbook.root.nodes[2],
			booksByDate: kbook.root.nodes[3],
			collections: kbook.root.nodes[4],
			bookmarks: kbook.root.nodes[5],
			nowPlaying: kbook.root.nodes[6],
			music: kbook.root.nodes[7],
			pictures: kbook.root.nodes[8],
			settings: kbook.root.nodes[9],
			advancedSettings: kbook.root.nodes[9].nodes[4]
		},
		
		// Creates "container" node, that displayes nodes in this.nodes[] array
		// Arguments:
		//	arg, can have the following fields:
		//		parent - parent node
		//		title - title of this node (shown on top of the screen, when inside the node)
		//		name - name of this node (shown in lists, if none supplied, title is used instead)
		//		comment - comment text (shown on the right bottom in list mode)
		//		kind - one of the NodeKinds, determines which icon to show
		//		sourceKind - one of the NodeSourceKinds, determines which small icon will be shown 
		//					on the right side of the list (eg small "sd" letters)
		//		separator - if equals to 1, node's bottom line will be shown in bold
		createContainerNode: function (arg) {
			var obj = Core.system.cloneObj(kbook.root.children.settings);
			Core.ui.ContainerNode.call(obj, undefined);
			if (typeof arg !== "undefined") {
				if (arg.hasOwnProperty("parent")) {obj.parent = arg.parent;}
				if (arg.hasOwnProperty("title")) {obj.title = arg.title;}
				if (arg.hasOwnProperty("name")) {
					obj.name = arg.name;
				} else {
					obj.name = arg.title;
				}
				if (arg.hasOwnProperty("comment") && (typeof arg.comment !== "undefined")) {
					obj._mycomment = arg.comment;
				} else {
					obj._mycomment = "";
				}
				if (arg.hasOwnProperty("kind")) {obj.kind = arg.kind;}
				if (arg.hasOwnProperty("sourceKind")) {obj._mysourceKind = arg.sourceKind;}
				if (arg.hasOwnProperty("separator")) {obj._myseparator = arg.separator;}
			}
			obj.nodes = [];
			
			return obj;
		}
	};
	
	
	// Container node, displays subnodes, takes care of paging etc
	/**
	 * @constructor
	 */
	Core.ui.ContainerNode = function (arg) {	
		var oldEnter = this.enter;
		var oldExit = this.exit;
	
		this.enter = function () {
			try {
				// Call construct
				if (typeof (this._myconstruct) == "function") {
					var endHere = this._myconstruct.apply(this, arguments);
					if (endHere === true) {
						return;
					}
				}
				
				// Restore item selection
				if (this.hasOwnProperty("selectionIndex") && this.hasOwnProperty("nodes")) {
					var nodeToSelect = this.nodes[this.selectionIndex];
					if (nodeToSelect) {
						nodeToSelect.selected = true;
					}
				}
			} catch (e) {
				log.error("error in ContainerNode.enter: " + e);
			}
			oldEnter.apply(this, arguments);
		};
	
		this.exit = function () {
			try {
				// Save parent's selection
				var nodes = this.nodes;
				this.selectionIndex = undefined;
				if (nodes) {
					for (var i = 0, n = nodes.length; i < n; i++) {
						if (nodes[i].selected) {
							this.selectionIndex = i;
							break;
						}
					}
				}
				
				if (this.hasOwnProperty("_myconstruct")) {
					delete this.nodes;
					this.nodes = [];
				}
			} catch (ignore) {
			}
			oldExit.apply(this, arguments);
		};	
	};
	Core.ui.ContainerNode.prototype = Core.system.cloneObj(kbook.root.children.settings); // settings node
	
	// Little hack to allow easy changing of node title, comment, kind etc
	kbook.tableData.oldGetValue = kbook.tableData.getValue;
	kbook.tableData.getValue = function (node, field) {
		try {
			var myVal = node["_my" + field];
			if (typeof myVal != "undefined") {
				if (typeof myVal == "function") {
					return myVal.call(node, arguments);
				}
				return myVal;
			}
		} catch (e) {
			log.error("in _my getValue: field '" + field + "' node '" + node.name + "': " + e);
		}
		try {
			return this.oldGetValue.apply(this, arguments);
		} catch (e2) {
			log.error("in getValue: " + e2);
			return "error: " + e2;
		}
	};
	
	kbook.tableData.oldGetKind = kbook.tableData.getKind;
	kbook.tableData.getKind = function () {
		try {
			var myVal = this.node._mykind;
			if (typeof myVal != "undefined") {
				if (typeof myVal == "function") {
					return myVal.call(this, arguments);
				}
				return myVal;
			}
		} catch (e) {
			log.error("in _my getKind: " + e);
		}
		return this.oldGetKind();
	};
} catch (e) {
	log.error("initializing core-ui", e);
}// Name: About
// Description: Adds PRS+ stuff to the about screen 
// Author: kartu
//
// History:
//	2010-03-14 kartu - Initial version, refactored from Utils
//	2010-04-05 kartu - Added Core.version object.
//	2010-04-17 kartu - Moved global vars into local functions context

try {
	// dummy function, to avoid introducing global vars
	tmp = function() {
		// About
		var getSoValue = Core.system.getSoValue;
		var about = kbook.model.container.ABOUT_GROUP.ABOUT;
		var data = getSoValue(about, "data");
		var records = getSoValue(data, "records");
		var duplicate = getSoValue(this, "Fskin.tableData.duplicate");
		var store = getSoValue(this, "Fskin.tableField.store");
		
		var record = duplicate.call(this, records[1]);
		var prspScriptVersion = Core.io.getFileContent("/Data/database/system/PRSPlus/prsp.ver", "n/a");
		var prspFirmwareVersion = Core.io.getFileContent("/opt/prspfw.ver", "n/a");
		store.call(this, record, "text", "PRS+ Script: " + prspScriptVersion +
			"\nPRS+ Firmware: " + prspFirmwareVersion +
			"\nAuthor: Mikheil Sukhiashvili aka kartu (kartu3@gmail.com) using work of: " + 
			"igorsk, boroda, obelix, pepak, llasram and others.\n" +
			"© GNU Lesser General Public License.");
		store.call(this, record, "kind", 4);
		records.splice(0, 0, record);
		
		about.dataChanged();
		Core.version = {
			script: prspScriptVersion,
			firmware: prspFirmwareVersion
		};
	};
	tmp();
} catch (e) {
	log.error("initializing core-hook2", e);
}
// Description: Localizaiton related code
// Author: kartu
//
// History:
//	2010-03-17 kartu - Initial version
//	2010-04-05 kartu - Finished localization
//	2010-04-10 kartu - Fixed collections localization (reported by kravitz)
//	2010-04-21 kartu - Localized. Fixed invisible "continue" comment bug.
//	2010-04-22 kartu - Added date customization
//	2010-04-24 kartu - Added Catalan, Georgian, German, Russian and Spanish locales
//	2010-04-24 kartu - Fixed SS_ON related bug (was set to min instead of max field)
//	2010-04-24 kartu - Changed language order
//	2010-04-25 kartu - Marked kbook.model.getDateAndClock as constructor
//	2010-04-25 kartu - Moved default UI localizer into separate file

tmp = function() {
	var _strings; // whatever is loaded from lang/<language>.js file
	var langL;
	var getDateFunc = function(format, separator) {
		var toDoubleDigit = function (num) {
					if (num < 10) {
						return "0" + num;
					} else {
						return num;
					}
		};		
		// "ddMMYY", "MMddYY", "YYMMdd", "ddMMMYY", "ddMONTHYY", "ddMMYYYY", "MMddYYYY", "YYYYMMdd", "ddMMMYYYY", "ddMONTHYYYY"
		return function() {
			try {
				var date = this;
				var day, month, nMonth, year, shortYear;
				day = toDoubleDigit(date.getDate());
				nMonth = date.getMonth() + 1;
				month = toDoubleDigit(nMonth); 
				year = date.getFullYear();
				shortYear = toDoubleDigit(year - Math.floor(year/100) * 100);
				switch (format) {
					case "ddMMYY":
						return day + separator + month + separator + shortYear;
					case "MMddYY":
						return month + separator + day + separator + shortYear;
					case "YYMMdd":
						return shortYear + separator + month + separator + day;
					case "ddMMMYY":
						return day + separator + langL("MONTH_SHORT_" + nMonth) + separator + shortYear;
					case "ddMONTHYY":
						return day + separator + langL("MONTH_" + nMonth) + separator + shortYear;
					case "ddMMYYYY":
						return day + separator + month + separator + year;
					case "MMddYYYY":
						return month + separator + day + separator + year;
					case "YYYYMMdd":
						return year + separator + month + separator + day;
					case "ddMMMYYYY":
						return day + separator + langL("MONTH_SHORT_" + nMonth) + separator + year;
					case "ddMONTHYYYY":
						return day + separator + langL("MONTH_" + nMonth) + separator + year;
					default:
						return day + separator + month + separator + shortYear;
				}
			} catch (e) {
				return e;
			}
		};
	};
	Core.lang = {
		// "fake" options, used only for loading stuff saved by other addon
		name: "Localization",
		optionDefs: [
			{
				name: "lang",
				defaultValue: "English.js"
			},
			{
				name: "dateFormat",
				defaultValue: "default"
			},
			{
				name: "dateSeparator",
				defaultValue: "/"
			}
		],
	
		init: function () {
			try {
				Core.settings.loadOptions(this);

				try {
					_strings = Core.system.callScript(Core.config.corePath + "lang/" + this.options.lang, log);
				} catch (e0) {
					log.error("Failed to load strings: ", e0);
				}
	
				// If locale is English, there is nothing to localize
				if ("English.js" !== this.options.lang) {
					try {
						// localize default ui
						var code = Core.io.getFileContent(Core.config.corePath + "lang/defaultUILocalizer.js"); 
						var localizeDefaultUI = new Function("_strings,Core", code);
						localizeDefaultUI(_strings, Core);
						delete localizeDefaultUI;
					} catch (e1) {
						log.error("Failed to localize default UI", e1);
					}
				}
				
				// Date
				if ("default" !== this.options.dateFormat) {
					var separator = "/";
					switch (this.options.dateSeparator) {
						case "minus":
							separator = "-";
							break;
						case "dot":
							separator = ".";
							break;
						case "space":
							separator = " ";
							break;
						case "none":
							separator = "";
							break;
					}
					Date.prototype.toLocaleDateString = getDateFunc(this.options.dateFormat, separator);
				}
				
				coreL = this.getLocalizer("Core"); // defined in core
				langL = this.getLocalizer("CoreLang");
			} catch (e) {
				log.error("in Core.lang.init: " + e);
			}
		},
		
		getStrings: function (category) {
			try {
				if (_strings !== undefined && _strings[category] !== undefined) {
					return _strings[category];
				} else {
					log.warn("Cannot find strings for category: " + category);
					return {};
				}
			} catch (e) {
				log.error("in getStrings: " + e);
			}
		},
		
		getLocalizer: function (category) {
			var createLocalizer = function(str, prefix) {
				var f = function(key) {
					if (str.hasOwnProperty(key)) {
						return str[key];
					} else {
						return prefix + key;
					}
				};
				return f;
			};
			return createLocalizer(this.getStrings(category), category + ".");
		}		
	};
	

	// Initialize lang
	Core.lang.init();
	
	// Language options
	var LangAddon = {
		name: "Localization",
		title:  langL("TITLE"),
		comment: langL("COMMENT"),
		optionDefs: [
			{
				name: "lang",
				title: langL("OPTION_LANG"),
				icon: "LIST",
				defaultValue: "English.js",
				values:	["Catalan.js", "German.js", "English.js",  "Spanish.js", "Georgian.js", "Russian.js"],
				valueTitles: {
					"Catalan.js": "Català",
					"German.js": "Deutsch",
					"English.js": "English",
					"Georgian.js": "ქ?რთული",
					"Russian.js": "Ру??кий",
					"Spanish.js": "Español"
				}
			},
			{
				name: "dateFormat",
				title: langL("OPTION_DATE_FORMAT"),
				defaultValue: "default",
				values: ["default", "ddMMYY", "MMddYY", "YYMMdd", "ddMMMYY", "ddMONTHYY", "ddMMYYYY", "MMddYYYY", "YYYYMMdd", "ddMMMYYYY", "ddMONTHYYYY"],
				valueTitles: {
					ddMMYY: "31/01/99", 
					MMddYY: "01/31/99", 
					YYMMdd: "99/01/31", 
					ddMMMYY: langL("ddMMMYY"), 
					ddMONTHYY: langL("ddMONTHYY"),
					ddMMYYYY: "31/01/1999", 
					MMddYYYY: "01/31/1999", 
					YYYYMMdd: "1999/01/31", 
					ddMMMYYYY: langL("ddMMMYYYY"), 
					ddMONTHYYYY: langL("ddMONTHYYYY")
				}
			},
			{
				name: "dateSeparator",
				title: langL("OPTION_DATE_SEPARATOR"),
				defaultValue: "backslash",
				values: ["default", "minus", "dot", "space", "none"],
				valueTitles: {
					"default": "/",
					"minus": "-",
					"dot": ".",
					"space": langL("VALUE_SPACE"),
					"none": langL("VALUE_NONE")
				}
			}			
		]		
	};

	Core.addAddon(LangAddon);
};

try {
	tmp();
} catch (e) {
	log.error("initializing core-lang", e);
}