// Name: Reading List
// Description: Replacement of original "Continue Reading" menu item by a folder with several books
// Author: kravitz
//
// HISTORY:
//	2010-03-20 kravitz - Initial version without list saving/loading
//	2010-03-22 kravitz - No activities when addon is disabled
//	2010-03-25 kravitz - Reading list saving/loading (saving by action)
//	2010-04-05 kravitz - Localized
//	2010-04-05 kravitz - Comment (books count) processing is changed
//	2010-04-16 kravitz - Optimized
//	2010-04-19 kravitz - Autosaving
//	2010-04-20 kravitz - Joined MenuSettings union
//	2010-04-20 kravitz - Check Reading list change before saving
//
// TODO:
//	- Fix handlers export
//	- Fix file opening with context cleaning

// Shortcuts
var log = Core.log.getLogger("ReadingList");
var getSoValue = Core.system.getSoValue;

// Default Reading list length
var RL_DEFAULT = 1;

var RL_TITLE = Core.ui.nodes["continue"].title;
var RL_FILE = "Reading.list";

// Localize
var str = Core.lang.getStrings("ReadingList");

var L = function (key) {
	if (str.hasOwnProperty(key)) {
		return str[key];
	} else {
		return "ReadingList." + key;
	}
};

var LF = function (key, param) {
	try {
		if (typeof str[key] == "function") {
			return str[key](param);
		} else if (typeof str[key] != "undefined") {
			return str[key];
		}
	} catch (e) {
		log.error("when calling " + key + " with param " + param + ": " + e);
	}
	return key;
};

// This addon
var ReadingList = {
	name: "ReadingList",
	title: RL_TITLE,
	union: "MenuSettings",
	optionDefs: [{
		name: "maxLength",
		title: RL_TITLE,
		icon: "CONTINUE",
		defaultValue: RL_DEFAULT,
		values:	[RL_DEFAULT, 3 , 10],
		valueTitles: {
			1: L("VALUE_DISABLED"),
			3: L("VALUE_3"),
			10: L("VALUE_10")
		}
	}] /*,
	actions: [{
		name: "saveReadingList",
		title: L("ACTION_SAVE"),
		group: "Utils",
		icon: "CONTINUE",
		action: function () {
			this.saveToFile();
			Core.ui.showMsg(["Reading list saved"]);
		}
	}]
*/
};

ReadingList.onChangeBook = function (bookNode) {
	if (this.options.maxLength == RL_DEFAULT) {
		// Reading list is disabled
		return;
	}
	var media = getFastBookMedia(bookNode);
	var source = media.source;
	var bookPath = source.path + media.path;
	var list = Core.ui.nodes.readingList;

	// Search current book in Reading list
	for (var i = 0, n = list.nodes.length; i < n; i++) {
		if (list.nodes[i]._bookPath == bookPath) { // Found...
			if (i) {
				list.nodes.unshift(list.nodes.splice(i, 1)[0]); // Move book on top
			}
			return;
		}
	}
	// Not found... add to ReadingList
	bookNode._bookPath = bookPath;
	list.nodes.unshift(bookNode);
	if (list.nodes.length > this.options.maxLength) {
		delete list.nodes.pop()[0]; // Remove last node from list
	}
};

ReadingList.onDeleteBook = function (bookNode) {
	if (this.options.maxLength == RL_DEFAULT) {
		// Reading list is disabled
		return;
	}
	var media = getFastBookMedia(bookNode);
	var source = media.source;
	var bookPath = source.path + media.path;
	var list = Core.ui.nodes.readingList;

	// Search current book in Reading list
	for (var i = 0, n = list.nodes.length; i < n; i++) {
		if (list.nodes[i]._bookPath == bookPath) { // Found...
			delete list.nodes.splice(i, 1)[0]; // Remove node from list
		}
		break;
	}
};

// Sets original or expanded Continue Reading style
ReadingList.setStyle = function (l) {
	kbook.root.nodes[0] = (l == RL_DEFAULT) ? Core.ui.nodes["continue"] : Core.ui.nodes.readingList;
};

// Truncates list to desired length
ReadingList.truncTo = function (l) {
	var list = Core.ui.nodes.readingList;
	var n = list.nodes.length - l;
	for (var i = 0; i < n; i++) {
		delete list.nodes.pop()[0]; // Remove last node from list
	}
};

ReadingList.onSettingsChanged = function (propertyName, oldValue, newValue) {
	if (oldValue === newValue) {
		return;
	}
	if (propertyName === "maxLength") {
		if (this.options.maxLength < RL_DEFAULT) {
			this.options.maxLength = RL_DEFAULT;
		}
		this.setStyle(this.options.maxLength);

		if (oldValue == RL_DEFAULT) {
			// Add current book to Reading list
			this.onChangeBook(kbook.model.currentBook);
		} else if (oldValue) {
			// Adjust Reading list length to new max
			this.truncTo((this.options.maxLength == RL_DEFAULT) ? 0 : this.options.maxLength);
		}
	}
};

ReadingList.saveToFile = function () {
	try {
		FileSystem.ensureDirectory(Core.config.settingsRoot);
		var list = Core.ui.nodes.readingList;
		var listFile = Core.config.settingsRoot + RL_FILE;
		var len = list.nodes.length;

		var current = "";
		for (var i = 0; i < len; i++) {
			current += list.nodes[i]._bookPath + "\r\n";
		}
		if (current.length == 0) {
			// List is empty - delete
			FileSystem.deleteFile(listFile);
			return;
		}
		// Load saved Reading list
		saved = Core.io.getFileContent(listFile, "");
		if (saved == current) {
			// Lists are equal
			return;
		}
		// ...aren't equal - save
		FileSystem.deleteFile(listFile); //FIXME
		Core.io.setFileContent(listFile, current);
	} catch (e) {
		log.error("Error in saveToFile(): " + e);
	}
};

ReadingList.loadFromFile = function () {
	try {
		var list = Core.ui.nodes.readingList;
		var listFile = Core.config.settingsRoot + RL_FILE;
		if (FileSystem.getFileInfo(listFile)) {
			var stream = new Stream.File(listFile);
			try {
				while (stream.bytesAvailable) {
					var path = stream.readLine();
					if (FileSystem.getFileInfo(path)) {
						// Create book node
						var node = _bf_pathToBookNode(path, list);
						if (node) {
							// Add to ReadingList
							node._bookPath = path;
							list.nodes.push(node);
						}
					}
				}
			} finally {
				stream.close();
			}
		}
	} catch (e) {
		log.error("Error in loadFromFile(): " + e);
	}
};

ReadingList.onInit = function () {
	// Reading list node
	var readingListNode = Core.ui.createContainerNode({
		parent: kbook.root,
		title: RL_TITLE,
		kind: Core.ui.NodeKinds.CONTINUE,
	});

	// Update from ContainerNode doesn't work for whatever reason, probably it is accessing the wrong "nodes"
	readingListNode.update = function(model) {
		for (var i = 0, n = this.nodes.length; i < n; i++) {
			if(this.nodes[i].update) {
				this.nodes[i].update.call(this.nodes[i], model);
			}
		}
	};

	readingListNode._mycomment = function () {
		return LF("FUNC_X_BOOKS", this.length);
	};

	// Global
	Core.ui.nodes.readingList = readingListNode;

	this.onSettingsChanged("maxLength", 0, this.options.maxLength);

	if (this.options.maxLength != RL_DEFAULT) {
		// Load saved Reading list
		this.loadFromFile();
	}
};

//FIXME Handlers export... O_o

_rl_onChangeBook = function (n) {
	ReadingList.onChangeBook(n);
};

_rl_onDeleteBook = function (n) {
	ReadingList.onDeleteBook(n);
};

_rl_saveToFile = function () {
	ReadingList.saveToFile();
};

//FIXME Move common strings and etc to core...

_rl_settingsCount = function (n) {
	return LF("FUNC_X_SETTINGS", n);
};

_rl_itemsCount = function (n) {
	return LF("FUNC_X_ITEMS", n);
};

return ReadingList;
