// Name: Read Mark
// Description: Marks book as already read
// Author: kravitz
//
// HISTORY:
//	2010-04-10 kravitz - Initial version
//	2010-04-19 kravitz - Bug in readMark() is fixed
//	2010-04-19 kravitz - Autosaving
//
// TODO:
//	- Add AlreadyRead book icon

// Shortcuts
var log = Core.log.getLogger("ReadMark");
var getSoValue = Core.system.getSoValue;
var cloneObj = Core.system.cloneObj;
var NodeKinds = Core.ui.NodeKinds;

var RM_FILE = "AlreadyRead.list";

// Localize
var str = Core.lang.getStrings("ReadMark");

var L = function (key) {
	if (str.hasOwnProperty(key)) {
		return str[key];
	} else {
		return "ReadMark." + key;
	}
};

// This addon
var ReadMark = {
	titles: [L("TITLE_UNREAD"), L("TITLE_READ")],
	readMarkNode: null /*,
	actions: [{
		name: "saveAlreadyRead",
		title: L("ACTION_SAVE"),
		group: "Utils",
		icon: "READ_BOOK",
		action: function () {
			this.saveToFile();
			Core.ui.showMsg(["Already read books saved"]);
		}
	}]
*/
};

ReadMark.bookNode = getSoValue("FskCache.tree.bookNode");
ReadMark.bookNode._mykind = function() {
	return getFastBookMedia(this)._read ? NodeKinds.READ_BOOK : NodeKinds.BOOK;
};

ReadMark.readMark = function (book) {
	var media = getFastBookMedia(book);
	media._read = !media._read;
	this.readMarkNode.name = this.titles[Number(media._read)];
};

ReadMark.onChangeBook = function (book) {
	var utilities = book.nodes[book.nodes.length - 1]; // Utilities node (must be last)
	if (typeof utilities.isReadMarkNode === "undefined") {
		// Insert ReadMark node into utilities
		utilities.nodes.splice(utilities.nodes.length - 1, 0, this.readMarkNode);
		utilities.isReadMarkNode = true;
	}
	// Temporarily change parent of ReadMark node
	this.readMarkNode.parent = utilities;
	this.readMarkNode.name = this.titles[getFastBookMedia(book)._read? 1 : 0];
};

ReadMark.saveToFile = function () {
	try {
		FileSystem.ensureDirectory(Core.config.settingsRoot);
		var listFile = Core.config.settingsRoot + RM_FILE;
		var records = kbook.model.cache.textMasters;
		// Create current AlreadyRead list
		var readBooks = "";
		for (var i = 0, n = records.count(); i < n; i++) {
			var media = records.getRecord(i);
			if (media._read) {
				readBooks += media.source.path + media.path + "\r\n";
			}
		}
		if (readBooks.length == 0) {
			// List is empty - delete
			FileSystem.deleteFile(listFile);
			return;
		}
		// Load saved AlreadyRead list
		savedBooks = Core.io.getFileContent(listFile, "");
		if (savedBooks == readBooks) {
			// Lists are equal
			return;
		}
		// ...aren't equal - save
		FileSystem.deleteFile(listFile); //FIXME
		Core.io.setFileContent(listFile, readBooks);
	} catch (e) {
		log.error("Error in saveToFile(): " + e);
	}
};

// Loads list and marks already read books
ReadMark.loadFromFile = function () {
	try {
		var listFile = Core.config.settingsRoot + RM_FILE;
		if (FileSystem.getFileInfo(listFile)) {
			var stream = new Stream.File(listFile);
			try {
				_bf_indexBooks(); //FIXME
				while (stream.bytesAvailable) {
					var path = stream.readLine();
					if (FileSystem.getFileInfo(path)) {
						var media = _bf_pathToBook(path); //FIXME
						if (media) {
							media._read = true;
						}
					}
				}
			} finally {
				stream.close();
			}
		}
	} catch (e) {
		log.error("Error in loadFromFile(): " + e);
	}
};

ReadMark.onInit = function () {
	var node = cloneObj(kbook.children.utilities.children.deleteBook);
	node.cache = kbook.model.cache;
	node.parent = null;
	FskCache.tree.xdbNode.construct.call(node);
	node.name = this.titles[0];
	node.title = node.name;
	node.kind = NodeKinds.CROSSED_BOX;
	node.nodes = [];
	node.onEnter = null;
	node.enter = function () {
		try {
			ReadMark.readMark(kbook.model.currentBook);
		} catch (e) {
			log.error("error in readMark: " + e);
		}
	};
	this.readMarkNode = node;

	this.loadFromFile();
};

/*** Public definitions ***/

// Accelerates media access
getFastBookMedia = function (book)  {
	if (book._myclass) {
		return book.media;
	}
	if (typeof book._media === "undefined") {
		book._media = getSoValue(book, "media");
	}
	return book._media;
};

//FIXME Handlers export...

_rm_onChangeBook = function (n) {
	ReadMark.onChangeBook(n);
};

_rm_saveToFile = function () {
	ReadMark.saveToFile();
};

return ReadMark;
