import struct, sys, binascii, StringIO, array

class FSKException(Exception):
    def __init__(self,msg):
        self.msg=msg
    def __str__(self):
        return repr(self.msg)

def getByte(f):
    return struct.unpack("<B",f.read(1))[0];

def getWord(f):
    return struct.unpack(">H",f.read(2))[0];

def getSWord(f):
    return struct.unpack(">h",f.read(2))[0];

def getDWord(f):
    return struct.unpack(">I",f.read(4))[0];

def getSDWord(f):
    return struct.unpack(">i",f.read(4))[0];

def getQWord(f):
    return struct.unpack(">Q",f.read(8))[0];

def getDouble(f):
    return struct.unpack(">d",f.read(8))[0];

def getFourcc(f):
    return f.read(4)

def getString(f):
    s = ""
    while True:
        c = f.read(1)
        if c=='\x00': break
        s += c
    return s

def strSByte(s,off=0):
    return struct.unpack(">b",s[off])[0];

def strSWord(s,off=0):
    return struct.unpack(">h",s[off:off+2])[0];

def strSDWord(s,off=0):
    return struct.unpack(">i",s[off:off+4])[0];

def strDWord(s,off=0):
    return struct.unpack(">L",s[off:off+4])[0];

def strDouble(s,off=0):
    return struct.unpack(">d",s[off:off+8])[0];

def strString(s,off=0):
    res = ""
    while True:
        c = s[off]
        if c=='\x00': break
        res += c
        off += 1
    return res

#   23=8C,2e=8D,34=8C,42=8F,55=90,64=91 -> ident
opcodeSynonyms = {0x8C:0x23,0x8D:0x2e,0x8C:0x34,0x8F:0x42,0x90:0x55,0x91:0x64}
opcodeTbl = {
  #opcode: (name, args size, args types)
  #i: identifier, r: relative offset, f: float, s: asciiz string, "l": list
  0x20: ("eof", 0),
  0x21: ("add", 0),
  0x24: ("and", 0),
  0x25: ("inv", 0),
  0x26: ("or", 0),
  0x27: ("xor", 0),
  0x31: ("dec", 0),
  0x36: ("div", 0),
  0x37: ("dup", 0),
  0x38: ("endscope", 0),
  0x39: ("enum", 0),
  0x3A: ("eq", 0),
  0x3C: ("false", 0),
  0x44: ("in", 0),
  0x45: ("inc", 0),
  0x46: ("instanceof", 0),
  0x47: ("newinstanceof", 0),
  0x49: ("endfinally", 0),
  0x4a: ("shl", 0),
  0x4b: ("less", 0),
  0x4c: ("leq", 0),
  0x4E: ("unaryminus", 0),
  0x4f: ("mod", 0),
  0x50: ("gr", 0),
  0x51: ("greq", 0),
  0x52: ("mul", 0),
  0x57: ("not", 0),
  0x58: ("neq", 0),
  0x59: ("null", 0),
  0x5A: ("arguments", 0),
  0x5B: ("unaryplus", 0),
  0x5C: ("pop", 0),
  0x5D: ("global", 0),
  0x5F: ("return", 0),
  0x66: ("shr", 0),
  0x68: ("fulleq", 0),
  0x69: ("fullneq", 0),
  0x6b: ("sub", 0),
  0x6C: ("swap", 0),
  0x6D: ("this", 0),
  0x6E: ("throw", 0),
  0x73: ("true", 0),
  0x74: ("typeof", 0),
  0x75: ("endcatch", 0),
  0x76: ("undefined", 0),
  0x78: ("sar", 0),
  0x7a: ("with", 0),
  0x7b: ("nop", 0),
  0x7d: ("makeref", 0),
  0x80: ("return", 0),
  0x81: ("void", 0),
  0x92: ("enc_func",0),
  0x94: ("dec_func",0),
  0x95: ("enc_check1",0x10),
  0x96: ("enc_script",0),
  0x97: ("switch_int",1),
  0x98: ("switch_str",1),

  0x22: ("args", 1, "l"),
  0x9c: ("sec_args", 1, "l"),
  0x2d: ("", 1),

  #deletes
  0x32: ("delete ident", 2, "i"),
  0x33: ("delete arg", 1),
  0x34: ("delete prop", 2, "i"),
  0x35: ("delete []", 0),

  #call expression
  0x2c: ("call ident", 2, "i"),
  0x2d: ("call arg", 1),
  0x2e: ("call prop", 2, "i"),
  0x2F: ("call []", 0),              #array, integer

  0x40: ("get ident", 2, "i"),
  0x41: ("get arg", 1),
  0x42: ("get prop", 2, "i"),
  0x43: ("get []", 0),              #array, integer

  #left hand side
  0x53: ("construct ident", 2, "i"),
  0x54: ("construct arg", 1),
  0x55: ("construct prop", 2, "i"),
  0x56: ("construct []", 0),              #array, integer

  #postfix
  0x62: ("set ident", 2, "i"),
  0x63: ("set arg", 1),
  0x64: ("set prop", 2, "i"),
  0x65: ("set []", 0),              #array, integer

  0x79: ("locals", 1, "l"),
  0x89: ("integerB", 1),

  0x23: ("fstart", 2, "i"),
  0x28: ("jmp", 2, "r"),
  0x29: ("je", 2, "r"),
  0x2a: ("jne", 2, "r"),
  0x30: ("catch", 2, "i"),
  0x3d: ("filename", 2, "i"),
  0x3f: ("function", 2, "r"),
  0x4d: ("line", 2),
  0x60: ("startscope", 2, "r"),
  0x67: ("catchblock", 2, "r"),

  0x82: ("xmlAttribute", 1, "c"),
  0x83: ("xmlElementData", 1, "c"),
  0x84: ("chunk84", 1, "c"),
  0x85: ("chunk85", 1, "c"),
  0x86: ("xmlElementOptional", 1, "c"),
  0x87: ("xmlElementRequired", 1, "c"),
  0x88: ("xmlElementZeroOrMore", 1, "c"),

  0x8a: ("integerW", 2),

  0x48: ("integerD", 4),
  0x7C: ("putID", 4, "p"),
  0x93: ("enc_check2", 4),

  0x5e: ("number", 8, "f"),

  0x6a: ("string", 0, "s"),
};

class Instruction:
    def __init__(self, f, getident):
        self.offset = f.tell()
        opcode = getByte(f)
        if opcode in opcodeSynonyms:
            opcode = opcodeSynonyms[opcode]
        self.opcode = opcode
        if not opcode in opcodeTbl:
            #raise FSKException("Unknown opcode: %x at offset %04X"%(opcode, self.offset))
            print "Warning: unknown opcode: %x at offset %04X"%(opcode, self.offset)
            info = ("", 0)
        else:
            info = opcodeTbl[opcode]
        self.name = info[0]
        self.type = "g" #general
        self.raw = f.read(info[1])
        if info[1]==0:
            self.arg = ""
        elif info[1]==1:
            self.arg = strSByte(self.raw)
        elif info[1]==2:
            self.arg = strSWord(self.raw)
        elif info[1]==4:
            self.arg = strSDWord(self.raw)
        elif info[1]==8:
            self.arg = strDouble(self.raw)
        else:
            self.arg = ""
        #    raise FSKException("Unknown argument type %s for opcode %02X at offset %04X"%(str(info[1]), opcode, self.offset))
        if opcode == 0x92:
            self.raw = f.read(0xa)
            add = strSWord(self.raw,4)+8*ord(self.raw[8])
            self.raw += f.read(add)
        elif opcode == 0x94:
            self.raw = f.read(0xa)
            add = 8*ord(self.raw[8])
            self.raw += f.read(add)

        if opcode==0x97:
            cnt = self.arg
            #print hex(cnt)
            self.arg = []
            while cnt>0:
                w = f.read(2)
                self.raw += w
                w = strSWord(w)
                #print w
                self.arg.append(w)
                cnt-=1
        
        if opcode==0x98 and self.arg==7:
            w = f.read(2)
            self.raw += w
            cnt = strSWord(w)
            #print hex(cnt)
            self.arg = []
            #self.type = "l"
            while cnt>0:
                w = f.read(2)
                self.raw += w
                self.arg.append(getident(strSWord(w)))
                w = f.read(2)
                self.raw += w
                self.arg.append(strSWord(w))
                cnt -= 1

        if len(info)>2:
            if info[2]=="i":
                self.type = "i"
                self.ident = getident(self.arg)
            elif info[2]=="r":
                self.type = "r"
            elif info[2]=="s":
                self.type = "s"
                self.arg = getString(f)
                self.raw = self.arg+'\0'
            elif info[2]=="p":
                self.type = "p"
                self.arg = [getident(strSWord(self.raw)), strSByte(self.raw[2]), strSByte(self.raw[3])]
            elif info[2]=="l":
                self.type="l"
                cnt = self.arg
                self.arg = []
                while cnt>0:
                    w = f.read(2)
                    self.raw += w
                    self.arg.append(getident(strSWord(w)))
                    cnt -= 1
            elif info[2]=="c":
                self.type="l"
                cnt = self.arg
                self.arg = []
                while cnt>0:
                    w = f.read(2)
                    self.raw += w
                    self.arg.append(getident(strSWord(w)))
                    w = f.read(2)
                    self.raw += w
                    self.arg.append(getident(strSWord(w)))
                    cnt -= 1
                #print "opcode %02X, args:"%opcode, self.arg

        self.size = f.tell()-self.offset

    def __str__(self):
        s = "%04X: %02X " % (self.offset, self.opcode)
        s+= binascii.hexlify(self.raw)
        if len(s)>20:
            s = s[:17]+"..."
        else:
            s = s.ljust(20)
        s += " ; "
        if len(self.name):
            s += " %s"%self.name
        else:
            s += " ??"
        if self.type=="g":
            s+= ' %s'%str(self.arg)
        elif self.type=="i":
            s+= ' %s'%str(self.ident)
        elif self.type=="r":
            s+= ' %04X'%(self.offset+self.size+self.arg)
        elif self.type=="s":
            s+= " %s"%repr(self.arg)
        elif self.type=="l":
            s+= " ("+", ".join(self.arg)+")"
        elif self.type=="p":
            s+= " ("+self.arg[0]+', '+str(self.arg[1])+', '+str(self.arg[2])+")"
        return s

class FskParser:
    def __init__(self,filename,filename2=None):
        if filename2:
            self.symbols = file(filename,"rb").read()
            self.code = file(filename2,"rb").read()
        else:
            self.filename = filename
            if filename[-4:].lower() == ".xsb":
                self.parseXsb()
            elif filename[-4:].lower() == ".dll":
                self.parseDll()
            elif filename[-3:].lower() == ".so":
                self.parseElf()
            else:
                raise FSKException("Don't know how to parse file %s"%filename)

        self.parseCode()

    def parseDll(self):
        """from ctypes import Structure, c_void_p, c_char, c_int, windll, POINTER
        class struct_grammar(Structure):
             _fields_ = [("pExecProc", c_void_p),
                         ("pSymbBlock", POINTER(c_char)),
                         ("nSymbSize", c_int),
                         ("pCodeBlock", POINTER(c_char)),
                         ("nCodeSize", c_int),
                         ]
        dll = windll[self.filename]
        grammar = POINTER(struct_grammar).in_dll(dll,"fskGrammar")[0]
        #print "Dll symbols: %d, code: %d"%(grammar.nSymbSize, grammar.nCodeSize)
        self.symbols = grammar.pSymbBlock[:grammar.nSymbSize]
        self.code = grammar.pCodeBlock[:grammar.nCodeSize]"""
        def getDwordRVA(pe, rva):
            return struct.unpack('<L',pe.get_data(rva,4))[0]
        def getVABytes(pe, va, size):
            return pe.get_data(va - pe.OPTIONAL_HEADER.ImageBase, size)

        from pefile import PE
        pe = PE(self.filename)
        g = [i.address for i in pe.DIRECTORY_ENTRY_EXPORT.symbols if i.name=='fskGrammar']
        if len(g):
            ptr = getDwordRVA(pe, g[0])
            #print hex(ptr)
            g = struct.unpack("5L",getVABytes(pe, ptr, 20))
            if g[1]==0 or g[3]==0:
                raise FSKException("Grammar doesn't define bytecode")
            self.symbols = getVABytes(pe, g[1],g[2])
            self.code = getVABytes(pe, g[3],g[4])

    def parseElf(self):
        from Elf import Elf
        elf = Elf(file(self.filename,"rb").read())
        g = [i for i in elf.dynamic_symbols if i.name=='fskGrammar']
        if len(g):
            g = g[0]
            d = elf.sections[g.st_shndx]
            ptr = d.getDword(g.st_value)
            #print "grammar address:",hex(ptr)
            g = struct.unpack("5L",d.getVABytes(ptr,20))
            if g[1]==0 or g[3]==0:
                raise FSKException("Grammar doesn't define bytecode")
            self.symbols = d.getVABytes(g[1],g[2])
            self.code = d.getVABytes(g[3],g[4])
            #file("t.sym","wb").write(sym)
            #file("t.code","wb").write(code)
        else:
            raise FSKException("%s doesn't export fskGrammar"%self.filename)


    def parseXsb(self):
        f = file(self.filename,"rb")
        try:
            xsl = getDWord(f)
            if getFourcc(f)!='XS11':
                raise FSKException("Bad file format")
            while f.tell()<xsl:
                off = f.tell()
                l = getDWord(f)
                t = getFourcc(f)
                #print "%s(%d)"%(t,l)
                if t=='SYMB':
                    self.symOffset = f.tell()
                    self.symbols = f.read(l-8)
                elif t=='CODE':
                    self.codeOffset = f.tell()
                    self.code = f.read(l-8)
                f.seek(off+l)
            #print syms
        finally:
            f.close()

    def getSym(self,idx):
        if idx==-1:
            return "-1"
        idx = idx&0x7FFF
        if idx<len(self.syms):
            return self.syms[idx]
        else:
            raise FSKException("Bad symref: %x"%idx)

    def parseCode(self):
        self.syms = []

        #f = StringIO.StringIO(self.symbols)
        c = strSWord(self.symbols, 0)
        #print "%d symbols"%c
        off = 2
        for i in xrange(c):
            sym = strString(self.symbols, off)
            #print "%d: %s"%(i, sym)
            self.syms.append(sym)
            off += len(sym)+1

        f = StringIO.StringIO(self.code)
        self.off2ins = {}
        self.inslist = []
        n = 0
        off = 0
        while off<len(self.code):
            f.seek(off)
            opcode = f.read(1)
            #print "%04X, %02X"%(off, ord(opcode))
            f.seek(off)
            if opcode=='\x92':
                #print "encrypted func at %X"%off
                encsize = strSWord(self.code, off+5)
                enctype = strSWord(self.code, off+7)
                extra = ord(self.code[off+9])
                dataoff = off+0xa
                encdata = array.array('B',self.code[dataoff+1+extra*8:dataoff+1+extra*8+encsize])
                if extra!=0:
                    found = False
                    j = 0
                    while j<extra:
                        m1 = self.code[dataoff:dataoff+4]
                        m2 = self.code[dataoff+4:dataoff+8]
                        if not found and m1=='\xFC\xA1\x3A\x61':
                            found = m2
                            #print "match: %r"%m2
                        dataoff+=8
                        j += 1
                    dataoff+=1
                    #print "Dataoff: %04X"%dataoff
                    if found:
                        k93 = strDWord(self.code,dataoff+encsize+0xc)
                        #print "k93: 0x%08X"%k93
                        key2 = k93^ (((encsize&~0xA0)<<0x10 | (encsize&0x1FE7))<<3)
                        #print "k2: 0x%08X"%key2
                        key4 = key2^0x031B11A7
                        k7 = (key4&0xFFFF)*0x55C2+(key4>>0x10)
                        k9 = (k7&0xFFFF)<<0x10 | 0x412D
                        #print "k9, m2: 0x%08X, 0x%08X"%(k9,strDWord(found))
                        k10 = k9^strDWord(found)
                        #print "k10: 0x%08X"%k10
                        key = [(k10>>0x18)&0xFF, (k10>>0x10)&0xFF, (k10>>8)&0xFF, k10&0xFF]
                        #print "decrypting 0x%x bytes with key 0x%08X"%(encsize, k10)
                        j = 0
                        while j<encsize:
                            #print "%02X^%02X = %02X"%(encdata[j],key[j%4],encdata[j]^key[j%4])
                            encdata[j]^=key[j%4]
                            #b = ord(self.code[dataoff+j])
                            #b^=key[j%4]
                            #f.write(chr(b))
                            j+=1
                    else:
                        print "Can't find known key for extra!=0"
                        f.seek(off)
                else:
                    dataoff+=1

                if self.code[dataoff-1]!='\0':
                    #print "Extra2!=0 !"
                    partlen = (encsize+3)/4
                    j=0
                    while j<partlen:
                        encdata[j]^=5
                        encdata[j+partlen]^=0xC
                        encdata[j+partlen*2]^=0xC0
                        if j+partlen*3<encsize:
                            encdata[j+partlen*3]^=0x50
                        j+=1
                    #f.seek(off)
                if enctype==0x111:
                    encsize -= 4
                    #f.seek(dataoff+encsize)
                    key = [encdata[encsize+j] for j in xrange(4)]
                    #print "decrypting 0x%x bytes with key %02X%02X%02X%02X"%(encsize, key[0], key[1], key[2], key[3])
                    j = 0
                    #print "Dataoff: %04X"%dataoff
                    #f.seek(dataoff)
                    while j<encsize:
                        #b = ord(f.read(1))
                        #print "%02X^%02X = %02X"%(encdata[j],key[j%4],encdata[j]^key[j%4])
                        #b^=key[j%4]
                        #f.seek(dataoff+j)
                        #f.write(chr(b))
                        encdata[j]^=key[j%4]
                        j+=1
                    f.seek(off)
                    f.write('\x94')
                    f.seek(off)
                else:
                    print "Unknown enctype!"

                f.seek(dataoff)
                f.write(encdata.tostring())
                f.seek(off)

            i = Instruction(f, self.getSym)
            i.num = n
            #print i
            self.off2ins[i.offset] = i
            self.inslist.append(i)
            n+=1
            off = f.tell()

    def dumpBytecode(self):
        return "\n".join(str(i) for i in self.inslist)

    def nextIns(self, i):
        return self.off2ins[i.offset+i.size]

    def prevIns(self, i):
        return self.inslist[i.num-1]

    def prevInsOff(self, off):
        return self.prevIns(self.off2ins[off])

    def getTarget(self, i):
        return i.offset+i.size+i.arg

    def getname(self,stack,i, args, locs):
        if i.opcode in [0x32,0x2c,0x40,0x62,0x53]:
            return i.ident
        elif i.opcode in [0x33,0x2d,0x41,0x54,0x63]:
            if i.arg>=0:
                return args[i.arg]
            else:
                return locs[-i.arg-1]
        elif i.opcode in [0x34,0x2e,0x42,0x55,0x64]:
            return stack.pop()+"."+i.ident
        elif i.opcode in [0x35,0x2f,0x43,0x56,0x65]:
            idx = stack.pop()
            return stack.pop()+"["+str(idx)+"]"
        else:
            raise FSKException("Bad opcode for getname: %02X!"%i.opcode);

    def obj2sting(self, obj):
        #print "obj:",obj
        m = obj["_ctr"]
        if m.endswith("Object()"):
            del obj["_ctr"]
            if len(obj):
                return "{ "+", ".join("%s: %s"%(p,v) for p,v in obj.items)+"}";
            else:
                return "{ }"
        else:
            return "new "+m

    def getargs(self, stack, objects):
        cnt = stack.pop()
        args = []
        while cnt>0:
            if len(stack):
                a = str(stack.pop())
            else:
                a = "<dummy>"
            a = self.var2sting(a, objects)
            args.append(a)
            cnt-=1
        return ", ".join(args)

    def arr2sting(self, arr):
        m = arr["_maxindex"]
        if m==0: return "new Array()"
        s = "[ "
        for i in xrange(m-1):
            if i in arr: s+=str(arr[i])+", "
        if m-1 in arr:
            s+=str(arr[m-1])+" "
        s+= "]"
        return s

    def var2sting(self, var, objects):
        if var[:4]=='<obj':
            pos = var.index('>',4)
            p = int(var[4:pos])
            if p in objects:
                return self.obj2sting(objects[p])+var[pos+1:]
        elif var[:4]=='<arr':
            pos = var.index('>',4)
            p = int(var[4:pos])
            if p in objects:
                return self.arr2sting(objects[p])+var[pos+1:]
        return var

    binaryops = {0x21: "+", 0x6B: "-", 0x3A: "==", 0x58: "!=",
                 0x4B: "<", 0x50: ">", 0x51: ">=", 0x24: "&",
                 0x78: ">>>", 0x4a: "<<", 0x66: ">>", 0x27: "^",
                 0x26: "|", 0x36: "/", 0x4f: "%", 0x44: "in",
                 0x52: "*", 0x68: "===", 0x69: "!==", 0x4C: "<=",
                 0x46: "instanceof"}
    constants = {0x6D: "this", 0x3C: "false", 0x73: "true", 0x5D: "<global>",
                 0x59: "null", 0x76: "undefined", 0x5A: "arguments"}

    def decompileBlock(self, start, end, args, locs, inSwitch=False, stack1 = None):
        off = start
        lines = []
        stack = []
        if stack1: stack=stack1
        objects = {}
        objcnt = 0
        #print "in decompileBlock(%04X-%04X)"%(start,end),"stack:",stack
        while off<end:
            try:
                i = self.off2ins[off]
            except:
                print "in decompileBlock(%04X-%04X)"%(start,end),"stack:",stack
                print "offset: %04X"%off
		#kartu - 
		off = off + 2
		continue
                #raise
            #print "%04X: %02X %r %r"%(off, i.opcode, stack[-10:], lines[-10:])
            if i.opcode in [0x93, 0x94]:
                off += i.size
                continue
            if i.opcode in [0x5e,0x48,0x8a,0x89]:
                stack.append(i.arg)
            elif i.opcode==0x6A:
                stack.append(repr(i.arg))
            elif i.opcode in self.constants:
                stack.append(self.constants[i.opcode])
            elif i.opcode in xrange(0x82,0x88+1):
                stack[-1] = opcodeTbl[i.opcode][0]+"<"+stack[-1]+", ("+", ".join(i.arg)+")>"
            elif i.opcode==0x37:
                if len(stack):
                    stack.append(stack[-1])
                else:
                    print "Stack underflow!"
                    stack = ['this','this']
            elif i.opcode==0x6C:
                a = stack[-2]
                stack[-2] = stack[-1]
                stack[-1] = a
            elif i.opcode==0x39:
                stack[-1] = "<enum>(%s)"%str(stack[-1])
            elif i.opcode==0x69: #!==
                inext = self.off2ins[off+i.size]
                if inSwitch:
                    #return the case label, the target address and the address of next instruction after jump
                    return (stack.pop(), self.getTarget(inext), inext.offset+inext.size)
                #if we have var, var, val in the stack and je next, this is probably a switch block
                if len(stack)>2 and stack[-2]==stack[-3] and inext.opcode==0x29:
                    #let's parse the switch block
                    #print "got switch"
                    #add first label
                    cases = [(stack.pop(), self.getTarget(inext))]
                    #print "%r -> %04X"%cases[0]
                    inext = self.off2ins[inext.offset+inext.size]
                    while True:
                        #collect all labels
                        #if next instruction is "dup", it's another case label
                        #if next instruction is jmp, it's jump to the default label
                        if inext.opcode == 0x37:
                            label = self.decompileBlock(inext.offset+inext.size, end, args, locs, True)
                            #print "%r -> %04X-%04X"%label
                            inext = self.off2ins[label[2]]
                            cases.append(label)
                        elif inext.opcode == 0x28:
                            label = ("<default>", self.getTarget(inext))
                            #print "%r -> %04X"%label
                            cases.append(label)
                            break
                        else:
                            print "Bad switch structure 1!"
                            return lines
                    #now go over all labels and output the blocks
                    switchvar = stack.pop()
                    stack.pop() #remove extra copy
                    lines.append("switch (%s) {"%switchvar)
                    t = 0
                    while t<len(cases):
                        case = cases[t]
                        if case[0]=="<default>":
                            #jmp just before the default case goes to the end of switch
                            previns = self.prevInsOff(case[1])
                            if previns.opcode == 0x28:
                                caseend = self.getTarget(previns)
                            elif previns.opcode in [0x5F, 0x6E]: #return or throw
                                caseend = end
                            else:
                                print "Bad switch structure 2!"
                                return lines
                            #print "default section: %04X-%04X"%(case[1], caseend)
                            if case[1]<caseend:
                                lines.append("default:")
                                lines.extend("\t"+str(k) for k in self.decompileBlock(case[1], caseend, args, locs, True))
                            break
                        lines.append("case %s:"%str(case[0]))
                        while t<len(cases)-1 and case[1]==cases[t+1][1]:
                            t+=1
                            #print "same as %s"%str(cases[t][0])
                            lines.append("case %s:"%str(cases[t][0]))
                        if t<len(cases)-1: caseend = cases[t+1][1]
                        #print "%s section: %04X-%04X"%(case[0], case[1], caseend)
                        lines.extend("\t"+str(k) for k in self.decompileBlock(case[1], caseend, args, locs, True))
                        t+=1
                    lines.append("}")
                    off = caseend
                else:
                    a = str(stack.pop())
                    b = str(stack.pop())
                    stack.append("%s !== %s"%(b,a))
            elif i.opcode in [0x97,0x98]: # switch
                inext = self.off2ins[off+i.size]
                #if we have var, var, val in the stack and je next, this is probably a switch block
                #print "got switch 97/98"
                if inext.opcode != 0x28:
                    print "97/98 should be followed by jmp!"
                    return lines
                cases = []
                if i.opcode==0x97:
                    for j in xrange(len(i.arg)):
                        casestart = off+4+2*j+i.arg[j]
                        if j<len(i.arg)-1:
                            caseend = off+6+2*j+i.arg[j+1]
                        else:
                            caseend = self.getTarget(inext)
                        #print "case %d: %04X-%04X"%(j, casestart, caseend)
                        cases.append((str(j), casestart, caseend))
                else:
                    tg = off+6
                    j = 1
                    while j<len(i.arg):
                        casestart = tg+2+i.arg[j]
                        if j<len(i.arg)-2:
                            caseend = tg+6+i.arg[j+2]
                        else:
                            caseend = self.getTarget(inext)
                        #print "case %d: %04X-%04X"%(j, casestart, caseend)
                        cases.append((i.arg[j-1], casestart, caseend))
                        tg += 4
                        j += 2
                cases.append(("<default>", self.getTarget(inext)))
                if True: #len(stack)>2 and stack[-2]==stack[-3] and inext.opcode==0x29:
                    #now go over all labels and output the blocks
                    switchvar = stack.pop()
                    stack.pop() #remove extra copy
                    lines.append("switch (%s) {"%switchvar)
                    t = 0
                    while t<len(cases):
                        case = cases[t]
                        if case[0]=="<default>":
                            #jmp just before the default case goes to the end of switch
                            previns = self.prevInsOff(case[1])
                            if previns.opcode == 0x28:
                                caseend = self.getTarget(previns)
                            elif previns.opcode in [0x5F, 0x6E]: #return or throw
                                caseend = end
                            else:
                                print "Bad switch structure 2!"
                                return lines
                            #print "default section: %04X-%04X"%(case[1], caseend)
                            if case[1]<caseend:
                                lines.append("default:")
                                lines.extend("\t"+str(k) for k in self.decompileBlock(case[1], caseend, args, locs, True))
                            break
                        lines.append("case %s:"%str(case[0]))
                        while t<len(cases)-1 and case[1]==cases[t+1][1]:
                            t+=1
                            #print "same as %s"%str(cases[t][0])
                            lines.append("case %s:"%str(cases[t][0]))
                        if t<len(cases)-1: caseend = cases[t+1][1]
                        #print "%s section: %04X-%04X"%(case[0], case[1], caseend)
                        lines.extend("\t"+str(k) for k in self.decompileBlock(case[1], caseend, args, locs, True))
                        t+=1
                    lines.append("}")
                    off = caseend
                    continue
                else:
                    a = str(stack.pop())
                    b = str(stack.pop())
                    stack.append("%s !== %s"%(b,a))
            elif i.opcode==0x28:# and off+i.size==end: #end of a case label/break/continue
                #print "cur block: %04X-%04X"%(start,end)
                #print "inSwitch:", inSwitch
                #print "target: %04X"%self.getTarget(i)
                if not inSwitch: #probably a break or continue
                    if self.getTarget(i)<off:
                        lines.append("continue;")
                    else:
                        lines.append("break;")
                break
            elif i.opcode in self.binaryops:
                a = str(stack.pop())
                if len(stack):
                    b = str(stack.pop())
                else:
                    b = '<dummy>'
                stack.append("%s %s %s"%(b,self.binaryops[i.opcode],a))
            elif i.opcode==0x74:
                stack[-1] = "typeof %s"%stack[-1]
            elif i.opcode==0x45:
                inext = self.off2ins[off+i.size]
                if len(stack)>1 and stack[-1]==stack[-2] and inext.opcode in xrange(0x62,0x66):
                    #stack.pop()
                    stack[-1] = stack[-1]+"++"
                    off = inext.offset+inext.size
                    continue
                else:
                    stack[-1] = stack[-1]+" + 1"
            elif i.opcode==0x31:
                inext = self.off2ins[off+i.size]
                if len(stack)>1 and stack[-1]==stack[-2] and inext.opcode in xrange(0x62,0x66):
                    #stack.pop()
                    stack[-1] = stack[-1]+"--"
                    off = inext.offset+inext.size
                    continue
                else:
                    stack[-1] = stack[-1]+" - 1"
            elif i.opcode in xrange(0x32,0x36):
                stack.append("delete "+self.getname(stack,i, args, locs))
            elif i.opcode in xrange(0x2c,0x30):
                stack.append(self.getname(stack,i, args, locs)+"("+self.getargs(stack, objects)+")")
            elif i.opcode in xrange(0x53,0x57):
                a = self.getname(stack,i, args, locs)+"("+self.getargs(stack, objects)+")"
                if a.endswith("Array()"):
                    ob = '<arr%d>'%objcnt
                    objects[objcnt]={"_maxindex":0}
                    #print ob+" = new "+a
                elif not a.endswith("Object()"):
                    ob = '<obj%d>'%objcnt
                    objects[objcnt]={"_ctr":a}
                    #print ob+" = new "+a
                else:
                    ob = '<obj%d>'%objcnt
                    #objects[objcnt]={"_ctr":a}
                    lines.append('%s = new %s'%(ob,a))
                objcnt+=1
                stack.append(ob)
            elif i.opcode in xrange(0x40,0x44):
                stack.append(self.getname(stack,i, args, locs))
            elif i.opcode in xrange(0x62,0x66):
                inext = self.off2ins[off+i.size]
                if i.opcode==0x65 and inext.opcode==0x5C and str(stack[-2]).startswith("<arr"):
                    idx = stack.pop()
                    a = stack.pop()
                    b = self.var2sting(str(stack.pop()),objects)
                    pos = a.index('>',4)
                    pos = int(a[4:pos])
                    #print "arr%d[%d] <- %s"%(pos,idx,b)
                    if idx+1>objects[pos]["_maxindex"]:
                        objects[pos]["_maxindex"] = idx+1
                    objects[pos][idx]=b
                    off = inext.offset+inext.size
                    continue
                elif inext.opcode==0x5C and str(stack[-1]).startswith("<arr"):
                    a = self.getname(stack, i, args, locs)
                    b = stack.pop()
                    pos = b.index('>',4)
                    pos = int(b[4:pos])
                    #print "arr%d: %s"%(pos, str(arrays[pos]))
                    stack.append(a + " = "+self.arr2sting(objects[pos]))
                    off = inext.offset+inext.size
                    continue
                else:
                    a = self.getname(stack, i, args, locs)
                    b = stack.pop()
                    #print a,b
                    if a[:4]=='<obj':
                        pos = a.index('>',4)
                        if int(a[4:pos]) in objects:
                            #<obj0>.prop = value
                            pos = int(a[4:pos])
                            objects[pos][a[pos+2:]]=b
                            #stack.append("<dummy>")
                        else:
                            stack.append(a + " = "+str(b))
                    else:
                        b = self.var2sting(str(b), objects)
                        if b.startswith(a):
                            if b.endswith(" + 1"):
                                stack.append(a + "++")
                            else:
                                stack.append(a + " = "+b)
                        else:
                            stack.append(a + " = "+b)
            elif i.opcode==0x47:
                a = stack.pop()
                ob = '<obj%d>'%objcnt
                objects[objcnt]={"_ctr":a}
                #print ob+" = "+a
                lines.append('%s = %s'%(ob,a))
                objcnt+=1
                stack.append(ob)
            elif i.opcode in [0x20]:
                pass
            elif i.opcode in [0x5C, 0x38]:
                if len(stack):
                    a = str(stack.pop())
                    a = self.var2sting(a, objects)
                    if len(stack) and (a[-2:]=='++' or a[-2]=='--') and (a[:-2]==stack[-1]):
                        stack[-1]=a
                    else:
                        if a!='<dummy>': lines.append(a+";")
            elif i.opcode==0x80:
                if len(stack):
                    a = self.var2sting(str(stack.pop()),objects)
                else:
                    a = "<bad>"
                lines.append("return "+a+";")
            elif i.opcode==0x5F:
                if len(lines)==0 or lines[-1][:6]!="return":
                    lines.append("return;")
            elif i.opcode==0x57:
                stack[-1]="!"+stack[-1]
            elif i.opcode==0x25:
                stack[-1]="~"+str(stack[-1])
            elif i.opcode==0x4E:
                stack[-1]="-"+str(stack[-1])
            elif i.opcode==0x6E:
                a = str(stack.pop())
                a = self.var2sting(a, objects)
                stack.append("throw "+a+";")
            elif i.opcode in [0x7D,0x7B]:
                #stack.append(stack[-1])
                pass
            elif i.opcode==0x7C:
                #print stack.pop(),".",i.arg[0],stack.pop()#+"// %d,%d"%i.arg[1:2]
                a = i.arg[0]      #property name
                b = stack.pop()   #object
                c = stack[-1]     #new property value
                #print b+"."+a+" <- "+c
                c = self.var2sting(c, objects)
                lines.append(b+"."+a+" = "+c)
                stack[-1]=b+"."+a
                inext = self.off2ins[off+i.size]
                if inext.opcode==0x5C:
                    stack.pop();
                    lines[-1]+=";"
                    off+=1
            elif i.opcode==0x3D:
                lines.append("//filename: %s"%i.ident)
            elif i.opcode==0x4D:
                lines.append("//line: %d"%i.arg)
            elif i.opcode==0x3F:
                fend = self.getTarget(i);
                stack.append("\n\t".join(self.decompileFunc(off+i.size)))
                off = fend
                continue
            elif i.opcode in [0x29,0x2A]:
                ifend = self.getTarget(i);
                cond = stack.pop()
                insend = self.off2ins[ifend]
                #print "if end ins:", insend
                while (insend.opcode in [0x29,0x2A]):# and len(stack)>1:# and (stack[-2]==stack[-1]):
                    #print "or/and condition: %04X-%04X"%(off+i.size, ifend)
                    stack.pop()
                    cond2 = self.decompileBlock(off+i.size+1, ifend, args, locs)
                    if i.opcode==0x29:
                        cond += " && " + cond2[0]
                    else:
                        cond += " || " + cond2[0]
                    i = insend
                    off = ifend
                    ifend = self.getTarget(i);
                    insend = self.off2ins[ifend]

                if ifend<off:
                    print "do-while: not implemented yet!!"
                    lines.append("while (%s)"%cond)
                    off+=i.size
                    continue

                iprevend = self.prevInsOff(ifend)
                #print "if end prev instr:", iprevend
                elend = 0
                if iprevend.opcode == 0x28:
                    if iprevend.arg<0:
                        #print "got while"
                        ifend = iprevend.offset
                    else:
                        #print "got else"
                        elend = self.getTarget(iprevend);
                        ifend = iprevend.offset
                #print "if/while block: %04X-%04X"%(off+i.size, ifend)
                ifb = self.decompileBlock(off+i.size, ifend, args, locs)
                off = ifend
                if iprevend.arg<0:
                    off = iprevend.offset+iprevend.size
                if elend:
                    #print "else block: %04X-%04X"%(off+i.size, elend)
                    elb = self.decompileBlock(off+i.size, elend, args, locs)
                    off = elend
                if elend and len(ifb)==1 and len(elb)==1 and ifb[0][-1]!=";" and elb[0][-1]!=";": #and ifb[0].find(" = ")==-1 and elb[0].find(" = ")==-1:
                    stack.append("(%s)?%s:%s"%(cond,ifb[0],elb[0]))
                else:
                    if iprevend.arg<0:
                        lines.append("while ("+cond+") {")
                    else:
                        lines.append("if ("+cond+") {")
                    lines.extend("\t"+k for k in ifb)
                    lines.append("}")
                    if elend:
                        lines.append("else {")
                        lines.extend("\t"+k for k in elb)
                        lines.append("}")
                continue
            elif i.opcode==0x60:
                bend = self.getTarget(i);
                if self.off2ins[bend].opcode in [0x67,0x60]: #catchblock, beginscope
                    lines.append("try {")
                    lines.extend("\t"+k for k in self.decompileBlock(off+i.size, bend, args, locs))
                    lines.append("}")
                    i = self.off2ins[bend]        #cathblock
                    i = self.off2ins[i.offset+i.size] #catch e
                    if i.opcode!=0x30:
                        #print "Bad catch 1!"
                        off = i.offset
                    else:
                        lines.append("catch (%s)"%i.ident)
                        i = self.off2ins[i.offset+i.size]
                        off = i.offset
                        if i.opcode!=0x60:
                            print "Bad catch 2!"
                            continue
                        bend = self.getTarget(i);
                if self.off2ins[bend].opcode==0x75: #endcatch
                    lines.append("{")
                    lines.extend("\t"+k for k in self.decompileBlock(off+i.size, bend, args, locs))
                    lines.append("}")
                    off = bend+1
                    i = self.off2ins[off]
                if i.opcode==0x60:
                    if self.off2ins[off+i.size].opcode==0x49:
                        #empty finally
                        off += i.size+1
                    else:
                        off1 = off+i.size
                        while off1<end and self.off2ins[off1].opcode!=0x49:
                            off1+=self.off2ins[off1].size
                        lines.append("finally {")
                        lines.extend("\t"+str(k) for k in self.decompileBlock(off+i.size, off1, args, locs))
                        lines.append("}")
                        off = off1+1
                else:
                    off = bend
                continue
            else:
		#kartu		
                print "Unknown opcode:", i
		break
                print "last stack:\n"+"\n".join(stack[-10:])
                print "last lines:\n"+"\n".join(lines[-10:])
                break
            off += i.size
        #print "about to leave decompileBlock(%04X-%04X)"%(start,end),"stack:",stack
        if len(stack):
            a = str(stack.pop())
            a = self.var2sting(a, objects)
            lines.append(a)
        #if len(stack):
        #    a = stack.pop()
        #    if a!='<dummy>': lines.append(str(a)+";")
        return lines

    def decompileFunc(self, offset):
        i = self.off2ins[offset]
        encrypted = False
        if i.opcode in [0x96,0x92,0x94]:
            encrypted = (i.opcode==0x92)
            i = self.nextIns(i)
        if i.opcode!=0x23:
            raise FSKException("Not a function at offset %02X!"%i.offset);
        fname = i.ident
        if fname=="-1": fname=""
        i = self.nextIns(i)
        if i.opcode not in [0x22, 0x9C]:
            raise FSKException("Bad function at offset %02X!"%offset);
        fargs = i.arg
        i = self.nextIns(i)
        if i.opcode!=0x79:
            raise FSKException("Bad function at offset %02X!"%offset);
        flocals = i.arg
        i = self.nextIns(i)
        if i.opcode!=0x60:
            raise FSKException("Bad function at offset %02X!"%offset);
        lines = []
        if i.arg==3 and self.nextIns(i).opcode==0x28: #global?
            i = self.nextIns(i)
            if i.opcode!=0x28 or i.arg!=1:
                raise FSKException("Bad global function at offset %02X!"%offset);
            i = self.nextIns(i)
            if i.opcode!=0x38:
                raise FSKException("Bad global function at offset %02X!"%offset);
            funcend = self.inslist[-1].offset;
            i = self.nextIns(i)
            glob = True
        else:
            lines.append("function %s(%s) {"%(fname,", ".join(fargs)))
            if encrypted: lines.append("//*** encrypted ***")
            funcend = self.getTarget(i)
            i = self.nextIns(i)
            glob = False
        if len(flocals):
            lines.append("\tvar "+", ".join(flocals)+';')
        lines.extend("\t"+k for k in self.decompileBlock(i.offset, funcend, fargs, flocals))
        i = self.off2ins[funcend]
        if i.opcode in [0x38,0x93,0x95] and not glob:
            lines.append("}")
        return lines

    def decompileCode(self, offset=0):
        return "\n".join(self.decompileFunc(offset))

def changeext(fname, newext):
    import os.path
    if newext[0]!='.': newext='.'+newext
    path, name = os.path.split(fname)
    pos = name.rfind('.')
    if pos!=-1:
        name = name[:pos]+newext
    return os.path.join(path,name)

if len(sys.argv)<2:
    print "DeFsk 0.2 (c) 2006 igorsk"
    print "Usage: DeFsk.py file.xsb|file.dll|file.so|(file.symb file.code)"
    sys.exit(1)

fsk = FskParser(*sys.argv[1:])
infile = sys.argv[1]
file(changeext(infile,".dmp"),"w").write(fsk.dumpBytecode())
file(changeext(infile,".xs"),"w").write(fsk.decompileCode())
