

#include "config.h"
#include "xournal.h"

#include <gtk/gtk.h>
#include <memory.h>
#include <liberutils/er_error.h>
#include <libermetadb/ermetadb.h>

#define ERSCRIBBLE_VERSION_MAJOR           "scribble_version_major"
#define ERSCRIBBLE_VERSION_MINOR           "scribble_version_minor"
#define ERSCRIBBLE_VERSION_COMPANY         "scribble_version_company"
#define ERSCRIBBLE_SCREEN_DPI              "scribble_screen_dpi"
#define ERSCRIBBLE_SCREEN_UNITS            "scribble_screen_units"

#define ERSCRIBBLE_ANNOTATION_TYPE         "scribble"

#define METADB_FILE_POSITION "file_position"
#define METADB_START_ANCHOR  "start_anchor"
#define METADB_END_ANCHOR    "end_anchor"
#define METADB_ANNOTATION_ID "annotation_id"
#define METADB_DATA          "data"

#define MAX_STROKES	0

typedef struct _ScbRect
{
    int     left;
    int     top;
    int     right;
    int     bottom;
} ScbRect;

typedef enum _ScbDevColor
{
    ERSCRIBBLE_DEV_COLOR_WHITE         = 0, 
    ERSCRIBBLE_DEV_COLOR_LIGHT_GRAY    = 1,
    ERSCRIBBLE_DEV_COLOR_DARK_GRAY     = 2,
    ERSCRIBBLE_DEV_COLOR_BLACK         = 3,
    ERSCRIBBLE_DEV_COLOR_UNKNOWN       = 0xffffffff
} ScbDevColor;

typedef struct _ScbColor
{
    unsigned long   pixel;      /**< for allocated colors, the value is used to draw
                                     this color on the screen */
    unsigned short  red;        /**< value of red weight */
    unsigned short  green;      /**< value of green weight */
    unsigned short  blue;       /**< value of blue weight */
} ScbColor;

typedef ScbColor * ScbColorPtr;

typedef struct _ScbPageStyle
{
    int             orientation;           /**< Orientation of a page, e.g. 0, 90, 270 */
    ScbColor        bgColor;               /**< Background color of a page */
} ScbPageStyle;

typedef struct _ScbPageAttributes
{
    unsigned short  minor_version;    /**< minor version number of the scribble library */
    ScbPageStyle    style;            /**< page style */
    ScbRect         area;             /**< page area  */
    int             strokes_number;   /**< number of the strokes in a page */
} ScbPageAttributes;

typedef struct _ScbStrokeAttributes
{
    ScbDevColor     color;          /**< Device color of the stroke */
    float           zoom;           /**< Zoom setting of the stroke */
    int             layer;          /**< Drawint layer of the stroke,
                                    "0" means drawing on the content */
    int             shape_id;       /**< ID of the drawing shape */
    int             size_id;        /**< ID of the shape size */
    int             points_number;  /**< Number of the stroke points */
} ScbStrokeAttributes;

typedef struct _ScbPoint
{
    int      x;         /**< x weight of the point */
    int      y;         /**< y weiget of the point */
} ScbPoint;

typedef struct _ScbDevPoint
{
    ScbPoint point;     /**< coorindate point */
    int      pressure;  /**< pressure */
} ScbDevPoint;

static void annotations_doc_open_with_database(const char *target_file_name, erMetadb *metadata_db);
static gboolean annotations_load_application_data(const char *target_file_name, erMetadb *metadata_db);
static metadata_table *annotations_load_document_data(const char *target_file_name, erMetadb *metadata_db);
static void annotations_load_pages_basic_data(const metadata_table *table, erMetadb *metadata_db);
static gboolean annotations_load_page_data(erMetadb *metadata_db, gint64 annotation_id, gint64 page_id);
static gboolean annotations_strokes_load(gint64 page_id, gchar* data, gsize *length, const int strokes_num, ScbPageAttributes *page_attributes);

static Layer *g_current_layer = NULL;
static Page  *g_current_page = NULL;

static int stroke_count;

int annotations_read_document(char* target_doc_path)
{
    gchar     *cp;

	//printf("annotations_read_document: %s\n", target_doc_path);
    // create a new metadata db for the document
    erMetadb *metadata_db = ermetadb_new();
	//printf("DB:0x%x\n", metadata_db);
	if (metadata_db == NULL)
	{
		printf("FATAL: Could not open database for :%s\n", target_doc_path);
		return 0;
	}

    // get folder
    cp = g_path_get_dirname(target_doc_path);
    GString *folder = g_string_new(cp);
    g_free(cp);

	//printf("opening: %s\n", folder->str);
    if (ermetadb_open_database(metadata_db, folder) == ER_OK)
    {
		//printf("DB opened\n");
        annotations_doc_open_with_database(target_doc_path, metadata_db);
    }
	else
	{
		printf("FATAL: no metadata for folder: %s\n", folder->str);
	}

    // clean up
    g_string_free(folder, TRUE);
	//printf("annotations_read_document-leave\n");

    return 1;
}

static void annotations_doc_open_with_database(const char *target_file_name, erMetadb *metadata_db)
{
    gboolean  ok;
    gchar     *cp;

	//printf("annotations_doc_open_with_database-enter\n");
	
	cp = g_path_get_basename(target_file_name);
    ok = annotations_load_application_data(cp, metadata_db);
    if (ok)
    {
		// load the document data from data base
		annotations_load_document_data(cp, metadata_db);
    }
	else
	{
		printf("annotations_load_application_data FAILED!\n");
	}
    g_free(cp);
}


static gboolean annotations_load_application_data(const char *target_file_name, erMetadb *metadata_db)
{
	unsigned short version_number_major;                    /**< major version number */
	unsigned short version_number_minor;                    /**< minor version number */
	GString*      version_company;                   /**< company name */
	unsigned int  screen_dpi;                       /**< DPI */
	GString*      screen_units;                     /**< Units used for calculation, such as "pixel" */

    // get document basic information from application_data table
	//printf("annotations_load_application_data-enter\n");

    // 1. create an empty table with required keys
    metadata_table *name_table = metadata_table_new();

    if (name_table == NULL)
    {
        return FALSE;
    }

    // 2. set new column of the table
    metadata_table_add_column(name_table, ERSCRIBBLE_VERSION_MAJOR);
    metadata_table_add_column(name_table, ERSCRIBBLE_VERSION_MINOR);
    metadata_table_add_column(name_table, ERSCRIBBLE_VERSION_COMPANY);
    metadata_table_add_column(name_table, ERSCRIBBLE_SCREEN_DPI);
    metadata_table_add_column(name_table, ERSCRIBBLE_SCREEN_UNITS);

    // 3. get the results
    GString *file_name = g_string_new(target_file_name);
    metadata_table *results_table = NULL;

    int ret = ermetadb_get_application_data(metadata_db
        , file_name
        , name_table
        , &results_table);

    // free the name table
    metadata_table_free(name_table);

    // free the file name
    g_string_free(file_name, TRUE);

    if (results_table == NULL)
    {
		printf("FATAL: could not find metadata for file: %s\n", target_file_name);
        return FALSE;
    }

    if (ret == ER_OK)
    {
        const metadata_cell *cell = NULL;
        // get major version number
        int index = metadata_table_find_column(results_table, ERSCRIBBLE_VERSION_MAJOR);
        if (index >= 0)
        {
            cell = metadata_table_get_cell(results_table, index);
            if (cell != NULL)
            {
                sscanf(cell->value.v_text->str, "%hd", &version_number_major);

                cell = NULL;
            }
        }

        //  get minor version number
        index = metadata_table_find_column(results_table, ERSCRIBBLE_VERSION_MINOR);
        if (index >= 0)
        {
            cell = metadata_table_get_cell(results_table, index);
            if (cell != NULL)
            {
                sscanf(cell->value.v_text->str, "%hd", &version_number_minor);

                cell = NULL;
            }
        }

        // get company name
        index = metadata_table_find_column(results_table, ERSCRIBBLE_VERSION_COMPANY);
        if (index >= 0)
        {
            cell = metadata_table_get_cell(results_table, index);
            if (cell != NULL)
            {
                version_company = g_string_new(cell->value.v_text->str);
                cell = NULL;
            }
        }

        // get screen dpi
        index = metadata_table_find_column(results_table, ERSCRIBBLE_SCREEN_DPI);
        if (index >= 0)
        {
            cell = metadata_table_get_cell(results_table, index);
            if (cell != NULL)
            {
                sscanf(cell->value.v_text->str, "%d", &screen_dpi);

                cell = NULL;
            }
        }

        // get screen units
        index = metadata_table_find_column(results_table, ERSCRIBBLE_SCREEN_UNITS);
        if (index >= 0)
        {
            cell = metadata_table_get_cell(results_table, index);
            if (cell != NULL)
            {
                screen_units = g_string_new(cell->value.v_text->str);
                cell = NULL;
            }
        }
    }
	
	//printf("Found: version: %d.%d %s DPI:%d Units:%s\n",
	//	   (int)version_number_major, (int)version_number_minor, version_company->str,
	//	   screen_dpi, screen_units->str);
	
    // free the results table
    metadata_table_free(results_table);

    return ret == ER_OK;
}

static metadata_table *annotations_load_document_data(const char *target_file_name, erMetadb *metadata_db)
{
    GString *filename = g_string_new(target_file_name);
    GString *annotation_type = g_string_new(ERSCRIBBLE_ANNOTATION_TYPE);
    metadata_table *results = NULL;

	//printf("annotations_load_document_data-enter\n");
    if (ermetadb_select_annotations (metadata_db,
                                     filename,
                                     annotation_type,
                                     NULL,
                                     -1,
                                     -1,
                                     &results) == ER_OK)
    {
        // load all of the pages data from results table
        annotations_load_pages_basic_data(results, metadata_db);
    }
	else
	{
		printf("FATAL: Could not load annotations for %s\n", target_file_name);
	}

    g_string_free(annotation_type, TRUE);
    g_string_free(filename, TRUE);
    metadata_table_free(results);

    return results;
}

static void annotations_load_pages_basic_data(const metadata_table *table, erMetadb *metadata_db)
{
    // The page instance would be constructed in this function
    // All of the basic data of a page would be set, except the BLOB

	//printf("annotations_load_pages_basic_data-enter\n");
    if (table == NULL || table->cell_data->len <= 0)
    {
        printf("Empty table of scribble\n");
        return;
    }

    gint64 position = 0;
    gint64 anno_id = 0;
	GString *anchor;
    guint  idx = 0;
	stroke_count = 0;
    for (; idx < table->n_rows; ++idx)
    {
        // read the file position -> page number
        int cell_idx = metadata_table_find_column(table, METADB_FILE_POSITION);
        //assert(cell_idx >= 0);
        const metadata_cell* cell = metadata_table_get_cell(table,
            metadata_table_cell_index(table, idx, cell_idx));
        //assert(cell->type == METADATA_INT64);
        position = cell->value.v_int64;

        // read the annotation id -> link to annotation id in annotations-table
        cell_idx = metadata_table_find_column(table, METADB_ANNOTATION_ID);
        //assert(cell_idx >= 0);
        cell = metadata_table_get_cell(table,
            metadata_table_cell_index(table, idx, cell_idx));
        //assert(cell->type == METADATA_INT64);
        anno_id = cell->value.v_int64;

        // read the start anchor (like 'pdf:/page:9' used for ? )
        cell_idx = metadata_table_find_column(table, METADB_START_ANCHOR);
        //assert(cell_idx >= 0);
        cell = metadata_table_get_cell(table,
            metadata_table_cell_index(table, idx, cell_idx));
        //assert(cell->type == METADATA_TEXT);
        anchor = g_string_new(cell->value.v_text->str);
        // create a new page with the start anchor and related information

		//printf("Page attributes : pos %d, anno_id:%d, anchor:%s\n", (int)position, (int)anno_id, anchor->str);
		// select correct page/layer from the journal
		g_current_page = g_list_nth_data(journal.pages, position-1);
		if (g_current_page)
		{
			g_current_layer = (Layer *)g_list_last(g_current_page->layers)->data;
		}
		else
		{
			g_current_layer = NULL;
		}
		//printf("Page: %x Layer %x\n", (int)g_current_page, (int)g_current_layer);
			
		// and load annotations
		annotations_load_page_data(metadata_db, anno_id, position);

		g_string_free(anchor, TRUE);
    }
}


static gboolean annotations_load_page_data(erMetadb *metadata_db, gint64 annotation_id, gint64 page_id)
{
    gint64 anno_list[1];
    anno_list[0] = annotation_id;

	//printf("load_page_data-enter\n");
    metadata_table *value = NULL;
    int ret = ermetadb_get_annotations(metadata_db,
        &anno_list[0], 1, &value);

    if (ret != ER_OK || value == NULL)
    {
		printf("FATAL: Cannot retrieve BLOB of page %d", (int)page_id);
        return FALSE;
    }

    // Should not retrieve more than one rows of data
    guint idx = 0;
    for (; idx < value->n_rows; ++idx)
    {
		char *data = NULL;
        int cell_idx = metadata_table_find_column(value, METADB_DATA);
        const metadata_cell *cell =
            metadata_table_get_cell(value, metadata_table_cell_index(value, idx, cell_idx));
        //assert(cell->type == METADATA_BLOB);
        gsize length = cell->value.v_blob.len;
		data = (char *)cell->value.v_blob.data;
		{
			ScbPageAttributes attributes;
			
			memcpy(&attributes, data, sizeof(attributes));
			//printf("PageAttrib: version: %d, ul_x:%d ul_y:%d lr_x:%d lr_y:%d strokes:%d\n", attributes.minor_version,
			//	   attributes.area.left, attributes.area.top, attributes.area.right, attributes.area.bottom, attributes.strokes_number);
			//printf("Xournal page: %f x %f offset (%f,%f)\n", g_current_page->width, g_current_page->height, g_current_page->hoffset, g_current_page->voffset);

			length -= sizeof(attributes);
			annotations_strokes_load(page_id, &data[sizeof(attributes)], &length, attributes.strokes_number, &attributes);
			//printf("X=[%d..%d] Y = [%d..%d]\n", min_x, max_x, min_y, max_y);
		}
    }

    return TRUE;
}

static gboolean annotations_strokes_load(gint64 page_id, gchar* data, gsize *length, const int strokes_num, ScbPageAttributes *page_attributes)
{
    //erscribble_strokes_clear(strokes);

    int i = 0;
    for (; i < strokes_num; ++i)
    {
		ScbStrokeAttributes attributes;
		struct Item *tmpItem;
		GnomeCanvasPoints *path;
		int j = 0;
        int min_x, max_x, min_y, max_y; // min/max per stroke

		memcpy(&attributes, data, sizeof(attributes));
		*length -= sizeof(attributes);
		data = &data[sizeof(attributes)];
		// TODO: use stroke attributes
		stroke_count++;
		
		//printf("[%d]StrokeAttrib: color:%d, zoom:%f width:%d points:%d\n",
		//	   stroke_count, (int)attributes.color, attributes.zoom, attributes.size_id, attributes.points_number);

		if ((MAX_STROKES == 0) || (stroke_count < MAX_STROKES))
		{
		// create item and fill it
		tmpItem = (struct Item *)g_malloc0(sizeof(struct Item));
		tmpItem->type = ITEM_STROKE;
		tmpItem->path = NULL;
		tmpItem->canvas_item = NULL;
		// we are assuming that the annotations only have 5 levels 1-5 )
		//tmpItem->brush.thickness = predef_thickness[0][attributes.size_id-1];
		tmpItem->brush.thickness = attributes.size_id * 0.42;
		tmpItem->brush.variable_width = FALSE;
		tmpItem->brush.color_no = 0;
		tmpItem->brush.color_rgba = predef_colors_rgba[0];
		tmpItem->brush.tool_type = 0; // PEN
		tmpItem->brush.thickness_no = 0;  // who cares ?
		tmpItem->brush.tool_options = 0;  // who cares ?
		tmpItem->brush.ruler = FALSE;
		tmpItem->brush.recognizer = FALSE;
		path = gnome_canvas_points_new(attributes.points_number);
		tmpItem->path = path;		
		// TODO: zoom factor for these strokes and canvas coordinates?
		}
		//printf("Created Item: %x with path: %x size %d\n", (int)tmpItem, (int)path, attributes.points_number);
		
		min_x = 5000; max_x = 0;
		min_y = 5000; max_y = 0;
		for (; j < attributes.points_number; ++j)
		{
			ScbDevPoint point;
			
			memcpy(&point, &data[j*sizeof(ScbDevPoint)], sizeof(ScbDevPoint));
			// only add a limited list of strokes!!
			if ((MAX_STROKES == 0) || (stroke_count < MAX_STROKES))
			{
				// TODO: store point, with attributes!
				//path->coords[j*2] = ((point.point.x * 100 / attributes.zoom) * g_current_page->width) / 
				//	                (page_attributes->area.right - page_attributes->area.left);
				//path->coords[j*2+1] = ((point.point.y * 100 / attributes.zoom) * g_current_page->height) /
				//	                (page_attributes->area.bottom - page_attributes->area.top);
				// Best results gives the following formula where:
				// points from file are scaled using the zoom factor of the stroke_device
				// then they are scaled from 'input' DPI (160) to 'output' DPI (72)
				// for some reason a factor of (1-1/16) is needed to scale all to 
				// the correct resolution (maybe related to margins?)
				path->coords[j*2] = ((point.point.x * 100.0 / attributes.zoom)) / 
									(160.0/72.0*(1-0.0625));
				path->coords[j*2+1] = ((point.point.y * 100.0 / attributes.zoom)) /
									(160.0/72.0*(1-0.0625));
				//printf("Point: (%d,%d)<=%d --> [%f,%f]\n", point.point.x, point.point.y, point.pressure, path->coords[j*2], path->coords[j*2+1]);
				if (point.point.x < min_x) min_x = point.point.x;
				if (point.point.x > max_x) max_x = point.point.x;
				if (point.point.y < min_y) min_y = point.point.y;
				if (point.point.y > max_y) max_y = point.point.y;
			}
		}
		// update the bounding box
		update_item_bbox(tmpItem);
		
		tmpItem->widths = NULL;
		
		*length -= (attributes.points_number * sizeof(ScbDevPoint));
		data = &data[(attributes.points_number * sizeof(ScbDevPoint))];
			
		if (g_current_layer)
		{
			if ((MAX_STROKES == 0) || (stroke_count < MAX_STROKES))
			{
				g_current_layer->items = g_list_append(g_current_layer->items, tmpItem);
				g_current_layer->nitems++;
			}
		}
    }

    return TRUE;
}






