package eu32k.manga2cbz;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.swing.JOptionPane;

import eu32k.manga2cbz.generated.Manga2CbzConfig;

public class MangaConverter {

   public static final String version = "v1.1";
   public static Method quantize = null;

   public MangaConverter(String[] args) {
      Manga2CbzConfig c = Config.getConfig();

      File lib = new File("gif4j_pro_trial_2.3.jar");
      if (lib.exists()) {
         try {
            URLClassLoader child = new URLClassLoader(new URL[] { lib.toURI().toURL() }, this.getClass().getClassLoader());
            Class<?> classToLoad = Class.forName("com.gif4j.quantizer.Quantizer", true, child);
            quantize = classToLoad.getMethod("quantize", int.class, BufferedImage.class, int.class);
            System.out.println("Library 'gif4j' found and ready.");
         } catch (Exception e) {
            // NOP
            System.out.println("Error: Library 'gif4j' not ready.");
         }
      } else {
         System.out.println("Library 'gif4j' NOT found. Color depth reduction disabled.");
      }

      File source = null;
      File destination = null;
      String name = null;
      if (args.length != 3) {
         source = UserInput.getSource();
         destination = UserInput.getDestination();
      } else {
         source = new File(args[0]);
         destination = new File(args[1]);
         name = args[2];
      }

      if (source == null || !source.exists() || !source.canRead()) {
         System.out.println("Error: Source not ready.");
         return;
      }

      if (destination == null) {
         System.out.println("Error: Bad destination");
         return;
      }

      if (!destination.exists()) {
         try {
            destination.mkdir();
         } catch (Exception e) {
            System.out.println("Error: Cannot create destination directory.");
            return;
         }
      }
      File destinationTemp = new File(destination.getAbsolutePath() + File.separator + "converted_temp_dir");
      try {
         destinationTemp.mkdir();
      } catch (Exception e) {
         System.out.println("Error: Cannot create destination temp directory.");
         return;
      }

      if (!destination.isDirectory() || !destination.canWrite()) {
         System.out.println("Error: Destination not ready.");
         return;
      }

      if (source.equals(destination)) {
         System.out.println("Error: Source and Destination cannot be the same directory.");
         return;
      }

      if (destination.toString().startsWith(source.toString())) {
         System.out.println("Error: Destination cannot be inside the Source directory.");
         return;
      }

      if (args.length != 3) {
         name = JOptionPane.showInputDialog("Enter a name for the cbz file:", "Name");
      }

      if (name == null || name.equals("")) {
         System.out.println("Error: Bad name.");
         return;
      }

      File decompressDir = new File(destination.getAbsolutePath() + File.separator + "unzip_temp_dir");
      decompressDir.mkdir();

      ArrayList<File> files = new ArrayList<File>();
      Util.scanDirectory(source, files, decompressDir);

      if (files.size() > 0) {

         System.out.println("Found " + files.size() + " image files.");

         int threads = c.getThreads() == 0 ? Runtime.getRuntime().availableProcessors() : c.getThreads();
         System.out.println("Starting " + threads + " threads.");
         convert(threads, files, destinationTemp, c);

         System.out.println("Deleting unizp folder...");

         Util.deleteDir(decompressDir);

         System.out.println("Zipping...");

         try {
            ZipOutputStream out = new ZipOutputStream(new FileOutputStream(destination.getAbsolutePath() + File.separator + name + ".cbz"));

            byte[] buf = new byte[1024];
            int len;
            CRC32 crc = new CRC32();
            FileInputStream in;

            for (File f : destinationTemp.listFiles()) {
               if (Util.isImage(f.getName())) {

                  ZipEntry entry = new ZipEntry(f.getName());
                  entry.setMethod(ZipEntry.STORED);
                  entry.setCompressedSize(f.length());
                  entry.setSize(f.length());

                  crc.reset();

                  in = new FileInputStream(f);
                  while ((len = in.read(buf)) > 0) {
                     crc.update(buf, 0, len);
                  }
                  in.close();

                  entry.setCrc(crc.getValue());

                  out.putNextEntry(entry);

                  // Transfer bytes from the file to the ZIP file
                  in = new FileInputStream(f);
                  while ((len = in.read(buf)) > 0) {
                     out.write(buf, 0, len);
                  }
                  in.close();

                  // Complete the entry
                  out.closeEntry();

                  // f.delete();
               }
            }

            // Complete the ZIP file
            out.close();
         } catch (IOException e) {
            e.printStackTrace();
         }

         System.out.println("Deleting temp image folder...");

         Util.deleteDir(destinationTemp);

         System.out.println("\nFinished. Have fun!\n");
      } else {
         System.out.println("No image files found.");
      }

      for (int i = 10; i >= 0; i--) {
         try {
            Thread.sleep(1000);
         } catch (InterruptedException e) {
            e.printStackTrace();
         }
         System.out.println("Closing in " + i + "...");
      }
      System.exit(0);
   }

   private void convert(final int max, final ArrayList<File> files, final File destination, final Manga2CbzConfig c) {
      ArrayList<Thread> threads = new ArrayList<Thread>();

      for (int t = 0; t < max; t++) {
         final int nr = t;
         Thread thread = new Thread(new Runnable() {
            @Override
            public void run() {
               for (int i = 0; i < files.size(); i++) {
                  File file = files.get(i);
                  if (i % max == nr) {
                     try {
                        System.out.println("Converting Image " + (i + 1) + " of " + files.size() + "...");
                        String imageType = c.getImageType().name().toLowerCase();
                        File target = new File(destination.getAbsoluteFile() + File.separator + Util.generateNumber(i, 8) + "." + imageType);
                        new MangaImage(file, target, imageType, c.getMaxWidth(), c.getMaxHeight(), c.getCropThreshold(), c.getRotateLandscape(), c.getSplitLandscape(), c.getGrayLevels());
                        System.out.println("Image " + (i + 1) + " OK.");
                     } catch (IOException e) {
                        e.printStackTrace();
                     }
                  }
               }
            }
         });
         threads.add(thread);
         thread.start();
      }

      for (Thread thread : threads) {
         try {
            thread.join();
         } catch (InterruptedException e) {
            e.printStackTrace();
         }
      }
   }

   public static void main(String[] args) {
      if (args.length == 0) {
         Util.createConsole();
      }
      System.out
            .println("'manga2cbz for iRex DR800S(G)' is a free & simple tool for converting manga and comics to view on the iRex DR800S(G).\n\nFeatures in v1.1:\n\n- Unpack folders or zip / cbz / rar / cbr files (or any mix of nested combinations)\n- Images: png, jpg, gif\n- Crop white borders\n- Rotate landscape images\n- Re-size images to match the screen resolution\n- Reduce colors to 4 - 256 levels of gray (additional library needed for < 256)\n- Save pics as png / jpg and pack them into a cbz file\n\nNew:\n- XML configuration (see automatically generated 'config.xml')\n- Split landscape images. (enable in XML)\n- Combining split and rotate is possible too. (split then rotate)\n- Mulit-threading according to the number of CPUs available (or setting in XML)\n- Bugfixes\n\nLimitations:\n- Still no split archives supported\n- Maybe some bugs\n\nUse it together with the AMAZING 'CBZ Document Plugin' written by Kolenka.\nsee: http://www.mobileread.com/forums/showthread.php?t=75348\n\n>>> You can enable much better compression (color depth reduction with Floyd-Steinberg dithering) by downloading the file 'gif4j_pro_trial_2.3.jar' from http://www.gif4j.com/download.htm and putting it into the same directory. <<<\n");
      new MangaConverter(args);
   }
}
