package eu32k.manga2cbz;

import java.awt.AlphaComposite;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import javax.imageio.ImageIO;

import eu32k.manga2cbz.generated.RotationMode;
import eu32k.manga2cbz.generated.SplitMode;

public class MangaImage {

   private BufferedImage img;

   public MangaImage(File source, File target, String type, int width, int height, int cropThreshold, RotationMode rotate, SplitMode split, int colorDepth) throws IOException {

      int colorBits = (int) Math.sqrt(colorDepth);

      read(source);

      crop(cropThreshold);

      if (split != SplitMode.OFF) {
         ArrayList<BufferedImage> newImages = split(width, height, split);
         if (newImages != null) {
            System.out.println("Splitting image.");
            for (int i = 0; i < newImages.size(); i++) {
               File newTarget = new File(target.getAbsoluteFile() + "_sp_" + Util.generateNumber(i, 3) + "." + type);
               ImageIO.write(newImages.get(i), type, newTarget);
               new MangaImage(newTarget, newTarget, type, width, height, cropThreshold, rotate, split, colorDepth);
            }
            return;
         }
      }

      if (rotate != RotationMode.OFF) {
         rotateIfLandscape(rotate == RotationMode.CW);
      }

      autoResize(width, height);

      if (MangaConverter.quantize != null && colorBits < 16) {
         try {
            this.img = (BufferedImage) MangaConverter.quantize.invoke(null, 8, this.img, colorBits);
            // System.out.println("q to " + this.colorBits);
         } catch (Exception e) {
            // NOP
         }
      }
      // this.img = com.gif4j.quantizer.Quantizer.quantize(8, this.img, 4);

      save(target, type);
   }

   private void read(File source) throws IOException {
      this.img = ImageIO.read(source);
   }

   private void save(File target, String type) throws IOException {
      ImageIO.write(this.img, type, target);
   }

   private void crop(int threshold) {
      int top = findTop(threshold);
      int bottom = findBottom(threshold);
      int left = findLeft(threshold);
      int right = findRight(threshold);

      if (left < right && top < bottom) {
         this.img = this.img.getSubimage(left, top, right - left, bottom - top);
      }
   }

   private int findTop(int threshold) {
      for (int i = 0; i < this.img.getHeight(); i++) {
         for (int j = 0; j < this.img.getWidth(); j++) {
            int color = this.img.getRGB(j, i);
            int r = (color & 0xFF0000) >> 16;
            int g = (color & 0xFF00) >> 8;
            int b = color & 0xFF;
            if (r < threshold || g < threshold || b < threshold) {
               return i;
            }
         }
      }
      return 0;
   }

   private int findBottom(int threshold) {
      for (int i = this.img.getHeight() - 1; i >= 0; i--) {
         for (int j = 0; j < this.img.getWidth(); j++) {
            int color = this.img.getRGB(j, i);
            int r = (color & 0xFF0000) >> 16;
            int g = (color & 0xFF00) >> 8;
            int b = color & 0xFF;
            if (r < threshold || g < threshold || b < threshold) {
               return i;
            }
         }
      }
      return 0;
   }

   private int findLeft(int threshold) {
      for (int i = 0; i < this.img.getWidth(); i++) {
         for (int j = 0; j < this.img.getHeight(); j++) {
            int color = this.img.getRGB(i, j);
            int r = (color & 0xFF0000) >> 16;
            int g = (color & 0xFF00) >> 8;
            int b = color & 0xFF;
            if (r < threshold || g < threshold || b < threshold) {
               return i;
            }
         }
      }
      return 0;
   }

   private int findRight(int threshold) {
      for (int i = this.img.getWidth() - 1; i >= 0; i--) {
         for (int j = 0; j < this.img.getHeight(); j++) {
            int color = this.img.getRGB(i, j);
            int r = (color & 0xFF0000) >> 16;
            int g = (color & 0xFF00) >> 8;
            int b = color & 0xFF;
            if (r < threshold || g < threshold || b < threshold) {
               return i;
            }
         }
      }
      return 0;
   }

   private void rotateIfLandscape(boolean cw) {
      if (this.img.getWidth() > this.img.getHeight()) {
         rotate90(cw);
      }
   }

   private void rotate90(boolean rotateCW) {
      int width = this.img.getWidth();
      int height = this.img.getHeight();
      BufferedImage rot = new BufferedImage(height, width, this.img.getType());
      if (rotateCW) {
         for (int i = 0; i < width; i++) {
            for (int j = 0; j < height; j++) {
               rot.setRGB(height - 1 - j, i, this.img.getRGB(i, j));
            }
         }
      } else {
         for (int i = 0; i < width; i++) {
            for (int j = 0; j < height; j++) {
               rot.setRGB(j, width - 1 - i, this.img.getRGB(i, j));
            }
         }
      }
      this.img = rot;
   }

   private ArrayList<BufferedImage> split(int width, int height, SplitMode split) {
      if (this.img.getWidth() < 2 * width) {
         System.out.println("no " + this.img.getWidth() + " " + 2 * width);
         return null;
      }

      ArrayList<BufferedImage> images = new ArrayList<BufferedImage>();
      int subImages = (int) Math.floor((double) this.img.getWidth() / (double) width);
      double dWidth = (double) this.img.getWidth() / (double) subImages;

      if (split == SplitMode.RIGHT) {

         for (int i = subImages; i > 0; i--) {
            int from = (int) Math.round(dWidth * (i - 1));
            int to = Math.min((int) Math.round(dWidth * i), this.img.getWidth());
            images.add(this.img.getSubimage(from, 0, to - from, this.img.getHeight()));
         }

      } else if (split == SplitMode.LEFT) {

         for (int i = 0; i < subImages; i++) {
            int from = (int) Math.round(dWidth * i);
            int to = Math.min((int) Math.round(dWidth * (i + 1)), this.img.getWidth());
            images.add(this.img.getSubimage(from, 0, to - from, this.img.getHeight()));
         }
      }
      return images;
   }

   private void autoResize(int width, int height) {
      int newWidth = this.img.getWidth();
      int newHeight = this.img.getHeight();

      if (this.img.getWidth() > width || this.img.getHeight() > height) {
         newWidth = width;
         newHeight = height;
         double thumbRatio = (double) newWidth / (double) newHeight;
         double imageRatio = (double) this.img.getWidth() / (double) this.img.getHeight();

         if (thumbRatio < imageRatio) {
            newHeight = (int) (newWidth / imageRatio);
         } else {
            newWidth = (int) (newHeight * imageRatio);
         }
      }
      resizeAndGrayScaleImage(newWidth, newHeight);
   }

   private void resizeAndGrayScaleImage(int width, int height) {
      BufferedImage resizedImage = new BufferedImage(width, height, BufferedImage.TYPE_BYTE_GRAY);
      Graphics2D g = resizedImage.createGraphics();
      g.setComposite(AlphaComposite.Src);
      g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
      g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
      g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      g.drawImage(this.img, 0, 0, width, height, null);
      g.dispose();
      this.img = resizedImage;
   }

}
