package eu32k.manga2cbz;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.swing.JOptionPane;

public class MangaConverter {

   private final int CROP_THRESHOLD = 200;
   private final int MAX_WIDTH = 768;
   private final int MAX_HEIGHT = 981;
   private final boolean ROTATE_CW = true;
   private final int THREADS = 4;
   private final String FILE_TYPE = "png";

   public MangaConverter(String[] args) {
      File source = null;
      File destination = null;
      String name = null;
      if (args.length != 3) {
         source = UserInput.getSource();
         destination = UserInput.getDestination();
      } else {
         source = new File(args[0]);
         destination = new File(args[1]);
         name = args[2];
      }

      if (source == null || !source.exists() || !source.canRead()) {
         System.out.println("Error: Source not ready.");
         return;
      }

      if (destination == null) {
         System.out.println("Error: Bad destination");
         return;
      }

      if (!destination.exists()) {
         try {
            destination.createNewFile();
         } catch (IOException e) {
            System.out.println("Error: Cannot create destination directory.");
            return;
         }
      }

      if (!destination.isDirectory() || !destination.canWrite()) {
         System.out.println("Error: Source not ready.");
         return;
      }

      if (args.length != 3) {
         name = JOptionPane.showInputDialog("Enter a name for the cbz file:", "Name");
      }

      if (name == null || name.equals("")) {
         System.out.println("Error: Bad name.");
         return;
      }

      File decompressDir = new File(destination.getAbsolutePath() + File.separator + "unzip_temp_dir");
      decompressDir.mkdir();

      ArrayList<File> files = new ArrayList<File>();
      Util.scanDirectory(source, files, decompressDir);

      System.out.println("Found " + files.size() + " image files.");

      convert(this.THREADS, files, destination);

      System.out.println("Deleting temp folder...");

      Util.deleteDir(decompressDir);

      System.out.println("Zipping...");

      try {
         ZipOutputStream out = new ZipOutputStream(new FileOutputStream(destination.getAbsolutePath() + File.separator + name + ".cbz"));

         byte[] buf = new byte[1024];
         int len;
         CRC32 crc = new CRC32();
         FileInputStream in;

         for (File f : destination.listFiles()) {
            if (Util.isImage(f.getName())) {

               ZipEntry entry = new ZipEntry(f.getName());
               entry.setMethod(ZipEntry.STORED);
               entry.setCompressedSize(f.length());
               entry.setSize(f.length());

               crc.reset();

               in = new FileInputStream(f);
               while ((len = in.read(buf)) > 0) {
                  crc.update(buf, 0, len);
               }
               in.close();

               entry.setCrc(crc.getValue());

               out.putNextEntry(entry);

               // Transfer bytes from the file to the ZIP file
               in = new FileInputStream(f);
               while ((len = in.read(buf)) > 0) {
                  out.write(buf, 0, len);
               }
               in.close();

               // Complete the entry
               out.closeEntry();

               f.delete();
            }
         }

         // Complete the ZIP file
         out.close();
      } catch (IOException e) {
         e.printStackTrace();
      }

      System.out.println("\nFinished. Have fun!\n");

      for (int i = 10; i >= 0; i--) {
         try {
            Thread.sleep(1000);
         } catch (InterruptedException e) {
            e.printStackTrace();
         }
         System.out.println("Closing in " + i + "...");
      }
      System.exit(0);
   }

   private void convert(final int max, final ArrayList<File> files, final File destination) {
      ArrayList<Thread> threads = new ArrayList<Thread>();

      for (int t = 0; t < max; t++) {
         final int nr = t;
         Thread thread = new Thread(new Runnable() {
            @Override
            public void run() {
               for (int i = 0; i < files.size(); i++) {
                  File file = files.get(i);
                  if (i % max == nr) {
                     try {
                        System.out.println("Converting Image " + (i + 1) + " of " + files.size() + "...");
                        File target = new File(destination.getAbsoluteFile() + File.separator + Util.generateNumber(i, 8) + "." + MangaConverter.this.FILE_TYPE);
                        new MangaImage(file, target, MangaConverter.this.FILE_TYPE, MangaConverter.this.MAX_WIDTH, MangaConverter.this.MAX_HEIGHT, MangaConverter.this.CROP_THRESHOLD,
                              MangaConverter.this.ROTATE_CW);
                        System.out.println("Image " + (i + 1) + " OK.");
                     } catch (IOException e) {
                        e.printStackTrace();
                     }
                  }
               }
            }
         });
         threads.add(thread);
         thread.start();
      }

      for (Thread thread : threads) {
         try {
            thread.join();
         } catch (InterruptedException e) {
            e.printStackTrace();
         }
      }
   }

   public static void main(String[] args) {
      if (args.length == 0) {
         Util.createConsole();
      }
      System.out
            .println("'manga2cbz for iRex DR800S(G)' is a free & simple tool for converting manga and comics to view on the iRex DR800S(G).\n\nVersion 1.01"
                  + "\nFeatures:"
                  + "\n- Unpacks zip / cbz / rar / cbr files"
                  + "\n- Crops white borders"
                  + "\n- Rotates wide images"
                  + "\n- Resizes images to match the screen resolution (768x981)"
                  + "\n- Reduces colors to 256 (or 16 with gif4j)\n- Uses 4 threads for converting"
                  + "\n- Saves pics as png and packs them into a cbz file\n\nUse together with the CBZ Document Plugin written by Kolenka.\nsee: http://www.mobileread.com/forums/showthread.php?t=75348\n"
                  + "\n>>> You can enable MUCH better compression (color depth reduction with Floyd-Steinberg dithering) by downloading\nthe file 'gif4j_pro_trial_2.3.jar' from http://www.gif4j.com/download.htm and putting it into the same directory.\n");
      new MangaConverter(args);
   }
}
