#include "pdf_doc.h"
#include "log.h"

#include <string.h> // for memset
#include <math.h>

GlobalParams *globalParams;	

GlobalParams::GlobalParams() {
	WARNPRINTF("Initializing global params");
	ctx = ddjvu_context_create("");
	mtx = new pdf::Mutex();
	if(!ctx) ERRORPRINTF("Error creating DjVu context");
}

GlobalParams::~GlobalParams() {	
	WARNPRINTF("Destructing global params");
	ddjvu_context_release(ctx);
	delete mtx;
}

void handleDdjvu(int wait) {
	ddjvu_context_t *ctx = globalParams->getContext();
	const ddjvu_message_t *msg;
	if (wait) msg = ddjvu_message_wait(ctx);
	while ((msg = ddjvu_message_peek(ctx))) {
	switch(msg->m_any.tag) {
        case DDJVU_ERROR:
		  WARNPRINTF("DjVu Error: %s\n", msg->m_error.message);
          if (msg->m_error.filename) WARNPRINTF("DjVu error in file: '%s:%d'\n",msg->m_error.filename, msg->m_error.lineno);
//          exit(10);
        default:
          break;
      }
      ddjvu_message_pop(ctx);
    }
}

bool endsWith(const char *a, const char *b) {
	if(strlen(a) < strlen(b)) return false;
	return strcmp (a+strlen(a)-strlen(b),b) == 0;
}

PDFDoc::PDFDoc(GooString* file) {
    WARNPRINTF("Opening DjVu document %s", file->getCString());
	pageWidths = pageHeights = pageDpis = pageRotations = 0;
	twoPageMode = endsWith(file->getCString(), "2pg.djvu");
	if(twoPageMode) WARNPRINTF("Opening in two page mode");
	ddjvu_context_t *ctx = globalParams->getContext();
	pdf::Mutex* mtx = globalParams->getMutex();
	mtx->lock();
	doc = ddjvu_document_create_by_filename(ctx, file->getCString(), TRUE);
	if(!doc) {
		ERRORPRINTF("Could not open DjVu document");
		ok = gFalse;
		mtx->unlock();
		return;
	} else {
		ok = gTrue;
	}
	while (!ddjvu_document_decoding_done(doc)) handleDdjvu(TRUE);
  	nPages = ddjvu_document_get_pagenum(doc);
	pageWidths    = new int[nPages];
	pageHeights   = new int[nPages];
	pageDpis      = new int[nPages];
	pageRotations = new int[nPages];
	for(int i=0;i<nPages;i++) pageWidths[i] = -1;
	mtx->unlock();
}

PDFDoc::~PDFDoc() {
	WARNPRINTF("Destructing PDFDoc");
	delete[] pageWidths;
	delete[] pageHeights;
	delete[] pageDpis;
	delete[] pageRotations;
}


RenderRet PDFDoc::displayPage(OutputDev *out, int page, double hDPI, double vDPI,
		int rotate, GBool useMediaBox, GBool Crop, GBool printing,
		GBool (*abortCheckCbk)(void *data),
		void* abortCheckCbkData,
		GBool (*annotDisplayDecideCbk) (Annot* annot, void *user_data),
		void *annotDisplayDecideCbkData) {
			pdf::Mutex* mtx = globalParams->getMutex();
			mtx->lock();
			// WARNPRINTF("PDFDoc::displayPage, page: %d, hDPI: %f, vDPI: %f, rotate: %d (stub)", page, hDPI, vDPI, rotate);
			bool leftPage = page % 2 != 0;
			if(twoPageMode) page = (page+1)/2;
			ddjvu_page_t *pg = ddjvu_page_create_by_pageno(doc, page-1);
			while (!ddjvu_page_decoding_done(pg)) handleDdjvu(TRUE);
			ddjvu_rect_t prect;
			prect.x = 0;
			prect.y = 0;
			prect.w = (int)(getPageCropWidth(page) * hDPI / 72.0);
			prect.h = (int)(getPageCropHeight(page) * vDPI / 72.0);
			// restrict max size for wonky dpi settings
			if(prect.w > 2000) { prect.h = (prect.h * 2000) / prect.w; prect.w = 2000; }
			if(prect.h > 2500) { prect.w = (prect.w * 2500) / prect.h; prect.h = 2500; }
			SplashBitmap *bmp = new SplashBitmap(prect.w, prect.h);
			unsigned char *data = bmp->getDataPtr();
			ddjvu_format_style_t style = DDJVU_FORMAT_GREY8;
			ddjvu_render_mode_t mode = DDJVU_RENDER_COLOR;
			ddjvu_format_t *fmt;
			fmt = ddjvu_format_create(style, 0, 0);
			ddjvu_format_set_row_order(fmt, 1);
			// ddjvu_format_set_gamma(fmt, 4.0);
			int rowsize = prect.w;
			ddjvu_rect_t rrect;
			rrect.y = 0;
			rrect.h = prect.h;
			if(twoPageMode) {
				rrect.w = prect.w;
				prect.w *= 2;
				rrect.x = leftPage ? 0 : prect.w - rrect.w;
			} else {
				rrect.x = 0;
				rrect.w = prect.w;
			}
			if(!ddjvu_page_render(pg, mode, &prect, &rrect, fmt, rowsize, (char*)data)) {
				unsigned char white = 0xFF;
				memset(data, white, rowsize * prect.h);
				WARNPRINTF("PDFDoc::displayPage: error displaying DjVu page %d, showing white bitmap", page);
				// ddjvu_context_t *ctx = globalParams->getContext();
				// ddjvu_message_t *msg = ddjvu_message_peek(ctx);
				// if(msg && msg->m_any.tag == DDJVU_ERROR) WARNPRINTF("ddjvu: %s\n", msg->m_error.message);
			}
			ddjvu_format_release(fmt);
			ddjvu_page_release(pg);
			// bitmap improvement?
			// experimental!
			int w = (int)rrect.w;
			for(int i=w*(rrect.h-1)-2;i>=w;i--) {
				// data[i] = (unsigned char)(256.0 * pow((data[i]/256.0),1.7));
				//if(data[i-1] > data[i] && data[i+1] > data[i]) data[i] >>= 1;
				// if(data[i-1] > data[i] && data[i+1] > data[i] && data[i+w] < 192 && data[i-w] < 192) data[i] >>= 1;
				//if(data[i] > 128) data[i] = 192 + (data[i]-128)/2;
				//else data[i] = data[i]/2;
			}
			// end of bitmap improvement
			out->setBitmap(bmp);
			mtx->unlock();
			return Render_Done;
		}

SplashBitmap::SplashBitmap(int wa, int ha) {
	// WARNPRINTF("creating SplashBitmap: w=%d h=%d", wa,ha);
	data = new unsigned char[wa*ha];
	w = wa;
	h = ha;
	rowSize = wa;
}

SplashBitmap* OutputDev::takeBitmap () {
	// WARNPRINTF("SplashOutputDev::takeBitmap");
	if(bmp) {
		SplashBitmap *tmp = bmp;
		bmp = 0;
		return tmp;
	}
	return 0;
}

SplashOutputDev::SplashOutputDev(SplashColorMode colorModeA, int bitmapRowPadA,
								 GBool reverseVideoA, SplashColorPtr paperColorA) : OutputDev() {
	WARNPRINTF("Creating SplashOutputDev");
	defCtm[0] = 1.0;
	defCtm[1] = 0.0;
	defCtm[2] = 0.0;
	defCtm[3] = 1.0;
	defCtm[4] = 0.0;
	defCtm[5] = 0.0;
	defIctm[0] = 1.0;
	defIctm[1] = 0.0;
	defIctm[2] = 0.0;
	defIctm[3] = 1.0;
	defIctm[4] = 0.0;
	defIctm[5] = 0.0;	
}

SplashOutputDev::~SplashOutputDev () {
	WARNPRINTF("Destructing SplashOutputDev");
}

TextOutputDev::TextOutputDev(char *fileName, GBool physLayoutA,
							 GBool rawOrderA, GBool append) : OutputDev() {
	WARNPRINTF("Creating TextOutputDev");					  
}

TextOutputDev::~TextOutputDev () {
	WARNPRINTF("Destructing TextOutputDev");
}


