/*
 * File Name: plugin_event.h
 */

/*
 * This file is part of uds-plugin-common.
 *
 * uds-plugin-common is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * uds-plugin-common is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#ifndef PLUGIN_EVENT_H_
#define PLUGIN_EVENT_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "plugin_type.h"
#include "plugin_unknown.h"

/**
 * @brief Type definition of plugin events.
 */
typedef enum
{
    EVENT_PAGINATE_START,             /**< Pagination starting Event */
    EVENT_PAGINATE_END,               /**< Pagination ending Event */
    EVENT_RENDERING_END,              /**< Rendering ending Event */
    EVENT_MARKER_READY,               /**< Marker(tree) ready Event */
    EVENT_MARKER_OUT_OF_DATE,         /**< Marker(tree) Out-Of-Date Event */
    EVENT_SEARCH_END,                 /**< Search ending Event */
    EVENT_SEARCH_ABORTED,             /**< Search aborted Event */
    EVENT_SEARCH_DICT_END,            /**< Search in dict ending Event */
    EVENT_LOOKUP_DICTS_END,           /**< Lookup in dict ending Event */
    EVENT_APPLICATION_COMMAND         /**< Mobipocket application command*/
} PluginEvent;

/**
 * @brief Define render status here.
 */
typedef enum
{
    RENDER_DONE,                        /**< Render finished. */
    RENDER_OUT_OF_MEMORY,               /**< Out of memory.   */
    RENDER_INVALID_PAGE,                /**< Invalid page.    */
    RENDER_FAIL                         /**< Render fails.    */
} PluginRenderStatus;

/**
 * @brief Pagination event, could be triggered when start and finished.
 */
typedef struct
{
    int current;    /**< The number of the latest paginated page */
    int total;      /**< The number of total pages that should be paginated */
} EventParmsPaginate_t;

/**
 * @brief Render event.
 */
typedef struct 
{
    unsigned long       rid;        /**< The reference ID of the rendered result */
    PluginRenderStatus  status;     /**< The status of render result. */
    IPluginUnknown*     result;     /**< The IPluginUnknown pointer of render result object */
} EventRenderEnd_t;

/**
 * @brief Marker(tree) generation done event.
 */
typedef struct 
{
    IPluginUnknown* result; /**< The IPluginUnknown pointer of the collection 
                                 which contains the requested marker(tree) */
} EventMarkerReady_t;

/**
 * @brief Search finishing event.
 */
typedef struct 
{
    unsigned long   search_id;  /**< ID of the search request*/
    IPluginUnknown* result;     /**< The IPluginUnknown pointer of the collection 
                                     which contains the requested search results. */
} EventSearchEnd_t;

typedef struct
{
    UDSString * dict_title;    /**< The dictionary title of  this index. */
    UDSString * language_in;   /**< The input language group of the 
                                    dictionary. Example: en_UK */
    UDSString * language_out;  /**< The output language group of the 
                                    dictionary. Example: nl_NL */
    UDSString * match_type;    /**< The match type during  a lookup. 
                                    Example: CANONIZATION, DESINFLECTION, UNKNOWN */
    UDSString * title;         /**< The title of this  item. */
} DictResult_t;

/**
 * @brief Search in dict event.
 */
typedef struct 
{
    IPluginUnknown* result; /**< The IPluginUnknown pointer of the collection 
                                 which contains the requested search in dict 
                                 results(DictResult_t) */
} EventSearchInDictReady_t;

/**
 * @brief Lookup in dict event.
 */
typedef struct 
{
    IPluginUnknown* result; /**< The IPluginUnknown pointer of the collection 
                                 which contains the requested lookup in dict 
                                 results(DictResult_t) */
} EventLookupInDictsReady_t;


/**
 * @brief Mobipocket application command event.
 */
typedef enum
{
    BUY_BOOK = 0,
    MAKE_PHONE_CALL,
    OPEN_EXTERNAL_LINK,
    OPEN_IMAGE,
    OPEN_INDEX,
    REFRESH_SCREEN,
    TOOLTIP,
}AppCmdType;

typedef struct
{
    AppCmdType type;

    union
    {
        UDSString * buy_book_url;
        UDSString * phone_number;
        UDSString * external_link_url;
        const char * image_data;
        void * index_descriptor;
        UDSString * tooltip_text;
    }command;
}EventApplicationCommand_t;

/**
 * @brief Put all above events into together.
 */
typedef union 
{
    EventParmsPaginate_t    paginate;       /**< Parameter of pagination  */
    EventRenderEnd_t        render_end;     /**< Parameter of rendering done */
    EventMarkerReady_t      marker_ready;   /**< Parameter of marker ready */
    EventSearchEnd_t        search_end;     /**< Parameter of search done */
    EventSearchInDictReady_t search_in_dict_ready;   /**< Parameter of search in dict ready */
    EventLookupInDictsReady_t lookup_in_dicts_ready;   /**< Parameter of lookup in ready */
    EventApplicationCommand_t app_cmd; /**< Parameter of application command */
} PluginEventAttrs;

/** 
 * @brief Define the callback function prototype.
 * @param thiz_sender The IPluginUnknown pointer of the object which sends 
 * the event.
 * @param handler_id Because UDS can register same event with same function 
    pointer twice, we use handler_id to distinguish them.
 * @param plugin_event The event which is sent by thiz_sender.
 * @param user_data User specified data.
 * @param plugin_data The data coming from plugin library. It would be used 
 * as parameters when broadcasting the event.
 */
typedef void (* EventFunc)( IPluginUnknown          *thiz_sender,
                            unsigned long           handler_id,
                            const PluginEvent       plugin_event,
                            void                    *user_data,
                            const PluginEventAttrs  *plugin_data );

/**
 * @brief Universal Document Shell Plugin Document Event Broadcaster interface.
 * Through IPluginEventBroadcaster, listeners can receive notification from 
 * plugin.
 */
typedef struct 
{
    /**
     * @brief Register call back function for specified event.
     * @param thiz The object identifier which implements the 
     * IPluginEventBroadcaster interface.
     * @param event The event on which the listener needs to register.
     * @param callback Callback function defined by caller.
     * @param user_data User specified data, plugin puts this variable 
     * to callback function as a parameter.
     * @param handler_id The identifier ID of this registration.
     * @return TODO. Add return code here.
     */
    PluginStatus (* add_event_receiver)( IPluginUnknown     *thiz,
                                         const PluginEvent  plugin_event,
                                         EventFunc          callback,
                                         void               *user_data,
                                         unsigned long      *handler_id);

    /**
     * @brief Remove a listener by specified ID.
     * @param thiz The object identifier which implements the 
     * IPluginEventBroadcaster interface.
     * @param handler_id The unique identifier of the registration, it is 
     * always generated by plugin while caller registers a callback 
     * function for specified event.
     * @return TODO. Add return code here.
     */
    PluginStatus (* remove_event_receiver)( IPluginUnknown  *thiz,
                                            unsigned long   handler_id );

} IPluginEventBroadcaster;

#ifdef __cplusplus
}
#endif 

#endif

