/**
 *
 * Copyright (c) 2007 Mitch Reierson.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "proxy1150.h"
#include "exception.h"

#include <iostream>
#include <sstream>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

using namespace std;

void Proxy1150::do_loop()
{
  bool bConnected = false;

  // Loop runs as a daemon without termination
  while (true) {

    try {
      if (!bConnected) {
	_dev1150.initialize();
	_dev1150.write("start,0,3,");
	_dev1150.write("1.0");
	bConnected = true;
      }

      string data;
      _dev1150.read(data);

      if (data.length() > 0) {

	// Parse and dispatch to the proxy methods      
	if (data.find("gethostbyname") != string::npos) {

	  gethostbyname(data);

	} else if (data.find("socket") != string::npos) {

	  socket(data);

	} else if (data.find("connect") != string::npos) {
	  
	  connect(data);

	} else if (data.find("send") != string::npos) {

	  send(data);

	} else if (data.find("recv") != string::npos) {
	
	  recv(data);

	} else if (data.find("close") != string::npos) {

	  close(data);

	}
      }

      usleep(500);

    } catch (DevOpenError) {
      cerr << "\nError: Device could not be opened.  Please make sure that this program is\n";
      cerr << "running with superuser privileges (as 'root').\n";
      bConnected = false;

    } catch (DevNotFoundError) {
      cerr << "\nError: Device was not found on the USB bus.  Please make sure that it is\n";
      cerr << "plugged into a USB port and turned on.\n";
      bConnected = false;

    } catch (DevReadError) {
      cerr << "\nError receiving data from device: " << usb_strerror() << endl;
      bConnected = false;

    } catch (DevWriteError) {
      cerr << "\nError sending data to device: " << usb_strerror() << endl;
      bConnected = false;

    }
    
    if (!bConnected) {
      sleep(5);
    }

  }
}

// Split the data message using a comma delimiter into n elements specified by num_splits
void Proxy1150::split(string &data, vector<string> &data_vector, int num_splits)
{
  string::size_type last_pos = 0;
  string::size_type pos = 0;
  bool bFinished = false;

  for (int i = 0; i < num_splits && !bFinished; ++i) {
    pos = data.find_first_of(",", last_pos);
    if ((i == (num_splits - 1)) || (pos == string::npos)) {
      data_vector.push_back(data.substr(last_pos));
      bFinished = true;
    } else {
      data_vector.push_back(data.substr(last_pos, pos - last_pos));
      last_pos = pos + 1;
    }		     
  }
}

// Acknowledge the message with the received message id, response bytes and response
void Proxy1150::ack(string &id, string resp)
{
  stringstream ack;
  ack << "ack," << id << "," << resp.length() << ",";
  write(ack.str());
  write(resp);
}

// Acknowledge the message with the received message id, response bytes and response
void Proxy1150::nak(string &id, string resp)
{
  stringstream nak;
  nak << "nak," << id << "," << resp.length() << ",";
  write(nak.str());
  write(resp);
}

// Send the IP address of the personal content server, if specified, otherwise do a lookup
void Proxy1150::gethostbyname(string &data)
{
  vector<string> data_vector;
  split(data, data_vector, 4);

  const char* host = data_vector[3].c_str();
  if (_host == "") {
    // Content server host was not specified, lookup the host requested by device
    struct hostent *hp;    
    hp = ::gethostbyname(host);
    if (hp) {
      _host = inet_ntoa( *(in_addr*) hp->h_addr );
    }
  }

  if (_host == "") {
    cerr << "Error: Host lookup failure: " << host << ", (errno = " << errno << ")\n";
    nak(data_vector[1]);
  } else {
    cout << "gethostbyname : " << _host << endl;
    ack(data_vector[1], _host);
  }
}

// Create a new socket and pass back the descriptor to the device
void Proxy1150::socket(string &data)
{
  vector<string> data_vector;
  split(data, data_vector, 3);

  _socket = ::socket(AF_INET, SOCK_STREAM, 0);

  if (_socket < 0) {
    cerr << "Error: Unable to open socket. (errno = " << errno << ")\n";
    nak(data_vector[1]);
  } else {
    cout << "Socket created " << _socket << endl;
    stringstream resp;
    resp << _socket;
    ack(data_vector[1], resp.str());
  }
}

// Open a TCP/IP connection to the content server
void Proxy1150::connect(string &data)
{
  vector<string> data_vector;
  split(data, data_vector, 6);

  int ret;
  struct sockaddr_in serv_addr;

  const char* host = data_vector[4].c_str();
  int port = atoi(data_vector[5].c_str());

  memset(&serv_addr, 0, sizeof(serv_addr));
  serv_addr.sin_family = AF_INET;
  serv_addr.sin_addr.s_addr = inet_addr(host);
  serv_addr.sin_port = htons(port);

  ret = ::connect(_socket, (const sockaddr *) &serv_addr, sizeof(serv_addr));
  if (ret < 0) {
    cerr << "Error: Connection refused to " << host << ":" << port << endl;
    nak(data_vector[1]);
  } else {
    cout << "Connected to " << host << ":" << port << endl;
    ack(data_vector[1]);
  }
}

// Forward the data message from the device to the content server
void Proxy1150::send(string &data)
{
  vector<string> data_vector;
  split(data, data_vector, 6);

  const char *buffer = data_vector[5].c_str();

  int ret;
  ret = ::send(_socket, buffer, strlen(buffer), 0);
  if (ret < 0) {
    cerr << "Error: Send failed (errno = " << errno << ")\n";
    nak(data_vector[1]);
  } else {
    cout << "Sent: " << ret << " byte(s)\n";
    stringstream resp;
    resp << ret;
    ack(data_vector[1], resp.str());
  }
}

// Pass the received data from the content server to the device
void Proxy1150::recv(string &data)
{
  vector<string> data_vector;
  split(data, data_vector, 5);

  int buffer_size = atoi(data_vector[4].c_str());
  char buffer[4092] = "";

  int ret;
  ret = ::recv(_socket, buffer, buffer_size, 0);
  if (ret < 0) {
    cerr << "Error: Recv failed (errno = " << errno << ")\n";
    nak(data_vector[1], "0");
  } else {
    string buf_str;
    buf_str.assign(buffer, ret);

    cout << "Recv: " << ret << " byte(s)\n";
    ack(data_vector[1], buf_str);
  }
}

// Close the socket
void Proxy1150::close(std::string &data)
{
  vector<string> data_vector;
  split(data, data_vector, 4);

  ::close(_socket);

  cout << "socket closed\n";

  ack(data_vector[1]);
}

